'''
Defines a L{Perk} for the gnome-terminal to correct caret event handling.

@todo: PP: does not handle the case where lots of output is immediately 
  followed by the appearance of the prompt again

@author: Peter Parente
@author: Brett Clippingdale
@author: Eitan Isaacson
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2007 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
import Task, Perk
import AEConstants
from POR import POR
from i18n import _
from copy import copy

__uie__ = dict(kind='perk', tier='gnome-terminal')

class _CaretEventSeq(object):
  '''
  Used to store and re-order caret event sequences such that the default 
  L{Perk}s can process them like normal.

  @cvar sequences: A dictionary with correct sequences for several caret 
    operations
  @type sequences: dictionary
  @cvar actions: A mapping of possible 'added' values with strings
  @type actions: dictionary

  @ivar caret_events: A dictionary of the latest caret events. The key is the 
  name of the action. The value is a tuple of task arguments, 
  and last key pressed.
  @type caret_events: dictionary
  @ivar last_text: Last text recorded. Used to fix the backspace and 
  delete operations.
  @type last_text: string
  '''
  sequences = {'move': ['move'],
               'insert': ['insert', 'move'],
               'backspace': ['delete', 'move'],
               'delete': ['delete']}

  actions = {None: 'move', 
             False: 'delete', 
             True: 'insert'}

  def __init__(self):
    self.caret_events = {}
    self.last_text = ''

  def reset(self):
    '''
    Reset the sequence object to zero sequences. And record the
    last text recieved.
    '''
 
    if 'move' in self.caret_events:
      # The last move op usually has the correct text.
      self.last_text = self.caret_events['move'][0]['text']
    elif 'delete' in self.caret_events:
      # If no move op, we assume that one char was deleted.
      offset = self.caret_events['delete'][0]['por'].char_offset
      try:
        self.last_text = '%s%s' % (self.last_text[:offset],
                                   self.last_text[offset+1:])
      except IndexError:
        pass
    self.caret_events = {}

  def add(self, kwargs, last_key):
    '''
    Add a caret event to the sequence.

    @param kwargs: Task arguments.
    @type kwargs: dictionary.
    @param last_key: Last key pressed.
    @type last_key: string.
    '''
    action = self.actions[kwargs['added']]
    self.caret_events[action] = (kwargs, last_key)

  def operationType(self):
    '''
    Determine what kind of operation the sequence is.

    @return: Operation name.
    @rtype: string
    '''
    keys_pressed = [c[1] for c in self.caret_events.values()]
    if 'space' in keys_pressed:
      return 'insert'
    elif len(self.caret_events) == 1 and 'move' in self.caret_events:
      return 'move'
    elif 'BackSpace' in keys_pressed:
      return 'backspace'
    elif 'Delete' in keys_pressed:
      return 'delete'
    elif 'insert' and 'move' in self.caret_events:
      return 'insert'
    # better that an exception ?
    else:
      return 'insert'
    

  def sequenceReorderIter(self, op_name):
    '''
    Iterates over stored events in the correct order for the given operation,

    @param op_name: Type of operation.
    @type op_name: string

    @return: Task arguments.
    @rtype: dictionary
    '''
    if 'delete' in self.caret_events:
      self._textDeleted(self.caret_events['delete'][0])
    for op in self.sequences[op_name]:
      task_data = self.caret_events.get(op)
      if task_data:
        yield task_data[0]
        
  def _textDeleted(self, kwargs):
    '''
    Fixes 'text' argument in delete events.

    @param kwargs: Task arguments.
    @type kwargs: dictionary
    '''
    por = kwargs['por']
    offset = por.char_offset
    try:
      text = self.last_text[offset:offset+1]
    except IndexError:
      text = ''
    kwargs['text'] = text

class GTerminalPerk(Perk.Perk):
  '''
  Corrects caret problems in in gnome-terminal.
  
  @ivar caret_ev_seq: Stores a sequence of events in a 20ms space.
  @type caret_ev_seq: L{_CaretEventSeq}
  '''
  def init(self):
    '''
    Registers a L{HandleCaretChange} task. Initializes an instance variable to
    store the previous caret L{POR}.
    '''
    self.caret_ev_seq = _CaretEventSeq()
    self.registerTask(HandleCaretChange('gterm caret'))
    self.ret_key_occured = False
    
  def getName(self):
    return _('GNOME terminal')

class HandleCaretChange(Task.CaretTask):
  '''
  Task that handles a L{AEEvent.CaretChange} in the terminal control by 
  storing events and later dispatching them on a timer in proper order.
  '''
  def execute(self, **kwargs):
    '''
    Stores events, and sets a timer to report them later
    @todo: SH fix copy mess in 'space' insertion logic
    '''
    if not self.hasAccRole('terminal') or None in self.getLastKey():
      # do normal processing for caret events outside the terminal control or
      # when there is no last key press
      return
    
    added = kwargs['added'] 
    # if move or delete
    if added is None or (added is False and self.getLastKey()[1] == 'Delete'):
      self.doTask('gterm move timer')
      self.registerTask(MoveTimer('gterm move timer', 20))
      self.perk.caret_ev_seq.reset()
    self.perk.caret_ev_seq.add(kwargs, self.getLastKey()[1])
    
    # An insertion event is not triggered from at-spi for a space after a 
    # backspace has occurred so we will manufacture one.
    if self.getLastKey()[1] == 'space' and kwargs['added'] == None:
      new_kwargs = {}
      for key, value in kwargs.iteritems():
        new_kwargs[key] = value
      new_kwargs['added'] = True
      new_kwargs['text'] = kwargs['text'][-2]
      newpor = POR()
      newpor.accessible = kwargs['por'].accessible
      newpor.item_offset = kwargs['por'].item_offset
      newpor.char_offset = kwargs['por'].char_offset-1
      new_kwargs['por'] = newpor
      self.perk.caret_ev_seq.add(new_kwargs, self.getLastKey()[1])
      
    # cause long output to be interrupted by next user input
    if self.getLastKey()[1] == 'Return':
      self.perk.ret_key_occured = True
    elif self.perk.ret_key_occured == True:
      self.stopNow()
      self.perk.ret_key_occured = False
      
    self.setTempVal('no update', True)
    return False
    
class MoveTimer(Task.TimerTask):
  '''
  Used to translate and forward a sequence of stored caret events to the 
  speech perk.
  '''
  def execute(self, **kwargs):
    op_name = self.perk.caret_ev_seq.operationType()
    self.inhibitMayStop()
    for task_info in self.perk.caret_ev_seq.sequenceReorderIter(op_name):
      self.doTask('read caret', **task_info)
    self.unregisterTask(self)
