'''
Defines a device that uses the Speech Dispatcher audio server via the speechd
package. Speech Dispatcher can be obtained from 
U{http://www.freebsoft.org/speechd}.

This device appears very buggy when using IBM TTS with the version from CVS.
Probably because Speech Dispatcher is still very early in development.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
import time
import speechd
import AEOutput
from i18n import _

__uie__ = dict(kind='device')

class SDSpeechStyle(AEOutput.StyleFlyweight):
  '''
  Overrides the base L{AEOutput.Style.StyleFlyweight} class to provide style
  settings relative to the default style for this device.
  '''
  ADDITIVE = ('Rate', 'Volume', 'Pitch')
  Rate = 0
  Volume = 0
  Pitch = 0
  
  def getSettings(self):
    '''
    Gets configurable relative settings affecting output from this device for
    particular types of information.
    
    @todo: PP: support in a later version
    
    @return: Group of all configurable settings per semantic information type
    @rtype: L{AEState.Setting.Group}
    '''
    raise NotImplementedError

class SDSpeechStyleDefault(AEOutput.AudioStyleDefault):
  '''
  Default style for Speech Dispatcher output.
  '''
  MinRate = -100
  MaxRate = 100
  MinVolume = -100
  MaxVolume = 100
  MinPitch = -100
  MaxPitch = 100
  Rate = 60
  Volume = 180 # about 90%
  Pitch = 0
  Voice = 0
  
  def getSettings(self):
    '''
    Gets configurable absolute settings affecting all output from this device.
    
    @return: Group of all configurable settings
    @rtype: L{AEState.Setting.Group}
    '''
    g = self._newGroup()
    a = g.newGroup(_('Speech'))
    a.newRange('Pitch', _('Pitch'), self.MinPitch, self.MaxPitch, 0,
               _('Baseline voice pitch'))
    a.newRange('Rate', _('Rate'), self.MinRate, self.MaxRate, 0,
                _('Speech rate'))
    a.newPercent('Volume', _('Volume'), self.MinVolume, self.MaxVolume, 0,
                 _('Speech volume as a percentage'))
    # generate a group for standard word parsing settings
    self._newWordGroup(g)
    return g

class SpeechDispatcher(AEOutput.Audio):
  '''
  Speech Dispatcher audio server.
  
  @ivar last_style: Last style object to be applied to output
  @type last_style: L{AEOutput.Style}
  @ivar speaker: Speaker that will synthesize speech from buffered text
  @type speaker: speechd.Client
  @ivar buffer: Buffer of text and styles to be sent to the device
  @type buffer: list
  @ivar voices: Constants for voices supported by the current speechd server
    engine
  @type voices: list
  @cvar ALL_VOICES: Constants representing all voices supported by speechd
  @type ALL_VOICES: list of string
  ''' 
  USE_THREAD = False
  ALL_VOICES = ['MALE1', 'FEMALE1', 'CHILD_MALE', 'MALE2', 'FEMALE2', 'MALE3', 
                'FEMALE3', 'CHILD_FEMALE', 'MALE4', 'FEMALE4']
  
  def _applyStyle(self, style):
    '''
    Applies a given style to this output device. All output following this 
    call will be presented in the given style until this method is called again
    with a different style.

    @param style: Style to apply to future output
    @type style: L{AEOutput.Style}
    '''
    self.speaker.set_voice(self.voices[style.Voice])
    self.speaker.set_rate(min(max(style.Rate, style.MinRate), style.MaxRate))
    self.speaker.set_pitch(min(max(style.Pitch,style.MinPitch),style.MaxPitch))
    self.speaker.set_volume(min(max(style.Volume, style.MinVolume), 
                                style.MaxVolume))

  def init(self):
    '''
    Initializes the speech driver through gnome-speech.

    @raise AEOutput.InitError: When the device can not be initialized
    '''
    try:
      self.speaker = speechd.Client('LSR')
    except:
      raise AEOutput.InitError
    self.default_style = SDSpeechStyleDefault()
    self.last_style = None
    self.buffer = []
    self.voices = []

  def _say(self, text):
    '''
    Encodes the text as utf-8 and speaks it. Filters out empty strings.

    @param text: Text to say
    @type text: string
    '''
    if not text:
      return
    try:
      text = text.encode('utf-8', 'replace')
      self.speaker.say(text)
    except (TypeError, UnicodeDecodeError), e:
      pass

  def close(self):
    '''
    Stops and closes the speech device.
    '''
    if self.speaker is not None:
      self.speaker.cancel()
    self.speaker.close()

  def sendStop(self, style=None):
    '''
    Stops speech immediately.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    if self.speaker is not None:
      self.speaker.cancel()
    self.buffer = []

  def sendTalk(self, style=None):
    '''
    Begins synthesis of the buffered text.
  
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    stream = []
    for text, style in self.buffer:
      if style.isDirty() or style != self.last_style:
        self._applyStyle(style)
        self.last_style = style
        self._say(' '.join(stream))
        stream = []
      if text is not None:
        stream.append(text)
    if stream:
      self._say(' '.join(stream))
    self.buffer = []

  def sendString(self, text, style):
    '''
    Adds the given string to the text to be synthesized. The string is buffered
    locally in this object since gnome-speech does not support buffering in
    the speech engine driver, even if the engine does support it.

    @param text: String to buffer on the device
    @type text: string
    @param style: Style with which this string should be output; None means no
      style change should be applied
    @type style: integer
    '''
    self.buffer.append((text, style.copy()))

  def isActive(self):
    '''
    Always returns False since Speech Dispatcher currently does not report its
    activity.

    @return: True when the speech device is synthesizing, False otherwise
    @rtype: boolean
    '''
    return False

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.

    @return: The localized name for the device
    @rtype: string
    '''
    return _('Speech Dispatcher')
   
  def sendIndex(self, style):
    '''
    Sends an index marker to the device driver. The driver should notify the
    device when the marker has been reached during output.
    
    @param style: Style indicating channel in which the marker will be appended
    @type style: L{AEOutput.Style}
    @raise NotImplementedError: Until Speech Dispatcher supports index markers
    '''
    raise NotImplementedError

  def createDistinctStyles(self, num_groups, num_layers):
    '''
    Creates distinct styles following the guidelines set forth in the base
    class. Only distinguishes groups in terms of voice, not layers.
    
    @param num_groups: Number of sematic groups the requestor would like to
      represent using distinct styles
    @type num_groups: integer
    @param num_layers: Number of content origins (e.g. output originating from
      a background task versus the focus) the requestor would like to represent
      using distinct styles
    @type num_layers: integer   
    @return: Styles
    @rtype: list of L{AEOutput.Style}
    '''
    styles = []
    # compute which voices are supported
    for name in self.ALL_VOICES:
      try:
        self.speaker.set_voice(name)
        self.voices.append(name)
      except AssertionError:
        pass
    # bounds on the number of voices
    self.default_style.MaxVoice = len(self.voices)-1
    self.default_style.MinVoice = 0
    for i in range(0, num_groups*num_layers):
      # compute a valid voice number
      v_num = i % len(self.voices)
      # create a style object
      s = SDSpeechStyle(self.default_style)
      # store the current voice number
      s.Voice = v_num
      # store relative values, but set them to zero to make them equivalent
      # to the default for now
      s.Rate = 0
      s.Volume = 0
      s.Pitch = 0
      styles.append(s)
    return styles
