'''
Defines an L{AEEvent} indicating a gesture has been performed on an L{AEInput} 
device.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
import AEInput
import Base, Constants

class InputGesture(Base.AccessEngineEvent):
  '''  
  Event that fires when some L{AEInput.Gesture} is detected on an L{AEInput}
  device. Always defaults to the L{Constants.LAYER_FOCUS}.
  
  This class registers its name and whether it should be monitored by default in
  an L{AEMonitor} using the L{Constants.registerEventType} function
  when this module is first imported. The L{AEMonitor} can use this
  information to build its menus.
  
  @ivar gesture: Gesture detected
  @type gesture: L{AEInput.Gesture}
  @ivar timestamp: Time at which the gesture was received on the device
  @type timestamp: float
  '''
  Constants.registerEventType('InputGesture', False)
  def __init__(self, gesture, timestamp, **kwargs):
    '''
    Calls the base class (which sets the event priority) and then stores the 
    L{AEInput.GestureList} that will be used to trigger a L{Task}.
    
    @param gesture: Gestures detected on an L{AEInput} device
    @type gesture: L{AEInput.Gesture}
    @param timestamp: Time at which the gesture was received on the device
    @type timestamp: float
    '''
    Base.AccessEngineEvent.__init__(self, focused=True, **kwargs)
    self.gesture = gesture
    self.timestamp = timestamp
    
  def __str__(self):
    '''
    Returns a human readable representation of this event including its name,
    its gesture codes, and its device.
    
    @return: Information about this event
    @rtype: string
    '''
    name = Base.AccessEngineEvent.__str__(self)
    return '%s:\n\tgesture: %s\n\tdevice: %s' % \
           (name, self.gesture, self.gesture.getDevice().getName())

  def execute(self, tier_manager, **kwargs):
    '''
    Contacts the L{TierManager} and asks it to manage this event as a gesture.
    
    @param tier_manager: TierManager that will handle the event
    @type tier_manager: L{TierManager}
    @param kwargs: Packed references to other managers not of interest here
    @type kwargs: dictionary
    @return: True to indicate the event executed properly
    @rtype: boolean
    '''
    tier_manager.manageGesture(self)
    return True
  
  def getTaskKey(self):
    '''
    Gets the L{AEInput.GestureList} that triggered this event. This 
    information is used to locate the L{Task} that should handle this event.
    
    @return: Gesture seen on an L{AEInput} device
    @rtype: L{AEInput.GestureList}
    '''
    g = self.gesture
    return AEInput.GestureList(g.getDevice(), gestures=[g])
  
  def getTimestamp(self):
    '''
    Gets the timestamp for when the event occurred. This timestamp is useful
    for connecting input events to changes on the desktop. For instance, the
    timestamp for keyboard input is needed when key presses open a new dialog
    on some platforms such that the window manager can activate the dialog 
    once it appears.
    
    @return: Gesture timestamp
    @rtype: float
    '''
    return self.timestamp
  
  def getDataForTask(self):
    '''
    Provides no information to the L{Task} about this event.
    
    @return: Empty dict
    @rtype: dictionary
    '''
    return {}