'''
Defines a gtk dialog L{AEChooser} for configuring LSR scripts, devices, 
profiles, and system settings. Uses a glade XML file to define the shell of the
dialog. Dynamically generates the contents of the script, device, and system
panes internal panes based on the available L{AEState} settings. Generates the
contents of the profile views using information from the L{UIRegistrar}.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import pygtk
pygtk.require('2.0')
import gtk, gobject, atk
import gtk.glade
from pyLinAcc import Atk
import AEChooser, AEState
from i18n import _, DOMAIN
from GTKUIEView import *

__uie__ = dict(kind='chooser')

class WidgetFactory(object):
  '''
  Widget factory for creating gtk views for L{AEState.Setting} objects. Has
  one public method for creating new widgets. All other methods are protected
  for internally handling widget events and updating their correspoding 
  settings.
  '''
  @classmethod
  def _updateBool(cls, widget, setting):
    setting.value = widget.get_active()
    
  @classmethod
  def _updateNumeric(cls, adj, setting):
    setting.value = adj.get_value()
    
  @classmethod
  def _updateFilename(cls, widget, setting):
    fn = widget.get_filename()
    if fn is not None:
      setting.value = fn
    
  @classmethod
  def _updateString(cls, widget, setting):
    setting.value = widget.get_text()

  @classmethod
  def _updateChoice(cls, widget, setting):
    index = widget.get_active()
    setting.value = setting.values[index]
  
  @classmethod
  def create(cls, setting):
    widget = None
    if isinstance(setting, AEState.Group):
      # another group
      widget = gtk.Frame(setting.name)
      widget.set_shadow_type(gtk.SHADOW_ETCHED_IN)
      return widget, None
    elif isinstance(setting, AEState.BoolSetting):
      # boolean
      widget = gtk.CheckButton()
      widget.set_active(setting.value)
      widget.connect('toggled', cls._updateBool, setting)
      # set the accessible name
      Atk.setNameDesc(widget, name=setting.label)
    elif isinstance(setting, AEState.FilenameSetting):
      # filename
      widget = gtk.FileChooserButton(setting.label)
      widget.set_filename(setting.value)
      widget.connect('selection-changed', cls._updateFilename, setting)
    elif isinstance(setting, AEState.StringSetting):
      # string
      widget = gtk.Entry()
      widget.set_text(setting.value)
      # @todo: PP: what signal? changed? appropriate for all updates? what 
      # about intermediate invalid values?
      widget.connect('changed', cls._updateString, setting)
    elif isinstance(setting, AEState.NumericSetting):
      # numeric or range
      if setting.precision == 0:
        step = 1
      else:
        step = 0.1**setting.precision
      page = int((setting.max-setting.min)/10.0)
      adj = gtk.Adjustment(setting.value, setting.min, setting.max, step, page)
      if isinstance(setting, AEState.RangeSetting):
        widget = gtk.HScale(adj)
      else:
        widget = gtk.SpinButton(adj)
      widget.set_digits(setting.precision)
      widget.set_value(setting.value)
      adj.connect('value-changed', cls._updateNumeric, setting)
    elif isinstance(setting, AEState.EnumSetting):
      # enum
      widget = gtk.combo_box_new_text()
      map(widget.append_text, setting.labels)
      widget.set_active(setting.values.index(setting.value))
      widget.connect('changed', cls._updateChoice, setting)
    elif isinstance(setting, AEState.ChoiceSetting):
      # choice
      widget = gtk.combo_box_new_text()
      map(widget.append_text, setting.values)
      widget.set_active(setting.values.index(setting.value))
      widget.connect('changed', cls._updateChoice, setting)
    else:
      raise NotImplementedError
    # create a label
    label = gtk.Label(setting.label)
    label.set_alignment(1.0, 0.5)   
    return widget, label

class SettingsChooser(AEChooser.AEChooser): 
  '''
  LSR settings dialog for configuring Perks, devices, profiles, and other 
  system settings.
  
  @ivar dialog: Dialog widget for configuring LSR settings
  @type dialog: gtk.Dialog
  @ivar system: LSR system wide settings
  @type system: L{AEState.Setting.Group}
  @ivar perks: Per L{Perk} settings keyed by Perk name
  @type perks: dictionary of string : L{AEState.Setting.Group}
  @ivar installed: All UIE class names, human readable names, and 
  descriptions installed keyed by their type
  @type installed: dictionary of string : 3-tuple of string
  @ivar associated: All UIE class names associated with this profile
  @type associated: list of string
  @ivar section_nb: Main tabbed panel
  @type section_nb: gtk.Notebook
  @ivar perk_tv: List of configurable L{Perk}s
  @type perk_tv: gtk.TreeView
  @ivar device_tv: List of configurable L{AEInput} and L{AEOutput} devices
  @type device_tv: gtk.TreeView
  @ivar system_vp: Panel for system settings
  @type system_vp: gtk.Viewport
  @ivar profile_perks: Aids selection of L{Perk}s to associate with the profile
  @type profile_perks: L{GTKUIEView.UIEView}
  @ivar profile_monitors: Aids selection of L{AEMonitor}s to associate with the
    profile
  @type profile_monitors: L{GTKUIEView.UIEView}
  @ivar profile_choosers: View of L{AEChooser}s installed
  @type profile_choosers: L{GTKUIEView.UIEView}
  @ivar profile_devices: Aids selection of preferred L{AEOutput} and L{AEInput}
    devices associated with this profile
  @type profile_devices: L{GTKUIEView.UIEView}
  @cvar FRAME_MARKUP: Pango markup for frame panels to make text larger
  @type FRAME_MARKUP: string
  @cvar PERK: Tab number for configuring Perks
  @type PERK: integer
  @cvar DEVICE: Tab number for configuring devices
  @type DEVICE: integer
  @cvar SYSTEM: Tab number for configuring system settings
  @type SYSTEM: integer
  @cvar PROFILE: Tab number for configuring the profile
  @type PROFILE: integer
  @cvar RAISE: Raise priority of a L{UIElement} for handling events or getting 
    loaded
  @type RAISE: integer
  @cvar LOWER: Lower priority of a L{UIElement} for handling events or getting 
    loaded
  @type LOWER: integer
  @cvar TO_LOAD: Indicates a L{UIElement} is now set to be loaded
  @type TO_LOAD: integer
  @cvar TO_UNLOAD: Indicates a L{UIElement} is now set to be unloaded
  @type TO_UNLOAD: integer
  ''' 
  SINGLETON = True
  PERK = 0
  DEVICE = 1
  SYSTEM = 2
  PROFILE = 3
  FRAME_MARKUP = '<span size="xx-large" weight="bold">%s</span>'
  
  # signal constants pulled from GTKUIEView
  RAISE = RAISE
  LOWER = LOWER
  TO_LOAD = TO_LOAD
  TO_UNLOAD = TO_UNLOAD
  
  def init(self, system, perks, installed, associated, profile, devices, 
           timestamp, **kwargs):
    '''
    Creates and shows the settings dialog and its components

    @param system: LSR system wide settings
    @type system: L{AEState.Setting.Group}
    @param perks: Per L{Perk} settings keyed by Perk name
    @type perks: dictionary of string : L{AEState.Setting.Group}
    @param devices: Per L{AEOutput}/L{AEInput} settings keyed by device name
    @type devices: dictionary of string : L{AEState.Setting.Group}
    @param installed: All UIE class names, human readable names, and 
      descriptions installed keyed by their type
    @type installed: dictionary of string : list of 3-tuple of string
    @param associated: All UIE class names, human readable names, and 
      descriptions associated with this profile keyed by their type
    @type associated: dictionary of string : list of 3-tuple of string
    @param timestamp: Time at which input was given indicating the start of
      this chooser
    @type timestamp: float
    @param profile: Name of the profile
    @type profile: string
    '''
    # store group objects
    self.system = system
    self.perks = perks
    self.devices = devices
    
    # load the glade file
    gtk.glade.set_custom_handler(self._createCustomWidget)
    source = gtk.glade.XML(self._getResource('lsr_settings.glade'), 
                           'main dialog', DOMAIN)

    # get the dialog widget
    self.dialog = source.get_widget('main dialog')
    try:
      # try to bring the window to the foreground if this feature is supported
      self.dialog.window.set_user_time(timestamp)
    except AttributeError:
      pass
    
    # get widget references
    self.tips = gtk.Tooltips()
    self.section_nb = source.get_widget('section notebook')
    self.perk_tv = source.get_widget('perk treeview')
    perk_label = source.get_widget('perk label')
    perk_vp = source.get_widget('perk viewport')
    self.device_tv = source.get_widget('device treeview')
    device_label = source.get_widget('device label')
    device_vp = source.get_widget('device viewport')
    self.system_vp = source.get_widget('system viewport')
    # put the name of the profile on the profile tab
    pl = source.get_widget('profile label')
    pl.set_text(pl.get_text() % profile.title())
   
    # create widget models
    self.perk_tv.set_model(gtk.ListStore(gobject.TYPE_STRING))
    self.perk_tv.set_data('subsection label', perk_label)
    self.perk_tv.set_data('subsection viewport', perk_vp)
    # @todo: PP: device might be a tree depending on where semantics are listed
    # in the future
    self.device_tv.set_model(gtk.ListStore(gobject.TYPE_STRING))
    self.device_tv.set_data('subsection label', device_label)
    self.device_tv.set_data('subsection viewport', device_vp)
    
    # connect all signal handlers
    source.signal_autoconnect(self)
    
    # populate all sections
    self._createPerkSettingsView()
    self._createDeviceSettingsView()
    self._createProfileView(associated, installed)
    self._createSystemSettingsView()
    
  def _createCustomWidget(self, glade, function_name, widget_name, *args):
    '''
    Creates a custom widget. Invoked during processing by glade.
    
    @param glade: glade XML parser
    @type glade: gtk.glade.XML
    @param widget_name: Name of the widget to create
    @type widget_name: string
    @param function_name: Name of the function to call to create this widget
    @type function_name: string
    @return: Custom widget
    @rtype: gtk.Widget
    '''
    return getattr(self, function_name)()
  
  def _createProfilePerkWidget(self):
    '''
    Creates a L{GTKUIEView.UIEView} instance to manage selection of 
    L{Perk}s.
    
    @return: Root widget of the UIE view
    @rtype: gtk.Widget
    '''
    self.profile_perks = UIEView(self, ordered=False)
    return self.profile_perks.getWidget()
    
  def _createProfileMonitorWidget(self):
    '''
    Creates a L{GTKUIEView.UIEView} instance to manage selection of 
    L{AEMonitor}s.
    
    @return: Root widget of the UIE view
    @rtype: gtk.Widget
    '''
    self.profile_monitors = UIEView(self, ordered=False)
    return self.profile_monitors.getWidget()
    
  def _createProfileChooserWidget(self):
    '''
    Creates a L{GTKUIEView.UIEView} instance to manage selection of 
    L{AEChooser}s.
    
    @return: Root widget of the UIE view
    @rtype: gtk.Widget
    '''
    self.profile_choosers = UIEView(self, ordered=False, 
                                           activatable=False)
    return self.profile_choosers.getWidget()
  
  def _createProfileDevicesWidget(self):
    '''
    Creates a L{GTKUIEView.UIEView} instance to manage selection of 
    L{AEInput} and L{AEOutput} devices.
    
    @return: Root widget of the UIE view
    @rtype: gtk.Widget
    '''
    self.profile_devices = UIEView(self, ordered=True)
    return self.profile_devices.getWidget()

  def _createSystemSettingsView(self):
    '''
    Populates the system settings profile panel.
    '''
    layout = self._populateSection(self.system, self.system_vp)
    self.system_vp.add(layout)
    self.system_vp.show_all()
    
  def _createProfileView(self, associated, installed):
    '''
    Populates all of the profile view tab panels.
    '''
    order = (('perks', self.profile_perks), 
             ('devices', self.profile_devices),
             ('monitors', self.profile_monitors),
             ('choosers', self.profile_choosers))
    for name, view in order:
      loaded = associated[name]
      unloaded = set(installed[name])-set(loaded)
      view.setData(loaded, unloaded)
    
  def _createPerkSettingsView(self):
    '''
    Populates the list of configurable L{Perk}s and selects the first to 
    generate its settings panel.
    '''
    # reset the view completely
    model = self.perk_tv.get_model()
    model.clear()
    cols = self.perk_tv.get_columns()
    map(self.perk_tv.remove_column, cols)
    
    # configure the perk list view
    crt = gtk.CellRendererText()
    tvc = gtk.TreeViewColumn('', crt, text=0)
    self.perk_tv.append_column(tvc)
    
    # populate the list of perks with settings
    if len(self.perks) > 0:
      for perk in self.perks.keys():
        model.append([perk])
      # set the first one as selected to cause an update of the panel
      self.perk_tv.set_cursor((0,))
      
  def _createDeviceSettingsView(self):
    '''
    Populates the list of configurable L{AEInput} and L{AEOutput} devices and 
    selects the first to generate its settings panel.
    '''
    # reset the view completely
    model = self.device_tv.get_model()
    model.clear()
    cols = self.device_tv.get_columns()
    map(self.device_tv.remove_column, cols)
    
    # configure the perk list view
    crt = gtk.CellRendererText()
    tvc = gtk.TreeViewColumn('', crt, text=0)
    self.device_tv.append_column(tvc)
    
    # populate the list of perks with settings
    if len(self.devices) > 0:
      for device in self.devices.keys():
        model.append([device])
      # set the first one as selected to cause an update of the panel
      self.device_tv.set_cursor((0,))

  def _onScrollToFocus(self, widget, direction, viewport):
    '''
    Scrolls a focused widget in a settings panel into view.
    
    @param widget: Widget that has the focus
    @type widget: gtk.Widget
    @param direction: Direction constant, ignored
    @type direction: integer
    @param viewport: Viewport to scroll to bring the widget into view
    @type viewport: gtk.Viewport
    '''
    x, y = widget.translate_coordinates(viewport, 0, 0)
    w, h = widget.window.get_geometry()[2:4]
    vw, vh = viewport.window.get_geometry()[2:4]
    adj = viewport.get_vadjustment()
    if y+40 > vh:
      adj.value += (y+40) - vh - 3
    elif y < 0:
      adj.value = max(adj.value + y, adj.lower)
    
  def _onSubSectionChange(self, tv):
    '''
    Prepares a settings panel for viewing when the name of a L{Perk} or 
    L{AEInput}/L{AEOutput} device is selected from the list of configurable
    elements.
    
    @param tv: Treeview in which a configurable item is selected
    @type tv: gtk.TreeView
    '''
    # get model data
    model = tv.get_model()
    label = tv.get_data('subsection label')
    viewport = tv.get_data('subsection viewport')
    # get selected category info
    path, col = tv.get_cursor()
    name = model[path][0]
    
    # throw away the previous contents
    child = viewport.get_child()
    if child:
      viewport.remove(child)
      
    # set the label on the section frame
    label.set_markup(self.FRAME_MARKUP % name)
    # populate the subsection
    if self.perk_tv == tv:
      layout = self._populateSection(self.perks[name], viewport)
      viewport.add(layout)
    elif self.device_tv == tv:
      layout = self._populateSection(self.devices[name], viewport)
      viewport.add(layout)
    viewport.show_all()

  def _populateSection(self, group, viewport):
    '''
    Populate a section pane with widgets representing the L{AEState.Setting}s 
    in the group. The container is the top level widget in which other widgets
    should be made. If recursive groups exist, their settings will be properly
    represented by widgets in a widget sub-container.

    @param group: Group of settings linked to actual settings values
    @type group: L{AEState.Setting.Group}
    '''
    n = len(group)
    # use a n by 2 table to order widgets nicely
    table = gtk.Table(n, 2)
    table.set_row_spacings(5)
    table.set_col_spacings(5)
    # nit: add in reverse order else controls show up backwards in acc. tree
    group = list(group)
    group.reverse()
    for row, setting in enumerate(group):
      widget, label = WidgetFactory.create(setting)
      if label is None:
        # if the widget is another group, recurse
        layout = self._populateSection(setting, viewport)
        layout.set_border_width(5)
        # add the sublayout to the group widget
        widget.add(layout)
        # attach the group widget to the table across both columns
        # nit: add widget on the right first then label on the left, else gail
        # screws up the order in the acc. hierarchy
        table.attach(widget, 0, 2, n-row-1, n-row, yoptions=gtk.FILL, 
                     xpadding=5)
      else:
        if setting.description:
          # set description as tooltip and accessible description
          self.tips.set_tip(widget, setting.description)
          Atk.setNameDesc(widget, description=setting.description)
        # relate the label to the widget
        Atk.setRelation(widget, atk.RELATION_LABELLED_BY, label)
        Atk.setRelation(label, atk.RELATION_LABEL_FOR, widget)
        # attach to the table
        # nit: add widget on the right first then label on the left, else gail
        # screws up the order in the acc. hierarchy
        table.attach(widget, 1, 2, n-row-1, n-row, yoptions=gtk.FILL)
        table.attach(label, 0, 1, n-row-1, n-row, gtk.FILL, gtk.FILL, 5, 0)
        # watch for focus on the widget so we can scroll
        widget.connect('focus', self._onScrollToFocus, viewport)
    return table
  
  def _getAssociated(self):
    '''
    Gets lists of L{UIElement} class names currently set to be associated with
    the active profile.
    
    @return: Dictionary of associated UIEs keyed by their kind
    @rtype: dictionary of string : list of string
    '''
    d = {}
    if self.profile_perks.isDirty():
      d['perks'] = self.profile_perks.getCurrentUIEs()[0]
    if self.profile_monitors.isDirty():
      d['monitors'] = self.profile_monitors.getCurrentUIEs()[0]
    if self.profile_devices.isDirty():
      d['devices'] = self.profile_devices.getCurrentUIEs()[0]
    return d

  def _onOK(self, widget):
    '''
    Closes the dialog and sends a signal indicating the OK action.
    
    @param widget: Source of GUI event
    @type widget: gtk.Widget
    '''
    self._signal(self.OK, system=self.system, perks=self.perks, 
                 devices=self.devices, associated=self._getAssociated())
    self.close()

  def _onCancel(self, widget):
    '''
    Closes the dialog and sends a signal indicating the Cancel action.
    
    @param widget: Source of GUI event
    @type widget: gtk.Widget
    '''
    self._signal(self.CANCEL, system=self.system, perks=self.perks,
                 devices=self.devices)
    self.close()

  def _onApply(self, widget):
    '''
    Sends a signal indicating the Apply action.
    
    @param widget: Source of GUI event
    @type widget: gtk.Widget
    '''
    # get all associated
    self._signal(self.APPLY, system=self.system, perks=self.perks,
                 devices=self.devices, associated=self._getAssociated())

  def close(self):
    '''
    Closes the chooser, preventing further chooser interaction with the user.
    '''
    self.tips = None
    gtk.glade.set_custom_handler(lambda x: None)
    self.dialog.destroy()

  def getName(self):
    '''
    Gets the name of the chooser.    
    
    @return: Human readable name of the chooser
    @rtype: string
    '''
    return _('LSR Settings')

  def updateDevices(self, devices):
    '''
    Updates the list of devices according to those that are now loaded.
    
    @param devices: Per L{AEOutput}/L{AEInput} settings keyed by device name
    @type devices: dictionary of string : L{AEState.Setting.Group}
    '''
    self.devices = devices
    self._createDeviceSettingsView()