'''
Defines a class representing a configurable setting in a L{AEState} object.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import os.path

class Group(list):
  '''  
  Named collection of related L{Setting}s. Provides the L{save} and L{restore}
  methods which act on all members in this group.
  
  @ivar context: Object whose properties are contained within this group
  @type context: L{AEState.AEState}
  @ivar name: Name of the group
  @type name: string
  '''
  def __init__(self, context, name=None):
    '''
    Stores the L{Group} context and its name.
    
    @param context: Object whose properties are contained within this group
    @type context: L{AEState.AEState}
    @param name: Name of the group
    @type name: string
    '''
    self.context = context    
    self.name = name
    
  def __str__(self):
    return 'Group %s: %s' % (self.name, list.__str__(self))
    
  def newGroup(self, name):
    '''
    Adds a subgroup to this group with the given name.
    
    @param name: Name of the new group
    @type name: string
    @return: The newly created group
    @rtype: L{Group}
    '''
    grp = Group(self.context, name)
    self.append(grp)
    return grp

  def newString(self, variable, label, description=''):
    '''
    Adds a new L{StringSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the string
      value
    @type variable: string
    @param label: Label for the new L{Setting}
    @type label: string
    @param description: Extended description of the L{Setting}
    @type description: string
    '''
    self.append(StringSetting(self.context, variable, label, description))
    
  def newFilename(self, variable, label, relative_path, description=''):
    '''
    Adds a new L{FilenameSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the filename
      value
    @type variable: string
    @param label: Label for the new L{Setting}
    @type label: string
    @param relative_path: Absolute path to which '.' or a filename with no 
      path refers
    @type relative_path: string
    @param description: Extended description of the L{Setting}
    @type description: string
    '''
    relative_path = os.path.realpath(relative_path)
    if os.path.isfile(relative_path):
      relative_path = os.path.dirname(relative_path)
    self.append(FilenameSetting(self.context, variable, label, description,
                                relative_path))
  
  def newBool(self, variable, label, description=''):
    '''
    Adds a new L{BoolSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the boolean
      value
    @type variable: string
    @param label: Label for the new L{Setting}
    @type label: string
    @param description: Extended description of the L{Setting}
    @type description: string
    '''
    self.append(BoolSetting(self.context, variable, label, description))
    
  def newNumeric(self, variable, label, min, max, precision, description=''):
    '''
    Adds a new L{NumericSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the numeric
      value
    @type variable: string
    @param label: Label for the new L{Setting}
    @type label: string
    @param min: Minimum value in the range
    @type min: number
    @param max: Maximum value in the range
    @type max: number
    @param precision: Number of decimal places
    @type precision: integer
    @param description: Extended description of the L{Setting}
    @type description: string
    '''
    self.append(NumericSetting(self.context, variable, label, description,
                               min, max, precision))

  def newRange(self, variable, label, min, max, precision, description=''):
    '''
    Adds a new L{RangeSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the numeric
      value
    @type variable: string
    @param label: Label for the new L{Setting}
    @type label: string
    @param min: Minimum value in the range
    @type min: number
    @param max: Maximum value in the range
    @type max: number
    @param precision: Number of decimal places
    @type precision: integer
    @param description: Extended description of the L{Setting}
    @type description: string
    '''
    self.append(RangeSetting(self.context, variable, label, description,
                             min, max, precision))
    
  def newPercent(self, variable, label, min, max, precision, description=''):
    '''
    Adds a new L{PercentRangeSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the numeric
      value
    @type variable: string
    @param label: Label for the new L{Setting}
    @type label: string
    @param min: Minimum value in the range to be shown as 0%
    @type min: number
    @param max: Maximum value in the range to be shown as 100%
    @type max: number
    @param precision: Number of decimal places
    @type precision: integer
    @param description: Extended description of the L{Setting}
    @type description: string
    '''
    self.append(PercentRangeSetting(self.context, variable, label, description,
                                    min, max, precision))
  
  def newChoice(self, variable, label, choices, description=''):
    '''
    Adds a new L{ChoiceSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the value
    @type variable: string
    @param label: Label for the new L{Setting}
    @type label: string
    @param choices: Collection of choices
    @type choices: list
    @param description: Extended description of the L{Setting}
    @type description: string
    '''
    self.append(ChoiceSetting(self.context, variable, label, description,
                              choices))
    
  def newEnum(self, variable, label, choices, description=''):
    '''
    Adds a new L{EnumSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the value
    @type variable: string
    @param label: Label for the new L{Setting}
    @type label: string
    @param choices: Collection of choices mapped to the values they represent
    @type choices: dictionary
    @param description: Extended description of the L{Setting}
    @type description: string
    '''
    self.append(EnumSetting(self.context, variable, label, description,
                            choices))

  def save(self):
    '''Invokes L{Setting.save} on all settings in the group.'''
    for s in self: s.save()

  def restore(self):
    '''Invokes L{Setting.restore} on all settings in the group.'''
    for s in self: s.restore()

class Setting(object):
  '''
  A proxy for a value in a L{AEState.AEState} object. Provides a description
  of the value. Defines methods for caching the current value and restoring it
  later.
  
  @ivar context: Object in which the variable containing the setting resides
  @type context: L{AEState.AEState}
  @ivar variable: Name of the variable to proxy
  @type variable: string
  @ivar label: Label of the variable
  @type label: string
  @ivar description: Extended description of the variable
  @type description: string
  @ivar cached: Cached value from the last L{save} invocation
  @type cached: object
  '''
  def __init__(self, context, variable, label, description):
    '''
    Initializes the instance variables.
    
    @param context: Object in which the variable containing the setting resides
    @type context: L{AEState.AEState}
    @param variable: Name of the variable to proxy
    @type variable: string
    @param label: Label of the variable
    @type label: string
    @param description: Extended description of the variable
    @type description: string
    '''
    self.context = context
    self.variable = variable
    self.label = label
    self.description = description
    self.cached = None
    
  def getValue(self):
    '''
    @return: Current value of the proxied variable
    @rtype: object
    '''
    return getattr(self.context, self.variable)
    
  def setValue(self, val):
    '''
    @param val: New value to store in the proxied variable
    @type val: object
    '''
    setattr(self.context, self.variable, val)
    
  value = property(getValue, setValue)
    
  def save(self):
    '''Saves the current variable value.'''
    self.cached = self.value
    
  def restore(self):
    '''Restores the previously saved variable value.'''
    if self.cached is not None:
      self.value = self.cached
    self.cached = None

class BoolSetting(Setting):
  '''Represents a boolean setting.'''
  def setValue(self, val):
    Setting.setValue(self, bool(val))
    
  value = property(Setting.getValue, setValue)

class StringSetting(Setting):
  '''Represents a string setting.'''
  def setValue(self, val):
    Setting.setValue(self, str(val))
    
  value = property(Setting.getValue, setValue)
  
class FilenameSetting(StringSetting):
  '''Represents a string filename setting.'''
  def __init__(self, context, variable, label, description, relative_path):
    Setting.__init__(self, context, variable, label, description)
    self.relative_path = relative_path
    
  def getValue(self):
    val = Setting.getValue(self)
    if not os.path.isfile(val):
      val = os.path.join(self.relative_path, os.path.basename(val))
    return val
  
  def setValue(self, val):
    if os.path.dirname(val) == self.relative_path:
      new = os.path.basename(val)
    StringSetting.setValue(self, val)

  value = property(getValue, setValue)

class NumericSetting(Setting):
  '''
  Represents a numeric setting where the bounds are arbitrary.
  
  @ivar min: Minimum value in the range.
  @type min: number
  @ivar max: Maximum value in the range
  @type max: number
  @ivar precision: Number of decimal places
  @type precision: integer
  '''
  def __init__(self, context, variable, label, description, min, max, 
               precision):
    Setting.__init__(self, context, variable, label, description)
    self.min = min
    self.max = max
    self.precision = precision
  
  def setValue(self, val):
    if val < self.min or val > self.max:
      raise ValueError
    if self.precision == 0:
      val = int(val)
    else:
      val = round(val, self.precision)
    Setting.setValue(self, val)
    
  value = property(Setting.getValue, setValue)

class RangeSetting(NumericSetting):
  '''
  Represents a numeric setting where the bounds are critical.
  '''
  pass
  
class PercentRangeSetting(RangeSetting):
  '''
  Represents a numeric setting that should be scaled for presentation to a 
  user as a percentage between 0% and 100%.
  
  @ivar true_min: Minimum value in the true range
  @type true_min: number
  @ivar true_max: Maximum value in the range
  @type true_max: number
  @ivar true_precision: Number of decimal places of a value in the true range
  @type true_precision: integer
  @ivar true_range: Size of the true range
  @type true_range: number
  '''
  def __init__(self, context, variable, label, description, min, max, 
               precision):
    RangeSetting.__init__(self, context, variable, label, description, 0, 100, 
                          0)
    self.true_min = min
    self.true_max = max
    self.true_precision = precision
    self.true_range = max - min
    
  def setValue(self, val):
    if val < self.min or val > self.max:
      raise ValueError
    val = (val/100.0 * self.true_range) + self.true_min
    if self.precision == 0:
      val = int(val)
    else:
      val = round(val, self.true_precision)
    Setting.setValue(self, val)
    
  def getValue(self):
    val = RangeSetting.getValue(self)
    return int(val/float(self.true_range)*100)
    
  value = property(getValue, setValue)

class ChoiceSetting(Setting):
  '''
  Represents a one of many choice setting.
  
  @ivar values: Collection of choice values
  @type values: list
  '''
  def __init__(self, context, variable, label, description, choices):
    Setting.__init__(self, context, variable, label, description)
    self.values = choices
    
  def setValue(self, val):
    if val not in self.values:
      raise ValueError
    Setting.setValue(self, val)
    
  value = property(Setting.getValue, setValue)
  
class EnumSetting(ChoiceSetting):
  '''
  Represents a one of many choice setting where text describes a non-text 
  value.
  
  @ivar labels: Collection of choice labels
  @type labels: list
  '''
  def __init__(self, context, variable, label, description, choices):
    # sort alphabetically
    self.labels = choices.keys()
    self.labels.sort()
    ChoiceSetting.__init__(self, context, variable, label, description, 
                           [choices[key] for key in self.labels])