'''
Defines L{Tools} for doing output.

@author: Peter Parente
@author: Pete Brunet
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import os
import LSRConstants
import Base, View
from i18n import _
from Error import *

class Output(Base.TaskTools):
  '''
  Provides methods for doing output on any L{AEOutput} device.
  
  @todo: PP: document or refactor font attributes

  @cvar may_stop: Count of the number of calls to L{inhibitMayStop}. When 
    greater than zero, the next call to L{mayStop} will decrement the count.
    When zero, L{mayStop} will send a stop to the output device.
  @type may_stop: integer
  ''' 
  may_stop = 0
  # Constants used for getting text attributes
  # attribute and value must observe i8n, need to be able to translate both
  FONT_SIZE = 'size'
  FONT_COLOR = 'fg-color'
  FONT_WEIGHT = 'weight'
  FONT_STYLE = 'style'
  FONT_UNDERLINE = 'underline'
  FONT_STYLES = {'italic':_('italic'), 'bold':_('bold'), 'single':_('single')}

  FONT_ATTRS = {FONT_SIZE:_('size'), FONT_COLOR:_('color'), 
                FONT_WEIGHT:_('weight'), FONT_STYLE:_('style'), 
                FONT_UNDERLINE:('underline')}
  FONT_NON_NUMERIC = [FONT_COLOR, FONT_STYLE, FONT_UNDERLINE]
  
  def getOutputDevice(self, name):
    '''
    Gets an L{AEOutput} device from the L{DeviceManager} given its name.
    
    @param name: UIE name of the L{AEOutput} device to get
    @type name: string
    @return: The named output device
    @rtype: L{AEOutput}
    @raise InvalidDeviceError: When a L{AEOutput} device with the given name is
      not registered
    '''
    device = self.device_man.getOutputByName(name)
    if device is None:
      # let the system report the device isn't available
      raise InvalidDeviceError
    return device
  
  def msg(self, text, sem=None, layer=None, dev=None):
    '''
    Outputs the text while disregarding settings.

    @param text: Text to output
    @type text: string
    @param sem: The semantic stream on which to send output; defaults to None 
      for the default semantic.
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.LAYER_FOCUS},
      L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    if layer is None:
      layer = self.layer
    self.device_man.sendString(dev, text, sem, layer, True)

  def out(self, text, sem=None, layer=None, dev=None):
    '''
    Outputs the text while disregarding settings and not sending a flush.

    @param text: Text to output
    @type text: string
    @param sem: The semantic stream on which to send output; defaults to None 
      for the default semantic.
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.LAYER_FOCUS},
      L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    if layer is None:
      layer = self.layer
    self.device_man.sendString(dev, text, sem, layer, False)
    
  def stopNow(self, dev=None, sem=None, layer=None):
    '''
    Tells the referenced output device to interrupt current output and clear 
    buffered data.
    
    @param dev: The device to send the stop to; defaults to None for the 
      default output device
    @type dev: L{AEOutput.Base}
    @param sem: The semantic information to stop; defaults to None for the
      default semantic
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.LAYER_FOCUS},
      L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    '''
    Output.may_stop = 0
    if not self.tier_man.getState().Stopping:
      return
    if layer is None:
      layer = self.layer
    self.device_man.sendStop(dev, sem, layer)
    
  def mayStop(self, dev=None, sem=None, layer=None):
    '''
    Stops the device if L{may_stop} is zero. When non-zero, decrements the
    count.

    @param dev: The device to send the stop to; defaults to None for the
      default output device
    @type dev: L{AEOutput.Base}
    @param sem: The semantic information to stop; defaults to None for the
      default semantic
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.LAYER_FOCUS},
      L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @return: Was the stop processed (True) or inhibited (False)?
    @rtype: boolean
    '''
    if layer is None:
      layer = self.layer
    if Output.may_stop == 0:
      self.stopNow(dev, sem, layer)
      return True
    else:
      Output.may_stop -= 1
      if Output.may_stop < 0:
        Output.may_stop = 0
      return False
      
  def inhibitMayStop(self):
    '''
    Prevents the next call to L{mayStop} from stopping the device provided to
    that call. Adds one to the L{may_stop} flag.
    '''
    Output.may_stop += 1
    
  def stopAll(self, dev=None):
    '''
    Tells the referenced output device to interrupt current output and clear 
    buffered data. Unlike L{stopNow}, this ignores the layer on which the 
    method was called and any semantic information. Everything on the device
    is stopped, even when L{TierManager.LSRState.Stopping} is False.
    
    @param dev: The device to send the stop to; defaults to None for the 
      default output device
    @type dev: L{AEOutput.Base}
    '''
    self.device_man.sendStop(dev, None, None)
    
  def observeStop(self, ob):
    self.observer
    
  def play(self, filename, sem=None, layer=None, dev=None):
    '''
    Outputs a file (sound, text, or ??) while respecting settings.

    @param filename: Name of the file to be output
    @type filename: string
    @param sem: The semantic stream on which to send output; defaults to None 
      for the default semantic.
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.LAYER_FOCUS},
      L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @param dev: The device to send the text to; defaults to None for the 
      default output device
    @type dev: L{AEOutput.Base}
    '''
    if layer is None:
      layer = self.layer
    if not os.path.exists(filename):
      filename = os.path.join(self.perk.getPath(), filename)
    self.device_man.sendFilename(dev, filename, sem, layer, True)
  
  def say(self, text, por=None, sem=None, layer=None, dev=None):
    '''
    Outputs the text with a talk at the end.

    @param text: Text to output
    @type text: string
    @param por: The point of regard of the start of text, or None when the text 
      is not from the screen
    @type por: L{POR}
    @param sem: The semantic stream on which to send output; defaults to None 
      for the default semantic.
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.LAYER_FOCUS},
      L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type layer: integer
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    if layer is None:
      layer = self.layer
    self.device_man.sendString(dev, text, sem, layer, True, por)
      
  def sayTextAttrs(self, por=None, text=None, template='%s', dev=None):
    '''
    Say font attributes (size, style, color, etc) of a given por.
    
    Attributes of note:
    (http://developer.gnome.org/doc/API/2.0/atk/AtkText.html#AtkTextAttribute)
    ATK_TEXT_ATTR_EDITABLE Either "true" or "false" indicating whether text is
    editable or not
    ATK_TEXT_ATTR_UNDERLINE "none", "single", "double" or "low"
    ATK_TEXT_ATTR_STRIKETHROUGH "true" or "false" whether text is strikethrough
    ATK_TEXT_ATTR_SIZE 	The size of the characters.
    ATK_TEXT_ATTR_SCALE The scale of the characters. The value is a string 
    representation of a double
    ATK_TEXT_ATTR_WEIGHT The numeric weight of the characters.  
    ATK_TEXT_ATTR_LANGUAGE The language used
    ATK_TEXT_ATTR_BG_COLOR The background color. The value is an RGB value of 
    the format "u,u,u"
    ATK_TEXT_ATTR_FG_COLOR The foreground color. The value is an RGB value of 
    the format "u,u,u"
    ATK_TEXT_ATTR_DIRECTION The direction of the text, if set. Values are 
    "none", "ltr" or "rtl"
    ATK_TEXT_ATTR_JUSTIFICATION The justification of the text, if set. Values
    are "left", "right", "center" or "fill"
    ATK_TEXT_ATTR_STYLE The slant style of the text, if set. Values are 
    "normal", "oblique" or "italic"
    
    @param por: L{POR} pointing at an accessible whose value will be read or 
      None when the L{task_por} should be used. The value is read only if the 
      text parameter is None.
    @type por: L{POR}
    @param text: Name text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''    
    if text is None:
      attrs = self.getAccAllTextAttrs(por)
      if attrs is None:
        # do nothing if no attributes available
        return False
      text = []
      for n, v in attrs.items():
        # non-numeric output (such as color) needs own check for i8n
        if n in self.FONT_NON_NUMERIC:
          if n == 'fg-color':
            v = self.getColorString(v)
          elif v in self.FONT_STYLES:
            v = self.FONT_STYLES[v]
        # else: leave v as it is (a numeric value)
        if n in self.FONT_ATTRS:
          text.append('%s %s' % (self.FONT_ATTRS[n], v))
        else:
          text.append('%s %s' % (n, v))
      text =', '.join(text)
    self.say(template % text, por, LSRConstants.SEM_TEXT_ATTR, self.layer, dev)
    return True
    
  #############################################################################
  # The following set/get methods are autogenerated by EmitGetSet.py                           
  # These are autogenerated in full (rather than with dynamic code generation 
  # or with other Python approaces, such as decorators) to allow epydoc to 
  # generate complete documentation for Perk writers
  #############################################################################
  def setContinuous(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Continuous for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Continuous.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Continuous is None:
      raise InvalidStyleError
    else:
      s.Continuous = val
  
  def getContinuous(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Continuous for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Continuous
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Continuous is None:
      raise InvalidStyleError
    else:
      return s.Continuous
  
  def setStoppable(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Stoppable for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Stoppable.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Stoppable is None:
      raise InvalidStyleError
    else:
      s.Stoppable = val
  
  def getStoppable(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Stoppable for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Stoppable
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Stoppable is None:
      raise InvalidStyleError
    else:
      return s.Stoppable
  
  def setSpeakable(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Speakable for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Speakable.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Speakable is None:
      raise InvalidStyleError
    else:
      s.Speakable = val
  
  def getSpeakable(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Speakable for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Speakable
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Speakable is None:
      raise InvalidStyleError
    else:
      return s.Speakable
  
  def setSoundable(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Soundable for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Soundable.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Soundable is None:
      raise InvalidStyleError
    else:
      s.Soundable = val
  
  def getSoundable(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Soundable for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Soundable
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Soundable is None:
      raise InvalidStyleError
    else:
      return s.Soundable
  
  def setMute(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Mute for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Mute.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Mute is None:
      raise InvalidStyleError
    else:
      s.Mute = val
  
  def getMute(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Mute for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Mute
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Mute is None:
      raise InvalidStyleError
    else:
      return s.Mute
  
  def setPosition(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Position for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Position.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Position is None:
      raise InvalidStyleError
    else:
      s.Position = val
  
  def getPosition(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Position for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Position
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Position is None:
      raise InvalidStyleError
    else:
      return s.Position
  
  def setLanguage(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Language for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Language.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Language is None:
      raise InvalidStyleError
    else:
      s.Language = val
  
  def getLanguage(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Language for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Language
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Language is None:
      raise InvalidStyleError
    else:
      return s.Language
  
  def getChannel(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Channel for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Channel
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Channel is None:
      raise InvalidStyleError
    else:
      return s.Channel
  
  def setChannel(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Channel for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Channel.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Channel is None:
      raise InvalidStyleError
    else:
      s.Channel = min(max(val, s.MinChannel), s.MaxChannel)
  
  def getMinChannel(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinChannel for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinChannel
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinChannel is None:
      raise InvalidStyleError
    else:
      return s.MinChannel
  
  def getMaxChannel(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxChannel for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxChannel
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxChannel is None:
      raise InvalidStyleError
    else:
      return s.MaxChannel
  
  def getVoice(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Voice for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Voice
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Voice is None:
      raise InvalidStyleError
    else:
      return s.Voice
  
  def setVoice(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Voice for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Voice.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Voice is None:
      raise InvalidStyleError
    else:
      s.Voice = min(max(val, s.MinVoice), s.MaxVoice)
  
  def getMinVoice(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinVoice for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinVoice
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinVoice is None:
      raise InvalidStyleError
    else:
      return s.MinVoice
  
  def getMaxVoice(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxVoice for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxVoice
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxVoice is None:
      raise InvalidStyleError
    else:
      return s.MaxVoice
  
  def getVolume(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Volume for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Volume
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Volume is None:
      raise InvalidStyleError
    else:
      return s.Volume
  
  def setVolume(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Volume for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Volume.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Volume is None:
      raise InvalidStyleError
    else:
      s.Volume = min(max(val, s.MinVolume), s.MaxVolume)
  
  def getMinVolume(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinVolume for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinVolume
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinVolume is None:
      raise InvalidStyleError
    else:
      return s.MinVolume
  
  def getMaxVolume(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxVolume for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxVolume
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxVolume is None:
      raise InvalidStyleError
    else:
      return s.MaxVolume
  
  def getRate(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Rate for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Rate
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Rate is None:
      raise InvalidStyleError
    else:
      return s.Rate
  
  def setRate(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Rate for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Rate.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Rate is None:
      raise InvalidStyleError
    else:
      s.Rate = min(max(val, s.MinRate), s.MaxRate)
  
  def getMinRate(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinRate for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinRate
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinRate is None:
      raise InvalidStyleError
    else:
      return s.MinRate
  
  def getMaxRate(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxRate for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxRate
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxRate is None:
      raise InvalidStyleError
    else:
      return s.MaxRate
  
  def getPitch(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Pitch for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Pitch
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Pitch is None:
      raise InvalidStyleError
    else:
      return s.Pitch
  
  def setPitch(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Pitch for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Pitch.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Pitch is None:
      raise InvalidStyleError
    else:
      s.Pitch = min(max(val, s.MinPitch), s.MaxPitch)
  
  def getMinPitch(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinPitch for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinPitch
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinPitch is None:
      raise InvalidStyleError
    else:
      return s.MinPitch
  
  def getMaxPitch(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxPitch for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxPitch
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxPitch is None:
      raise InvalidStyleError
    else:
      return s.MaxPitch
  
  def getGender(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Gender for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Gender
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Gender is None:
      raise InvalidStyleError
    else:
      return s.Gender
  
  def setGender(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Gender for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Gender.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Gender is None:
      raise InvalidStyleError
    else:
      s.Gender = min(max(val, s.MinGender), s.MaxGender)
  
  def getMinGender(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinGender for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinGender
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinGender is None:
      raise InvalidStyleError
    else:
      return s.MinGender
  
  def getMaxGender(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxGender for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxGender
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxGender is None:
      raise InvalidStyleError
    else:
      return s.MaxGender
  
  def getAspiration(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Aspiration for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Aspiration
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Aspiration is None:
      raise InvalidStyleError
    else:
      return s.Aspiration
  
  def setAspiration(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Aspiration for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Aspiration.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Aspiration is None:
      raise InvalidStyleError
    else:
      s.Aspiration = min(max(val, s.MinAspiration), s.MaxAspiration)
  
  def getMinAspiration(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinAspiration for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinAspiration
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinAspiration is None:
      raise InvalidStyleError
    else:
      return s.MinAspiration
  
  def getMaxAspiration(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxAspiration for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxAspiration
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxAspiration is None:
      raise InvalidStyleError
    else:
      return s.MaxAspiration
  
  def getFrication(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Frication for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Frication
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Frication is None:
      raise InvalidStyleError
    else:
      return s.Frication
  
  def setFrication(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Frication for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Frication.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Frication is None:
      raise InvalidStyleError
    else:
      s.Frication = min(max(val, s.MinFrication), s.MaxFrication)
  
  def getMinFrication(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinFrication for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinFrication
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinFrication is None:
      raise InvalidStyleError
    else:
      return s.MinFrication
  
  def getMaxFrication(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxFrication for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxFrication
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxFrication is None:
      raise InvalidStyleError
    else:
      return s.MaxFrication
  
  def getIntonation(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Intonation for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Intonation
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Intonation is None:
      raise InvalidStyleError
    else:
      return s.Intonation
  
  def setIntonation(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Intonation for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Intonation.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Intonation is None:
      raise InvalidStyleError
    else:
      s.Intonation = min(max(val, s.MinIntonation), s.MaxIntonation)
  
  def getMinIntonation(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinIntonation for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinIntonation
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinIntonation is None:
      raise InvalidStyleError
    else:
      return s.MinIntonation
  
  def getMaxIntonation(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxIntonation for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxIntonation
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxIntonation is None:
      raise InvalidStyleError
    else:
      return s.MaxIntonation
  
  def getHeadSize(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of HeadSize for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of HeadSize
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.HeadSize is None:
      raise InvalidStyleError
    else:
      return s.HeadSize
  
  def setHeadSize(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of HeadSize for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new HeadSize.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.HeadSize is None:
      raise InvalidStyleError
    else:
      s.HeadSize = min(max(val, s.MinHeadSize), s.MaxHeadSize)
  
  def getMinHeadSize(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinHeadSize for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinHeadSize
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinHeadSize is None:
      raise InvalidStyleError
    else:
      return s.MinHeadSize
  
  def getMaxHeadSize(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxHeadSize for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxHeadSize
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxHeadSize is None:
      raise InvalidStyleError
    else:
      return s.MaxHeadSize
  
  def getInstrument(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Instrument for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of Instrument
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Instrument is None:
      raise InvalidStyleError
    else:
      return s.Instrument
  
  def setInstrument(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Instrument for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Instrument.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to set the value, focus, tier, or background
    @type layer: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.Instrument is None:
      raise InvalidStyleError
    else:
      s.Instrument = min(max(val, s.MinInstrument), s.MaxInstrument)
  
  def getMinInstrument(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinInstrument for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MinInstrument
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MinInstrument is None:
      raise InvalidStyleError
    else:
      return s.MinInstrument
  
  def getMaxInstrument(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxInstrument for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, focus, tier, or background
    @type layer: integer
    @return: The current value of MaxInstrument
    @rtype: integer
    '''
    layer = layer or self.layer
    s = self.device_man.getStyle(dev, sem, layer)
    if s is None or s.MaxInstrument is None:
      raise InvalidStyleError
    else:
      return s.MaxInstrument
  
  ##############################################################################
  # The following say___() methods are autogenerated by EmitSay.py                           
  # These are autogenerated in full (rather than with dynamic code generation or
  # with other Python approaches, such as decorators) to allow epydoc to 
  # generate complete documentation for Perk writers
  ##############################################################################

  def sayHardline(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the text from the given L{POR} to the next hard line break.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Hardline text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      end_por = self.getEndofHardline(por)
      text = self.getAccTextBetween(por, end_por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_HARDLINE, self.layer, dev)
    return True
  
  def sayWord(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the word at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Word text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getWordText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_WORD, self.layer, dev)
    return True
  
  def sayChar(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the character at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Char text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getCharText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_CHAR, self.layer, dev)
    return True
  
  def sayName(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the name of the accessible at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Name text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccName(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_NAME, self.layer, dev)
    return True
  
  def sayLabel(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the label at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Label text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccLabel(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_LABEL, self.layer, dev)
    return True
  
  def sayItem(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the item at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Item text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getItemText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_ITEM, self.layer, dev)
    return True
  
  def sayWindow(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the window title containing the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Window text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getWindowTitle(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_WINDOW, self.layer, dev)
    return True
  
  def saySection(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the label of the section containing the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Section text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccLabel(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_SECTION, self.layer, dev)
    return True
  
  def sayApp(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the name of the app containing the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: App text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAppName(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_APP, self.layer, dev)
    return True
  
  def sayFont(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the name of the font at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Font text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      # @todo: DBC: this will not work for all applications, improve this
      text = self.getAccTextAttr('family-name', por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_FONT, self.layer, dev)
    return True
  
  def sayColor(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the color of the text at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Color text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccTextAttr('fg-color', por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_COLOR, self.layer, dev)
    return True
  
  def sayRole(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the role of the accessible at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Role text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccRoleName(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_ROLE, self.layer, dev)
    return True
  
  def sayLevel(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the level number at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Level text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccLevel(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_LEVEL, self.layer, dev)
    return True
  
  def sayStatus(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the status message at the current L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Status text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getItemText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_STATUS, self.layer, dev)
    return True
  
  def sayState(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs states of user interest at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: State text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getStateText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_STATE, self.layer, dev)
    return True
  
  def sayDesc(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the description at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Desc text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccDesc(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_DESC, self.layer, dev)
    return True
  
  def sayHotkey(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the "click" action hotkey at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Hotkey text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccClickKey(por)
      # remove extra characters from the text; this will probably break hotkeys
      # that use < and >; can fix with a regex
      try:
        text = text.replace('<', '')
        text = text.replace('>', ', ')
      except AttributeError:
        pass
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_HOTKEY, self.layer, dev)
    return True
  
  def sayError(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs an LSR system error message. L{POR} ignored.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Error text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
  
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_ERROR, self.layer, dev)
    return True
  
  def sayInfo(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs an informational LSR system message. L{POR} ignored.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Info text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
  
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_INFO, self.layer, dev)
    return True
  
  def sayConfirm(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs a confirmation message for a user action. L{POR} ignored.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Confirm text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
  
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_CONFIRM, self.layer, dev)
    return True
  
  def sayValue(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the floating point value at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Value text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccFloatValue(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_VALUE, self.layer, dev)
    return True
  
  def sayValueExtents(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the extents of the float value at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: EXTENTS text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccFloatValueExtents(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_EXTENTS, self.layer, dev)
    return True
  
