'''
Maps the commands registered in L{DefaultPerk} to keys with Alt-Shift as the
standard modifier.

Keys registered

Left and Right Ctrl: Mute/unmute output
Ctrl: Stop output
Alt-Shift-PageUp: Increase speech rate
Alt-Shift-PageDown: Decrease speech rate
Alt-Shift-F1: Where Am I?
Alt-Shift-5: Focus to pointer
Alt-Shift-3: Pointer to focus
Alt-Shift-Y: Read top of view (e.g. window caption)
Alt-Shift-D: Read accessible description
Alt-Shift-B: Read text attributes
Alt-Shift-U: Previous item
Alt-Shift-I: Current item
Alt-Shift-O: Next item
Alt-Shift-J: Previous word
Alt-Shift-K: Current word
Alt-Shift-L: Next word
Alt-Shift-M: Previous character
Alt-Shift-,: Current character
Alt-Shift-.: Next character
Alt-Shift-P: Show the PerkChooser for this Tier

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import Perk

__uie__ = dict(kind='perk', all_tiers=True)

class AltShiftPerk(Perk.Perk):
  '''
  Defines keyboard mappings for features in DefaultPerk and DefaultDialogPerk
  using Alt-Shift as the modifier.
  '''
  def init(self):
    '''    
    Gets a reference to the L{Keyboard} device. Sets the Alt and Shift keys as
    modifier keys. Registers named L{Task}s in the L{DefaultPerk} as commands.
    '''
    # get the Keyboard device and register modifiers and commands
    kbd = self.getInputDevice('Keyboard')
    self.addInputModifiers(kbd, kbd.AEK_ALT_L, kbd.AEK_SHIFT_L, kbd.AEK_ALT_R,
                           kbd.AEK_SHIFT_R, kbd.AEK_CONTROL_L, 
                           kbd.AEK_CONTROL_R)

    self.registerCommand(kbd, 'stop now', True, [kbd.AEK_CONTROL_R])
    self.registerCommand(kbd, 'stop now', True, [kbd.AEK_CONTROL_L])
    
    # register commands for either left Alt-Shift or right Alt-Shift
    pairs = [[kbd.AEK_ALT_L, kbd.AEK_SHIFT_L], 
             [kbd.AEK_ALT_R, kbd.AEK_SHIFT_R]]
    for pair in pairs:
      self.registerCommand(kbd, 'increase rate', False,pair+[kbd.AEK_PAGE_UP])
      self.registerCommand(kbd, 'decrease rate',False,pair+[kbd.AEK_PAGE_DOWN])
      self.registerCommand(kbd, 'previous item', False, pair+[kbd.AEK_U])
      self.registerCommand(kbd, 'current item', False, pair+[kbd.AEK_I])
      self.registerCommand(kbd, 'next item', False, pair+[kbd.AEK_O])
      self.registerCommand(kbd, 'previous word', False, pair+[kbd.AEK_J])
      self.registerCommand(kbd, 'current word', False, pair+[kbd.AEK_K])
      self.registerCommand(kbd, 'next word', False, pair+[kbd.AEK_L])
      self.registerCommand(kbd, 'previous char', False, pair+[kbd.AEK_M])
      self.registerCommand(kbd, 'current char', False, pair+[kbd.AEK_COMMA])
      self.registerCommand(kbd, 'next char', False, pair+[kbd.AEK_PERIOD])
      self.registerCommand(kbd, 'where am i', False, pair+[kbd.AEK_F1])
      self.registerCommand(kbd, 'focus to pointer', False, pair+[kbd.AEK_5])
      self.registerCommand(kbd, 'pointer to focus', False, pair+[kbd.AEK_3])
      self.registerCommand(kbd, 'read top', False, pair+[kbd.AEK_Y])
      self.registerCommand(kbd, 'read text attributes',False, pair+[kbd.AEK_B])
      self.registerCommand(kbd, 'read description', False, pair+[kbd.AEK_D])
      self.registerCommand(kbd, 'read all', False, pair+[kbd.AEK_A])
      self.registerCommand(kbd, 'show perk chooser', False, pair+[kbd.AEK_P])
      self.registerCommand(kbd, 'show settings chooser', False, 
                           pair+[kbd.AEK_S])
