'''
Defines tools for assisting L{Perk} developers.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
# import useful modules for Perks
import Perk, Task
from POR import POR
from i18n import _

class DeveloperPerk(Perk.Perk):
  '''
  Defines keyboard commands for L{Perk} developers that report all L{Perk}s in
  the current L{Tier}, refresh all L{Perk}s in the current L{Tier}, and 
  show/hide all monitors registered in the current profile.
  '''
  def init(self):
    '''
    Registers keys for all L{Task}s in the module.
    '''
    kbd = self.getInputDevice('Keyboard')
    # set Alt+CapsLock as the modifier
    kbd.addModifier(kbd.AEK_ALT_L)
    kbd.addModifier(kbd.AEK_ALT_R)
    kbd.addModifier(kbd.AEK_CAPS_LOCK)
  
    # register named tasks
    self.registerNamedTask(SayPerks(self), 'Say Perks')
    self.registerNamedTask(ReloadPerks(self), 'Reload Perks')
    self.registerNamedTask(ShowHideMonitors(self), 'Show/Hide Monitors')
    
    # map named tasks to commands
    pairs = [[kbd.AEK_ALT_L, kbd.AEK_CAPS_LOCK],
             [kbd.AEK_ALT_R, kbd.AEK_CAPS_LOCK]]
    for pair in pairs:
      self.registerCommandTask(kbd, [pair+[kbd.AEK_P]], 'Say Perks')
      self.registerCommandTask(kbd, [pair+[kbd.AEK_R]], 'Reload Perks')
      self.registerCommandTask(kbd, [pair+[kbd.AEK_M]], 'Show/Hide Monitors')

class SayPerks(Task.InputTask):
  '''
  Says the number of L{Perk}s loaded on the active L{Tier} followed by their
  names.
  '''
  def execute(self, **kwargs):
    '''
    Stops speech then makes the announcement.
    '''
    self.stopNow()
    names = self.getPerkNames()
    n = len(names)
    self.sayInfo(text=self.getAppName(), template=str(n)+' perks in %s.')
    self.sayInfo(text=', '.join(names))

class ReloadPerks(Task.InputTask):
  '''
  Reloads all L{Perk}s in the current L{Tier}. Useful during L{Perk} 
  development when changes have been made to a L{Perk} and those changes should
  be tested without restarting LSR.
  '''
  def execute(self, **kwargs):
    '''
    Reloads L{Perk}s then makes an announcement.
    '''
    self.reloadPerks()
    self.stopNow()
    self.sayInfo(text=self.getAppName(), template='Reloaded perks in %s')

class ShowHideMonitors(Task.InputTask):
  '''
  Shows all monitors associated with the current profile if all are hidden.
  Hides all monitors associated with the current profile if any one is shown.
  '''
  def execute(self, **kwargs):
    '''
    Reloads L{Perk}s then makes an announcement.
    '''
    self.stopNow()
    self.inhibitMayStop()
    if not self.loadMonitors():
      self.unloadMonitors()
      self.sayInfo(text=_('Hiding monitors'))
    else:
      self.sayInfo(text=_('Showing monitors'))