'''
Defines a class representing a multichannel audio output device. Uses pyibmtts
and pySonicEx for speech synthesis and mixing respectively.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import AEOutput
import pySonicEx
import pyibmtts
import math, time, threading, Queue
from i18n import _

BUFFER_SIZE = 2**16

class CliqueAudioStyle(AEOutput.AudioStyleDefault):
  '''
  Overrides the base L{AEOutput.Style.AudioStyleDefault} class, filling in 
  fields for supported style properties with their appropriate values. After
  this class is defined, it is immediately turned into a singleton so it may
  be used ubiquitously as the default style for this device.
  '''
  def __init__(self):
    self.MaxVoice = 8
    self.MinVoice = 1
    self.MaxVolume = 65535
    self.MinVolume = 0
    self.Volume = 40000
    self.MinPitch = 40
    self.MaxPitch = 442
    self.Pitch = 100
    self.MaxRate = 1297
    self.MinRate = 176
    self.Rate = 300
    self.Position = (0, 0, 0)
    self.MaxChannel = 8
    self.MinChannel = 0
CliqueAudioStyle = CliqueAudioStyle() # singleton

class CliqueAudio(AEOutput.Audio):
  '''
  Defines a class to send multichannel audio output from LSR to the FMODEx mixer
  using the pySonicEx wrapper. Uses the IBM TTS engine to generate speech 
  output, though other engines could be used instead.

  To be compliant with LSR requirements, this class implements the interface
  defined in the L{AEOutput.Audio} class.
  
  @ivar mixer: Multichannel sound mixer
  @type mixer: pySonicEx.System
  @ivar channels: Concurrent output channels
  @type channels: list of L{Channel}
  @ivar last_chan: Index of the last channel to be sent output
  @type last_chan: integer
  '''
  USE_THREAD = False
  
  def _localize(self, text):
    '''
    Converts the given unicode text to latin-1, the encoding expected by the
    IBM TTS runtime.

    @todo: PP: Temporary kludge to convert all text to Latin-1 as IBM TTS is
      expecting that encoding instead of UTF-8. Should look at the style 
      language attribute in the future to do the correct encoding.
    @param text: Text to localize
    @type text: unicode
    @return: Localized string
    @rtype: string
    '''
    try:
      # kludge!
      return text.encode('latin-1', 'replace')
    except (TypeError, UnicodeDecodeError), e:
      return text
 
  def init(self):
    '''
    Initializes the pySonicEx system.
    '''
    # initialize the pySonicEx mixer
    self.mixer = pySonicEx.System_Create()
    self.mixer.Output = pySonicEx.OUTPUTTYPE_ALSA
    self.mixer.Init()
    self.channels = []
    self.last_chan = None

  def createDistinctStyles(self, num_styles):
    '''
    Creates distinct output styles differentiated by voice, pitch, position, and
    channel up to the maximum voices given by the default style for this device. 
    Styles are repeated beyond that threshold.
      
    @param num_styles: Number of styles that the requestor would like the
      device to initialize
    @type num_styles: integer
    @return: Styles
    @rtype: list of L{AEOutput.Style}
    '''
    styles = []
    size = len(self.channels)
    dd = math.pi / num_styles
    for i in xrange(size, num_styles+size):
      # create a new channel and style
      ch = Channel(self.mixer)
      s = AEOutput.Style(CliqueAudioStyle)
      # get a unique voice within this set
      v_num = (i+1) % s.MaxVoice
      v = ch.tts.getVoice(v_num)
      s.Voice = v_num
      s.Pitch = v.pitchBaseline
      s.Volume = v.volume
      # position in a semicircle in front of the user
      a = i * (-1)**i
      # avoid being really precise to prevent shifts in perceived location
      x = round(math.cos(math.pi/2+(dd*a)), 4)
      y = round(math.sin(math.pi/2+(dd*a)), 4)
      s.Position = (x, y, 0)
      #s.Channel = i
      # save the channels 
      self.channels.append(ch)
      styles.append(s)
    return styles

  def close(self):
    '''
    Loops over all L{Channel}s and closes them. When all have been closed,
    deletes the L{mixer}.
    '''
    for ch in self.channels:
      ch.close()
      # wait until the thread terminates
      ch.join()
    del self.mixer

  def getName(self):
    '''
    @return: Localized name of this device
    @rtype: string
    '''
    return _('Clique audio device')

  def sendString(self, text, style):
    '''
    Adds the given string to the text to be synthesized.

    @param text: String to buffer on the device
    @type text: string
    @param style: Style with which this string should be output; None means no
      style change should be applied
    @type style: integer
    '''
    if style is not None:
      # keep track of the last channel in case we don't get another style
      self.last_chan = style.Channel
    # have the channel handle the text and style
    self.channels[self.last_chan].put(self._localize(text), style)

  def sendStop(self, style=None):
    '''
    Stops speech immediately on the channel indicated by the given style or
    on all channels if it is None.
    
    @param style: Style indicating which channel will be stopped
    @type style: L{AEOutput.Style}
    '''
    if style is None:
      for ch in self.channels:
        ch.stop()
    else:
      self.channels[style.Channel].stop()

  def sendTalk(self, style=None):
    '''
    Starts speech immediately on the channel indicated by the given style or
    on all channels if it is None.
    
    @param style: Style indicating which channel will be started
    @type style: L{AEOutput.Style}
    '''
    if style is None:
      for ch in self.channels:
        ch.talk()
    else:
      self.channels[style.Channel].talk()
  
  def isActive(self):
    '''
    Gets if any L{Channel} is active meaning it has queued text or is busy
    synthesizing.
    
    @return: Is any channel active?
    @rtype: boolean
    '''
    for ch in self.channels:
      if ch.active():
        return True
    return False
  
  def getDefaultStyle(self):
    '''
    Gets the singleton default style for this device.
    
    @return: Default style of the device
    @rtype: L{AEOutput.AudioStyleDefault}
    '''
    return CliqueAudioStyle
  
class Msg(object):
  '''
  Struct class for storing commands and data to be processed later by a 
  a separate thread.
  
  @cvar TALK: Begin output
  @type TALK: integer
  @cvar STOP: Stop current output
  @type STOP: integer
  @cvar TEXT: Buffer some text
  @type TEXT: integer
  @cvar QUIT: Close the thread
  @type QUIT: integer
  @ivar cmd: One of the commands defined as a class variable in this class
  @type cmd: integer
  @ivar text: Text to buffer or None if no text with this command
  @type text: string
  @ivar style: Style to apply or None if no style with this command
  @type style: L{AEOutput.Style}
  '''
  TALK, STOP, TEXT, QUIT = 0,1,2,3
  def __init__(self, cmd, text=None, style=None):
    '''
    Store instance data. See instance variables for parameter descriptions.
    '''
    self.cmd = cmd
    self.text = text
    self.style = style

class Channel(threading.Thread):
  '''
  Manages a single output stream. Multiple concurrent streams are possible 
  using multiple L{Channel} objects.
  
  @ivar queue: Queue of device L{Msg}s to process
  @type queue: L{Queue.Queue}
  @ivar mixer: Multichannel sound mixer
  @type mixer: pySonicEx.System
  @ivar ch: Audio channel on which output will be done
  @type ch: pySonicEx.Channel
  @ivar want_stop: Has a stop command been received?
  @type want_stop: boolean
  @ivar last_pos: Spatial position of the channel
  @type last_pos: 3-tuple of float
  @ivar tts: IBM TTS engine
  @type tts: pyibmtts.Engine
  @ivar buffer: Queue of text and style objects to be synthesized
  @type buffer: Queue.Queue
  @ivar mode: Mode flags for the type of sound data to be mixed
  @type mode: integer
  '''
  def __init__(self, mixer):
    '''
    Initializes this channel and starts it running as a separate thread. 
    Creates a Queue object to buffer commands until they should be applied.
    Creates an instance of the IBM TTS engine and gives it a memory buffer
    into which it should synthesize speech.
    
    @param mixer: Reference to the mixer
    @type mixer: pySonicEx.System
    '''
    # initialize the thread
    threading.Thread.__init__(self)
    # initialize the queues
    self.queue = Queue.Queue()
    self.want_stop = False
    self.last_pos = None
    # store the mixer instance
    self.mixer = mixer
    self.ch = None
    # initialize the TTS engine
    self.tts = pyibmtts.Engine()
    self.tts.realWorldUnits = True
    self.buffer = pyibmtts.SynthBuffer(self.tts)
    self.tts.setListener(self.buffer)
    self.tts.setOutputBuffer(BUFFER_SIZE)
    # settings used by pySonicEx
    self.mode = pySonicEx.OPENRAW|pySonicEx.OPENMEMORY|pySonicEx.ThreeD
    self.start()
        
  def put(self, text, style):
    '''
    Buffers text and style information for later synthesis. If the style 
    indicates a change in position, an implicit talk command is given since the
    current implementation cannot support position changes mid-stream.
    
    @param text: Text to synthesize
    @type text: string
    @param style: Style to apply to the text to output
    @type style: L{AEOutput.Style}
    '''
    self.queue.put(Msg(Msg.TEXT, text, style))
    
  def talk(self):
    '''
    Buffers a talk command indicating synthesis and output should begin on this
    channel.
    '''
    self.queue.put(Msg(Msg.TALK))
  
  def stop(self):
    '''
    Clear all buffered text and messages yet to be applied. Set the 
    L{want_stop} flag so the next processed message will cause a stop to be
    sent to the channel. Also put a stop message in the old queue so the thread
    will awake if it is waiting on the old queue.
    '''
    old = self.queue
    self.queue = Queue.Queue()
    self.want_stop = True
    old.put_nowait(Msg(Msg.STOP))
    self.queue.put_nowait(Msg(Msg.STOP))
    
  def close(self):
    '''
    Set the stop flag and put a quit message in the queue.
    '''
    self.want_stop = True
    self.queue.put(Msg(Msg.QUIT))
    
  def active(self):
    '''
    Gets if the channel is active meaning it either is synthesizing or has
    buffered text or commands.
    
    @return: Is the channel active?
    @rtype: boolean
    '''
    rv = self.queue.qsize() != 0 or self.tts.speaking()
    if self.ch is not None:
      return rv or self.ch.IsPlaying()
    else:
      return rv
    
  def run(self):
    '''
    Runs the thread loop. The loops sleeps while waiting for a L{Msg} to be
    posted to the L{queue}. When a message is posted, the thread awakes and
    handles it as a text, talk, stop, or quit command.
    '''
    while 1:
      # reset the talk and want stop flags
      talk = False
      # wait for the next message
      msg = self.queue.get()
      if msg.cmd == msg.QUIT:
        # kill this channel thread
        break
      elif msg.cmd == msg.STOP or self.want_stop:
        # clear buffered speech text and currently playing speech
        if self.tts.speaking():        
          self.tts.stop()
        if self.ch and self.ch.IsPlaying():
          self.ch.Stop()
        self.want_stop = False
      elif msg.cmd == msg.TEXT:
        # apply a new style
        if msg.style is not None:
          # talk immediately if a pySonicEx style property changed
          talk = self._applyStyle(msg.style)
        # queue new text
        if msg.text is not None:
          self.tts.addText(msg.text)
      elif (msg.cmd == msg.TALK or talk == True) and self.tts.speaking():
        # synthesize speech synchrnously to the buffer
        self.tts.synthSync()
        # create a struct describing the sound format
        info = pySonicEx.CreateSoundExInfo(defaultfrequency=11025,
                                           numchannels=1,
                                           format=pySonicEx.SOUND_FORMAT_PCM16,
                                           length=len(self.buffer.pcm))
        # create the sound object for the synthed PCM data
        snd = self.mixer.CreateSound(self.buffer.pcm, self.mode, info)
        # create a channel to play the sound, but leave it paused
        self.ch = self.mixer.PlaySound(snd, paused=True, channel=self.ch)
        # set the channel position
        if self.last_pos:
          self.ch.ThreeDPosition = self.last_pos
        # unpause the channel to start output
        self.ch.Paused = False
        # wait until the sound is done or we want to stop
        while self.ch.IsPlaying() and not self.want_stop:
          self.mixer.Update()
          time.sleep(1e-5)
        if self.want_stop and self.ch:
          # do an immediate stop to avoid lag
          self.ch.Stop()
        # reset the buffer for future synthesis
        self.buffer.reset()

  def _applyStyle(self, style):
    '''
    Appplies a new style immediately to the speech engine instance in this
    L{Channel}. Also respects mixer spatial positioning by storing a new 
    position in L{last_pos} so that it may be applied later at the proper point
    in the output.
    
    @param style: Style object to apply
    @type style: L{AEOutput.Style}
    @return: Was a new position stored?
    @rtype: boolean
    '''
    try:
      v = self.tts.getVoice(style.Voice)
    except pyibmtts.ECIException:
      pass
    # try to get the voice to activate
    v = self.tts.setActiveVoice(v)
    # restore voice independent parameters
    try:
      v.speed = style.Rate
    except pyibmtts.ECIException:
      pass
    try:
      v.volume = style.Volume
    except pyibmtts.ECIException:
      pass
    try:
      v.pitchBaseline = style.Pitch
    except pyibmtts.ECIException:
      pass
    
    if self.last_pos is None:
      # if position was never specified, store it now but indicate it has not
      # changed
      self.last_pos = style.Position
      return False
    elif self.last_pos != style.Position:
      # if the new position differs from the previous, store it and indicate
      # a talk should be done implicitly to respect the new position
      self.last_pos = style.Position
      return True
    else:
      # otherwise, just return
      return False