'''
Defines L{AccAdapt.Adapter}s for AT-SPI accessibles that manage their 
descendants.

@todo: PP: add more methods to override DefaultInfo implementations that should
  return info about items, not the container itself

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import pyLinAcc
from DefaultInfo import *
from DefaultNav import *
from DefaultAction import *
from LSRInterfaces import *
from pyLinAcc import Interfaces, Constants

class ContainerAccInfoAdapter(DefaultAccInfoAdapter):
  '''
  Overrides L{DefaultAccInfoAdapter} to provide information specific to 
  containers that have STATE_MANAGES_DESCENDANTS. Expects the subject to be
  a L{POR}.
  '''
  provides = [IAccessibleInfo]
  
  @staticmethod
  def when(subject):
    '''
    Tests if the given subject can be adapted by this class.
    
    @param subject: L{POR} containing an accessible to test
    @type subject: L{POR}
    @return: True when the subject meets the condition named in the docstring
    for this class, False otherwise
    @rtype: boolean
    '''
    ss = subject.accessible.getState()
    return ss.contains(Constants.STATE_MANAGES_DESCENDANTS)
  
  def _getSubItems(self, acc):
    '''
    Gets subitems of an item in a container that manages its descendants.
    Subitems appear to be used to indicate items that cannot be selected 
    independently of each other in a list or tree control so they are made
    children of a dummy accessible. This is most often observed when a list or
    tree has an icon + plus text in one of its cells.
    
    @param acc: Accessible that may or may not have subitems
    @type acc: L{pyLinAcc.Accessible}
    @return: List of accessibles, including the given one
    @rtype: list of L{pyLinAcc.Accessible}
    '''
    si = [acc] + [acc.getChildAtIndex(i) for i in xrange(acc.childCount)]
    return si
  
  @pyLinAcc.errorToLookupError
  def getAccSelection(self):
    '''  
    Gets a list L{POR}s referring to the selected items within the subject 
    accessible.
    
    @return: Points of regard to selected items
    @rtype: list of L{POR}s
    @raise LookupError: When the subject accessible is dead
    '''
    try:
      sel = Interfaces.ISelection(self.accessible)
    except NotImplementedError:
      return []
    # return PORs that have the item index set properly
    return [POR(self.accessible, sel.getSelectedChild(i).getIndexInParent(), 0)
            for i in range(sel.nSelectedChildren)]
    
  @pyLinAcc.errorToLookupError
  def getAccRoleName(self):
    '''
    Gets the accessible role name of the subject if item offset is None else 
    the role name of the given item. The name is localized.
    
    @return: Accessible name of requested object
    @rtype: string
    @raise LookupError: When the subject accessible or the child is dead
    @raise IndexError: When the item offset is outside the bounds of the 
      number of children of the subject accessible
    '''
    if self.item_offset is None:
      return super(ContainerAccInfoAdapter, self).getAccRoleName()
    try:
      c = self.accessible.getChildAtIndex(self.item_offset)
      # @todo: PP: just report the parent role for now, otherwise we get 
      # lots of repeats; revisit
      #l = [i.getLocalizedRoleName() for i in self._getSubItems(c)]
      return unicode(c.getLocalizedRoleName(), 'utf-8')
    except AttributeError:
      raise IndexError
  
  @pyLinAcc.errorToLookupError
  def getAccName(self):
    '''
    Gets the accessible name of the subject if item offset is None else the 
    name of the given item.
    
    @return: Accessible name of requested object
    @rtype: string
    @raise LookupError: When the subject accessible or the child is dead
    @raise IndexError: When the item offset is outside the bounds of the 
      number of children of the subject accessible
    '''
    if self.item_offset is None:
      return super(ContainerAccInfoAdapter, self).getAccName()
    try:
      c = self.accessible.getChildAtIndex(self.item_offset)
      l = [i.name for i in self._getSubItems(c)]
      return unicode(' '.join(l), 'utf-8')
    except AttributeError:
      raise IndexError
      
  @pyLinAcc.errorToLookupError  
  def getAccDescription(self):
    '''
    Gets the accessible description of the subject if item offset is None else 
    the name of the given item.
    
    @return: Accessible description of requested object
    @rtype: string
    @raise LookupError: When the subject accessible or the child is dead
    @raise IndexError: When the item offset is outside the bounds of the 
      number of children of the subject accessible
    '''
    if self.item_offset is None:
      return super(ContainerAccInfoAdapter, self).getAccDescription()
    try:
      c = self.accessible.getChildAtIndex(self.item_offset)
      l = [i.description for i in self._getSubItems(c)]
      return unicode(' '.join(l), 'utf-8')
    except AttributeError:
      raise IndexError
      
  @pyLinAcc.errorToLookupError
  def getAccItemText(self):
    '''
    Gets the accessible name of the subject if item offset is None else the
    name of the given child accessible.

    @return: Accessible description of requested object
    @rtype: string
    @raise LookupError: When the subject accessible or the child is dead
    @raise IndexError: When the item offset is outside the bounds of the 
      number of children of the subject accessible
    '''
    if self.item_offset is None:
      return super(ContainerAccInfoAdapter, self).getAccItemText()
    try:
      c = self.accessible.getChildAtIndex(self.item_offset)
      # use the IAccessibleInfo interface to get the text from the child to 
      # account for the possibility that the child implements the AT-SPI Text 
      # interface
      l = [IAccessibleInfo(POR(i)).getAccItemText() 
           for i in self._getSubItems(c)]
      return u' '.join(l)
    except AttributeError:
      raise IndexError
  
  @pyLinAcc.errorToLookupError
  def getAccStates(self):
    '''
    Gets a list of names of states indicating the current state of the given 
    accessible.

    @return: Names of all states
    @rtype: list of string
    @raise LookupError: When the subject accessible or the child is dead
    @raise IndexError: When the item offset is outside the bounds of the 
      number of children of the subject accessible
    '''
    if self.item_offset is None:
      return super(ContainerAccInfoAdapter, self).getAccStates()
    try:
      c = self.accessible.getChildAtIndex(self.item_offset)
    except AttributeError:
      raise IndexError
    return IAccessibleInfo(POR(c, self.item_offset)).getAccStates()
  
  @pyLinAcc.errorToLookupError 
  def getAccIndex(self):
    '''
    Gets the index of an item as the item offset.

    @return: Zero indexed item offset
    @rtype: integer
    @raise LookupError: When the table or item is no longer valid
    '''
    return self.item_offset
  
class ContainerAccActionAdapter(DefaultAccActionAdapter):
  '''
  Overrides L{DefaultAccActionAdapter} to provide information specific to 
  containers that have STATE_MANAGES_DESCENDANTS. Expects the subject to be a
  L{POR}.
  '''
  provides = [IAccessibleAction]
  
  @staticmethod
  def when(subject):
    '''
    Tests if the given subject can be adapted by this class.

    @param subject: L{POR} containing an accessible to test
    @type subject: L{POR}
    @return: True when the subject meets the condition named in the docstring
    for this class, False otherwise
    @rtype: boolean
    '''
    acc = subject.accessible
    c = Constants
    ss = acc.getState()
    return ss.contains(c.STATE_MANAGES_DESCENDANTS)
  
  @pyLinAcc.errorToLookupError
  def selectChild(self):
    '''
    Selects the accessible object implementing this interface when the param
    is None or the child item otherwise.
    
    @return: Did accessible accept (True) or refuse (False) the selection?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead
    @raise NotImplementedError: When the accessible does not support an 
      interface for setting the selection
    '''
    if self.item_offset is None:
      return super(ContainerAccActionAdapter,self).selectChild(self.item_offset)
    else:
      c = Interfaces.ISelection(self.accessible)
      return c.selectChild(self.item_offset)