'''
Defines a class for managing the L{Perk}s loaded for a single instance of an
application.

@author: Peter Parente
@author: Pete Brunet
@author: Larry Weiss
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging
import LSRSettings, AEInput, Task

log = logging.getLogger('Tier')

class Tier(object):
  '''
  Manages the L{Perk}s for a single instance of a running program (a process). 
  Supports the addition and removal of L{Perk}s from its stack.  Executes 
  L{Task}s registered in each L{Perk} in response to L{AEEvent}s. Provides
  methods to assist L{Perk}s and L{Task}s in finding registered L{Task}s 
  throughout the L{Tier}.

  @ivar perks: List of L{Perk}s treated as a stack (last in, first executed)
  @type perks: list
  @ivar wanted_events: Lookup table for what L{AEEvent} are desired by any
    L{Task} in any L{Perk} in this L{Tier}. Used to optimize event dispatch and
    processing. Unwanted events are ignored.
  @type wanted_events: dictionary
  @ivar tier_manager: L{TierManager} that created this L{Tier}
  @type tier_manager: L{TierManager}
  @ivar state: Object storing the current state of this L{Tier}
  @type state: L{AEState}
  @ivar focus_por: The L{POR} for the current focus. This POR contains a
    reference to the accessible that last received focus, the item within that
    acc. that has either the selector or caret, and the char offset within the
    Item that represents the caret position. L{Task}s need it to perform a user
    action based on the focus. L{Tier} needs it to set the "current position"
    when the Mode setting (Pointer/Focus) is in Focus mode, prior to Task
    execution. L{Perk}s need it during their initialization for the same reason
    Tasks do. The tool that displays the View to developers should use it to
    ensure that element is displayed/highlighted. The Task that launches the
    View hiearchy tool must provide the focus POR to the tool.
  @type focus_por: L{POR} 
  @ivar name: Name of this L{Tier}
  @type name: string
  @ivar aid: ID uniquely identifying this L{Tier} from all other L{Tier}s
  @type aid: opaque
  '''
  def __init__(self, tier_manager, name, aid):
    '''
    Stores a reference to the L{TierManager} that created this L{Tier}. Creates
    an empty list of L{Perk}s. Initializes the stored focus L{POR} to None.
    Creates a new L{LSRSettings} object to be provided to L{Task.Tools} at L{Task}
    execution time.
    
    @todo: PP: LSRSettings won't be created directly here but rather loaded
      from disk once persistence is implemented
    
    @param tier_manager: Manager that created this L{Tier}
    @type tier_manager: L{TierManager}
    @param name: Name of this L{Tier}
    @type name: string
    @param aid: ID uniquely identifying this L{Tier} from all other L{Tier}s
    @type aid: opaque
    '''
    self.tier_manager = tier_manager
    self.perks = []
    self.wanted_events = {}
    self.focus_por = None
    self.name = name
    self.aid = aid
    self.state = LSRSettings.LSRSettings()
    
  def pushPerk(self, ae, *perks):
    '''
    Adds one or more L{Perk}s to the top of the stack. If more than one L{Perk}
    is specified, the last specified L{Perk} will be at the top of the stack 
    when this method completes. That is, the behavior of pushing more than one
    L{Perk} at a time is the same as if each L{Perk} were pushed individually.
    
    @param ae: The AccessEngine context to use for initializing the Perk
    @type ae: L{AccessEngine}
    @param perks: L{Perk}s to add
    @type perks: list of L{Perk}
    '''
    for perk in perks:
      # top of the stack
      self.perks.insert(0, perk)
      # let the Perk initialize itself
      perk.preExecute(ae, self)
      perk.init()
      perk.postExecute(False)
      
  def popPerk(self, perk=None):
    '''
    Removes the L{Perk} at the top of the stack.
    
    @param perk: L{Perk} to remove or None to remove the top L{Perk} (defaults 
        to None)
    @type perk: L{Perk}
    @raise IndexError: When the stack is empty
    @raise ValueError: When specified L{Perk} is not found
    '''
    if perk is None:
      self.perks.pop(0)
    else:
      self.perks.remove(perk)
      
  def getPerks(self):
    '''
    Gets all L{Perk}s pushed onto this L{Tier} in execution order.
    
    @return: All L{Perk}s pushed onto this L{Tier}
    @rtype: list of L{Perk}
    '''
    return self.perks
  
  def setEventInterest(self, kind, wants):
    '''
    Updates the L{wanted_events} dictionary to indicate that a L{Task} in a 
    L{Perk} in this L{Tier} has been registered for a particular kind of 
    L{AEEvent}. This information is used for optimization purposes such that no 
    processing will occur on the event unless at least one L{Task} is registered
    that will use it.
    
    @param kind: Kind of L{AEEvent} of interest to a L{Task}
    @type kind: L{AEEvent} class
    @param wants: Does a L{Perk} want an event (i.e. a L{Task} is registering 
      for it or no longer want an event (i.e. a L{Task} is unregistering for 
      it)?
    @type wants: boolean
    '''
    count = self.wanted_events.setdefault(kind, 0)
    if wants:
      count += 1
    else:
      count -= 1
    if count <= 0:
      del self.wanted_events[kind]
    else:
      self.wanted_events[kind] = count
    # inform the TierManager of the new event interest
    self.tier_manager.setEventInterest(kind, wants)
      
  def wantsEvent(self, event):
    '''
    Gets if this L{Tier} wants a particular kind of L{AEEvent} given that one of
    its L{Perk}s has a L{Task} that wants to handle it.
    
    @param event: Event to be tested
    @type event: L{AEEvent}
    '''
    return self.wanted_events.has_key(type(event))
  
  def getCommandPerkAndTask(self, gesture_list):
    '''
    Gets a L{Task} registered to execute in response to the L{AEInput.Gesture}
    and the L{Perk} in which it is registered. None is returned if not found.
    
    @param gesture_list: Gestures and device expressed as a list of virtual key 
      codes
    @type gesture_list: L{AEInput.Gesture}
    @return: L{Task} set to execute in response to the input gesture and
      L{Perk} in which it is registered or (None, None)
    @rtype: 2-tuple of (L{Perk.Perk}, L{Task.Base.Task})
    '''
    for perk in self.perks:
      task = perk.getCommandTaskLocal(gesture_list)
      if task is not None:
        return perk, task
    return (None, None)

  def getNamedTask(self, name):
    '''
    Gets a L{Task} with the given name by iterating through the registered 
    L{Perk}s until one is found containing a L{Task} registered under the given
    name. None is returned if not found.
    
    Called by L{Perk.Perk.getNamedTask}.
    
    @param name: Name of the L{Task} to find 
    @type name: string
    @return: L{Task} with the given name or None
    @rtype: L{Task.Base.Task}
    '''
    for perk in self.perks:
      task = perk.getNamedTaskLocal(name)
      if task is not None:
        return task
    return None
    
  def getAllEventTasks(self, event_type, task_layer):
    '''    
    Gets all L{Task}s registered to handle the given type of event on the
    given layer by iterating through the registered {Perk}s. The L{Task}s are
    returned in the order they will be executed both within and across
    L{Perk}s in this L{Tier}.
    
    Called by L{Perk.Perk.getAllEventTasks}.

    @param event_type: Desired type of L{AEEvent}
    @type event_type: L{AEEvent} class
    @param task_layer: Layer on which the desired L{Task}s are registered, one 
      of L{Task.FOCUS_LAYER}, L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type task_layer: integer
    @return: List of all L{Task}s registered to handle the given type of event
      on the given layer
    @rtype: list of L{Task.Base.Task}
    @see: L{Perk.Perk.getAllEventTasks}
    '''
    tasks = []
    for perk in self.perks:
      tasks.extend(perk.getEventTasksLocal(event_type, task_layer))
    return tasks
  
  def _executeTask(self, ae, task, por, layer, task_data, propagate):
    '''
    Executes the given L{Task} in response to the given L{AEEvent}.
    
    @param ae: Reference to AccessEngine for event context
    @type ae: L{AccessEngine}
    @param task: A L{Task} registered to handle the given event
    @type task: L{Task}
    @param por: Point of regard where the event occurred
    @type por: L{POR}
    @param propagate: Should this event be propagated to the 
      L{Task.Base.Task.execute} method or should we call 
      L{Task.Base.Task.update} instead?
    @type propagate: boolean
    @param layer: Layer on which the L{AEEvent} occurred, one of 
      L{AEEvent.FOCUS_LAYER}, L{AEEvent.TIER_LAYER}, L{AEEvent.BACKGROUND_LAYER}
    @type layer: integer
    @param task_data: Keyword arguments to be provided to the L{Task}
    @type task_data: dictionary
    @return: Should the next registered L{Task} be executed?
    @rtype: boolean
    '''
    # pass this Tier to the task so it can access this Tier's instance variables
    # like state and focus_por
    task.preExecute(ae, self, por)
    try:
      if propagate:
        # unpack the dictionary and execute the task
        rv = task.execute(layer=layer, **task_data)
      else:
        # unpack the dictionary and update the task
        task.update(layer=layer, **task_data)
        rv = False
    except Task.ToolsError, ex:
      # speak the Task.ToolsError if state.Trap is True
      if self.state.Trap:
        task.say(str(ex))
      else:
        log.exception(ex)
      rv = True
    except Exception:
      # log any other Perk exception
      log.exception('Tier exception')
      # continue processing
      rv = True
    if rv is None:
      # continue processing if no return value is specified
      rv = True
    # uninitialize the Task, potentially storing the POR only if the event was
    # in the focus layer
    task.postExecute(layer == Task.FOCUS_LAYER)
    return rv
  
  def manageEvent(self, acc_eng, event):
    '''
    Manages an event by iterating through the L{Perk} stack (top to bottom) and
    checking for registered L{Task}s of the given type. Executes the registered
    L{Task}s (last registered, first executed) in each L{Perk} until one of the 
    following conditions is met:
      - All L{Task}s have executed
      - A L{Task} returns False
      - A L{Task} raises an exception
    
    In the latter two cases, no additional L{Task}s in the current L{Perk} or 
    additional L{Perk}s in this L{Tier} are executed. Instead the 
    L{Task.Base.Task.update} methods are called to allow housekeeping operations 
    (e.g updating state) to be performed.
    
    If a L{Task} returns neither True or False (e.g. it returns None) a 
    warning is logged and the return value is treated as if it were True. This
    likely means the L{Task} forgot to specify a return value.
    
    @param acc_eng: Reference to AccessEngine for event context
    @type acc_eng: L{AccessEngine}
    @param event: Event to process
    @type event: L{AEEvent.Base.AccessEngineEvent}
    '''
    try:
      # grab POR from any focus type event
      self.focus_por = event.getFocusPOR()
    except AttributeError:
      pass
    
    # use type as an indicator of what Task will handle
    if not self.wantsEvent(event):
      # quit immediately if no Task wants this event
      return
    
    # show the event in registered monitors
    self.tier_manager.showEvent(event, self.name)
    
    # get these only once as their values are constant for every Task executed
    por = event.getPOR()
    task_layer = event.getLayer()
    task_data = event.getDataForTask()
    event_type = type(event)

    # run through all the registered Perks
    propagate = True
    for perk in self.perks:
      perk.preExecute(acc_eng, self, por)
      # run through all Tasks of the given type in this Perk
      for task in perk.getEventTasksLocal(event_type, task_layer):
        propagate = self._executeTask(acc_eng, task, por, task_layer, task_data,
                                      propagate)
        self.tier_manager.showTask(event, perk.getName(), task.getName(), 
                                   propagate)
      perk.postExecute(False)
        
  def manageGesture(self, acc_eng, event):
    '''
    Manages an event by getting the L{AEInput.Gesture} that triggered it and
    locating a L{Task} registered to execute in response to it. If a L{Task} 
    could not be found for the given event, the L{Task} registered for invalid
    gestures is executed instead.
    
    @param acc_eng: Reference to AccessEngine for event context
    @type acc_eng: L{AccessEngine}
    @param event: Event to process
    @type event: L{AEEvent.Base.AccessEngineEvent}
    '''
    # look for a Perk with a Task registered to respond to this input command
    # for now, construct a gesture list for comparison
    g = event.getGesture()
    gl = AEInput.GestureList(g.getDevice(), gestures=[g])
    perk, task = self.getCommandPerkAndTask(gl)
    # show the event in registered monitors
    self.tier_manager.showEvent(event, self.name)
    if task is not None:
      perk.preExecute(acc_eng, self)
      # execute that Task
      self._executeTask(acc_eng, task, None, event.getLayer(), {}, True)
      self.tier_manager.showTask(event, perk.getName(), task.getName(), False)
      perk.postExecute(False)
      
  def getState(self):
    '''
    @return: Returns the L{LSRSettings} for this Tier.
    @rtype: L{LSRSettings}
    '''
    return self.state
  
  def setState(self, state):
    '''
    Sets this Tier's state to the provided (non-None) object.

    @param state: The new state to set.
    @type state: L{LSRSettings}
    '''
    self.state = state
  
  def getFocusPOR(self):
    '''
    @return: Returns the current focus L{POR} for this Tier
    @rtype: L{POR.POR}
    '''
    return self.focus_por
  