'''
Defines the main function which parses command line arguments and acts on them.
Creates an instance of the L{UIRegistrar} to add a new user interface element
(UIE) to the repository on disk when commands for the L{UIRegistrar} are
present. If the kill switch is found, kills the running instance of LSR started
by this user. Otherwise, starts an instance of the L{AccessEngine} and runs the
screen reader.

The L{main} function in this module parses the command line parameters and
dispatches command line options to the other top level functions defined here.
The other top level functions split up the work of installing/uninstalling new
UIEs (L{install}), associating/unassociating (L{associate}), creating/removing
profiles (L{profile}), initializing the L{UIRegistrar} repository on first run
(L{initRepository}) killing the screen reader (L{kill}), and running the screen
reader (L{run}). Other top level functions are provided for convenience such as
L{checkA11y} and L{generate}.

See the LSR man pages for help with using command line parameters.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at 
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
try:
  # little bit of fun to see how psyco improves things
  import psyco
  psyco.profile()
except ImportError:
  pass

import logging, optparse, os, signal, sys, subprocess
import UIRegistrar, Adapters, LSRConstants, AEOutput
from i18n import _
from LSRConstants import HOME_USER, HOME_LSR, LSR_VERSION, LSR_DATE, TEMPLATES
from LSRInterfaces import implements

# define the format for log entries and timestamps on each entry
# see http://docs.python.org/lib/node352.html
log_format = '%(asctime)s %(levelname)s %(name)s %(message)s'
time_format = '%a, %d %b %Y %H:%M:%S'

class PrintLogger(object):
  '''
  Provides a dirt-simple interface compatible with stdout and stderr. When
  assigned to sys.stdout or sys.stderr, an instance of this class redirects 
  print statements to the logging system. This means the result of the 
  print statements can be silenced, sent to a file, etc. using the command
  line options to LSR. The log level used is defined by the L{LEVEL} constant
  in this class.
  
  @cvar LEVEL: Logging level for writes directed through this class
  @type LEVEL: integer
  @ivar log: Reference to the Print log channel
  @type log: logging.Logger
  '''
  LEVEL = 15
  
  def __init__(self):
    '''
    Create the logger.
    '''
    self.log = logging.getLogger('Print')
    self.chunks = []
    
  def write(self, data):
    '''
    Write the given data at the debug level to the logger. Stores chunks of
    text until a new line is encountered, then sends to the logger.
    
    @param data: Any object that can be converted to a string
    @type data: stringable
    '''
    s = data.encode('utf-8')
    if s.endswith('\n'):
      self.chunks.append(s[:-1])
      s = ''.join(self.chunks)
      self.log.log(self.LEVEL, s)
      self.chunks = []
    else:
      self.chunks.append(s)

def welcome():
  '''
  Prints copyright, license, and version info.
  '''
  print LSRConstants.NAME
  print LSRConstants.COPYRIGHT
  print LSRConstants.LICENSE
  print
  
def install(options):
  '''
  Installs a new UIE in the repository via the L{UIRegistrar}.
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When the required params are not specified
  '''
  # make sure the default UIEs have been installed at least once in the past
  initRepository()
  if None in [options.uie_name, options.uie_path, options.uie_kind]:
    raise ValueError(_('Name, kind, and path are required'))

  reg = UIRegistrar.UIRegistrar()
  reg.install(options.uie_kind, options.uie_name, options.uie_path)
  return _('A %s with name %s was installed' % 
           (options.uie_kind, options.uie_name))

def uninstall(options):
  '''
  Uninstalls a UIE from the repository via the L{UIRegistrar}.

  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When the required params are not specified
  '''
  # make sure the default UIEs have been installed at least once in the past
  initRepository()
  if not options.uie_name or not options.uie_kind:
    raise ValueError(_('Name and kind are required'))

  reg = UIRegistrar.UIRegistrar()
  reg.uninstall(options.uie_kind, options.uie_name)
  return _('A %s with name %s was uninstalled' % 
           (options.uie_kind, options.uie_name))
  
def associate(options):
  '''  
  Associates a UIE with a profile so that it is loaded at a particular time
  (e.g. on startup, when any tier is created, or when a particular tier is
  created).
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When the required params are not specified
  '''
  # make sure the default UIEs have been installed at least once in the past
  initRepository()
  if None in [options.uie_name, options.uie_kind, options.uie_when,
              options.profile]:
    raise ValueError(_('Name, kind, profile, and when are required'))
  if options.uie_when == UIRegistrar.ONE_TIER and not options.uie_tier:
    raise ValueError(_('Tier name is required when associating with one tier'))
  
  reg = UIRegistrar.UIRegistrar()
  reg.associate(options.uie_kind, options.uie_name, options.profile, 
                options.uie_when, options.uie_tier, options.uie_index)
  return _('A %s with name %s was added to profile %s' % 
           (options.uie_kind, options.uie_name, options.profile))

def disassociate(options):
  ''' 
  Disassociates a UIE with a profile so that it is no longer loaded at a
  particular time.
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When the required params are not specified
  '''
  # make sure the default UIEs have been installed at least once in the past
  initRepository()
  if None in [options.uie_name, options.uie_kind, options.uie_when,
              options.profile]:
    raise ValueError(_('Name, kind, profile, and when are required'))
  if options.uie_when == UIRegistrar.ONE_TIER and not options.uie_tier:
    raise ValueError(_('Tier name is required when disassociating from one tier'))
  
  reg = UIRegistrar.UIRegistrar()
  reg.disassociate(options.uie_kind, options.uie_name, options.profile, 
                   options.uie_when, options.uie_tier)
  return _('A %s with name %s was removed from profile %s' % 
           (options.uie_kind, options.uie_name, options.profile))

def generate(options):
  ''' 
  Generates a template file for the given kind of UIE. Installs the template
  UIE immediately. Associates it if the name of a profile is also specified.
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When required params are not specified
  '''
  if None in [options.uie_name, options.uie_path, options.uie_kind]:
    # check for install errors up front
    raise ValueError(_('Name, path, and kind are required'))
  if options.profile is not None:
    # check for association errors up front
    if None in [options.profile, options.uie_when]:
      raise ValueError(_('Name, kind, path, profile, and when are required'))
    if options.uie_when == UIRegistrar.ONE_TIER and not options.uie_tier:
      raise ValueError(_('Tier name is required when associating with one tier'))
  # fill in the template and copy it to the appropriate location
  try:
    temp = file(os.path.join(TEMPLATES, options.uie_kind), 'r').read()
  except IOError:
    raise ValueError(_('No template for %s') % options.uie_kind)
  uie = temp % dict(name=options.uie_name)
  try:
    file(options.uie_path, 'w').write(uie)
  except IOError:
    raise ValueError(_('Could not create %s file') % options.uie_kind)
  # install the UIE
  rv = install(options)
  if options.profile is not None:
    # associate the UIE with the given profile
    return associate(options)
  else:
    return rv
  
def createProfile(options):
  '''
  Creates a new, empty profile.
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When the required params are not specified
  '''
  # make sure the default UIEs have been installed at least once in the past
  initRepository()
  if not options.profile:
    raise ValueError(_('Profile name is required'))
  reg = UIRegistrar.UIRegistrar()
  reg.createProfile(options.profile)
  return _('Profile %s was created') % options.profile

def removeProfile(options):
  '''
  Removes an existing profile.
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When the required params are not specified
  '''
  # make sure the default UIEs have been installed at least once in the past
  initRepository()
  if not options.profile:
    raise ValueError(_('Profile name is required'))
  reg = UIRegistrar.UIRegistrar()
  reg.removeProfile(options.profile)
  return _('Profile %s was removed') % options.profile
    
def kill(options):
  '''
  Kills all running instances of the screen reader. Reads the directory names in
  ~/.lsr/pids and treats them as the process IDs (PIDs) for the running 
  instances of LSR. Ignores any errors accessing PIDs stored on disk or 
  permissions for stopping a process.
  
  @warning: Any directory name that evaluates to an integer in the PIDs 
      directory will cause a process running under that PID to be killed.
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  '''
  pid_path = os.path.join(HOME_USER, 'pids')
  try:
    # get a list of all the running LSR processes
    pids = os.listdir(pid_path)
  except OSError:
    return
  for pid in pids:
    try:
      # kill the process
      os.kill(int(pid), signal.SIGTERM)
    except OSError:
      pass
    try:
      # delete the directory representing the running process
      os.rmdir(os.path.join(pid_path, pid))
    except OSError:
      pass
  return _('Killed LSR')
  
def say(options):
  '''
  Uses the default L{AEOutput} device to output a string.
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  '''
  reg = UIRegistrar.UIRegistrar()
  prof = options.profile or 'user'
  for dev in reg.loadAssociated(UIRegistrar.DEVICE, prof, UIRegistrar.STARTUP):
    # find the first output device
    if implements(dev, AEOutput.AEOutput):
      # speak the utf-8 encoded string seen on the command line
      s = options.action_value
      try:
        dev.init()
      except:
        # try another device
        continue
      dev.sendStringSync(s)
      dev.close()
      return _('Said: %s') % s
  return _('No default output device')
    
def show(options):
  '''
  Shows all installed scripts and profiles, indicating all associated UIE's
  via the L{UIRegistrar}.
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When the required params are not specified
  '''
  # make sure the default UIEs have been installed at least once in the past
  initRepository()

  reg = UIRegistrar.UIRegistrar()
  print _('Installed UIEs')
  i_count = 0
  for kind in UIRegistrar.ALL_KINDS:
    names = reg.listInstalled(kind)
    i_count += len(names)
    print kind+':', names
  print 
  
  print _('Associated UIEs')
  p_count = 0
  for name in reg.listProfiles():
    p_count += 1
    print 'profile:', name
    for kind in UIRegistrar.ALL_KINDS:
      print '  '+kind
      for when in UIRegistrar.ALL_TIMES:
        print '    '+when+':', reg.listAssociated(kind, name, when)
    print
  
  return _('%d installed UIEs, %d profiles') % (i_count, p_count)

def run(options):
  '''
  Runs the screen reader. Stores the process ID (PID) in the name of a 
  directory in L{HOME_USER} so it can be terminated later by this same script.
  
  @param options: Options seen on the command line
  @type options: optparse.Values
  @return: Description of the action
  @rtype: string
  @raise ValueError: When anything goes wrong during initialization
  '''
  # make sure the default UIEs have been installed at least once in the past
  initRepository()
  # make sure the profile exists
  profile = options.profile or 'user'
  if profile not in UIRegistrar.UIRegistrar().listProfiles():
    raise ValueError(_('Profile %s does not exist') % options.profile)
  # store the pid of this process in the user's .lsr directory
  pid = os.path.join(HOME_USER, 'pids', str(os.getpid()))
  os.makedirs(pid)
  # import AE here so we can do other ops outside a Gnome session
  import AccessEngine
  # use the profile specified on the command line or the default
  ae = AccessEngine.AccessEngine(profile)
  # create a print log channel for redirecting print statements to debug
  sys.stdout = PrintLogger()
  rv = ae.run()
  os.rmdir(pid)
  return rv

def initRepository():
  '''
  Checks if the L{UIRegistrar} repository has been initialized with two
  default profiles, user and developer, and all the L{Perk}s, devices, choosers,
  and monitors that ship with LSR. Stores a version number in the repository as
  a way of tracking if new defaults should be installed when LSR is upgraded.
  Also stores the L{HOME_LSR} path so that the links to the default UIEs are
  updated to point at either the installed version of LSR or the one being run
  from source at some path. Ignores all errors indicating a default is already 
  installed.
  
  If new UIEs are shipped with LSR that should be added to the repository by
  default, the L{LSRConstants.LSR_VERSION} or L{LSRConstants.LSR_DATE}
  strings must change so the repository is rebuilt.
  
  The repository is automatically rebuit if the L{LSRConstants.HOME_LSR} path 
  for this running instance of LSR is different from the one from the one that
  was stored when the repository was last built.
  
  The initial repository layout is given by L{LSRInitConfig}.
  '''
  version = '%s:%s:%s' % (HOME_LSR, LSR_VERSION, LSR_DATE)
  reg = UIRegistrar.UIRegistrar()
  if reg.getVersion() == version:
    # do nothing if up to date
    return

  # reset the repository if the version string doesn't match
  reg.removeAll() 
  
  import LSRInitConfig as cfg
  
  # install all UIEs
  for kind in UIRegistrar.ALL_KINDS:
    for name in getattr(cfg.install, kind):
      reg.install(kind, name, os.path.join(HOME_LSR, kind.title()+'s', 
                                           name+'.py'))
  
  # create all profiles
  for prof in cfg.profiles:
    reg.createProfile(prof.name)
    
    # handle all kinds except for Perks
    for kind in UIRegistrar.ALL_KINDS[:-1]:
      for name in getattr(prof, kind):
        reg.associate(kind, name, prof.name, UIRegistrar.STARTUP)
        
    # handle default Perks
    for name in prof.perk.get(None, []):
      reg.associate(UIRegistrar.PERK, name, prof.name, UIRegistrar.ANY_TIER)
    # handle Tier specific Perks
    for tier in (t for t in prof.perk if t is not None):
      for name in prof.perk[tier]:
        reg.associate(UIRegistrar.PERK, name, prof.name, UIRegistrar.ONE_TIER,
                      tier)

  # only set the version once everything has succeeded
  reg.setVersion(version)

def getAction(option, opt_str, value, parser):
  '''
  Maps a command line param to the name of a function in this module that will
  handle it. For instance, the command line parameter for installing a UIE is
  mapped to the L{install} function which will carry out the command.
  '''
  parser.values.action = option.default
  parser.values.action_value = value
  
def checkA11y():
  '''
  Runs gconf to check if desktop accessibility support is enabled or not. If 
  not, enables it and asks the user to logout.
  
  @raise ValueError: When accessibility support is not enabled
  '''
  # check that accessibility is enabled in gconf and enable it if not
  gconf = subprocess.Popen(['gconftool-2', '-g', 
                            '/desktop/gnome/interface/accessibility'], 
                           stdout=subprocess.PIPE)
  rv = gconf.communicate()[0]
  if not rv.startswith('true'):
    gconf = subprocess.Popen(['gconftool-2', '-s', '-t', 'boolean',
                            '/desktop/gnome/interface/accessibility', 'true'])
    raise ValueError(_('Desktop accessibility support not enabled. Enabling...'\
                       '\nPlease logout and run LSR again.'))
    
def parseOptions():
  '''
  Parses the command line arguments based on the definition of the expected
  arguments provided here.
  
  @return: Parsed options
  @rtype: optparse.Values
  '''
  # create a command line option parser
  op = optparse.OptionParser()
  # add all the command line options to the parser
  op.add_option('-k', '--kill', action='callback', callback=getAction,
                help=_('kill a running instance of LSR'), default=kill)
  op.add_option('-g', '--generate', action='callback', callback=getAction,
                help=_('generate a template for a new UIE'), default=generate)
  op.add_option('-i', '--install', action='callback', callback=getAction,
                help=_('install a UIE'), default=install)
  op.add_option('-u', '--uninstall', action='callback', callback=getAction,
                help=_('uninstall a UIE'), default=uninstall)
  op.add_option('-a', '--associate', action='callback', callback=getAction,
                help=_('associate a UIE with a profile'), default=associate)
  op.add_option('-d', '--disassociate', action='callback', callback=getAction,
                help=_('disassociate a UIE from a profile'), 
                default=disassociate)
  op.add_option('-c', '--create-profile', action='callback', callback=getAction,
                help=_('create a new user profile'), default=createProfile)
  op.add_option('-r', '--remove-profile', action='callback', callback=getAction,
                help=_('remove a user profile'), default=removeProfile)
  op.add_option('-s', '--show', action='callback', callback=getAction,
                default=show,
                help=_('show installed perks, profiles, and associated UIEs'))
  op.add_option('-p', '--profile', action='store', dest='profile',
                help=_('name of the profile (required for --create-profile, ' \
                       '--remove-profile, --associate, --disassociate; ' \
                       'optional when running LSR or using --say or --generate'))
  op.add_option('-y', '--say', action='callback', callback=getAction, type='str',
                help=_('output a string using the default device in the given'
                        'profile (defaults to user)'), default=say)
  op.add_option('--name', dest='uie_name', action='store',
                help=_('name of the UIE (required for --generate, --install, '\
                        '--associate, --uninstall, --disassociate'))
  op.add_option('--kind', dest='uie_kind', action='store',
                choices=UIRegistrar.ALL_KINDS,
                help=_('kind of UIE: [%s] (required for --generate, --install,'\
                       '--uninstall, --associate, --disassociate') % \
                       ', '.join(UIRegistrar.ALL_KINDS))
  op.add_option('--path', dest='uie_path', action='store', 
                help=_('path to the UIE (required for --generate, --install)'))
  op.add_option('--when', dest='uie_when', action='store', 
                choices=UIRegistrar.ALL_TIMES,
                help=_('when the UIE is loaded: [%s] (required for '\
                       '--associate, --disassociate, optional for --generate)' % 
                       ', '.join(UIRegistrar.ALL_TIMES)))
  op.add_option('--index', dest='uie_index', action='store', type='int',
                help=_('load order index for the UIE (optional for --generate,'\
                       ' --associate)'))
  op.add_option('--tier', dest='uie_tier', action='store', type='str',
                help=_('name of the Tier for the UIE (required for --when=onetier'))
  op.add_option('-l', '--log-level', dest='log_level', action='store',
                choices=['none', 'debug', 'print', 'info', 'warning', 'error'],
                help=_('level of log messages: ')+' [debug, print, info, ' \
                'warning, error, critical]')
  op.add_option('--log-channel', dest='log_channels', action='append',
                help=_('channel of log messages, any of: ')+'[Perk, Tier, ...]')
  op.add_option('--log-file', dest='log_file', action='store', type='str',
                help=_('filename for the log, defaults to stderr if not '\
                       'specified'))
  (options, args) = op.parse_args()
  return options

def configLogging(options):
  '''
  Configures the logging system based on the setting specified on the command
  line. Defaults to logging exceptions only to stdout.
  
  @param options: Parsed options
  @type options: optparse.Values
  '''  
  if options.log_level is None:
    # default to logging critical messages only
    options.log_level = 'error'
  if options.log_level == 'print':
    # set the log level to print statements if specified
    level = PrintLogger.LEVEL
  else:
    # get the logging constant from the logging module by name; it's an
    # uppercase version of the log_level string provided on the command line
    level = getattr(logging, options.log_level.upper())
  # get the root logger
  root = logging.getLogger()
  # log to a file or to stderr
  if options.log_file:
    handler = logging.FileHandler(options.log_file, 'w')
  else:
    handler = logging.StreamHandler(sys.stderr)
  # set the format and handler for messages
  formatter = logging.Formatter(log_format, time_format)
  root.addHandler(handler)
  root.setLevel(level)
  handler.setFormatter(formatter)
  # add filters to the root handler
  if options.log_channels is not None:
    for name in options.log_channels:
      handler.addFilter(logging.Filter(name))

def main():
  '''
  Print the welcome message, parses command line arguments, and configures the
  logging system. Dispatches control to one of the top-level functions in this
  module depending on the command line parameters specified. 

  If the LSR L{AccessEngine} should be run (i.e. not configured or tested), 
  checks that a11y support is enabled in gconf. If not, enables it but exits
  with an error.
  '''
  # show the welcome message
  welcome()
  # parse the command line options
  options = parseOptions()
  # set up the logging system
  configLogging(options)
  
  try:
    # see if an action has been specified
    mtd = options.action
  except AttributeError:
    # if not, run LSR
    mtd = run

  if mtd is run:
    # make sure accessibility is enabled on the desktop
    checkA11y()
    # run the AccessEngine
    mtd(options)
  else:
    # execute the function for the action
    print mtd(options)

if __name__ == '__main__':
  main()
