'''
Defines L{AccAdapt.Adapter}s for AT-SPI table accessibles. Tables implement the
Table interface but not the Selection interface.

@author: Pete Brunet
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

from POR import POR
from AEEvent import *
from LSRInterfaces import *
from DefaultEventHandler import *
from DefaultNav import *
from pyLinAcc import Constants, Interfaces
import pyLinAcc

class TableNavAdapter(DefaultNavAdapter):
  '''
  Overrides L{DefaultNavAdapter} to provide navigation over table cells as items
  and to avoid traversing cells as separate accessible children in the
  L{IAccessibleNav} interface. Expects the subject to be a L{POR}.
  
  @todo: PP: implement item traversal
  
  Adapts accessibles that provide the L{Interfaces.ITable} interface
  and do not provide the L{Interfaces.ISelection} interface.
  '''
  provides = [IAccessibleNav, IItemNav]
  
  @staticmethod
  def when(subject):
    '''
    Tests if the given subject can be adapted by this class.
    
    @param subject: L{POR} containing an accessible to test
    @type subject: L{POR}
    @return: True when the subject meets the condition named in the docstring
      for this class, False otherwise
    @rtype: boolean
    '''
    acc = subject.accessible
    try:
      Interfaces.ISelection(acc)
      return False
    except NotImplementedError:
      pass
    return Interfaces.ITable(acc)
  
  @pyLinAcc.errorToLookupError 
  def getFirstAccChild(self):
    '''
    Always raises LookupError. Tables have items but no children.
    
    @raise LookupError: Always
    '''
    raise LookupError

  @pyLinAcc.errorToLookupError 
  def getLastAccChild(self):
    '''
    Always raises LookupError. Tables have items but no children.
    
    @raise LookupError: Always
    '''
    raise LookupError

  @pyLinAcc.errorToLookupError  
  def getChildAcc(self, index):
    '''
    Always raises LookupError. Tables have items but no children.
    
    @raise LookupError: Always
    '''
    raise LookupError

class TableEventHandlerAdapter(DefaultEventHandlerAdapter):
  '''
  Overrides L{DefaultEventHandlerAdapter} to fire L{AEEvent.SelectorChange} 
  events on focus and descendant changes for tables. Expects the subject to be
  a L{pyLinAcc.Accessible}.
  
  Adapts subject accessibles that provide the L{Interfaces.ITable}
  interface and do not provide the L{Interfaces.ISelection} interface.
  '''
  @staticmethod
  def when(subject):
    '''
    Tests if the given subject can be adapted by this class.
    
    @param subject: Accessible to test
    @type subject: L{pyLinAcc.Accessible}
    @return: True when the subject meets the condition named in the docstring
    for this class, False otherwise
    @rtype: boolean
    '''
    try:
      Interfaces.ISelection(subject)
      return False
    except NotImplementedError:
      pass
    return Interfaces.ITable(subject) is not None
      
  def _handleFocusEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.FocusChange} indicating that the accessible being
    adapted has gained the focus. Also creates a L{AEEvent.SelectorChange}.
    These two L{AEEvent}s will be posted by the caller.
    
    @param event: Raw focus change event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.FocusChange} and L{AEEvent.SelectorChange}
    @rtype: tuple of L{AEEvent}
    '''
    por = POR(self.subject, None, 0)
    return (FocusChange(por, True, **kwargs), 
            self._getSelectorChange(event, **kwargs))
  
  def _handleDescendantEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.SelectorChange} indicating the "selector" moved in this
    accessible.
    
    @param event: Raw decendent changed event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.SelectorChange}
    @rtype: tuple of L{AEEvent}
    '''
    return (self._getSelectorChange(event, **kwargs),)
  
  def _getSelectorChange(self, event, **kwargs):
    '''
    Creates an L{AEEvent.SelectorChange} indicating the "selector" moved in this
    accessible.
    
    @param event: Raw decendant changed or focus changed event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: Selection event
    @rtype: L{AEEvent.SelectorChange}
    
    @todo: PB: Handle more than the selection of a single cell.
    @todo: PB: If there is no accessible text, call getItem instead of using 
      accessible name?  If so, then need getItem here with code that
      understands table interfaces.
    @todo: PP: how to get selected cell(s)? getSelectedRows and 
      getSelectedColumns don't appear to be the right ones
    '''
    # use the accessible returned with the event if it's available
    if event.any_data is not None:
      acc_cell = event.any_data
      item_offset = event.detail1
    else:
      # otherwise, get the active cell using the table interface
      table = Interfaces.ITable(self.subject)
      #row_list = table.getSelectedRows()
      #ri = row_list[0]
      #col_list = table.getSelectedColumns()
      #ri = col_list[0]
      #acc_cell = table.getAccessibleAt(ri, ci)
      #item_offset = table.getIndexAt(ri, ci)
      #temp kludge until table interfaces work properly
      acc_cell = table.getAccessibleAt(0,0)
      item_offset = table.getIndexAt(0, 0)

    # determine if there is a Text interface
    try:
      text = Interfaces.IText(acc_cell)
      acc_cell_text = acc_cell.name or text.getText(0, text.characterCount)
    except NotImplementedError:
      acc_cell_text = acc_cell.name

    # create a POR, pass it and the item text at the POR to the Tier
    por = POR(self.subject, item_offset, 0)
    return SelectorChange(por, unicode(acc_cell_text, 'utf-8'), **kwargs)