'''
Defines a class responsible for managing the saving and loading of settings
from disk.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import LSRConstants
import shelve, os

# paths to parts of the settings repository
ROOT_PATH = os.path.join(LSRConstants.HOME_USER, 'settings')

class SettingsManager(object):
  '''
  Manages the persistence of L{AEState} objects. Has methods for loading and
  saving state under a given name in the profile under which LSR is running.
  
  @ivar acc_eng: Reference to the L{AccessEngine} that created this manager
  @type acc_eng: L{AccessEngine}
  @ivar states: Hash of persistent L{AEState} objects keyed by their name
  @type states: L{shelve.Shelf}
  '''
  def __init__(self, acc_eng):
    '''
    Stores the reference to the L{AccessEngine}. Opens the shelved data on disk
    in the profile currently used by the L{AccessEngine}.
    
    @param acc_eng: Reference to the L{AccessEngine} that created this manager
    @type acc_eng: L{AccessEngine}
    '''
    self.states = None
    self.acc_eng = acc_eng
    try:
      # make sure there's a folder for this profile
      os.makedirs(ROOT_PATH)
    except OSError:
      pass
    
  def init(self, **kwargs):
    '''
    Opens the saved state from disk. Called by L{AccessEngine} at startup.
    '''
    s = os.path.join(ROOT_PATH, self.acc_eng.getProfile())
    self.states = shelve.open(s, protocol=-1)
    
  def close(self):
    '''
    Closes the handle to the shelved data on disk.
    '''
    self.states.close()    
    
  def saveState(self, name, state):
    '''
    Saves the given L{AEState} object under the given name in the profile 
    currently used by the L{AccessEngine}.
    
    @param name: Name under which to save the L{AEState}
    @type name: string
    @param state: State object to store
    @type state: L{AEState}
    '''
    self.states[name] = state
  
  def loadState(self, name):
    '''    
    Loads the L{AEState} object under the given name in the profile currently
    used by the L{AccessEngine}.
    
    @param name: Name under which the L{AEState} was previously stored
    @type name: string
    @return: State object to load
    @rtype: L{AEState}
    @raise KeyError: When the name is not a valid key
    '''
    return self.states[name]