'''
Contains the L{GSpeech} abstract base class which provides support for all
gnome-speech devices. Device definitions using gnome-speech should derive from
this class and override the L{GSpeech.useThread} and L{GSpeech.getCommandChars}
methods. This module should never be directly registered as its own speech
device with L{UIRegistrar} as the GSpeech class cannot be directly
initialized.

@var MAX_BUFFER: Maximum number of characters to buffer before forcing a 
  L{Devices.IBMSpeech.writeTalk} call.
@type MAX_BUFFER: integer
@var SAY_WAIT_RETRY: Time to wait between calls to say if say fails
@type SAY_WAIT_RETRY: float

@author: Larry Weiss
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import time
import AEOutput
from i18n import _

import ORBit, bonobo
# try to get the typelib
ORBit.load_typelib('GNOME_Speech')
import GNOME.Speech, GNOME__POA.Speech

# constants specific to GSpeech devices
SAY_WAIT_RETRY = 0.01
MAX_BUFFER = 80

class GSpeech(AEOutput.Speech):
  '''
  Defines an abstract base class to send output from LSR to a speech device via
  gnome-speech U{http://cvs.gnome.org/viewcvs/gnome-speech/}.

  To be compliant with LSR requirements, this implements the interface defined 
  in the L{AEOutput.Base} and L{AEOutput.Speech} classes.
  
  @ivar driver: Reference to a speech engine server
  @type driver: GNOME.Speech.SynthesisDriver
  @ivar speaker: Speaker that will synthesize speech from buffered text
  @type speaker: GNOME.Speech.Speaker
  @ivar buffer: Buffer of characters to be sent to the device
  @type buffer: list
  @ivar current_voice: Index of current voice - initialized to -1
  @type current_voice: integer
  @ivar voice_count: Count of available voices - initialized to 0
  @type voice_count: integer
  @cvar DEVICE_IID: Interface identifier for the gnome-speech device. Defaults 
    to None and should be overridden in a subclass.
  @type DEVICE_IID: string
  @cvar _cmd_map: Mapping from L{AEOutput.Constants} commands for speech to 
    string parameter names used by gnome-speech
  @type _cmd_map: dictionary
  '''
  # mapping from LSR constants to gnome-speech command strings
  _cmd_map = {AEOutput.Constants.CONTEXT_VOLUME : 'volume',
              AEOutput.Constants.CONTEXT_RATE : 'rate',
              AEOutput.Constants.CONTEXT_ASPIRATION : 'breathiness',
              AEOutput.Constants.CONTEXT_FRICATION : 'roughness',
              AEOutput.Constants.CONTEXT_INTONATION : 'pitch fluctuation',
              AEOutput.Constants.CONTEXT_PITCH : 'pitch',
              AEOutput.Constants.CONTEXT_HEADSIZE : 'head size',
              AEOutput.Constants.CONTEXT_GENDER : 'gender'
              }  
  
  DEVICE_IID = None
  def __init__(self):
    '''
    Initializes the instance and it's variables, but not the device. Use 
    L{deviceInit} to initialize the device.
    '''
    # allow parent to initialize vars
    AEOutput.Speech.__init__(self)
    self.driver = None
    self.speaker = None
    self.buffer = []
    self.current_voice = -1
    self.voice_count = 0
    
  def _say(self, text):
    '''
    Loops until L{speaker}.say succeeds by returning a non-negative value. 

    @note: gnome-speech returns a negative value when there is a backlog of 
        events that must be processed by a GNOME.Speech.SpeechCallback before
        say will succeed
    @todo: PP: store the return value of say to correlate callback markers
    @param text: Text to say
    @type text: string
    '''
    while 1:
      try:
        text = text.encode('utf-8', 'replace')
        sid = self.speaker.say(text)
      except (TypeError, UnicodeDecodeError), e:
        print e
        break
      if sid >= 0: break
      time.sleep(SAY_WAIT_RETRY)
      
  def _commandToParam(self, cmd):
    '''
    Maps the given integer L{AEOutput.Constants} speech command to a 
    gnome-speech string parameter. Tries to convert the command to a string 
    representing a parameter string itself if it is not a known command.
    
    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: integer
    @raise ValueError: When the command is unknown and cannot be treated as 
      a string parameter in its own right
    '''
    try:
      # look up the command in the command to param dictionary
      param = self._cmd_map[cmd]
    except KeyError:
      # try to treat the command as a string
      try:
        param = str(cmd)
      except TypeError:
        raise ValueError(cmd)
    return param
  
  def _changeVoice(self, value):
    '''
    Creates a new speaker having the given voice index. Destroys the old speaker
    if the new one could be created

    @note: This should be using self.driver.getVoices and giving it a
      GNOME.Speech.VoiceInfo object, but that method appears to always return
      all the available voices regardless of what we ask for with VoiceInfo.
    @param value: Index of the voice to set
    @type value: integer
    @raise ValueError: When the voice index is out of range
    '''
    # get all available voices
    voices = self.driver.getAllVoices()
    try:
      rate = self.getCommandLevel(AEOutput.Constants.CONTEXT_RATE)
    except ValueError:
      rate = None
    try:
      vol = self.getCommandLevel(AEOutput.Constants.CONTEXT_VOLUME)
    except ValueError:
      vol = None
    try:
      speakr = self.driver.createSpeaker(voices[value])
      # if no exception, unref old speaker and save the new
      self.speaker.unref()
    except IndexError:
      raise ValueError(AEOutput.Constants.CONTEXT_VOICE, value)
    self.speaker = speakr
    self.current_voice = value
    # and keep the rate and volume constant
    if rate is not None:
      self.writeCommand(AEOutput.Constants.CONTEXT_RATE, rate)
    if vol is not None:
      self.writeCommand(AEOutput.Constants.CONTEXT_VOLUME, vol)

  def deviceInit(self):
    '''
    Initializes the speech driver through gnome-speech.

    Overrides L{AEOutput.Speech.deviceInit}

    @raise AEOutput.InitError: When the device can not be initialized
    '''
    # initialize CORBA
    ORBit.CORBA.ORB_init()
    # try to activate Festival
    self.driver = bonobo.activation.activate_from_id(self.DEVICE_IID, 0, False)
    try:
      # check if already initialized
      bInit = self.driver.isInitialized()
    except AttributeError:
      # driver is None when the engine is not available
      raise AEOutput.InitError
    # could have been initialized by someone else
    if not bInit:
      if not self.driver.driverInit():
        # driverInit fails when engine is not in a "good state"
        raise AEOutput.InitError
    # no speaker if initialized by someone else
    if not self.speaker:
      # create a default speaker
      voices = self.driver.getAllVoices()
      self.voice_count = len(voices)
      self.speaker = self.driver.createSpeaker(voices[0])
      self.current_voice = 0
      if self.speaker is None:
        # speaker is None when something failed
        raise AEOutput.InitError

  def deviceClose(self):
    '''
    Stops and closes the speech device.

    Overrides L{AEOutput.Speech.deviceClose}
    '''
    self.speaker.stop()
    del self.speaker
    del self.driver

  def writeStop(self):
    '''
    A blocked stop command to the device that returns only when the device is
    ready to receive more data. This does not use L{writeCommand}.

    Overrides L{AEOutput.Speech.writeStop}
    '''
    self.speaker.stop()
    self.buffer = []

  def writeTalk(self):
    '''
    Ensures that received content should be presented to the user. Returns only
    when all buffered content has been sent to the output device. This does not 
    use L{writeCommand} and has no effect when there is no text pending.

    Overrides L{AEOutput.Speech.writeTalk}
    '''
    if len(self.buffer) > 0:
      self._say(''.join(self.buffer))
      self.buffer = []

  def writeString(self, string):
    '''
    Buffers the given character. Ensures the buffered content is presented to 
    the user when the buffer size exceeds L{MAX_BUFFER} and the given character
    is a space.

    Overrides L{AEOutput.Speech.writeString}
    
    @param string: String to buffer
    @type string: string
    '''
    # just buffer the char since gnome-speech will try speaking it right away
    #   actually send the text in writeTalk or a writeCommand
    self.buffer.append(string)
    # do some "dumb" buffer management
    if len(self.buffer) > MAX_BUFFER:
      self.writeTalk()

  def writeCommand(self, cmd, value):
    ''' 
    Sends a command at the given specified value. Returns only once the new
    value has been set.

    Overrides L{AEOutput.Speech.writeCommand}

    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: integer
    @param value: New value to set for the given command
    @type value: object
    @raise ValueError: When the command or value is invalid
    '''
    # send what we have so far at the old value
    self.writeTalk()
    if cmd == AEOutput.Constants.CONTEXT_VOICE:
      # call _changeVoice to handle voice commands
      self._changeVoice(value-1)  # change value to an index
    else:
      param = self._commandToParam(cmd)
      if not self.speaker.setParameterValue(param, value):
        raise ValueError(cmd, value)

  def deviceSpeaking(self):
    '''
    Indicates whether the device is active.

    Overrides L{AEOutput.Speech.deviceSpeaking}

    @return: True when the speech device is synthesizing, False otherwise
    @rtype: boolean
    '''
    return self.speaker.isSpeaking()

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included.

    Overrides L{AEOutput.Base.AEOutput.getName}

    @return: The localized name for the device
    @rtype: string
    '''
    return '%s %s' % (self.driver.synthesizerName,
                      self.driver.synthesizerVersion)

  def getCommandName(self, cmd, value):
    '''
    Gets the user displayable (localized) name for the specified command at
    the specified value.

    Overrides L{AEOutput.Base.AEOutput.getCommandName}

    @todo: PP: add localization support based on the current locale
    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: integer
    @param value: Value of the command to describe
    @type value: object
    @return: Localized name for the command at the specified value
    @rtype: string
    @raise ValueError: When the command is invalid
    '''
    # voice is a special case
    if cmd == AEOutput.Constants.CONTEXT_VOICE:
      voices = self.driver.getAllVoices()
      try:
        return voices[value].name
      except IndexError:
        return ''
    param = self._commandToParam(cmd)
    return self.speaker.getParameterValueDescription(param, value)

  def getCommandLevel(self, cmd):
    '''
    Gets the current value of the specified command.

    Overrides L{AEOutput.Base.AEOutput.getCommandLevel}

    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: integer
    @raise ValueError: When the command is invalid
    '''
    # voice is a special case
    if cmd == AEOutput.Constants.CONTEXT_VOICE:
      return (self.current_voice + 1)  # current_voice is index
    param = self._commandToParam(cmd)
    rv = self.speaker.getParameterValue(param)
    if rv < 0:
      # anything less than zero is an error
      raise ValueError(cmd)
    else:
      # return the value value
      return rv

  def getLevelCount(self, cmd):
    '''
    Gets the maximum settable value for the specified command. A return of 0 or 
    1 indicates that a single value is available.

    Overrides L{AEOutput.Base.AEOutput.getLevelCount}

    @param cmd: L{AEOutput.Constants} representing a speech command
    @type cmd: integer
    @return: Minimum and maximum settable value for the given command
    @rtype: 2-tuple of integer
    @raise ValueError: When the command is invalid
    @see: L{sendCommand}
    '''
    # voice is a special case
    if cmd == AEOutput.Constants.CONTEXT_VOICE:
      return (1, self.voice_count)
    # map the command integer value to the gnome-speech string
    param = self._commandToParam(cmd)
    # find the parameter with the given name
    for sp in self.speaker.getSupportedParameters():
      if sp.name == param:
        # return the minimum and maximum values
        return (sp.min, sp.max)
    raise ValueError
