#!/usr/bin/perl
#
# run-test.pl: runs a number of regresion tests for libgnomeprint
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, 
# Boston, MA 02111-1307, USA. 
#
# Author:
#    Chema Celorio <chema@ximian.com>
#


##
## FIXME: test pdf's with pdf2ps from cups (Chema)
##

my $test_result = "output/test.result";

# Runs a command and returns is retval
# if @quiet = 1, does not emit the error on the console
# if $spacial case = 1, 1 is also a success return code, used by run_gpa_test
sub my_run_command
{
    my ($command, $quiet, $special_case) = (@_);
    my $ret = system ($command) / 256;

    if ($quiet ne 1) {
	 if ($special_case) {
	   if ($ret eq 99) {
		return $ret;
	   }
	 }
	   
	 if ($ret > 0) {
	   print "\n\n*** *** Error *** ***\n\n";
	   print "Command failed \"$command\"\nWith return code: " . $ret . "\n";
	   print "Error message:\n";
	   print "-------------------------------\n";
	   system ("cat " . $test_result);
	   print "-------------------------------\n\n";
	   print "run-test.pl unsucsessfull\n";
	 }
	 
    }
    
    return $ret;
}

sub run_acroread5
{
    my ($pdf_file, $quiet, $output_file_path) = (@_);
    return run_acroread ($pdf_file, $quiet, $output_file_path, 1);
}

# Test if $pdf_file is a good pdf according to acroread
#
# The only way to test from the command line if acroread was able to parse
# the pdf file is to run it and see if it generated error messages since
# it returns 0 even on success
# So what we do is run it and see if $test_result is an empty file or not
sub run_acroread
{
    my ($pdf_file, $quiet, $output_file_path, $version5) = (@_);
    my $ACROREAD_COMMAND;    
    if ($version5) {
	   $ACROREAD_COMMAND = "acroread5 -toPostScript -pairs ";
    } else {
	   $ACROREAD_COMMAND = "acroread4 -toPostScript -pairs ";
    }
    my $output_file;
    if ($output_file_path) {
	   $output_file = $output_file_path;
    } else {
	   $output_file = $pdf_file;
	   $output_file =~ s/\.pdf/\.ps/g;
    }
    my $command = $ACROREAD_COMMAND . $pdf_file . " " . $output_file .  " > " . $test_result . " 2>&1";

    if ($pdf_file eq "") {
	   print "Error while running acroread, missing argument pdf_file\n";
	   return 1;
    }

    $ret = &my_run_command ($command, $quiet);

    my $size = (stat ($test_result))[7];

    if ($size gt 0) {
	   $ret = 1;
	   if ($quiet ne 1) {
		  print "\n\n*** *** Error *** ***\n\n";
		  print "Command failed \"$command\"\nWith return code: " . $ret . "\n";
		  print "Error message:\n";
		  print "-------------------------------\n";
		  system ("cat " . $test_result);
		  print "-------------------------------\n\n";
		  print "run-test.pl unsucsessfull\n";
	   }
    } else {
	   $ret = 0;
    }

    return $ret;
}

# Test if $pdf_file is a good pdf according to pdfcheck
#
sub run_pdfcheck
{
    my ($pdf_file, $quiet) = (@_);
    my $command;

    $command = "tools/pdfcheck $pdf_file";
    $ret = &my_run_command ($command, $quiet);

    return $ret;
}

#
# We test the meta backend by generating a metafile and replaying it with
# generate, then we compare the output generated by the metafile
# with the one previously generated postscript output, the files
# should be identical. We have to clean "%%CreationDate: D:" first
#
sub run_meta
{
    my ($meta_file, $quiet) = (@_);
    my $command;
    my $output_file = $meta_file . ".ps";
    $command  = "./generate --backend=ps";
    $command .= " --replay=" . $meta_file . " ";
    $command .= $output_file . " ";
    $command .= "> " . $test_result . " 2>&1";

    # Call generate, to generate a ps file
    $ret = &my_run_command ($command);
    if ($ret ne 0) {
	   return $ret;
    }

    # Before we compare the files, we need to modify them
    # to remove the CreationDate string, which will differ
    # We use some sed action for that, boy am i lazy ;-)
    $command  = "sed -e 's#%%CreationDate: D:[0-9]*##g' ";
    $command .= " < " . $output_file . " > " . $output_file . ".clean";
    $ret = &my_run_command ($command);
    if ($ret ne 0) {
	   return $ret;
    }
    my $ps_file = $meta_file;
    $ps_file =~ s/meta/ps/;
    $command  = "sed -e 's#%%CreationDate: D:[0-9]*##g' ";
    $command .= " < " . $ps_file . " > " . $ps_file . ".clean";
    $ret = &my_run_command ($command);
    if ($ret ne 0) {
	   return $ret;
    }
    
    # Compare the files originaly created and the meta-rendered ones
    $command = "diff --brief " . $output_file . ".clean " . $ps_file . ".clean ";
    $command .= " > " . $test_result . " 2>&1";
    $ret = &my_run_command ($command);
    if ($ret ne 0) {
	   return $ret;
    }
    
    return $ret;
}

# Test is $ps_file is good according to ghostscript
sub run_gs
{
    my ($ps_file, $quiet) = (@_);
    my $GS_COMMAND = "gs -q -dNOPAUSE -dNODISPLAY -dQUIET ";
    my $command = $GS_COMMAND . $ps_file . " < files/quit.txt > " . $test_result . " 2>&1";

    if ($ps_file eq "") {
	   print "Error while running gs, missing argument ps_file\n";
	   return 1;
    }
    
    return &my_run_command ($command, $quiet);
}

#
# Calls ./generate . composes the command line from the function args
#
sub generate_file
{
    my ($sequence, $backend, $quiet, $output_file) = (@_);
    my $command;
    
    $command  = "./generate --backend=" . $backend;
    $command .= " --sequence=" . $sequence . " ";
    $command .= $output_file . " ";
    $command .= "> " . $test_result . " 2>&1";

    my $ret = &my_run_command ($command, $quiet);

    return $ret;
}

#
# Calls ./gpa-test . composes the command line from the function args
#
sub run_gpa_test
{
    my ($sequence, $quiet) = (@_);
    my $command;
    
    $command  = "./gpa-test ";
    $command .= " --num=" . $sequence . " > " . $test_result . " 2>&1";
    
    my $ret = &my_run_command ($command, $quiet, 1);

    return $ret;
}


#
# Runs ./generate with sequences that should generate an error when run
# if generate does not return an error, the test fails.
# adjust max to match the numbe of tests that generate.c implements
#
sub my_check_errors
{
    my $max = 3;
    my $i = 0;

    print "\nTesting invalid drawing commands ...\n";
    
    while ($i < $max) {
	   my $test_name = "error \#" . $i;
	   my $output_file = "output/error_" . $i . "\.error";
	   my $sequence = sprintf ("error_%02d", $i);
		  
	   print "  Testing \"". $test_name ."\"...";
	   $ret = &generate_file ($sequence, "ps", 1, $output_file);

	   if ($ret eq 0) {
		  print "Error ... command should have failed for $test_name\n";
		  print "We used sequence=" . $sequence . " and backend=ps\n";
		  exit 1;
	   }
	   print "pass\n";
	   $i ++;
    }
}			    


#
# Thes the gpa stuff
#
sub my_check_gpa
{
    my $last = 0;
    my $i = 0;
    my $ret;

    print "\nTesting gpa sequences ...\n";
    
    while (not $last) {

	 print "  Testing sequence #". $i ."...";
	 
	 $ret = run_gpa_test ($i, 0);

	 if ($ret eq 99) {
	   $last = 1;
	   $ret = 0;
	 }

	 if ($ret ne 0) {
	   print "Error.. command failed\n";
	   exit 1;
	 }
	 
	 print "pass\n";
	 $i ++;
    }
}			    

#
# Generates valid output and verifies if it is valid, for example by
# having it beein read by ghostscript or acroread
#
sub my_check_output
{
    my $max = 7;
    my (@tests) = (
		   ["ps",   \&run_gs],
		   ["meta", \&run_meta], # Always run meta before acroread because
			                    # acroread overwrites the .ps files
		   ["pdf",  \&run_acroread,  "acroread4"],
		   ["pdf",  \&run_acroread5, "acroread5"],
		   ["pdf",  \&run_pdfcheck,  "pdfcheck"],
		   ["pdf",  \&run_gs,        "ghostscript"],
    );

    print "\nTesting output generation ...\n";
    
    foreach $test_item (@tests) {
	   for ($i = 0; $i < $max; $i++) {
		  my $output_file = sprintf ("output/out_%02d.%s", $i, $$test_item[0]);
		  my $proc = $$test_item[1];
		  my $test_name;

		  # Compose the test name
		  $test_name = $i . " " . $$test_item[0];
		  if (defined $$test_item[2]) {
			 $test_name .= " on " . $$test_item[2];
		  }
		  
		  # Generate the file that we'll test
		  print "  Testing \"". $test_name ."\"...";
		  my $sequence = $i;
		  my $backend  = $$test_item[0];
		  $ret = &generate_file ($sequence, $backend, 0, $output_file);
		  if ($ret ne 0) {
			 print "Error.. command failed\n";
			 exit 1;
		  }
		  
		  # Run the test on the file
		  if (defined $proc) {
			 if (&$proc ($output_file)) {
				exit 1;
			 } else {
				print "pass\n";
			 }
		  } else {
			 print "  skipped. Test for: $output_file does not have a test function\n";
		  }
	   }
		  
    }
}


#
# Scan thru the list of fonts on the system and test the output generated
# when embeding each font in a ps and pdf job
#
sub my_check_fonts
{
    my $max = system ("./fonts -n") / 256;
    my $i = 0;
    my $font_file = "output/_fontlist.txt";
    my $failed = 0;
    
    system ("./fonts -d > " . $font_file);
    
    open (MY_FILE, $font_file);
    @fonts = <MY_FILE>;
    close (MY_FILE);
	   
    print "\nTesting fonts ... (" . $max . ")\n";

    while ($i < $max) {
	   my $font = $fonts[$i];
	   my $command;
	   my $ret;
	   my $output_file;

	   chomp ($font);

	   print "  Testing font \#" . ($i + 1) . " [" . $font . "] on ps ... ";
	   $output_file = sprintf ("output/font_test_%03d.ps", ($i + 1));
	   $command =  "./fonts --generate=" . ($i + 1);
	   $command .= " --output=" . $output_file . " > " . $test_result;

	   $ret = &my_run_command ($command);
	   if ($ret eq 0) {
		  $ret = run_gs ($output_file, 0);
	   }
	   if ($ret ne 0) {
		  $failed++;
		  print "FAILED\n";
	   } else {
		  print "pass\n";
	   }

	   $output_file = sprintf ("output/font_test_%03d.pdf", ($i + 1));
	   $command =  "./fonts --generate=" . ($i + 1);
	   $command .= " --output=" . $output_file . " --pdf > " . $test_result;
	   $ret = &my_run_command ($command);
	   if ($ret ne 0) {
		  return $ret;
	   }
	   if ($ret eq -5) {
		  # Crashed
		  exit -5;
	   }
	   
	   print "  Testing font \#" . ($i + 1) . " [" . $font . "] on pdf ghostscript... ";
	   $ret = run_gs ($output_file);
	   if ($ret ne 0) {
		  $failed++;
		  print "FAILED\n";
	   } else {
		  print "pass\n";
	   }

	   print "  Testing font \#" . ($i + 1) . " [" . $font . "] on pdf acroread4... ";
	   $ret = run_acroread ($output_file);
	   if ($ret ne 0) {
		  $failed++;
		  print "FAILED\n";
	   } else {
		  print "pass\n";
	   }

	   print "  Testing font \#" . ($i + 1) . " [" . $font . "] on pdf acroread5... ";
	   $ret = run_acroread5 ($output_file);
	   if ($ret ne 0) {
		  $failed++;
		  print "FAILED\n";
	   } else {
		  print "pass\n";
	   }
	   
	   $i++;
    }

    if ($failed ne 0) {
	   print "\n";
	   print "$failed out out of " . ($max * 2). " Font tests failed\n\n";
 	   print "run-test.pl unsucsessfull\n";
	   exit 1;
    }
    
}

#
# Check that the catalog generated is valid
#
sub my_check_catalog
{
    my $output_file;
    my $command;
    my $ret;
    my $failed = 0;
    
    print "\nTesting catalog ...\n";

    print "  Testing catalog on ps ... ";
    $output_file = "output/catalog.ps";
    $command =  "./fonts --catalog ";
    $command .= "--output=" . $output_file . " > " . $test_result;

    $ret = &my_run_command ($command);
    if ($ret eq 0) {
	   $ret = run_gs ($output_file, 0);
    }
    if ($ret ne 0) {
	   $failed++;
	   print "FAILED\n";
    } else {
	   print "pass\n";
    }

    print "  Testing catalog on pdf acroread4... ";
    $output_file = "output/catalog.pdf";
    $command =  "./fonts --catalog ";
    $command .= "--output=" . $output_file . " --pdf > " . $test_result;
    $ret = &my_run_command ($command);
    if ($ret ne 0) {
	   return $ret;
    }
    if ($ret eq -5) {
	   # Crashed
	   exit -5;
    }
	   
    $ret = run_acroread ($output_file);
    if ($ret ne 0) {
	   $failed++;
	   print "FAILED\n";
    } else {
	   print "pass\n";
    }

    print "  Testing catalog on pdf acroread5... ";
    $ret = run_acroread5 ($output_file);
    if ($ret ne 0) {
	   $failed++;
	   print "FAILED\n";
    } else {
	   print "pass\n";
    }
	   
    if ($failed ne 0) {
	   print "\n";
	   print "Catalog test failed\n\n";
 	   print "run-test.pl unsucsessfull\n";
	   exit 1;
    }
    
}

# Check that we can compile a gnome-print app, in this case we
# test generate.c
sub my_compile
{
    print "\nTesting compilation ...";

    my $command;

    $command  = "make clean ";
    $command .= "> " . $test_result . " 2>&1 && ";
    $command .= "make ";
    $command .= "> " . $test_result . " 2>&1";

    my $ret = &my_run_command ($command);
    if ($ret ne 0) {
	   print "failed\n";
	   exit 1;
    }

    $command  = "(cd tools; make clean ";
    $command .= "> ../" . $test_result . " 2>&1 && ";
    $command .= "make ";
    $command .= "> ../" . $test_result . " 2>&1)";

    my $ret = &my_run_command ($command);
    if ($ret ne 0) {
	   print "failed\n";
	   exit 1;
    }
    
    print "pass\n";
    
    return $ret;
}

# We try running sutff that we know is good to check that the test is positive
# and tests known to fail to verify that the error is found.
sub my_sanity_check
{
    my $ret;

    print "\nRunning sanity check... ";
    
    # This are know to be successfull
    $ret = &run_gs ("files/good.ps", 0);
    if ($ret eq 1) {
	   print " failed (on good ps)\n";
	   exit 1;
    } else {
	   print "1";
    }
    $ret = &run_acroread ("files/pdf_good.pdf", 0, "output/pdf_good.pdf.ps");
    if ($ret eq 1) {
	   print " failed (on good pdf)\n";
	   exit 1;
    } else {
	   print "2";
    }

    # This are known to fail
    $ret = &run_gs ("files/bad.ps", 1);
    if ($ret eq 0) {
	   print " failed (on bad ps)\n";
	   exit 1;
    } else {
	   print "3";
    }
    $ret = &run_acroread ("files/pdf_bad.pdf", 1);
    if ($ret eq 0) {
	   print " failed (on bad pdf)\n";
	   exit 1;
    } else {
	   print "4";
    }

    # This should crash, i.e. fail.
    $ret = &run_gpa_test (-1, 1);
    if ($ret eq 0) {
	   print " failed (gpa-test with -1 did not crashed)\n";
	   exit 1;
    } else {
	   print "5";
    }

    # Check that ./fonts work
    $ret = system ("./fonts -n") / 256;
    if ($ret < 3) {
	   print " failed ./fonts -n [with $ret]\n";
	   exit 1;
    } else {
	   print "6";
    }
    
    print " pass\n";
}

# Checks the environment and dependencies
sub my_check_environment
{
    local *FILE;
    my $check_result = "output/check.result";

    # Make the output directory if it doesn't exist
    if (not stat ("output")) {
	   mkdir ("output", 0744);
    }
    
    print "\nChecking environment ...\n";

    # Check gs
    print "  Checking for gs...";
    my $command = "gs -v > " . $check_result . " 2>&1";
    my $ret = &my_run_command ($command, 1);
    if ($ret ne 0) {
	   print ("..failed\n\n");
	   exit 1;
	}
    if (not open (FILE, $check_result)) {
	   print ("..failed\n\n");
	   exit 1;
    }
    my $ver = <FILE>;
    unless ($ver =~ m/Ghostscript [67]/) {
	   print ("..failed    You need ghostscript 6.x or 7.x to run this tests\n\n");
	   exit 1;
    }
    close FILE;
    print "pass\n";

    # Check for acroread4
    # You should link acroread 4.x to a binary named acroread4
    print "  Checking for acroread4 ...";
    my $command = "acroread4 --help > " . $check_result . " 2>&1";
    my $ret = &my_run_command ($command, 1);
    if ($ret ne 0) {
	   print ("..failed\n\n");
	   exit 1;
	}
    if (not open (FILE, $check_result)) {
	   print ("..failed\n\n");
	   exit 1;
    }
    close FILE;
    print "pass\n";
    # Check for acroread5
    # You should link acroread 5.x to a binary named acroread5
    print "  Checking for acroread5 ...";
    my $command = "acroread5 --help > " . $check_result . " 2>&1";
    my $ret = &my_run_command ($command, 1);
    if ($ret ne 0) {
	   print ("..failed\n\n");
	   exit 1;
	}
    if (not open (FILE, $check_result)) {
	   print ("..failed\n\n");
	   exit 1;
    }
    close FILE;
    print "pass\n";

    # Check diff
    print "  Checking for diff...";
    my $command = "diff --version > " . $check_result . " 2>&1";
    my $ret = &my_run_command ($command, 1);
    if ($ret ne 0) {
	   print ("..failed\n\n");
	   exit 1;
	}
    if (not open (FILE, $check_result)) {
	   print ("..failed\n\n");
	   exit 1;
    }
    close FILE;
    print "pass\n";

    # Check sed
    print "  Checking for sed...";
    my $command = "sed --version > " . $check_result . " 2>&1";
    my $ret = &my_run_command ($command, 1);
    if ($ret ne 0) {
	   print ("..failed\n\n");
	   exit 1;
	}
    if (not open (FILE, $check_result)) {
	   print ("..failed\n\n");
	   exit 1;
    }
    close FILE;
    print "pass\n";
}

sub my_usage
{
    print "\n";
    print "Usage: ./run-test.pl [tests]\n";
    print "\n";
    print "Where test can be any set of:\n";
    print "\tall check gpa errors output catalog fonts\n";
    print "\n";
    print "Test \"all\" is ran if no tests are specified on the command line\n";
    print "\n";
    exit 0;
}

sub my_check_all
{
    &my_check_environment ();
    &my_sanity_check ();
    &my_check_gpa ();
    &my_check_errors ();
    &my_check_output ();
    &my_check_catalog ();
    &my_check_fonts ();
}

sub my_clean
{
    system("rm output/*.ps output/*.pdf output/*.result output/*.error > /dev/null 2>&1");
}

my $arg;

&my_clean ();

if (scalar (@ARGV) eq 0) {
    &my_check_all ();
} else {
    while ($arg = shift (@ARGV)) {
	   if ($arg eq "all") {
		  &my_check_all ();
	   }    
	   if ($arg eq "check") {
		  &my_sanity_check ();
	   }
	   if ($arg eq "gpa") {
		  &my_check_gpa ();
	   }
	   if ($arg eq "errors") {
		  &my_check_errors ();
	   }
	   if ($arg eq "output") {
		  &my_check_output ();
	   }
	   if ($arg eq "catalog") {
		  &my_check_catalog ();
	   }
	   if ($arg eq "fonts") {
		  &my_check_fonts ();
	   }
	   if ($arg eq "-h" || $arg eq "--help" || $arg eq "help") {
		  &my_usage ();
	   }
    }
}

print "\nrun-test.pl successful\n\n";

print "add ps2pdf check\n";

exit 0;
