#define __GNOME_RFONT_C__

/*
 *  Copyright (C) 2000-2001 Ximian Inc. and authors
 *
 *  Authors:
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  GnomeFontFace - grid fitted font
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <string.h>
#include <ctype.h>
#include <libgnomeprint/gnome-rfont.h>
#include <libart_lgpl/art_misc.h>
#include <libart_lgpl/art_affine.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_vpath_bpath.h>
#include <libart_lgpl/art_svp_vpath.h>
#include <libart_lgpl/art_rect_svp.h>
#include <libart_lgpl/art_gray_svp.h>
#include <libart_lgpl/art_svp_wind.h>
#include <libgnomeprint/gnome-font-private.h>
#include <libgnomeprint/gnome-pgl-private.h>

#define noVERBOSE

struct _GnomeRFont {
	GObject object;
	GnomeFont *font;
	gdouble transform[6];
	GHashTable * bpaths;
	GHashTable * svps;
	GHashTable * graymaps;
	GHashTable * dimensions;
	GHashTable * displays;
	gchar * gdk_name;
};

struct _GnomeRFontClass {
	GObjectClass parent_class;
};

typedef struct {
	guchar * pixels;
	gint x0, y0;
	gint width, height, rowstride;
} GnomeRFontGrayMap;

typedef struct {
	ArtDRect ddim;
	ArtIRect idim;
} GnomeRFontDimension;

static GHashTable * rfonts = NULL;

static void gnome_rfont_class_init (GnomeRFontClass * klass);
static void gnome_rfont_init (GnomeRFont * rfont);

static void gnome_rfont_finalize (GObject * object);

static const GnomeRFontGrayMap * gnome_rfont_get_glyph_graymap (const GnomeRFont * rfont, gint glyph);
static const GnomeRFontDimension * gnome_rfont_get_glyph_dimension (const GnomeRFont * rfont, gint glyph);

static gboolean rfont_free_dimension (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_graymap (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_svp (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_bpath (gpointer key, gpointer value, gpointer data);

static guint rfont_hash (gconstpointer key);
static gboolean rfont_equal (gconstpointer key1, gconstpointer key2);

static GObjectClass * parent_class;

GType
gnome_rfont_get_type (void)
{
	static GType rfont_type = 0;
	if (!rfont_type) {
		static const GTypeInfo rfont_info = {
			sizeof (GnomeRFontClass),
			NULL, NULL,
			(GClassInitFunc) gnome_rfont_class_init,
			NULL, NULL,
			sizeof (GnomeRFont),
			0,
			(GInstanceInitFunc) gnome_rfont_init
		};
		rfont_type = g_type_register_static (G_TYPE_OBJECT, "GnomeRFont", &rfont_info, 0);
	}
	return rfont_type;
}

static void
gnome_rfont_class_init (GnomeRFontClass * klass)
{
	GObjectClass * object_class;

	object_class = (GObjectClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_rfont_finalize;
}

static void
gnome_rfont_init (GnomeRFont * rfont)
{
	art_affine_identity (rfont->transform);
	rfont->bpaths = g_hash_table_new (NULL, NULL);
	rfont->svps = g_hash_table_new (NULL, NULL);
	rfont->graymaps = g_hash_table_new (NULL, NULL);
	rfont->dimensions = g_hash_table_new (NULL, NULL);
	rfont->displays = g_hash_table_new (NULL, NULL);
}

static void
gnome_rfont_finalize (GObject * object)
{
	GnomeRFont * rfont;

	rfont = (GnomeRFont *) object;

	g_hash_table_remove (rfonts, rfont);

	if (rfont->gdk_name) g_free (rfont->gdk_name);

	if (rfont->dimensions) {
		g_hash_table_foreach_remove (rfont->dimensions, rfont_free_dimension, NULL);
		g_hash_table_destroy (rfont->dimensions);
		rfont->dimensions = NULL;
	}
	if (rfont->graymaps) {
		g_hash_table_foreach_remove (rfont->graymaps, rfont_free_graymap, NULL);
		g_hash_table_destroy (rfont->graymaps);
		rfont->graymaps = NULL;
	}
	if (rfont->svps) {
		g_hash_table_foreach_remove (rfont->svps, rfont_free_svp, NULL);
		g_hash_table_destroy (rfont->svps);
		rfont->svps = NULL;
	}
	if (rfont->bpaths) {
		g_hash_table_foreach_remove (rfont->bpaths, rfont_free_bpath, NULL);
		g_hash_table_destroy (rfont->bpaths);
		rfont->bpaths = NULL;
	}

	if (rfont->font) {
		g_object_unref (G_OBJECT (rfont->font));
		rfont->font = NULL;
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GnomeRFont *
gnome_font_get_rfont (GnomeFont * font, const gdouble * transform)
{
	GnomeRFont search;
	GnomeRFont * rfont;

	g_return_val_if_fail (font != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_FONT (font), NULL);
	g_return_val_if_fail (transform != NULL, NULL);

	if (rfonts == NULL) {
		rfonts = g_hash_table_new (rfont_hash, rfont_equal);
	}

	search.font = font;
	memcpy (search.transform, transform, 4 * sizeof (gdouble));
	search.transform[4] = search.transform[5] = 0.0;

	rfont = g_hash_table_lookup (rfonts, &search);

	if (rfont != NULL) {
#ifdef VERBOSE
		g_print ("found cached rfont %s\n", gnome_font_get_name (font));
#endif
		gnome_rfont_ref (rfont);
		return rfont;
	}

	rfont = g_object_new (GNOME_TYPE_RFONT, NULL);

	rfont->font = font;
	g_object_ref (G_OBJECT (font));
	memcpy (rfont->transform, transform, 4 * sizeof (gdouble));
	rfont->transform[4] = rfont->transform[5] = 0.0;

	g_hash_table_insert (rfonts, rfont, rfont);

	return rfont;
}

GnomeFont *
gnome_rfont_get_font (const GnomeRFont * rfont)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	return rfont->font;
}

GnomeFontFace *
gnome_rfont_get_face (const GnomeRFont * rfont)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	return rfont->font->face;
}

gdouble *
gnome_rfont_get_matrix (const GnomeRFont * rfont, gdouble * matrix)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);
	g_return_val_if_fail (matrix != NULL, NULL);

	memcpy (matrix, rfont->transform, 4 * sizeof (gdouble));

	return matrix;
}

ArtPoint *
gnome_rfont_get_glyph_stdadvance (GnomeRFont * rfont, gint glyph, ArtPoint * advance)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	gnome_font_get_glyph_stdadvance (rfont->font, glyph, advance);
	art_affine_point (advance, advance, rfont->transform);

	return advance;
}

ArtDRect *
gnome_rfont_get_glyph_stdbbox (GnomeRFont * rfont, gint glyph, ArtDRect * bbox)
{
	const GnomeRFontDimension * dim;

	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	dim = gnome_rfont_get_glyph_dimension (rfont, glyph);

	* bbox = dim->ddim;

	return bbox;
}

ArtPoint *
gnome_rfont_get_glyph_stdkerning (GnomeRFont * rfont, gint glyph0, gint glyph1, ArtPoint * kerning)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);
	g_return_val_if_fail (glyph0 > 0, NULL);
	g_return_val_if_fail (glyph1 > 0, NULL);
	g_return_val_if_fail (kerning != NULL, NULL);

	if (!gnome_font_get_glyph_stdkerning (rfont->font, glyph0, glyph1, kerning)) {
		g_warning ("file %s: line %d: Font stdkerning failed", __FILE__, __LINE__);
		return NULL;
	}

	art_affine_point (kerning, kerning, rfont->transform);

	return kerning;
}

const ArtBpath *
gnome_rfont_get_glyph_bpath (GnomeRFont * rfont, gint glyph)
{
	ArtBpath * bpath;
	gdouble affine[6];
	gdouble size;

	bpath = g_hash_table_lookup (rfont->bpaths, GINT_TO_POINTER (glyph));

	if (bpath) return bpath;

	size = gnome_font_get_size (rfont->font);
	affine[0] = rfont->transform[0] * size * 0.001;
	affine[1] = rfont->transform[1] * size * 0.001;
	affine[2] = rfont->transform[2] * size * 0.001;
	affine[3] = rfont->transform[3] * size * 0.001;
	affine[4] = affine[5] = 0.0;

	bpath = (ArtBpath *) gnome_font_face_get_glyph_stdoutline (rfont->font->face, glyph);

	g_return_val_if_fail (bpath != NULL, NULL);

	bpath = art_bpath_affine_transform (bpath, affine);

	g_hash_table_insert (rfont->bpaths, GINT_TO_POINTER (glyph), bpath);

	return bpath;
}

const ArtSVP *
gnome_rfont_get_glyph_svp (GnomeRFont * rfont, gint glyph)
{
	ArtSVP * svp, * svp1;
	ArtBpath * bpath;
	ArtVpath * vpath, * vpath1;

	svp = g_hash_table_lookup (rfont->svps, GINT_TO_POINTER (glyph));

	if (svp) return svp;

	/* fixme: */

	bpath = (ArtBpath *) gnome_rfont_get_glyph_bpath (rfont, glyph);

	g_return_val_if_fail (bpath != NULL, NULL);

	vpath = art_bez_path_to_vec (bpath, 0.25);
	vpath1 = art_vpath_perturb (vpath);
	art_free (vpath);
	svp = art_svp_from_vpath (vpath1);
	art_free (vpath1);
	svp1 = art_svp_uncross (svp);
	art_svp_free (svp);
	svp = art_svp_rewind_uncrossed (svp1, ART_WIND_RULE_ODDEVEN);
	art_svp_free (svp1);

	g_hash_table_insert (rfont->svps, GINT_TO_POINTER (glyph), svp);

	return svp;
}

static const GnomeRFontGrayMap *
gnome_rfont_get_glyph_graymap (const GnomeRFont * rfont, gint glyph)
{
	GnomeRFontGrayMap * gmap;
	ArtSVP * svp;
	ArtDRect bbox;
	ArtIRect ibox;

	gmap = g_hash_table_lookup (rfont->graymaps, GINT_TO_POINTER (glyph));

	if (gmap) return gmap;

	/* fixme: */

	svp = (ArtSVP *) gnome_rfont_get_glyph_svp (rfont, glyph);
	art_drect_svp (&bbox, svp);
	art_drect_to_irect (&ibox, &bbox);

	gmap = g_new (GnomeRFontGrayMap, 1);
	gmap->width = gmap->rowstride = ibox.x1 - ibox.x0;
	gmap->height = ibox.y1 - ibox.y0;
	gmap->x0 = ibox.x0;
	gmap->y0 = ibox.y0;
	gmap->pixels = g_new0 (guchar, gmap->width * gmap->height);

	art_gray_svp_aa (svp,
		ibox.x0, ibox.y0, ibox.x1, ibox.y1,
		gmap->pixels, gmap->rowstride);

	g_hash_table_insert (rfont->graymaps, GINT_TO_POINTER (glyph), gmap);

	return gmap;
}

static const GnomeRFontDimension *
gnome_rfont_get_glyph_dimension (const GnomeRFont * rfont, gint glyph)
{
	GnomeRFontDimension * dim;
	ArtSVP * svp;

	dim = g_hash_table_lookup (rfont->dimensions, GINT_TO_POINTER (glyph));

	if (dim) return dim;

	/* fixme: */

	svp = (ArtSVP *) gnome_rfont_get_glyph_svp (rfont, glyph);

	dim = g_new (GnomeRFontDimension, 1);

	art_drect_svp (&dim->ddim, svp);
	art_drect_to_irect (&dim->idim, &dim->ddim);

	g_hash_table_insert (rfont->dimensions, GINT_TO_POINTER (glyph), dim);

	return dim;
}

void
gnome_rfont_render_glyph_rgba8 (GnomeRFont * rfont, gint glyph,
				guint32 rgba,
				gdouble x, gdouble y,
				guchar * buf,
				gint width, gint height, gint rowstride,
				guint flags)
{
	const GnomeRFontGrayMap * gmap;
	gint xp, yp, x0, y0, x1, y1;
	guint inkr, inkg, inkb, inka;
	guchar * s, * s0, * d, * d0;

	gmap = gnome_rfont_get_glyph_graymap (rfont, glyph);

	xp = (gint) floor (x + 0.5);
	yp = (gint) floor (y + 0.5);

	x0 = MAX (0, xp + gmap->x0);
	y0 = MAX (0, yp + gmap->y0);
	x1 = MIN (width, xp + gmap->x0 + gmap->width);
	y1 = MIN (height, yp + gmap->y0 + gmap->height);

	inkr = rgba >> 24;
	inkg = (rgba >> 16) & 0xff;
	inkb = (rgba >> 8) & 0xff;
	inka = rgba & 0xff;

	d0 = d = buf + y0 * rowstride + x0 * 4;
	s0 = s = gmap->pixels + (y0 - yp - gmap->y0) * gmap->rowstride + (x0 - xp - gmap->x0);

	for (y = y0; y < y1; y++) {
		for (x = x0; x < x1; x++) {
			guint bgr, bgg, bgb, bga;
			guint pixr, pixg, pixb;
			guint fgr, fgg, fgb;
			guint alpha;

			bgr = * (d + 0);
			bgg = * (d + 1);
			bgb = * (d + 2);
			bga = * (d + 3);

			alpha = (inka * (*s++) + 0x80) >> 8;

			pixr = (bgr * bga + 0x80) >> 8;
			pixg = (bgg * bga + 0x80) >> 8;
			pixb = (bgb * bga + 0x80) >> 8;

			fgr = ((inkr - pixr) * alpha + 0x80) >> 8;
			fgg = ((inkg - pixg) * alpha + 0x80) >> 8;
			fgb = ((inkb - pixb) * alpha + 0x80) >> 8;

			*d++ = pixr + fgr;
			*d++ = pixg + fgg;
			*d++ = pixb + fgb;
			*d++ = bga + (((0xff - bga) * alpha + 0x80) >> 8);
		}

		s = s0 += gmap->rowstride;
		d = d0 += rowstride;
	}
}

void
gnome_rfont_render_glyph_rgb8 (GnomeRFont * rfont, gint glyph,
				guint32 rgba,
				gdouble x, gdouble y,
				guchar * buf,
				gint width, gint height, gint rowstride,
				guint flags)
{
	const GnomeRFontGrayMap * gmap;
	gint xp, yp, x0, y0, x1, y1;
	guint inkr, inkg, inkb, inka;
	guchar * s, * s0, * d, * d0;

	gmap = gnome_rfont_get_glyph_graymap (rfont, glyph);

	xp = (gint) floor (x + 0.5);
	yp = (gint) floor (y + 0.5);

	x0 = MAX (0, xp + gmap->x0);
	y0 = MAX (0, yp + gmap->y0);
	x1 = MIN (width, xp + gmap->x0 + gmap->width);
	y1 = MIN (height, yp + gmap->y0 + gmap->height);

	inkr = rgba >> 24;
	inkg = (rgba >> 16) & 0xff;
	inkb = (rgba >> 8) & 0xff;
	inka = rgba & 0xff;

	d0 = d = buf + y0 * rowstride + x0 * 3;
	s0 = s = gmap->pixels + (y0 - yp - gmap->y0) * gmap->rowstride + (x0 - xp - gmap->x0);

	for (y = y0; y < y1; y++) {
		for (x = x0; x < x1; x++) {
			guint bgr, bgg, bgb;
			guint fgr, fgg, fgb;
			guint alpha;

			bgr = * (d + 0);
			bgg = * (d + 1);
			bgb = * (d + 2);
			alpha = (inka * (*s++) + 0x80) >> 8;

			fgr = (inkr * alpha + 0x80) >> 8;
			fgg = (inkg * alpha + 0x80) >> 8;
			fgb = (inkb * alpha + 0x80) >> 8;

			*d++ = ((bgr * (0xff - alpha) + 0x80) >> 8) + fgr;
			*d++ = ((bgg * (0xff - alpha) + 0x80) >> 8) + fgg;
			*d++ = ((bgb * (0xff - alpha) + 0x80) >> 8) + fgb;
		}

		s = s0 += gmap->rowstride;
		d = d0 += rowstride;
	}
}

/*
 * These are somewhat tricky, as you cannot do arbotrarily transformed
 * fonts with Pango. So be cautious and try to figure out the best
 * solution.
 */

PangoFont *
gnome_rfont_get_closest_pango_font (const GnomeRFont *rfont, PangoFontMap *map)
{
	gdouble dx, dy, dpi;

	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);
	g_return_val_if_fail (map != NULL, NULL);
	g_return_val_if_fail (PANGO_IS_FONT_MAP (map), NULL);

	dx = rfont->transform[2] - rfont->transform[0];
	dx = dx * dx;
	dy = rfont->transform[1] - rfont->transform[3];
	dy = dy * dy;

	dpi = sqrt (dx * dy * 0.5);

	return gnome_font_get_closest_pango_font (rfont->font, map, dpi);
}

PangoFontDescription *
gnome_rfont_get_pango_description (const GnomeRFont *rfont)
{
	gdouble dx, dy, dpi;

	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	dx = rfont->transform[2] - rfont->transform[0];
	dx = dx * dx;
	dy = rfont->transform[1] - rfont->transform[3];
	dy = dy * dy;

	dpi = sqrt (dx * dy * 0.5);

	return gnome_font_get_pango_description (rfont->font, dpi);
}

/* Helpers */

static gboolean
rfont_free_dimension (gpointer key, gpointer value, gpointer data)
{
	g_free (value);

	return TRUE;
}

static gboolean
rfont_free_graymap (gpointer key, gpointer value, gpointer data)
{
	GnomeRFontGrayMap * gmap;

	gmap = (GnomeRFontGrayMap *) value;

	if (gmap->pixels) g_free (gmap->pixels);
	g_free (gmap);

	return TRUE;
}

static gboolean
rfont_free_svp (gpointer key, gpointer value, gpointer data)
{
	art_svp_free ((ArtSVP *) value);

	return TRUE;
}

static gboolean
rfont_free_bpath (gpointer key, gpointer value, gpointer data)
{
	art_free (value);

	return TRUE;
}

static guint
rfont_hash (gconstpointer key)
{
	GnomeRFont * rfont;

	rfont = (GnomeRFont *) key;

	return (guint) rfont->font;
}

static gint
rfont_equal (gconstpointer key1, gconstpointer key2)
{
	GnomeRFont * f1, * f2;

	f1 = (GnomeRFont *) key1;
	f2 = (GnomeRFont *) key2;

	if (f1->font != f2->font) return FALSE;

	return art_affine_equal (f1->transform, f2->transform);
}

