#define __GNOME_PRINT_MASTER_C__

/*
 *  Copyright (C) 2000 Ximian Inc.
 *
 *  Authors:
 *    Michael Zucchi <notzed@ximian.com>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  A system print interface.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libgnomeprint/gnome-print-meta.h>
#include "gnome-print-master-private.h"

static void gnome_print_master_class_init (GnomePrintMasterClass *klass);
static void gnome_print_master_init (GnomePrintMaster *gpm);
static void gnome_print_master_finalize (GObject *object);

static GObjectClass *parent_class;

guint
gnome_print_master_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomePrintMasterClass),
			NULL, NULL,
			(GClassInitFunc) gnome_print_master_class_init,
			NULL, NULL,
			sizeof (GnomePrintMaster),
			0,
			(GInstanceInitFunc) gnome_print_master_init
		};
		type = g_type_register_static (G_TYPE_OBJECT, "GnomePrintMaster", &info, 0);
	}
	return type;
}

static void
gnome_print_master_class_init (GnomePrintMasterClass *klass)
{
	GObjectClass *object_class;
	
	object_class = (GObjectClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_print_master_finalize;
}

static void
gnome_print_master_init (GnomePrintMaster *gpm)
{
	gpm->config = NULL;

	gpm->meta = gnome_print_meta_new ();
}

static void
gnome_print_master_finalize (GObject *object)
{
	GnomePrintMaster *gpm;

	gpm = GNOME_PRINT_MASTER(object);

	if (gpm->config) {
		gnome_print_config_unref (gpm->config);
	}

	if (gpm->meta != NULL) {
		g_object_unref (G_OBJECT (gpm->meta));
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

/**
 * gnome_print_master_new:
 * 
 * Create a new GnomePrintMaster.  All values are initialised
 * to sensible defaults.
 * 
 * Return value: A new GnomePrintMaster.
 **/

GnomePrintMaster *
gnome_print_master_new (void)
{
	GnomePrintMaster *gpm;

	gpm = g_object_new (GNOME_TYPE_PRINT_MASTER, NULL);

	gpm->config = gnome_print_config_default ();

	return gpm;
}

/**
 * gnome_print_master_new_from_dialog:
 * @dialog: %A GnomePrintDialog
 * 
 * Create a new GnomePrintMaster based on the values in the
 * %GnomePrintDialog.  Range values are initialised to sensible
 * defaults.  Other values are initialised from the given dialog.
 * 
 * Return value: A new GnomePrintMaster.
 **/

GnomePrintMaster *
gnome_print_master_new_from_config (GnomePrintConfig *config)
{
	GnomePrintMaster *gpm;
	
	g_return_val_if_fail (config != NULL, NULL);

	gpm = g_object_new (GNOME_TYPE_PRINT_MASTER, NULL);

	gpm->config = gnome_print_config_ref (config);

	return gpm;
}

GnomePrintConfig *
gnome_print_master_get_config (GnomePrintMaster *gpm)
{
	g_return_val_if_fail (gpm != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), NULL);

	if (gpm->config) gnome_print_config_ref (gpm->config);

	return gpm->config;
}

/**
 * gnome_print_master_get_context:
 * @gpm: An initialised GnomePrintMaster.
 * 
 * Retrieve the GnomePrintContext which applications
 * print to.
 * 
 * Return value: The printing context.
 **/

GnomePrintContext *
gnome_print_master_get_context (GnomePrintMaster *gpm)
{
	g_return_val_if_fail (gpm != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), NULL);

	if (gpm->meta) g_object_ref (G_OBJECT (gpm->meta));

	return gpm->meta;
}

/**
 * gnome_print_master_get_pages:
 * @gpm: An initialised and closed GnomePrintMaster.
 * 
 * Find the number of pages stored in a completed printout.
 * 
 * Return value: If @gpm has not been closed using
 * gnome_print_master_close(), then 0, otherwise the number
 * of pages created by the application.
 **/

int
gnome_print_master_get_pages (GnomePrintMaster *gpm)
{
	g_return_val_if_fail (gpm != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), 0);

	return gnome_print_meta_get_pages (GNOME_PRINT_META (gpm->meta));
}

/**
 * gnome_print_master_set_paper:
 * @gpm: An initialised GnomePrintMaster.
 * @paper: Paper size required.
 * 
 * Set the paper size for the print.
 **/

gint
gnome_print_master_set_paper (GnomePrintMaster *gpm, const guchar *paper)
{
	gboolean ret;

	g_return_val_if_fail (gpm != NULL, GNOME_PRINT_ERROR_UNKNOWN);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), GNOME_PRINT_ERROR_UNKNOWN);
	g_return_val_if_fail (paper != NULL, GNOME_PRINT_ERROR_BADVALUE);

	ret = gnome_print_config_set (gpm->config, GNOME_PRINT_KEY_PAPER_SIZE, paper);

	return ret ? GNOME_PRINT_OK : GNOME_PRINT_ERROR_BADVALUE;
}

const guchar *
gnome_print_master_get_paper (const GnomePrintMaster *gpm)
{
	g_return_val_if_fail (gpm != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), NULL);

	/* fixme: memleak */
	return gnome_print_config_get (gpm->config, GNOME_PRINT_KEY_PAPER_SIZE);
}

/**
 * gnome_print_master_set_printer:
 * @gpm: An initialised GnomePrintMaster.
 * @printer: Printer device to use.
 * 
 * Sets physical printer used to print to.  If this is not set,
 * then a dialogue will be presented on each print.
 *
 * This functions takes ownership of the printer reference.
 **/

gint
gnome_print_master_set_printer (GnomePrintMaster *gpm, const guchar *printer)
{
	gboolean ret;

	g_return_val_if_fail (gpm != NULL, GNOME_PRINT_ERROR_UNKNOWN);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), GNOME_PRINT_ERROR_UNKNOWN);
	g_return_val_if_fail (printer != NULL, GNOME_PRINT_ERROR_BADVALUE);

	ret = gnome_print_config_set (gpm->config, "Printer", printer);

	return ret ? GNOME_PRINT_OK : GNOME_PRINT_ERROR_BADVALUE;
}

/**
 * gnome_print_master_set_copies:
 * @gpm: A GnomePrintContext.
 * @copies: Number of copies of each page to print.
 * @iscollate: Whether page copies are collated.
 * 
 * Set the number of copies to print, and whether pages are collated.
 * If @iscollate is set, then a multiple copy print to a physical
 * printer will order pages as "1,2,3,...1,2,3,..." otherwise they
 * will be ordered "1,1,...,2,2...".
 **/

gint
gnome_print_master_set_copies (GnomePrintMaster *gpm, gint copies, gboolean collate)
{
	gboolean ret;

	g_return_val_if_fail (gpm != NULL, GNOME_PRINT_ERROR_UNKNOWN);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), GNOME_PRINT_ERROR_UNKNOWN);

	/* fixme: key */
	ret = gnome_print_config_set_int (gpm->config, GNOME_PRINT_KEY_NUM_COPIES, copies);
	g_return_val_if_fail (ret, GNOME_PRINT_ERROR_UNKNOWN);
	/* fixme: key */
	ret = gnome_print_config_set_boolean (gpm->config, GNOME_PRINT_KEY_COLLATE, collate);
	g_return_val_if_fail (ret, GNOME_PRINT_ERROR_UNKNOWN);

	return GNOME_PRINT_OK;
}

/**
 * gnome_print_master_close:
 * @gpm: A GnomePrintMaster which has had printing performed
 * on it.
 * 
 * Closes the GnomePrintMaster @gpm, ready for printing
 * or previewing.
 **/

gint
gnome_print_master_close (GnomePrintMaster *gpm)
{
	g_return_val_if_fail (gpm != NULL, GNOME_PRINT_ERROR_UNKNOWN);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), GNOME_PRINT_ERROR_UNKNOWN);

	return gnome_print_context_close (gpm->meta);
}

#if 0
/* quick show preview window */
int
gnome_print_master_preview(GnomePrintMaster *gpm, char *title)
{
	char *text;
	GnomePrintMasterPreview *pmp;

	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), -1);

	pmp = gnome_print_master_preview_new(gpm, title);
	gtk_widget_show(pmp);

	return 0;
}
#endif

/**
 * gnome_print_master_print:
 * @gpm: A completed GnomePrintMaster.
 * 
 * Print the pages stored in the GnomePrintMaster to
 * the phyisical printing device.
 *
 * If no printer has been set, then a dialogue is presented,
 * asking the user for the printer to print to.
 * 
 * Return value: Returns -1 on error.
 **/

gint
gnome_print_master_print (GnomePrintMaster *gpm)
{
	GnomePrintContext *ctx;
	gint copies, nstacks, npages, nsheets;
	gboolean collate;
	gint s;
	const guchar *buf;
	gint blen;
	gint ret;

	g_return_val_if_fail (gpm != NULL, GNOME_PRINT_ERROR_UNKNOWN);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), GNOME_PRINT_ERROR_UNKNOWN);

	ctx = gnome_print_context_new (gpm->config);
	g_return_val_if_fail (ctx != NULL, GNOME_PRINT_ERROR_UNKNOWN);

	collate = FALSE;
	gnome_print_config_get_boolean (gpm->config, GNOME_PRINT_KEY_COLLATE, &collate);
	copies = 1;
	gnome_print_config_get_int (gpm->config, GNOME_PRINT_KEY_NUM_COPIES, &copies);

	if (collate) {
		nstacks = copies;
		nsheets = 1;
	} else {
		nstacks = 1;
		nsheets = copies;
	}
	
	npages = gnome_print_master_get_pages (gpm);
	buf = gnome_print_meta_get_buffer (GNOME_PRINT_META (gpm->meta));
	blen = gnome_print_meta_get_length (GNOME_PRINT_META (gpm->meta));

	for (s = 0; s < nstacks; s++) {
		gint p;
		for (p = 0; p < npages; p++) {
			gint i;
			for (i = 0; i < nsheets; i++) {
				ret = gnome_print_meta_render_data_page (ctx, buf, blen, p, TRUE);
				g_return_val_if_fail (ret == GNOME_PRINT_OK, ret);
			}
		}
	}

	ret = gnome_print_context_close (ctx);
	g_object_unref (G_OBJECT (ctx));

	return ret;
}
