/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "examplewindow.h" 

#include <libgdamm.h>
#include <iostream>

#ifdef GLIBMM_EXCEPTIONS_ENABLED
void ExampleWindow::create_grid(const Glib::RefPtr<Gnome::Gda::Dict>& dict)
#else
void ExampleWindow::create_grid(const Glib::RefPtr<Gnome::Gda::Dict>& dict, std::auto_ptr<Glib::Error>& error)
#endif // GLIBMM_EXCEPTIONS_ENABLED
{
  // Select relevant order fields from database
  Glib::RefPtr<Gnome::Gda::Query> query = Gnome::Gda::Query::create(dict);
#ifdef GLIBMM_EXCEPTIONS_ENABLED
  query->set_sql_text("SELECT id, customer, creation_date FROM orders");
#else
  query->set_sql_text("SELECT id, customer, creation_date FROM orders", error);
  if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED
  m_model = Gnome::Gda::DataModelQuery::create(query);

  // This selects fields from the customers table to restrict the customer
  // field in the datamodel above to the values that are in the customers
  // table. We also select name to show the user the name of the customer
  // in the grid instead of just its ID (as it would appear in the orders
  // table).
  Glib::RefPtr<Gnome::Gda::Query> restr_query = Gnome::Gda::Query::create(dict);

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  restr_query->set_sql_text("SELECT id, name FROM customers");
#else
  restr_query->set_sql_text("SELECT id, name FROM customers", error);
  if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED

  Glib::RefPtr<Gnome::Gda::DataModel> restr = Gnome::Gda::DataModelQuery::create(restr_query);

  const Glib::ustring update_query = 
                 "UPDATE orders set "
                 "customer=##/*name:'+1' type:gint*/ "
                 "WHERE id=##/*name:'-0' type:gint*/";

  const Glib::ustring delete_query =
                 "DELETE FROM orders WHERE id=##/*name:'-0' type:gint*/";

  const Glib::ustring insert_query =
                 "INSERT INTO orders (customer) "
                 "VALUES (##/*name:'+1' type:gint*/)";

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  m_model->set_modification_query(update_query);
  m_model->set_modification_query(delete_query);
  m_model->set_modification_query(insert_query);
#else
  m_model->set_modification_query(update_query, error);
  m_model->set_modification_query(delete_query, error);
  m_model->set_modification_query(insert_query, error);
  if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED

  m_grid = Gtk::manage(new Gnome::Db::Grid(m_model));
  Glib::RefPtr<Gnome::Gda::DataModelIter> iter = m_grid->get_raw_grid()->get_current_data();

  // This function first looks up the parameter by its field name "customer"
  // and then restricts the values it can hold to those in the first
  // column (index 0) of the customers table.
  //
  // There is also some gda magic involved so that the grid actually
  // shows the name of the customer of the ID. I am not absolutely sure
  // how this works.
#ifdef GLIBMM_EXCEPTIONS_ENABLED
  iter->restrict_param_by_name("customer", restr, 0);
#else
  iter->restrict_param_by_name("customer", restr, 0, error);
#endif // GLIBMM_EXCEPTIONS_ENABLED
}

ExampleWindow::ExampleWindow(const Glib::RefPtr<Gnome::Gda::Dict>& dict)
{    
  m_box.set_border_width(6);
  m_box.pack_start(m_label, Gtk::PACK_SHRINK);

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    create_grid(dict);
  }
  catch(const Glib::Error& err)
  {
    std::cerr << "Exception caught: " << err.what() << std::endl;
    exit(1);
  }
#else
  std::auto_ptr<Glib::Error> error;
  create_grid(dict, error);
  if(error.get() != NULL)
  {
    std::cerr << "Exception caught: " << error->what() << std::endl;
    exit(1);
  }
#endif

  m_box.pack_start(*m_grid);
  add(m_box);
  set_default_size(400, 400);
  show_all();
}
