/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include <libgnomedbmm.h>
#include <libgdamm.h>

#include <iostream>

#ifdef GLIBMM_EXCEPTIONS_ENABLED
void do_example()
#else
void do_example(std::auto_ptr<Glib::Error>& error)
#endif // GLIBMM_EXCEPTIONS_ENABLED
{
  Glib::RefPtr<Gnome::Gda::Dict> dict = Gnome::Gda::Dict::create();
  Glib::RefPtr<Gnome::Gda::Client> client = Gnome::Gda::Client::create();
  Glib::ustring connection_string = "DB_DIR=" LIBGNOMEDB_DATADIR ";DB_NAME=demo_db";

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  Glib::RefPtr<Gnome::Gda::Connection> cnc = client->open_connection_from_string("SQLite", connection_string, "" /* username */, "" /* password */);
  dict->set_connection(cnc);
  dict->update_dbms_meta_data();
#else
  Glib::RefPtr<Gnome::Gda::Connection> cnc = client->open_connection_from_string("SQLite", connection_string, "", "", Gnome::Gda::ConnectionOptions(0), error);
  if(error.get() != NULL) return;

  dict->set_connection(cnc);
  dict->update_dbms_meta_data(error);
  if(error.get() != NULL) return;
#endif // GLIBMM_EXCEPTIONS_ENABLED

  // Specifies that this is an UPDATE query.
  Glib::RefPtr<Gnome::Gda::Query> query = Gnome::Gda::Query::create(dict, Gnome::Gda::QUERY_TYPE_UPDATE);

  // Update queries may only have one target.
  Glib::RefPtr<Gnome::Gda::QueryTarget> products = Gnome::Gda::QueryTarget::create(query, "products");

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  query->add_target(products);
#else
  query->add_target(products, error);
  if(error.get() != NULL) return;
#endif

  // Create a condition that tests two fields for equality, to not update
  // each row in the table to the same values.
  Glib::RefPtr<Gnome::Gda::QueryCondition> condition = Gnome::Gda::QueryCondition::create(query, Gnome::Gda::QUERY_CONDITION_LEAF_EQUAL);
  // Left operand is the ref field in the table
  condition->leaf_set_operator(Gnome::Gda::QUERY_CONDITION_OP_LEFT, Gnome::Gda::QueryFieldField::create(query, "ref", products));
  // Right operand is the constant '42'. This way, all rows whose ref field
  // contains 42 will be deleted.
  condition->leaf_set_operator(Gnome::Gda::QUERY_CONDITION_OP_RIGHT, Gnome::Gda::QueryFieldValue::create(query, Gnome::Gda::Value(42)));

  // Associate the condition to the query
  query->set_condition(condition);

  // Changes the price and wh_stored fields of the rows matching the above
  // condition to the specified values.
  query->add_field(Gnome::Gda::QueryFieldField::create(query, "price", products, Gnome::Gda::QueryFieldValue::create(query, Gnome::Gda::Value(21.99))));
  query->add_field(Gnome::Gda::QueryFieldField::create(query, "wh_stored", products, Gnome::Gda::QueryFieldValue::create(query, Gnome::Gda::Value(5))));

  // Prints "UPDATE products SET price=21.990000, wh_stored=5 WHERE ref=42"
  std::cout << query->get_sql_text() << std::endl;
}

int main(int argc, char* argv[])
{
  Gnome::Db::init("Queries example", "1.0", argc, argv);

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    do_example();
  }
  catch(const Glib::Error& err)
  {
    std::cerr << "Exception caught: " << err.what() << std::endl;
    return 1;
  }
#else
  std::auto_ptr<Glib::Error> error;
  do_example(error);
  if(error.get() != NULL)
  {
    std::cerr << "Exception caught: " << error->what() << std::endl;
    return 1;
  }
#endif // GLIBMM_EXCEPTIONS_ENABLED

  return 0;
}
