
#include <libgnomedbmm.h>

class SameDataDemo: public Gtk::Window
{
public:
  static const char* FILE;

  SameDataDemo(const Glib::RefPtr<Gnome::Gda::Dict>& dict);

private:
  void restrict_default_served_by_field(Gnome::Db::DataWidget& data_widget, const Glib::RefPtr<Gnome::Gda::DataModel>& restrict_with, int restrict_col);
  void on_grid_row_changed(int row);
  void on_form_row_changed(int row);

  Glib::RefPtr<Gnome::Gda::DataModelIter> m_grid_iter;
  Glib::RefPtr<Gnome::Gda::DataModelIter> m_form_iter;

  sigc::connection m_grid_iter_connection;
  sigc::connection m_form_iter_connection;

  Gtk::ToggleButton* m_keep_sync;
};

const char* SameDataDemo::FILE = __FILE__;

SameDataDemo::SameDataDemo(const Glib::RefPtr<Gnome::Gda::Dict>& dict):
  Gtk::Window(Gtk::WINDOW_TOPLEVEL)
{
  Gtk::VBox* box = Gtk::manage(new Gtk::VBox(false, 5));
  box->set_border_width(5);

  Gtk::Label* label = Gtk::manage(new Gtk::Label("The following GnomeDbForm and GnomeDbGrid widgets\n"
                                                 "display data from the 'customers' and 'salesrep' tables."));
  box->pack_start(*label, Gtk::PACK_SHRINK);

  const Glib::ustring select_query =
                      "SELECT c.id, c.name, c.default_served_by as \"SalesRep\" "
                      "FROM customers c "
                      "LEFT JOIN salesrep s ON (s.id=c.default_served_by)";

  const Glib::ustring update_query =
                      "UPDATE customers SET "
                      "id=##/*name:'+0' type:'gint'*/, "
                      "name=##/*name:'+1' type:gchararray*/, "
                      "default_served_by=##/*name:'+2' type:gint*/ "
                      "WHERE id=##/*name:'-0' type:gint*/";

  const Glib::ustring delete_query =
                      "DELETE FROM customers "
                      "WHERE id=##/*name:'-0' type:gint*/";

  const Glib::ustring insert_query =
                      "INSERT INTO customers (id, name, default_served_by) "
                      "VALUES ("
                      "##/*name:'+0' type:gint*/, "
                      "##/*name:'+1' type:gchararray*/, "
                      "##/*name:'+2' type:gint*/)";

  Glib::RefPtr<Gnome::Gda::Query> query;
  Glib::RefPtr<Gnome::Gda::DataModelQuery> model;
  Glib::RefPtr<Gnome::Gda::DataModelQuery> sr_model;

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    query = Gnome::Gda::Query::create(dict);
    query->set_sql_text(select_query);
    model = Gnome::Gda::DataModelQuery::create(query);

    model->set_modification_query(update_query);
    model->set_modification_query(delete_query);
    model->set_modification_query(insert_query);

    /* Create a data model for the salesrep */
    query = Gnome::Gda::Query::create(dict);
    query->set_sql_text("SELECT id, name FROM salesrep");
    sr_model = Gnome::Gda::DataModelQuery::create(query);
  }
  catch(const Glib::Error& e)
  {
    std::cerr << e.what() << std::endl;
  }
#else
  std::auto_ptr<Glib::Error> error;
  query = Gnome::Gda::Query::create(dict);
  query->set_sql_text(select_query, error);
  model = Gnome::Gda::DataModelQuery::create(query);

  model->set_modification_query(update_query, error);
  model->set_modification_query(delete_query, error);
  model->set_modification_query(insert_query, error);

  /* Create a data model for the salesrep */
  query = Gnome::Gda::Query::create(dict);
  query->set_sql_text("SELECT id, name FROM salesrep", error);
  sr_model = Gnome::Gda::DataModelQuery::create(query);

  if(error.get() != NULL)
    std::cerr << error->what() << std::endl;
#endif // GLIBM_EXCEPTIONS_ENABLED

  label = Gtk::manage(new Gtk::Label("<b>GnomeDbGrid:</b>", 0, 0));
  label->set_use_markup(true);
  box->pack_start(*label, Gtk::PACK_SHRINK);

  /* Create grid widget */
  Gnome::Db::Grid* grid = Gtk::manage(new Gnome::Db::Grid(model));
  box->pack_start(*grid, Gtk::PACK_EXPAND_WIDGET);

  /* Restrict the c.default_served_by field in the grid to be within the sr_model */
  Gnome::Db::RawGrid* rawgrid = grid->get_raw_grid();
  restrict_default_served_by_field(*rawgrid, sr_model, 0);
  m_grid_iter = rawgrid->get_current_data();
  m_grid_iter_connection = m_grid_iter->signal_row_changed().connect(sigc::mem_fun(*this, &SameDataDemo::on_grid_row_changed));

  /* Create form widget which uses the same data model as the grid */
  label = Gtk::manage(new Gtk::Label("<b>GnomeDbForm:</b>", 0, 0));
  label->set_use_markup(true);
  box->pack_start(*label, Gtk::PACK_SHRINK);

  Glib::RefPtr<Gnome::Gda::DataProxy> proxy = rawgrid->get_proxy();
  Gnome::Db::Form* form = Gtk::manage(new Gnome::Db::Form(proxy));
  box->pack_start(*form, Gtk::PACK_SHRINK);

  /* Restrict the c.default_served_by field in the form to be within the sr_model */
  Gnome::Db::RawForm* rawform = form->get_raw_form();
  restrict_default_served_by_field(*rawform, sr_model, 0);
  m_form_iter = rawform->get_current_data();
  m_form_iter_connection = m_form_iter->signal_row_changed().connect(sigc::mem_fun(*this, &SameDataDemo::on_form_row_changed));

  /* Optional synchronization of the selections */
  label = Gtk::manage(new Gtk::Label("<b>Selected rows synchronization option:</b>\n"
                                     "<small>Effective only at the next selected row change</small>", 0.0, 0.0));
  label->set_use_markup(true);
  box->pack_start(*label, Gtk::PACK_SHRINK);

  m_keep_sync = Gtk::manage(new Gtk::CheckButton("Keep selected rows synchronized"));
  box->pack_start(*m_keep_sync, Gtk::PACK_SHRINK);

  add(*box);
  show_all_children();
}

void SameDataDemo::restrict_default_served_by_field(Gnome::Db::DataWidget& data_widget, const Glib::RefPtr<Gnome::Gda::DataModel>& restrict_with, int restrict_col)
{
  Glib::RefPtr<Gnome::Gda::DataModelIter> iter = data_widget.get_current_data();

#ifdef GLIBMM_EXCEPTIONS_ENABLED
  try
  {
    iter->restrict_param_by_name("SalesRep", restrict_with, restrict_col);
  }
  catch(const Glib::Error& e)
  {
    std::cerr << e.what() << std::endl;
  }
#else
  std::auto_ptr<Glib::Error> error;
  iter->restrict_param_by_name("SalesRep", restrict_with, restrict_col, error);
  if(error.get() != NULL)
    std::cerr << error->what() << std::endl;
#endif
}

void SameDataDemo::on_grid_row_changed(int row)
{
  if(m_keep_sync->get_active())
  {
    m_form_iter_connection.block();
    m_form_iter->set_at_row(row);
    m_form_iter_connection.unblock();
  }
}

void SameDataDemo::on_form_row_changed(int row)
{
  if(m_keep_sync->get_active())
  {
    m_grid_iter_connection.block();
    m_grid_iter->set_at_row(row);
    m_grid_iter_connection.unblock();
  }
}
