/* gnome-db-canvas-query-struct.c
 *
 * Copyright (C) 2002 - 2006 Vivien Malerba
 * Copyright (C) 2002 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include "marshal.h"
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-canvas-query-struct.h"
#include "gnome-db-canvas-entity.h"
#include "gnome-db-canvas-join.h"
#include "gnome-db-canvas-field.h"

static void gnome_db_canvas_query_struct_class_init (GnomeDbCanvasQueryStructClass * class);
static void gnome_db_canvas_query_struct_init       (GnomeDbCanvasQueryStruct * canvas);
static void gnome_db_canvas_query_struct_dispose   (GObject *object);
static void gnome_db_canvas_query_struct_finalize   (GObject *object);

/* virtual functions */
static void       create_canvas_items (GnomeDbCanvas *canvas);
static void       clean_canvas_items  (GnomeDbCanvas *canvas);
static void       graph_item_added    (GnomeDbCanvas *canvas, GdaGraphItem *item);
static void       graph_item_dropped  (GnomeDbCanvas *canvas, GdaGraphItem *item);
static GtkWidget *build_context_menu  (GnomeDbCanvas *canvas);
static void       set_pixels_per_unit (GnomeDbCanvas *canvas, gdouble n);

static void       query_destroyed_cb     (GdaQuery *query, GnomeDbCanvas *canvas);
static void       query_join_added_cb (GdaQuery *query, GdaQueryJoin *join, GnomeDbCanvas *canvas);

static void       drag_action_dcb     (GnomeDbCanvas *canvas, GnomeDbCanvasItem *from_item, GnomeDbCanvasItem *to_item);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;


struct _GnomeDbCanvasQueryStructPrivate
{
	GdaQuery     *query;
	GSList      *ent_items;
	GHashTable  *hash_targets; /* key = GdaQueryTarget, value = GnomeDbCanvasEntity */
	GHashTable  *hash_joins; /* key = GdaQueryJoin, value = GnomeDbCanvasJoin */
};

GType
gnome_db_canvas_query_struct_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbCanvasQueryStructClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_canvas_query_struct_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCanvasQueryStruct),
			0,
			(GInstanceInitFunc) gnome_db_canvas_query_struct_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_CANVAS, "GnomeDbCanvasQueryStruct", &info, 0);
	}
	return type;
}

static void
gnome_db_canvas_query_struct_init (GnomeDbCanvasQueryStruct * canvas)
{
	canvas->priv = g_new0 (GnomeDbCanvasQueryStructPrivate, 1);
	canvas->priv->hash_targets = g_hash_table_new (NULL, NULL);
	canvas->priv->hash_joins = g_hash_table_new (NULL, NULL);
	canvas->priv->query = NULL;
}

static void
gnome_db_canvas_query_struct_class_init (GnomeDbCanvasQueryStructClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	/* GnomeDbCanvas virtual functions */
	GNOME_DB_CANVAS_CLASS (class)->create_canvas_items = create_canvas_items;
	GNOME_DB_CANVAS_CLASS (class)->clean_canvas_items = clean_canvas_items;
	GNOME_DB_CANVAS_CLASS (class)->graph_item_added = graph_item_added;
	GNOME_DB_CANVAS_CLASS (class)->graph_item_dropped = graph_item_dropped;
	GNOME_DB_CANVAS_CLASS (class)->build_context_menu = build_context_menu;
	GNOME_DB_CANVAS_CLASS (class)->set_pixels_per_unit = set_pixels_per_unit;

	GNOME_DB_CANVAS_CLASS (class)->drag_action = drag_action_dcb;
	
	object_class->dispose = gnome_db_canvas_query_struct_dispose;
	object_class->finalize = gnome_db_canvas_query_struct_finalize;
}

static void
gnome_db_canvas_query_struct_dispose (GObject *object)
{
	GnomeDbCanvasQueryStruct *canvas;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS_QUERY_STRUCT (object));

	canvas = GNOME_DB_CANVAS_QUERY_STRUCT (object);

	if (canvas->priv) {
		if (canvas->priv->query)
			query_destroyed_cb (canvas->priv->query, GNOME_DB_CANVAS (canvas));
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
gnome_db_canvas_query_struct_finalize (GObject *object)
{
	GnomeDbCanvasQueryStruct *canvas;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS_QUERY_STRUCT (object));

	canvas = GNOME_DB_CANVAS_QUERY_STRUCT (object);

	if (canvas->priv) {
		g_hash_table_destroy (canvas->priv->hash_targets);
		g_hash_table_destroy (canvas->priv->hash_joins);

		g_free (canvas->priv);
		canvas->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

static void
drag_action_dcb (GnomeDbCanvas *canvas, GnomeDbCanvasItem *from_item, GnomeDbCanvasItem *to_item)
{
	GdaEntityField *f_field = NULL, *t_field = NULL;
	GdaQueryTarget *f_target = NULL, *t_target = NULL;
	GdaGraphItem *gitem;
	GnomeDbCanvasItem *citem;
	GdaQueryJoin *join;
	GdaQuery *query;
	GdaQueryCondition *cond, *newcond;
	GdaQueryField *qfield;
	
	/* fields retreival */
	if (GNOME_DB_IS_CANVAS_FIELD (from_item))
		f_field = gnome_db_canvas_field_get_field (GNOME_DB_CANVAS_FIELD (from_item));
	if (GNOME_DB_IS_CANVAS_FIELD (to_item))
		t_field = gnome_db_canvas_field_get_field (GNOME_DB_CANVAS_FIELD (to_item));
	
	if (!f_field || !t_field)
		return;
	
	/* targets retreival */
	citem = (GnomeDbCanvasItem *) gnome_db_canvas_field_get_parent_item (GNOME_DB_CANVAS_FIELD (from_item));
	gitem = gnome_db_canvas_item_get_graph_item (citem);
	f_target = (GdaQueryTarget *) gda_graph_item_get_ref_object (gitem);

	citem = (GnomeDbCanvasItem *) gnome_db_canvas_field_get_parent_item (GNOME_DB_CANVAS_FIELD (to_item));
	gitem = gnome_db_canvas_item_get_graph_item (citem);
	t_target = (GdaQueryTarget *) gda_graph_item_get_ref_object (gitem);

	if (!f_target || !GDA_IS_QUERY_TARGET (f_target) ||
	    !t_target || !GDA_IS_QUERY_TARGET (t_target))
		return;

	if (f_target == t_target) {
		GtkWidget *dlg;
		gchar *msg = g_strdup_printf ("<big>%s</big>\n\n%s",
					      _("Can not create join:"),
					      _("A join must be between two different targets. If the "
						"same table or view must be joinned to itself, then "
						"create another target for that table or view before "
						"creating the new join."));
		dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
							  GTK_BUTTONS_CLOSE, msg);
		g_free (msg);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);

		return;
	}

	query = GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->query;
	join = gda_query_get_join_by_targets (query, f_target, t_target);
	if (!join) {
		/* create a join */
		join = gda_query_join_new_with_targets (query, f_target, t_target);
		gda_query_join_set_join_type (join, GDA_QUERY_JOIN_TYPE_INNER);
		gda_query_add_join (query, join);
		g_object_unref (join);
	}

	/* add an EQUAL condition */
	newcond = gda_query_condition_new (query, GDA_QUERY_CONDITION_LEAF_EQUAL);
	qfield = gda_query_get_field_by_ref_field (query, f_target, f_field, GDA_ENTITY_FIELD_ANY);
	if (!qfield) {
		qfield = gda_query_field_field_new (query, NULL);
		g_object_set (G_OBJECT (qfield), "target", f_target, "field", f_field, NULL);
		gda_query_field_set_visible (qfield, FALSE);
		gda_entity_add_field (GDA_ENTITY (query), GDA_ENTITY_FIELD (qfield));
		g_object_unref (qfield);
	}
	gda_query_condition_leaf_set_operator (newcond, GDA_QUERY_CONDITION_OP_LEFT, qfield);

	qfield = gda_query_get_field_by_ref_field (query, t_target, t_field, GDA_ENTITY_FIELD_ANY);
	if (!qfield) {
		qfield = gda_query_field_field_new (query, NULL);
		g_object_set (G_OBJECT (qfield), "target", t_target, "field", t_field, NULL);
		gda_query_field_set_visible (qfield, FALSE);
		gda_entity_add_field (GDA_ENTITY (query), GDA_ENTITY_FIELD (qfield));
		g_object_unref (qfield);
	}
	gda_query_condition_leaf_set_operator (newcond, GDA_QUERY_CONDITION_OP_RIGHT, qfield);

	/* integrate newcond in the join's condition */
	cond = gda_query_join_get_condition (join);
	if (cond) {
		if (gda_query_condition_get_cond_type (cond) != GDA_QUERY_CONDITION_NODE_AND) {
			GdaQueryCondition *cond2;

			cond2 = gda_query_condition_new (query, GDA_QUERY_CONDITION_NODE_AND);
			g_return_if_fail (gda_query_condition_node_add_child (cond2, cond, NULL));
			gda_query_join_set_condition (join, cond2);
			g_object_unref (cond2);
			cond = cond2;
		}

		g_return_if_fail (gda_query_condition_node_add_child (cond, newcond, NULL));
		g_object_unref (newcond);
	}
	else {
		gda_query_join_set_condition (join, newcond);
		g_object_unref (newcond);
	}

#ifdef debug
	gda_object_dump (GDA_OBJECT (query), 0);
#endif
}

/**
 * gnome_db_canvas_query_struct_new
 * @query: a #GdaQuery object
 * @graph: a #GdaGraph object, or %NULL
 *
 * Creates a new canvas widget to display the targets and joins of @query
 *
 * @graph contains all the targets's graphical representations and their respective
 * locations on the canvas, or can be %NULL (in which case nothing is displayed)
 *
 * Returns: a new #GnomeDbCanvasQueryStruct widget
 */
GtkWidget *
gnome_db_canvas_query_struct_new (GdaQuery *query, GdaGraph *graph)
{
	GObject *obj;
	GnomeDbCanvasQueryStruct *canvas;
	GSList *joins, *list;

	g_return_val_if_fail (query && GDA_IS_QUERY (query), NULL);
	if (graph) {
		g_return_val_if_fail (GDA_IS_GRAPH (graph), NULL);
		g_return_val_if_fail (gda_object_get_dict (GDA_OBJECT (query)) == gda_object_get_dict (GDA_OBJECT (graph)), NULL);
	}

        obj = g_object_new (GNOME_DB_TYPE_CANVAS_QUERY_STRUCT, "aa", FALSE, NULL);
	gnome_canvas_set_center_scroll_region (GNOME_CANVAS (obj), TRUE);
	canvas = GNOME_DB_CANVAS_QUERY_STRUCT (obj);

	/* connecting to the GdaQuery */
	canvas->priv->query = query;
	gda_object_connect_destroy (query, G_CALLBACK (query_destroyed_cb), obj);
	g_signal_connect (G_OBJECT (query), "join_added",
			  G_CALLBACK (query_join_added_cb), obj);

	/* populating the canvas */
	g_object_set (obj, "graph", graph, NULL);

	/* adding all the joins */
	joins = gda_query_get_joins (query);
	list = joins;
	while (list) {
		query_join_added_cb (query, (GdaQueryJoin *) (list->data), (GnomeDbCanvas *) obj);
		list = g_slist_next (list);
	}
	g_slist_free (joins);

	return GTK_WIDGET (obj);
}

static void
query_destroyed_cb (GdaQuery *query, GnomeDbCanvas *canvas)
{
	/* disconnecting signals */
	GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->query = NULL;
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (query_destroyed_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (query_join_added_cb), canvas);

	/* clean the canvas */
	clean_canvas_items (canvas);
}

static void
query_join_added_cb (GdaQuery *query, GdaQueryJoin *join, GnomeDbCanvas *canvas)
{
	GnomeDbCanvasItem *canvas_join;
	GnomeCanvasItem *root, *canvas_item;

	root = GNOME_CANVAS_ITEM (gnome_canvas_root (GNOME_CANVAS (canvas)));

	canvas_join = g_hash_table_lookup (GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_joins, join);
	g_return_if_fail (!canvas_join);
		
	canvas_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (root),
					     GNOME_DB_TYPE_CANVAS_JOIN,
					     "join", join,
					     NULL);
		
	g_hash_table_insert (GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_joins, 
			     join, canvas_item);
}

/*
 * Add all the required GnomeCanvasItem objects for the associated #GdaGraph object 
 */
static void
create_canvas_items (GnomeDbCanvas *canvas)
{
	GSList *list, *graph_items;
	GdaGraph *graph = gnome_db_canvas_get_graph (canvas);

	graph_items = gda_graph_get_items (graph);
	list = graph_items;
	while (list) {
		graph_item_added (canvas, GDA_GRAPH_ITEM (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (graph_items);
}

static void
clean_canvas_items (GnomeDbCanvas *canvas)
{
	/* destroy items */
	while (GNOME_CANVAS_GROUP (GNOME_CANVAS (canvas)->root)->item_list) 
		gtk_object_destroy (GTK_OBJECT ((GNOME_CANVAS_GROUP (GNOME_CANVAS (canvas)->root)->item_list)->data));

	/* clean memory */
	g_hash_table_destroy (GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_targets);
	g_hash_table_destroy (GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_joins);
	GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_targets = g_hash_table_new (NULL, NULL);
	GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_joins = g_hash_table_new (NULL, NULL);
}

/*
 * Add the GnomeDbCanvasEntity corresponding to the graph item
 */
static GtkWidget *canvas_entity_popup_func (GnomeDbCanvasEntity *ce);
static void
graph_item_added (GnomeDbCanvas *canvas, GdaGraphItem *item)
{
	GnomeCanvasItem *canvas_item, *root;
	GdaObject *ref_obj = gda_graph_item_get_ref_object (item);

	root = GNOME_CANVAS_ITEM (gnome_canvas_root (GNOME_CANVAS (canvas)));
	if (GDA_IS_QUERY_TARGET (ref_obj)) {
		/* GnomeDbCanvasEntity for the table */
		canvas_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (root),
						     GNOME_DB_TYPE_CANVAS_ENTITY,
						     "popup_menu_func", canvas_entity_popup_func,
						     "target", ref_obj,
						     "x", 50.,
						     "y", 50.,
						     "graph_item", item,
						     NULL);
		gnome_db_canvas_declare_item (canvas, GNOME_DB_CANVAS_ITEM (canvas_item));
		g_hash_table_insert (GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_targets, ref_obj, canvas_item);
		gnome_canvas_update_now (GNOME_CANVAS (canvas)); /* forced here because of a GnomeCanvas bug */
	}
}

static void popup_func_delete_cb (GtkMenuItem *mitem, GnomeDbCanvasEntity *ce);
static GtkWidget *
canvas_entity_popup_func (GnomeDbCanvasEntity *ce)
{
	GtkWidget *menu, *entry;

	menu = gtk_menu_new ();
	entry = gtk_menu_item_new_with_label (_("Remove"));
	g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_func_delete_cb), ce);
	gtk_menu_append (GTK_MENU (menu), entry);
	gtk_widget_show (entry);

	return menu;
}

static void
popup_func_delete_cb (GtkMenuItem *mitem, GnomeDbCanvasEntity *ce)
{
	GdaGraphItem *gitem;
	GdaQueryTarget *target;
#ifdef debug
	GdaQuery *query;
#endif

	gitem = gnome_db_canvas_item_get_graph_item (GNOME_DB_CANVAS_ITEM (ce));
	target = (GdaQueryTarget *) gda_graph_item_get_ref_object (gitem);

	g_assert (target && GDA_IS_QUERY_TARGET (target));
#ifdef debug
	query = gda_query_target_get_query (target);
#endif
	gda_object_destroy (GDA_OBJECT (target));

#ifdef debug
	gda_object_dump (GDA_OBJECT (query), 0);
#endif
}

/*
 * Remove the GnomeDbCanvasEntity corresponding to the graph item
 */
static void
graph_item_dropped (GnomeDbCanvas *canvas, GdaGraphItem *item)
{
	GdaObject *ref_obj = gda_graph_item_get_ref_object (item);
	
	if (GDA_IS_QUERY_TARGET (ref_obj)) {
		GtkObject *canvas_item = g_hash_table_lookup (GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_targets,
							      ref_obj);
		if (canvas_item) {
			gtk_object_destroy (canvas_item);
			g_hash_table_remove (GNOME_DB_CANVAS_QUERY_STRUCT (canvas)->priv->hash_targets, ref_obj);
		}
	}
}

static void popup_add_target_cb (GtkMenuItem *mitem, GnomeDbCanvasQueryStruct *canvas);
static GtkWidget *
build_context_menu (GnomeDbCanvas *canvas)
{
	GtkWidget *menu, *mitem, *submenu, *submitem;
	GSList *tables, *list;
	GnomeDbCanvasQueryStruct *qstruct = GNOME_DB_CANVAS_QUERY_STRUCT (canvas);
	GdaObjectRef *refbase;
	GdaDict *dict = gda_object_get_dict (GDA_OBJECT (qstruct->priv->query));
	gboolean other_tables = FALSE;

	tables = gda_dict_database_get_tables (gda_dict_get_database (dict));

	menu = gtk_menu_new ();
	submitem = gtk_menu_item_new_with_label (_("New target from table"));
	gtk_widget_show (submitem);
	gtk_menu_append (menu, submitem);
	if (tables) {
		submenu = gtk_menu_new ();
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (submitem), submenu);
		gtk_widget_show (submenu);
		
		/* add a menu item for each table not currently shown displayed.
		 * WARNING: if a GdaDictTable is detroyed while that menu is displayed, it can still be selected
		 * from within the menu even though it will not do anything.
		 */
		list = tables;
		while (list) {
			mitem = gtk_menu_item_new_with_label (gda_object_get_name (GDA_OBJECT (list->data)));
			gtk_widget_show (mitem);
			gtk_menu_append (submenu, mitem);
			
			refbase = GDA_OBJECT_REF (gda_object_ref_new (dict));
			gda_object_ref_set_ref_object (refbase, GDA_OBJECT (list->data));
			g_object_set_data_full (G_OBJECT (mitem), "table", refbase, g_object_unref);
			
			g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (popup_add_target_cb), canvas);
			other_tables = TRUE;

			list = g_slist_next (list);
		}
		g_slist_free (tables);
	}

	/* sub menu is incensitive if there are no more tables left to add */
	gtk_widget_set_sensitive (submitem, other_tables);

	return menu;
}

static void
popup_add_target_cb (GtkMenuItem *mitem, GnomeDbCanvasQueryStruct *canvas)
{
	GdaObjectRef *refbase;
	GdaObject *ref_obj;

	refbase = g_object_get_data (G_OBJECT (mitem), "table");
	ref_obj =  gda_object_ref_get_ref_object (refbase);

	if (ref_obj && GDA_IS_DICT_TABLE (ref_obj)) {
		GdaDictTable *table = GDA_DICT_TABLE (ref_obj);
		GdaQueryTarget *target;
		GdaGraphItem *gitem;

		GSList *constraints, *list;
		GSList *targets = gda_query_get_targets (canvas->priv->query), *tlist;

		/* actual target and corresponding GdaGraphItem creation */
		target = gda_query_target_new (canvas->priv->query, gda_object_get_name (GDA_OBJECT (table)));
		gitem = GDA_GRAPH_ITEM (gda_graph_item_new (gda_object_get_dict (GDA_OBJECT (canvas->priv->query)), GDA_OBJECT (target)));
		gda_graph_item_set_position (gitem, GNOME_DB_CANVAS (canvas)->xmouse, GNOME_DB_CANVAS (canvas)->ymouse);
		gda_graph_add_item (gnome_db_canvas_get_graph (GNOME_DB_CANVAS (canvas)), gitem);
		g_object_unref (G_OBJECT (gitem));
		gda_query_add_target (canvas->priv->query, target, NULL);
		g_object_unref (target);
		
		/* using database FK constraints to create joins */
		constraints = gda_dict_database_get_tables_fk_constraints (gda_dict_get_database (gda_object_get_dict (GDA_OBJECT (table))),
								     table, NULL, FALSE);
		list = constraints;
		while (list) {
			GdaDictConstraint *cons = GDA_DICT_CONSTRAINT (list->data);
			GdaDictTable *otable;
			GdaQueryTarget *otarget = NULL;
			gboolean table_is_pkey = TRUE;
			
			otable = gda_dict_constraint_get_table (cons);
			if (otable == table) {
				table_is_pkey = FALSE;
				otable = gda_dict_constraint_fkey_get_ref_table (cons);
			}
			
			/* find a suitable target to make a join with */
			tlist = targets;
			while (tlist && !otarget) {
				if (gda_query_target_get_represented_entity (GDA_QUERY_TARGET (tlist->data)) == 
				    (GdaEntity *) otable)
					otarget = GDA_QUERY_TARGET (tlist->data);
				tlist = g_slist_next (tlist);
			}
			
			if (otarget) {
				GdaQueryJoin *join;
				
				/* actual join */
				join = gda_query_join_new_with_targets (canvas->priv->query, otarget, target);
				gda_query_join_set_join_type (join, GDA_QUERY_JOIN_TYPE_INNER);
				gda_query_add_join (canvas->priv->query, join);
				gda_query_join_set_condition_from_fkcons (join);
				g_object_unref (join);
			}
			
			list = g_slist_next (list);
		}
		g_slist_free (constraints);
		g_slist_free (targets);
	}

	/* REM: ref_obj could also be in the future a SELECT #GdaQuery */
}

static void
set_pixels_per_unit (GnomeDbCanvas *canvas, gdouble n)
{
	GList *list;

	list = GNOME_CANVAS_GROUP (GNOME_CANVAS (canvas)->root)->item_list;
	while (list) {
		if (GNOME_DB_IS_CANVAS_ENTITY (list->data))
			g_object_set (G_OBJECT (list->data), "scale", n, NULL);
		list = g_list_next (list);
	}
}
