/* utility.c
 *
 * Copyright (C) 2003 - 2006 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <glib/gprintf.h>
#include "utility.h"
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-data-entry.h"
#include "gnome-db-decl.h"

/**
 * utility_entry_build_actions_menu
 * @obj_data:
 * @attrs:
 * @function:
 *
 * Creates a GtkMenu widget which contains the possible actions on a data entry which 
 * attributes are @attrs. After the menu has been displayed, and when an action is selected,
 * the @function callback is called with the 'user_data' being @obj_data.
 *
 * The menu item which emits the signal has a "action" attribute which describes what action the
 * user has requested.
 *
 * Returns: the new menu
 */
GtkWidget *
utility_entry_build_actions_menu (GObject *obj_data, guint attrs, GCallback function)
{
	GtkWidget *menu, *mitem;
	gchar *str;
	gboolean nullact = FALSE;
	gboolean defact = FALSE;
	gboolean reset = FALSE;

	gboolean value_is_null;
	gboolean value_is_modified;
	gboolean value_is_default;

	menu = gtk_menu_new ();

	/* which menu items to make sensitive ? */
	value_is_null = attrs & GDA_VALUE_ATTR_IS_NULL;
	value_is_modified = ! (attrs & GDA_VALUE_ATTR_IS_UNCHANGED);
	value_is_default = attrs & GDA_VALUE_ATTR_IS_DEFAULT;

	if (! (attrs & GDA_VALUE_ATTR_NO_MODIF)) {
		if ((attrs & GDA_VALUE_ATTR_CAN_BE_NULL) && 
		    !(attrs & GDA_VALUE_ATTR_IS_NULL))
			nullact = TRUE;
		if ((attrs & GDA_VALUE_ATTR_CAN_BE_DEFAULT) && 
		    !(attrs & GDA_VALUE_ATTR_IS_DEFAULT))
			defact = TRUE;
		if (!(attrs & GDA_VALUE_ATTR_IS_UNCHANGED)) {
			if (attrs & GDA_VALUE_ATTR_HAS_VALUE_ORIG) 
				reset = TRUE;
		}
	}

	/* set to NULL item */
	str = g_strdup (_("Unset"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem),
					value_is_null);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GDA_VALUE_ATTR_IS_NULL));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, nullact);

	/* default value item */
	str = g_strdup (_("Set to default value"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), 
					value_is_default);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GDA_VALUE_ATTR_IS_DEFAULT));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, defact);
		
	/* reset to original value item */
	str = g_strdup (_("Reset to original value"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), 
					!value_is_modified);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GDA_VALUE_ATTR_IS_UNCHANGED));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, reset);

	return menu;
}


/**
 * utility_entry_build_info_colors_array
 * 
 * Creates an array of colors for the different states of an entry:
 *    Valid   <-> No special color
 *    Null    <-> Green
 *    Default <-> Blue
 *    Invalid <-> Red
 * Each status (except Valid) is represented by two colors for GTK_STATE_NORMAL and
 * GTK_STATE_PRELIGHT.
 *
 * Returns: a new array of 6 colors
 */
GdkColor **utility_entry_build_info_colors_array ()
{
	GdkColor **colors;
	GdkColor *color;
	
	colors = g_new0 (GdkColor *, 6);
	
	/* Green color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_NULL, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[0] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_NULL, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[1] = color;
	
	
	/* Blue color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_DEFAULT, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[2] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_DEFAULT, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[3] = color;
	
	
	/* Red color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_INVALID, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[4] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_INVALID, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[5] = color;

	return colors;
}

/**
 * utility_proxy_compute_attributes_for_group
 *
 * Computes an attributes from the individual attributes of the values stored in @store and
 * corresponding to the columns for the parameters in @group (as described by @model_iter), 
 * at row pointed by @iter
 *
 * Returns: the attributes
 */
guint
utility_proxy_compute_attributes_for_group (GdaParameterListGroup *group, GnomeDbDataStore *store, 
					    GdaDataModelIter *model_iter, GtkTreeIter *tree_iter, 
					    gboolean *to_be_deleted)
{
	guint attributes = GDA_VALUE_ATTR_IS_NULL | GDA_VALUE_ATTR_CAN_BE_NULL |
                GDA_VALUE_ATTR_IS_DEFAULT | GDA_VALUE_ATTR_CAN_BE_DEFAULT |
                GDA_VALUE_ATTR_IS_UNCHANGED | GDA_VALUE_ATTR_HAS_VALUE_ORIG;
        gboolean to_del = TRUE, local_to_del;
	gboolean new_row = FALSE;
        GSList *list;
        gint col;
        gint offset;
        guint localattr;
	GdaDataProxy *proxy;
	gint row;

	proxy = gnome_db_data_store_get_proxy (store);
        offset = gda_data_proxy_get_proxied_model_n_cols (proxy);

        /* list the values in proxy_model for each param in GDA_PARAMETER_LIST_NODE (group->nodes->data)->params */
	attributes = 0;
        for (list = group->nodes; list; list = list->next) {
		col = gda_data_model_iter_get_column_for_param (model_iter, GDA_PARAMETER_LIST_NODE (list->data)->param);
                gtk_tree_model_get (GTK_TREE_MODEL (store), tree_iter,
                                    DATA_STORE_COL_TO_DELETE, &local_to_del,
                                    offset + col, &localattr, -1);
		if (list == group->nodes)
			attributes = localattr;
		else
			attributes &= localattr;
                to_del = to_del && local_to_del;
		new_row = (row < 0) ? TRUE : FALSE;
        }

	if (to_be_deleted)
                *to_be_deleted = to_del;

        return attributes;
}

/**
 * utility_proxy_compute_values_for_group
 * Computes a list of values containing the individual values stored in @store and
 * corresponding to the columns for the parameters if @group, at row pointed by @tree_iter.
 * 
 * If @model_values is TRUE, then the values in the returned list are the values of the
 * @group->nodes_source->data_model, at the @group->nodes_source->shown_cols_index columns
 *
 * For both performances reasons and because the contents of the @group->nodes_source->data_model
 * may change, this function uses the gda_data_proxy_get_model_row_value() function.
 *
 * The returned list must be freed by the caller, but the values in the list must not be modified or
 * freed.
 */
GList *
utility_proxy_compute_values_for_group (GdaParameterListGroup *group, GnomeDbDataStore *store, 
					GdaDataModelIter *model_iter, 
					GtkTreeIter *tree_iter, gboolean model_values)
{
	gint offset;
	GList *retval = NULL;
	GdaDataProxy *proxy;

	proxy = gnome_db_data_store_get_proxy (store);
	offset = gda_data_proxy_get_proxied_model_n_cols (proxy);
	if (!model_values) {
		GSList *list;
		GValue *value;

		list = group->nodes;
		while (list) {
			gint col;

			col = gda_data_model_iter_get_column_for_param (model_iter, GDA_PARAMETER_LIST_NODE (list->data)->param);
			gtk_tree_model_get (GTK_TREE_MODEL (store), tree_iter,
					    col, &value, -1);
			retval = g_list_append (retval, value);
			
			list = g_slist_next (list);
		}
	}
	else {
		gint col, i, proxy_row;
		GdaParameterListSource *source;
		const GValue *value;
		gboolean slow_way = FALSE;
		gboolean ret_null = FALSE;

		proxy_row = gnome_db_data_store_get_row_from_iter (store, tree_iter);
		source = group->nodes_source;
		for (i = 0 ; (i < source->shown_n_cols)  && !ret_null; i++) {
			col = source->shown_cols_index[i];
#ifdef PROXY_STORE_EXTRA_VALUES
			if (!slow_way) {
				value = gda_data_proxy_get_model_row_value (proxy, source->data_model, proxy_row, col);
				if (value)
					retval = g_list_append (retval, (GValue *) value);
				else {
					if (gda_data_proxy_get_assigned_model_col (proxy, source->data_model, col) < 0)
						slow_way = TRUE;
					else
						retval = g_list_append (retval, NULL);
				}
			}
#endif
			slow_way = TRUE;
			
			if (slow_way) {
				/* may be a bit slow: try to find the values in the model, it's better if the user
				 * uses gda_data_proxy_assign_model_col()!  */
				GSList *key_values = NULL;
				gint row, *cols_index;
				GSList *list;
				gint j;
				
				cols_index = g_new0 (gint, g_slist_length (group->nodes));
				list = group->nodes;
				j = 0;
				while (list) {
					gint colno;
					colno = gda_data_model_iter_get_column_for_param (model_iter, 
									 GDA_PARAMETER_LIST_NODE (list->data)->param);
					cols_index [j] = GDA_PARAMETER_LIST_NODE (list->data)->source_column;
					gtk_tree_model_get (GTK_TREE_MODEL (store), tree_iter,
							    colno, &value, -1);
					key_values = g_slist_append (key_values, (GValue *) value);
					list = g_slist_next (list);
					j++;
				}
				
				row = gda_data_model_get_row_from_values (GDA_DATA_MODEL (source->data_model), 
									  key_values, cols_index);
				if (row >= 0) {
					value = gda_data_model_get_value_at (GDA_DATA_MODEL (source->data_model),
									     col, row);
					retval = g_list_append (retval, (GValue *) value);
				}
				else {
#ifdef DEBUG_WARNING
					g_warning ("Could not find requested value in restricting data model");
					g_print ("Requested: ");
					list = key_values;
					j = 0;
					while (list) {
						gchar *str;
						
						if (value) {
							str = gda_value_stringify ((GValue *) list->data);
							g_print ("/%s @col %d", str, cols_index [j]);
							g_free (str);
						}
						else
							g_print ("/NULL @col %d", cols_index [j]);
						list = g_slist_next (list);
						j++;
					}
					g_print (" in data model: \n");
					gda_data_model_dump (source->data_model, stdout);
#endif
					ret_null = TRUE;
				}
				g_slist_free (key_values);
			}
		}

		if (ret_null) {
			g_list_free (retval);
			retval = NULL;
		}
	}

	return retval;
}


/**
 * utility_table_field_attrs_stringify
 */
gchar *
utility_table_field_attrs_stringify (guint attributes)
{
	if (attributes & FIELD_AUTO_INCREMENT)
		return g_strdup ("AUTO_INCREMENT");

	return NULL;
}

/**
 * utility_table_field_attrs_parse
 */
guint
utility_table_field_attrs_parse (const gchar *str)
{
	if (!strcmp (str, "AUTO_INCREMENT"))
		return FIELD_AUTO_INCREMENT;

	return 0;
}


/**
 * utility_build_encoded_id
 *
 * Creates a BASE64 kind encoded string. It's not really a BASE64 because:
 * - the characters + and / of BASE64 are replaced with - and _
 * - no padding is done using the = character
 *
 * The created string is a valid NCName XML token.
 */
static inline unsigned char
to_uchar (gchar ch)
{
	return ch;
}

gchar *
utility_build_encoded_id (const gchar *prefix, const gchar *id)
{
	const gchar conv[64] = 
		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";
	gchar *str, *out;
	const gchar *in;
	int ln = 0, pln = 0;
	int offset;

	/* size computation */
	if (prefix) {
		pln = strlen (prefix);
		ln = pln;
	}

	ln += (strlen (id) * 4 + 2) / 3 + 1;
	str = g_new0 (char, ln);

	/* copy prefix */
	out = str;
	if (prefix) {
		strcpy (str, prefix);
		out += pln;
	}

	/* create data */
	offset = 4;
	for (in = id; offset == 4; in += 3) {
		offset = 0;
		
		if (in[0]) {
			offset = 2;
			
			out[0] = conv [to_uchar (in[0]) >> 2];

			if (in[1]) {
				offset = 3;
				out[1] = conv [((to_uchar (in[0]) << 4) + 
						(to_uchar (in[1]) >> 4)) & 0x3f];

				if (in[2]) {
					offset = 4;

					out[2] = conv [((to_uchar (in[1]) << 2) + 
							(to_uchar (in[2]) >> 6)) & 0x3f];
					out[3] = conv [to_uchar (in[2]) & 0x3f];
				}
				else
					out[2] = conv [(to_uchar (in[1]) << 2) & 0x3f];
			}
			else
				out[1] = conv [(to_uchar (in[0]) << 4) & 0x3f];


			out += offset;
		}
        }

	return str;
}

#define B64(x)	     \
  ((x) == 'A' ? 0    \
   : (x) == 'B' ? 1  \
   : (x) == 'C' ? 2  \
   : (x) == 'D' ? 3  \
   : (x) == 'E' ? 4  \
   : (x) == 'F' ? 5  \
   : (x) == 'G' ? 6  \
   : (x) == 'H' ? 7  \
   : (x) == 'I' ? 8  \
   : (x) == 'J' ? 9  \
   : (x) == 'K' ? 10 \
   : (x) == 'L' ? 11 \
   : (x) == 'M' ? 12 \
   : (x) == 'N' ? 13 \
   : (x) == 'O' ? 14 \
   : (x) == 'P' ? 15 \
   : (x) == 'Q' ? 16 \
   : (x) == 'R' ? 17 \
   : (x) == 'S' ? 18 \
   : (x) == 'T' ? 19 \
   : (x) == 'U' ? 20 \
   : (x) == 'V' ? 21 \
   : (x) == 'W' ? 22 \
   : (x) == 'X' ? 23 \
   : (x) == 'Y' ? 24 \
   : (x) == 'Z' ? 25 \
   : (x) == 'a' ? 26 \
   : (x) == 'b' ? 27 \
   : (x) == 'c' ? 28 \
   : (x) == 'd' ? 29 \
   : (x) == 'e' ? 30 \
   : (x) == 'f' ? 31 \
   : (x) == 'g' ? 32 \
   : (x) == 'h' ? 33 \
   : (x) == 'i' ? 34 \
   : (x) == 'j' ? 35 \
   : (x) == 'k' ? 36 \
   : (x) == 'l' ? 37 \
   : (x) == 'm' ? 38 \
   : (x) == 'n' ? 39 \
   : (x) == 'o' ? 40 \
   : (x) == 'p' ? 41 \
   : (x) == 'q' ? 42 \
   : (x) == 'r' ? 43 \
   : (x) == 's' ? 44 \
   : (x) == 't' ? 45 \
   : (x) == 'u' ? 46 \
   : (x) == 'v' ? 47 \
   : (x) == 'w' ? 48 \
   : (x) == 'x' ? 49 \
   : (x) == 'y' ? 50 \
   : (x) == 'z' ? 51 \
   : (x) == '0' ? 52 \
   : (x) == '1' ? 53 \
   : (x) == '2' ? 54 \
   : (x) == '3' ? 55 \
   : (x) == '4' ? 56 \
   : (x) == '5' ? 57 \
   : (x) == '6' ? 58 \
   : (x) == '7' ? 59 \
   : (x) == '8' ? 60 \
   : (x) == '9' ? 61 \
   : (x) == '-' ? 62 \
   : (x) == '_' ? 63 \
   : -1)

static const signed char b64[0x100] = {
  B64 (0), B64 (1), B64 (2), B64 (3),
  B64 (4), B64 (5), B64 (6), B64 (7),
  B64 (8), B64 (9), B64 (10), B64 (11),
  B64 (12), B64 (13), B64 (14), B64 (15),
  B64 (16), B64 (17), B64 (18), B64 (19),
  B64 (20), B64 (21), B64 (22), B64 (23),
  B64 (24), B64 (25), B64 (26), B64 (27),
  B64 (28), B64 (29), B64 (30), B64 (31),
  B64 (32), B64 (33), B64 (34), B64 (35),
  B64 (36), B64 (37), B64 (38), B64 (39),
  B64 (40), B64 (41), B64 (42), B64 (43),
  B64 (44), B64 (45), B64 (46), B64 (47),
  B64 (48), B64 (49), B64 (50), B64 (51),
  B64 (52), B64 (53), B64 (54), B64 (55),
  B64 (56), B64 (57), B64 (58), B64 (59),
  B64 (60), B64 (61), B64 (62), B64 (63),
  B64 (64), B64 (65), B64 (66), B64 (67),
  B64 (68), B64 (69), B64 (70), B64 (71),
  B64 (72), B64 (73), B64 (74), B64 (75),
  B64 (76), B64 (77), B64 (78), B64 (79),
  B64 (80), B64 (81), B64 (82), B64 (83),
  B64 (84), B64 (85), B64 (86), B64 (87),
  B64 (88), B64 (89), B64 (90), B64 (91),
  B64 (92), B64 (93), B64 (94), B64 (95),
  B64 (96), B64 (97), B64 (98), B64 (99),
  B64 (100), B64 (101), B64 (102), B64 (103),
  B64 (104), B64 (105), B64 (106), B64 (107),
  B64 (108), B64 (109), B64 (110), B64 (111),
  B64 (112), B64 (113), B64 (114), B64 (115),
  B64 (116), B64 (117), B64 (118), B64 (119),
  B64 (120), B64 (121), B64 (122), B64 (123),
  B64 (124), B64 (125), B64 (126), B64 (127),
  B64 (128), B64 (129), B64 (130), B64 (131),
  B64 (132), B64 (133), B64 (134), B64 (135),
  B64 (136), B64 (137), B64 (138), B64 (139),
  B64 (140), B64 (141), B64 (142), B64 (143),
  B64 (144), B64 (145), B64 (146), B64 (147),
  B64 (148), B64 (149), B64 (150), B64 (151),
  B64 (152), B64 (153), B64 (154), B64 (155),
  B64 (156), B64 (157), B64 (158), B64 (159),
  B64 (160), B64 (161), B64 (162), B64 (163),
  B64 (164), B64 (165), B64 (166), B64 (167),
  B64 (168), B64 (169), B64 (170), B64 (171),
  B64 (172), B64 (173), B64 (174), B64 (175),
  B64 (176), B64 (177), B64 (178), B64 (179),
  B64 (180), B64 (181), B64 (182), B64 (183),
  B64 (184), B64 (185), B64 (186), B64 (187),
  B64 (188), B64 (189), B64 (190), B64 (191),
  B64 (192), B64 (193), B64 (194), B64 (195),
  B64 (196), B64 (197), B64 (198), B64 (199),
  B64 (200), B64 (201), B64 (202), B64 (203),
  B64 (204), B64 (205), B64 (206), B64 (207),
  B64 (208), B64 (209), B64 (210), B64 (211),
  B64 (212), B64 (213), B64 (214), B64 (215),
  B64 (216), B64 (217), B64 (218), B64 (219),
  B64 (220), B64 (221), B64 (222), B64 (223),
  B64 (224), B64 (225), B64 (226), B64 (227),
  B64 (228), B64 (229), B64 (230), B64 (231),
  B64 (232), B64 (233), B64 (234), B64 (235),
  B64 (236), B64 (237), B64 (238), B64 (239),
  B64 (240), B64 (241), B64 (242), B64 (243),
  B64 (244), B64 (245), B64 (246), B64 (247),
  B64 (248), B64 (249), B64 (250), B64 (251),
  B64 (252), B64 (253), B64 (254), B64 (255)
};

#define isbase64(x) ((to_uchar (x) <= 255) && (0 <= b64[to_uchar (x)]))


/**
 * utility_build_decoded_id
 *
 * Reverse of utility_build_encoded_id()
 */
gchar *
utility_build_decoded_id (const gchar *prefix, const gchar *id)
{
	gchar *str;
	gchar *out;
	const gchar *in;
	int ln = 0;
	int offset;

	/* go to the first byte of actual data */
	in = id;
	if (prefix) {
		const gchar *tmp;
		tmp = prefix;
		while (*tmp) {
			in++;
			tmp++;
		}
	}

	ln = (strlen (in) * 3) / 4 + 3;
	str = g_new0 (char, ln);
	out = str;

	/* create data */
	offset = 3;
	for (; offset == 3; in += 4) {
		offset = 0;
		
		if (isbase64 (in[0]) && isbase64 (in[1])) {
			out[0] = ((b64 [to_uchar (in[0])] << 2) | 
				  (b64 [to_uchar (in[1])] >> 4));
			
			if (isbase64 (in[2])) {
				out[1] = (((b64 [to_uchar (in[1])] << 4) & 0xf0) |
					  (b64 [to_uchar (in[2])] >> 2));

				if (isbase64 (in[3])) {
					out[2] = (((b64 [to_uchar (in[2])] << 6) & 0xc0) |
						  b64 [to_uchar (in[3])]);
					offset = 3;
				}
			}
		}

		out += offset;
        }

	return str;
}
