/* GNOME DB library
 * Copyright (C) 2005 The GNOME Foundation
 *
 * AUTHORS:
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktable.h>
#include <libgda/libgda.h>
#include <libgnomedb/gnome-db-dsn-spec.h>
#include <libgnomedb/gnome-db-provider-selector.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomedb/gnome-db-basic-form.h>
#include <glib/gi18n-lib.h>

#define PARENT_TYPE GTK_TYPE_VBOX

typedef enum {
	NO_PROVIDER,
	OLD_IFACE,
	PROVIDER_FORM,
	PROVIDER_ERROR
} WidgetType;

struct _GnomeDbDsnSpecPrivate {
	GdaClient *client; /* created only for this widget */
	gchar     *provider;

	WidgetType type;
	GtkWidget *form; /* what it really is is determined by @type */
	gchar     *cnc_string; /* as it was last updated */
};

static void gnome_db_dsn_spec_class_init (GnomeDbDsnSpecClass *klass);
static void gnome_db_dsn_spec_init       (GnomeDbDsnSpec *spec,
					  GnomeDbDsnSpecClass *klass);
static void gnome_db_dsn_spec_finalize   (GObject *object);
static void gnome_db_dsn_spec_dispose    (GObject *object);

enum {
	CHANGED,
	LAST_SIGNAL
};


static gint gnome_db_dsn_spec_signals[LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

/*
 * GnomeDbDsnSpec class implementation
 */

static void
gnome_db_dsn_spec_class_init (GnomeDbDsnSpecClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = gnome_db_dsn_spec_dispose;
	object_class->finalize = gnome_db_dsn_spec_finalize;
	klass->changed = NULL;

	/* add class signals */
	gnome_db_dsn_spec_signals[CHANGED] =
		g_signal_new ("changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbDsnSpecClass, changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
}

static void
update_form_contents (GnomeDbDsnSpec *spec)
{
	/*g_print ("DSN: %s\n", spec->priv->cnc_string);*/
	switch (spec->priv->type) {
	case OLD_IFACE:
		g_assert (spec->priv->form);
		gtk_entry_set_text (GTK_ENTRY (spec->priv->form), 
				    spec->priv->cnc_string ? spec->priv->cnc_string : "");
		break;
	case PROVIDER_FORM: {
		/* update data set in form */
		GdaParameterList *dset;
		GSList *params_set = NULL;
		GSList *list;
		g_assert (spec->priv->form);

		dset = gnome_db_basic_form_get_data_set (GNOME_DB_BASIC_FORM (spec->priv->form));

		/* split array in a list of named parameters, and for each parameter value, set the correcponding
		   parameter in @dset */
		/* FIXME: do something better here for array splitting  */
		if (spec->priv->cnc_string) {
			gchar **array = NULL;
			array = g_strsplit (spec->priv->cnc_string, ";", 0);
			if (array) {
				gint index = 0;
				gchar *tok;
				gchar *value;
				gchar *name;

				for (index = 0; array[index]; index++) {
					name = strtok_r (array [index], "=", &tok);
					if (name)
						value = strtok_r (NULL, "=", &tok);
					else
						value = NULL;
					if (name && value) {
						GdaParameter *param;

						param = gda_parameter_list_find_param (dset, name);
						if (param)
							if (gda_parameter_set_value_str (param, value))
								params_set = g_slist_prepend (params_set, param);
					}
				}

				g_strfreev (array);
			}
		}

		list = dset->parameters;
		while (0 && list) {
			if (!params_set || !g_slist_find (params_set, list->data)) {
				/* empty parameter */
				gda_parameter_set_value (GDA_PARAMETER (list->data), NULL);
			}
			list = g_slist_next (list);
		}
		g_slist_free (params_set);
		break;
	}
	default:
		/* no change here */
		break;
	}
}

static void
dsn_form_changed (GnomeDbBasicForm *form, GdaParameter *param, gboolean is_user_modif, GnomeDbDsnSpec *spec)
{
	if (! is_user_modif)
		return;

	g_signal_emit (spec, gnome_db_dsn_spec_signals[CHANGED], 0, NULL);
}

static void
dsn_entry_changed (GtkEntry *entry, GnomeDbDsnSpec *spec)
{
	g_signal_emit (spec, gnome_db_dsn_spec_signals[CHANGED], 0, NULL);
}

static void
adapt_form_widget (GnomeDbDsnSpec *spec)
{
	gchar *str;

	/* destroy any previous widget */
	if (spec->priv->form) {
		gtk_container_foreach (GTK_CONTAINER (spec), (GtkCallback) gtk_widget_destroy, NULL);
		spec->priv->form = NULL;
	}
	spec->priv->type = NO_PROVIDER;
	
	if (!spec->priv->provider) 
		return;
	
	/* create new widget */
	str = gda_client_get_dsn_specs (spec->priv->client, spec->priv->provider);
	if (str) {
		GdaParameterList *dset;
		GtkWidget *wid;
		GError *error = NULL;
		
		dset = GDA_PARAMETER_LIST (gda_parameter_list_new_from_spec_string (NULL, str, &error));
		if (dset) {
			spec->priv->type = PROVIDER_FORM;

			wid = gnome_db_basic_form_new (dset);
			gnome_db_basic_form_show_entries_actions (GNOME_DB_BASIC_FORM (wid), FALSE);
			g_object_unref (dset);

			spec->priv->form = wid;
			update_form_contents (spec);
			g_signal_connect (G_OBJECT (wid), "param_changed",
					  G_CALLBACK (dsn_form_changed), spec);
		}
		else {
			gchar *str2;
			spec->priv->type = PROVIDER_ERROR;

			str2 = g_strdup_printf (_("Provider internal error: %s"), 
						error && error->message ? error->message : "???" );
			wid = gtk_label_new (str2);
			g_free (str2);
		}
		
		gtk_widget_show (wid);
		gtk_container_add (GTK_CONTAINER (spec), wid);
		
		g_free (str);
	}
	else {
		/* plugin_get_dsn_specs() is not supported, use old method */
		GtkWidget *hbox, *label;
		GtkWidget *entry;
		
		spec->priv->type = OLD_IFACE;

		hbox = gtk_hbox_new (FALSE, 0);
		gtk_widget_show (hbox);
		gtk_container_add (GTK_CONTAINER (spec), hbox);		

		label = gnome_db_new_label_widget (_("Connection _string:"));
		gtk_widget_show (label);
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

		entry = gtk_entry_new ();
		gtk_widget_show (entry);
		gtk_box_pack_start (GTK_BOX (hbox), entry, FALSE, FALSE, 0);
		spec->priv->form = entry;

		update_form_contents (spec);
		g_signal_connect (G_OBJECT (entry), "changed",
				  G_CALLBACK (dsn_entry_changed), spec);

	}
}


static void
gnome_db_dsn_spec_init (GnomeDbDsnSpec *spec, GnomeDbDsnSpecClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_DSN_SPEC (spec));

	spec->priv = g_new0 (GnomeDbDsnSpecPrivate, 1);
	spec->priv->client = gda_client_new ();
	spec->priv->type = NO_PROVIDER;
}

static void
gnome_db_dsn_spec_dispose (GObject *object)
{
	GnomeDbDsnSpec *spec = (GnomeDbDsnSpec *) object;

	g_return_if_fail (GNOME_DB_IS_DSN_SPEC (spec));

	/* free memory */
	if (spec->priv->client) {
		g_object_unref (spec->priv->client);
		spec->priv->client = NULL;
	}

	/* chain to parent class */
	parent_class->dispose (object);
}

static void
gnome_db_dsn_spec_finalize (GObject *object)
{
	GnomeDbDsnSpec *spec = (GnomeDbDsnSpec *) object;

	g_return_if_fail (GNOME_DB_IS_DSN_SPEC (spec));

	/* free memory */
	if (spec->priv->cnc_string)
		g_free (spec->priv->cnc_string);
	if (spec->priv->provider)
		g_free (spec->priv->provider);

	g_free (spec->priv);
	spec->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_dsn_spec_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDsnSpecClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_dsn_spec_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDsnSpec),
			0,
			(GInstanceInitFunc) gnome_db_dsn_spec_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbDsnSpec",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_dsn_spec_new
 * @provider: the provider to be used 
 *
 * Creates a new #GnomeDbDsnSpec widget
 *
 * Returns:
 */
GtkWidget *
gnome_db_dsn_spec_new (const gchar *provider)
{
	GnomeDbDsnSpec *spec;

	spec = g_object_new (GNOME_DB_TYPE_DSN_SPEC, NULL);
	if (provider)
		spec->priv->provider = g_strdup (provider);
	adapt_form_widget (spec);

	return GTK_WIDGET (spec);
}

static gchar *
params_to_string (GnomeDbDsnSpec *spec)
{
	GString *string = NULL;
	gchar *str;
	GdaParameterList *dset;
	GSList *list;

	g_assert (spec->priv->form);
	if (! GNOME_DB_IS_BASIC_FORM (spec->priv->form))
		return NULL;

	dset = gnome_db_basic_form_get_data_set (GNOME_DB_BASIC_FORM (spec->priv->form));
	list = dset->parameters;
	while (list) {
		GdaParameter *param;

		param = GDA_PARAMETER (list->data);
		if (gda_parameter_is_valid (param)) {
			const GValue *value;
			value = gda_parameter_get_value (param);
			str = NULL;
			if (value && !gda_value_is_null ((GValue *) value)) {
				GdaDataHandler *dh;
				GType dtype;

				dtype = gda_parameter_get_g_type (param);
				dh = gda_dict_get_default_handler (default_dict, dtype);
				str = gda_data_handler_get_str_from_value (dh, value);
			}
			if (str && *str) {
				gchar *name;
				if (!string)
					string = g_string_new ("");
				else
					g_string_append_c (string, ';');
				g_object_get (G_OBJECT (list->data), "string_id", &name, NULL);
				g_string_append_printf (string, "%s=%s", name, str);
			}
			g_free (str);
		}
		list = g_slist_next (list);
	}

	str = string ? string->str : NULL;
	if (string)
		g_string_free (string, FALSE);
	return str;
}

/**
 * gnome_db_dsn_spec_set_provider
 * @spec: a #GnomeDbDsnSpec widget
 * @provider: the provider to be used 
 *
 * Updates the displayed fields in @spec to represent the required
 * and possible arguments that a connection to a database through 
 * @provider would require
 */
void
gnome_db_dsn_spec_set_provider (GnomeDbDsnSpec *spec, const gchar *provider)
{
	g_return_if_fail (GNOME_DB_IS_DSN_SPEC (spec));
	g_return_if_fail (spec->priv);

	if (spec->priv->provider)
		g_free (spec->priv->provider);
	spec->priv->provider = NULL;

	if (provider)
		spec->priv->provider = g_strdup (provider);
	adapt_form_widget (spec);
}

/**
 * gnome_db_dsn_spec_get_specs
 * @spec: a #GnomeDbDsnSpec widget
 *
 * Get the currently displayed provider's specific
 * connection string
 *
 * Returns: a new string, or %NULL if no provider have been specified
 */
gchar *
gnome_db_dsn_spec_get_specs (GnomeDbDsnSpec *spec)
{
	g_return_val_if_fail (GNOME_DB_IS_DSN_SPEC (spec), NULL);
	g_return_val_if_fail (spec->priv, NULL);

	switch (spec->priv->type) {
	case OLD_IFACE:
		return g_strdup (gtk_entry_get_text (GTK_ENTRY (spec->priv->form)));
	case PROVIDER_FORM:
		return params_to_string (spec);
	default:
		return NULL;
	}
}

/**
 * gnome_db_dsn_spec_set_specs
 * @spec: a #GnomeDbDsnSpec widget
 * @specs: 
 *
 * Sets the connection string to be displayed in the widget
 */
void
gnome_db_dsn_spec_set_specs (GnomeDbDsnSpec *spec, const gchar *specs_string)
{
	g_return_if_fail (GNOME_DB_IS_DSN_SPEC (spec));
	g_return_if_fail (spec->priv);

	/* save DSN string */
	if (spec->priv->cnc_string)
		g_free (spec->priv->cnc_string);
	spec->priv->cnc_string = NULL;

	if (specs_string)
		spec->priv->cnc_string = g_strdup (specs_string);

	update_form_contents (spec);
}
