/* GNOME DB library
 * Copyright (C) 1999 - 2006 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Vivien Malerba <malerba@gnome-db.org>
 *      Daniel Espinosa <esodan@gmail.com>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <libgda/gda-config.h>
#include <libgnomedb/gnome-db-data-source-selector.h>
#include <libgnomedb/gnome-db-util.h>

#define PARENT_TYPE GTK_TYPE_COMBO_BOX

struct _GnomeDbDataSourceSelectorPrivate {
	guint config_listener;
};

static void gnome_db_data_source_selector_class_init (GnomeDbDataSourceSelectorClass *klass);
static void gnome_db_data_source_selector_init       (GnomeDbDataSourceSelector *selector,
						      GnomeDbDataSourceSelectorClass *klass);
static void gnome_db_data_source_selector_finalize   (GObject *object);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
show_data_sources (GnomeDbDataSourceSelector *selector, const gchar *name)
{
	GdaDataSourceInfo *dsn_info = NULL;
	GtkListStore *model;
	GtkTreeIter iter;

	g_return_if_fail (GNOME_DB_IS_DATA_SOURCE_SELECTOR (selector));

	model = gtk_list_store_new (1, G_TYPE_STRING);

	if (name) {
		dsn_info = gda_config_find_data_source (name);
				
		if (dsn_info) {
			
			gtk_list_store_append (model, &iter);
	                    
			gtk_list_store_set (model, &iter, 0, dsn_info->name, -1);
			
			gtk_combo_box_set_model (GTK_COMBO_BOX (selector), GTK_TREE_MODEL (model));
			gtk_combo_box_set_active_iter (GTK_COMBO_BOX (selector), &iter);
			
			gda_data_source_info_free (dsn_info);
		}
	} 
	else {
		GList *dsn_list, *l;
                
                dsn_list = gda_config_get_data_source_list ();
		
		for (l = dsn_list; l != NULL; l = l->next) {
			dsn_info = (GdaDataSourceInfo *) l->data;
			if (!dsn_info)
				continue;
                        
			/* add the data source to the option menu */
			
			gtk_list_store_append (model, &iter);
			
			gtk_list_store_set (model, &iter, 0, dsn_info->name, -1);
		}
		
		gtk_combo_box_set_model (GTK_COMBO_BOX(selector), GTK_TREE_MODEL(model));
		gtk_combo_box_set_active (GTK_COMBO_BOX (selector), 0);

		gda_config_free_data_source_list (dsn_list);
	}
}

/*
 * GnomeDbDataSourceSelector class implementation
 */

static void
gnome_db_data_source_selector_class_init (GnomeDbDataSourceSelectorClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_data_source_selector_finalize;
}

static void
config_changed_cb (const gchar *path, gpointer user_data)
{
	const gchar *current_dsn;
	GnomeDbDataSourceSelector *selector = (GnomeDbDataSourceSelector *) user_data;

	g_return_if_fail (GNOME_DB_IS_DATA_SOURCE_SELECTOR (selector));

	current_dsn = gnome_db_data_source_selector_get_dsn (selector);
	show_data_sources (selector, NULL);
	gnome_db_data_source_selector_set_dsn (selector, current_dsn);
	g_free (current_dsn);
}

static void
gnome_db_data_source_selector_init (GnomeDbDataSourceSelector *selector,
				    GnomeDbDataSourceSelectorClass *klass)
{
	GtkCellRenderer *renderer;

	g_return_if_fail (GNOME_DB_IS_DATA_SOURCE_SELECTOR (selector));

	/* allocate private structure */
	selector->priv = g_new0 (GnomeDbDataSourceSelectorPrivate, 1);

	/* add listener for configuration */
	selector->priv->config_listener = gda_config_add_listener (
		"/apps/libgda/Datasources",
		(GdaConfigListenerFunc) config_changed_cb,
		selector);

	/* add textual renderer */
	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (selector), renderer, TRUE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (selector), renderer,
				       "text", 0);
	
}

static void
gnome_db_data_source_selector_finalize (GObject *object)
{
	GnomeDbDataSourceSelector *selector = (GnomeDbDataSourceSelector *) object;

	g_return_if_fail (GNOME_DB_IS_DATA_SOURCE_SELECTOR (selector));

	if (selector->priv->config_listener != 0) {
		gda_config_remove_listener (selector->priv->config_listener);
		selector->priv->config_listener = 0;
	}

	/* free memory */
	g_free (selector->priv);
	selector->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_data_source_selector_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDataSourceSelectorClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_data_source_selector_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDataSourceSelector),
			0,
			(GInstanceInitFunc) gnome_db_data_source_selector_init
		};
		type = g_type_register_static (PARENT_TYPE,
					       "GnomeDbDataSourceSelector",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_data_source_selector_new
 * @name: name of data source to display.
 *
 * Create a new #GnomeDbDataSourceSelector, which is just a #GtkComboBox
 * which displays, as its items, all the data sources currently
 * configured in the system. It is useful for connection and configuration
 * screens, where the user has to choose a data source to work with.
 *
 * Returns: the newly created widget.
 */
GtkWidget *
gnome_db_data_source_selector_new (const gchar *name)
{
	GnomeDbDataSourceSelector *selector;

	selector = g_object_new (GNOME_DB_TYPE_DATA_SOURCE_SELECTOR, NULL);
	show_data_sources (selector, name);

	return GTK_WIDGET (selector);
}

/**
 * gnome_db_data_source_selector_get_dsn
 * @name: name of data source to display.
 *
 * Get the Data Source Name (DSN) actualy selected in the #GnomeDbDataSourceSelector.
 *
 * Returns: the DSN name actualy selected.
 */
gchar *
gnome_db_data_source_selector_get_dsn (GnomeDbDataSourceSelector *selector)
{
	GtkTreeIter iter;
	gchar *dsn;
        
        gtk_combo_box_get_active_iter (GTK_COMBO_BOX (selector), &iter);
        
        gtk_tree_model_get (gtk_combo_box_get_model (GTK_COMBO_BOX(selector)), &iter, 0, &dsn, -1);
        

	return dsn;
}

/**
 * gnome_db_data_source_selector_set_dsn
 * @name: name of data source to display.
 *
 * Set the selected Data Source Name (DSN) in the #GnomeDbDataSourceSelector.
 *
 */
void
gnome_db_data_source_selector_set_dsn (GnomeDbDataSourceSelector *selector, const gchar *dsn)
{
        GtkTreeIter iter;
        GtkTreeModel *model;
        gboolean valid;
        gchar *actual_dsn;
        
        model = gtk_combo_box_get_model(GTK_COMBO_BOX(selector));
        
        valid = gtk_tree_model_get_iter_first (model, &iter);
        while(valid) {
                gtk_tree_model_get (model, &iter, 0, &actual_dsn, -1);
                if (!g_ascii_strcasecmp (dsn, actual_dsn)) {
                        gtk_combo_box_set_active_iter (GTK_COMBO_BOX(selector), &iter);
                        break;
                }
                valid = gtk_tree_model_iter_next (model, &iter);
        }
}
