/* gnome-db-handler-passmd5.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-handler-passmd5.h"
#include "gnome-db-entry-passmd5.h"
#include <libgnomedb/gnome-db-server.h>
#include <gmodule.h>

static void gnome_db_handler_passmd5_class_init (GnomeDbHandlerPassmd5Class * class);
static void gnome_db_handler_passmd5_init (GnomeDbHandlerPassmd5 * wid);
static void gnome_db_handler_passmd5_dispose (GObject   * object);

/* plugin initialisation */
GnomeDbDataHandler* plugin_init (GnomeDbServer *srv, GModule *module);

/* GnomeDbDataHandler interface */
static void         gnome_db_handler_passmd5_data_handler_init      (GnomeDbDataHandlerIface *iface);
static GnomeDbDataEntry *gnome_db_handler_passmd5_get_entry_from_value   (GnomeDbDataHandler *dh, const GdaValue *value, 
							      GdaValueType type);
static gchar       *gnome_db_handler_passmd5_get_sql_from_value     (GnomeDbDataHandler *dh, const GdaValue *value);
static gchar       *gnome_db_handler_passmd5_get_str_from_value     (GnomeDbDataHandler *dh, const GdaValue *value);
static GdaValue    *gnome_db_handler_passmd5_get_value_from_sql     (GnomeDbDataHandler *dh, const gchar *sql, 
							      GdaValueType type);
static GdaValue    *gnome_db_handler_passmd5_get_sane_init_value    (GnomeDbDataHandler * dh, GdaValueType type);

static guint        gnome_db_handler_passmd5_get_nb_gda_types       (GnomeDbDataHandler *dh);
static GdaValueType gnome_db_handler_passmd5_get_gda_type_index     (GnomeDbDataHandler *dh, guint index);
static gboolean     gnome_db_handler_passmd5_accepts_gda_type       (GnomeDbDataHandler * dh, GdaValueType type);

static const gchar *gnome_db_handler_passmd5_get_descr              (GnomeDbDataHandler *dh);
static const gchar *gnome_db_handler_passmd5_get_descr_detail       (GnomeDbDataHandler *dh);
static const gchar *gnome_db_handler_passmd5_get_version            (GnomeDbDataHandler *dh);
static gboolean     gnome_db_handler_passmd5_is_plugin              (GnomeDbDataHandler *dh);
static const gchar *gnome_db_handler_passmd5_get_plugin_name        (GnomeDbDataHandler *dh);
static const gchar *gnome_db_handler_passmd5_get_plugin_file        (GnomeDbDataHandler *dh);
static gchar       *gnome_db_handler_passmd5_get_key                (GnomeDbDataHandler *dh);


/* signals */
enum
{
	LAST_SIGNAL
};

static gint gnome_db_handler_passmd5_signals[LAST_SIGNAL] = { };

struct  _GnomeDbHandlerPassmd5Priv {
	gchar          *detailled_descr;
	guint           nb_gda_types;
	GdaValueType   *valid_gda_types;
	GnomeDbServer       *srv;
	GnomeDbDataHandler  *string_handler;

	GModule        *module;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

GType
gnome_db_handler_passmd5_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbHandlerPassmd5Class),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_handler_passmd5_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbHandlerPassmd5),
			0,
			(GInstanceInitFunc) gnome_db_handler_passmd5_init
		};		

		static const GInterfaceInfo data_entry_info = {
			(GInterfaceInitFunc) gnome_db_handler_passmd5_data_handler_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbHandlerPassmd5", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_DATA_HANDLER, &data_entry_info);
	}
	return type;
}

static void
gnome_db_handler_passmd5_data_handler_init (GnomeDbDataHandlerIface *iface)
{
	iface->get_entry_from_value = gnome_db_handler_passmd5_get_entry_from_value;
	iface->get_sql_from_value = gnome_db_handler_passmd5_get_sql_from_value;
	iface->get_str_from_value = gnome_db_handler_passmd5_get_str_from_value;
	iface->get_value_from_sql = gnome_db_handler_passmd5_get_value_from_sql;
	iface->get_value_from_str = NULL;
	iface->get_sane_init_value = gnome_db_handler_passmd5_get_sane_init_value;
	iface->get_nb_gda_types = gnome_db_handler_passmd5_get_nb_gda_types;
	iface->accepts_gda_type = gnome_db_handler_passmd5_accepts_gda_type;
	iface->get_gda_type_index = gnome_db_handler_passmd5_get_gda_type_index;
	iface->get_descr = gnome_db_handler_passmd5_get_descr;
	iface->get_descr_detail = gnome_db_handler_passmd5_get_descr_detail;
	iface->get_version = gnome_db_handler_passmd5_get_version;
	iface->is_plugin = gnome_db_handler_passmd5_is_plugin;
	iface->get_plugin_name = gnome_db_handler_passmd5_get_plugin_name;
	iface->get_plugin_file = gnome_db_handler_passmd5_get_plugin_file;
	iface->get_key = gnome_db_handler_passmd5_get_key;
}


static void
gnome_db_handler_passmd5_class_init (GnomeDbHandlerPassmd5Class * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_handler_passmd5_dispose;
}

static void
gnome_db_handler_passmd5_init (GnomeDbHandlerPassmd5 * hdl)
{
	/* Private structure */
	hdl->priv = g_new0 (GnomeDbHandlerPassmd5Priv, 1);
	hdl->priv->detailled_descr = _("Data handler for passwords stored as MD5 strings");
	hdl->priv->nb_gda_types = 1;
	hdl->priv->valid_gda_types = g_new0 (GdaValueType, 1);
	hdl->priv->valid_gda_types[0] = GDA_VALUE_TYPE_STRING;
	hdl->priv->srv = NULL;
	hdl->priv->string_handler = NULL;
	hdl->priv->module = NULL;

	gnome_db_base_set_name (GNOME_DB_BASE (hdl), gnome_db_handler_passmd5_get_plugin_name (hdl));
	gnome_db_base_set_description (GNOME_DB_BASE (hdl), _("MD5 password handling"));
}

static void
gnome_db_handler_passmd5_dispose (GObject *object)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_HANDLER_PASSMD5 (object));

	hdl = GNOME_DB_HANDLER_PASSMD5 (object);

	if (hdl->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));
		
		g_free (hdl->priv->valid_gda_types);
		hdl->priv->valid_gda_types = NULL;
		
		if (hdl->priv->srv)
			g_object_remove_weak_pointer (G_OBJECT (hdl->priv->srv),
						      (gpointer *) & (hdl->priv->srv));

		g_free (hdl->priv);
		hdl->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

/*
 * Plugin initialization
 */
GnomeDbDataHandler*
plugin_init (GnomeDbServer *srv, GModule *module)
{
	GnomeDbHandlerPassmd5 *passmd5;

	passmd5 = GNOME_DB_HANDLER_PASSMD5 (gnome_db_handler_passmd5_new (srv));
	passmd5->priv->module = module;
	
	return GNOME_DB_DATA_HANDLER (passmd5);
}


/**
 * gnome_db_handler_passmd5_new
 * @srv: a #GnomeDbServer object
 *
 * Creates a data handler for passmd5s
 *
 * Returns: the new object
 */
GObject *
gnome_db_handler_passmd5_new (GnomeDbServer *srv)
{
	GObject *obj;
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	obj = g_object_new (GNOME_DB_HANDLER_PASSMD5_TYPE, NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (obj);

	g_object_add_weak_pointer (G_OBJECT (srv), (gpointer *) &(hdl->priv->srv));
	hdl->priv->srv = srv;

	/* get a pointer to the String Handler which we will use */
	hdl->priv->string_handler = gnome_db_server_get_handler_by_gda (srv, GDA_VALUE_TYPE_STRING);
	g_assert (hdl->priv->string_handler);

	return obj;
}


/* Interface implementation */
static GnomeDbDataEntry *
gnome_db_handler_passmd5_get_entry_from_value (GnomeDbDataHandler *iface, const GdaValue *value, GdaValueType type)
{
	GnomeDbHandlerPassmd5 *hdl;
	GnomeDbDataEntry *de;
	GdaValueType real_type;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	if (value && (gda_value_get_type (value) != GDA_VALUE_TYPE_NULL)) {
                real_type = gda_value_get_type (value);
                g_return_val_if_fail (gnome_db_data_handler_accepts_gda_type (iface, type), NULL);
        }
        else
                real_type = type;

        de = GNOME_DB_DATA_ENTRY (gnome_db_entry_passmd5_new (iface, real_type));
        if (value && (gda_value_get_type (value) != GDA_VALUE_TYPE_NULL))
                gnome_db_data_entry_set_value (de, value);
        else 
                gnome_db_data_entry_set_value (de, NULL);

        return de;
}

static gchar *
gnome_db_handler_passmd5_get_sql_from_value (GnomeDbDataHandler *iface, const GdaValue *value)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return gnome_db_data_handler_get_sql_from_value (hdl->priv->string_handler, value);
}

static gchar *
gnome_db_handler_passmd5_get_str_from_value (GnomeDbDataHandler *iface, const GdaValue *value)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return gnome_db_data_handler_get_str_from_value (hdl->priv->string_handler, value);
}

static GdaValue *
gnome_db_handler_passmd5_get_value_from_sql (GnomeDbDataHandler *iface, const gchar *sql, GdaValueType type)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return gnome_db_data_handler_get_value_from_sql (hdl->priv->string_handler, sql, type);
}

static GdaValue *
gnome_db_handler_passmd5_get_sane_init_value (GnomeDbDataHandler *iface, GdaValueType type)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return gnome_db_data_handler_get_sane_init_value (hdl->priv->string_handler, type);
}

static guint
gnome_db_handler_passmd5_get_nb_gda_types (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), 0);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, 0);

	return hdl->priv->nb_gda_types;
}


static gboolean
gnome_db_handler_passmd5_accepts_gda_type (GnomeDbDataHandler *iface, GdaValueType type)
{
	GnomeDbHandlerPassmd5 *hdl;
	guint i = 0;
	gboolean found = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), FALSE);
	g_return_val_if_fail (type != GDA_VALUE_TYPE_UNKNOWN, FALSE);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, 0);

	while (!found && (i < hdl->priv->nb_gda_types)) {
		if (hdl->priv->valid_gda_types [i] == type)
			found = TRUE;
		i++;
	}

	return found;
}

static GdaValueType
gnome_db_handler_passmd5_get_gda_type_index (GnomeDbDataHandler *iface, guint index)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), GDA_VALUE_TYPE_UNKNOWN);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, GDA_VALUE_TYPE_UNKNOWN);
	g_return_val_if_fail (index < hdl->priv->nb_gda_types, GDA_VALUE_TYPE_UNKNOWN);

	return hdl->priv->valid_gda_types[index];
}

static const gchar *
gnome_db_handler_passmd5_get_descr (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return gnome_db_base_get_description (GNOME_DB_BASE (hdl));
}

static const gchar *
gnome_db_handler_passmd5_get_descr_detail (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return hdl->priv->detailled_descr;
}

static const gchar *
gnome_db_handler_passmd5_get_version (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);
	
	return g_strdup ("V1R0");
}

static gboolean
gnome_db_handler_passmd5_is_plugin (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), FALSE);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, FALSE);

	return TRUE;
}

static const gchar *
gnome_db_handler_passmd5_get_plugin_name (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);
	
	return "Passmd5 Plugin";
}

static const gchar *
gnome_db_handler_passmd5_get_plugin_file (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);
	
	if (hdl->priv->module)
		return g_module_name (hdl->priv->module);
	else
		return NULL;
}

static gchar *
gnome_db_handler_passmd5_get_key (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerPassmd5 *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_PASSMD5 (iface), NULL);
	hdl = GNOME_DB_HANDLER_PASSMD5 (iface);
	g_return_val_if_fail (hdl->priv, NULL);
	
	return g_strdup (gnome_db_base_get_name (GNOME_DB_BASE (hdl)));
}

