/* gnome-db-graphviz.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "gnome-db-graphviz.h"
#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include "gnome-db-join.h"
#include "gnome-db-entity.h"
#include "gnome-db-qfield.h"
#include "gnome-db-field.h"
#include "gnome-db-qf-all.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-qf-value.h"
#include "gnome-db-qf-func.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-condition.h"
#include "gnome-db-renderer.h"
#include "gnome-db-referer.h"
#include "gnome-db-data-set.h"
#include "gnome-db-parameter.h"

/* 
 * Main static functions 
 */
static void gnome_db_graphviz_class_init (GnomeDbGraphvizClass * class);
static void gnome_db_graphviz_init (GnomeDbGraphviz * srv);
static void gnome_db_graphviz_dispose (GObject   * object);
static void gnome_db_graphviz_finalize (GObject   * object);

static void gnome_db_graphviz_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_graphviz_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

static void weak_obj_notify (GnomeDbGraphviz *graph, GObject *obj);
/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;


/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbGraphvizPrivate
{
	GSList *graphed_objects;
};


/* module error */
GQuark gnome_db_graphviz_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_graphviz_error");
	return quark;
}


GType
gnome_db_graphviz_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGraphvizClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_graphviz_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGraphviz),
			0,
			(GInstanceInitFunc) gnome_db_graphviz_init
		};

		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbGraphviz", &info, 0);
	}
	return type;
}

static void
gnome_db_graphviz_class_init (GnomeDbGraphvizClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_graphviz_dispose;
	object_class->finalize = gnome_db_graphviz_finalize;

	/* Properties */
	object_class->set_property = gnome_db_graphviz_set_property;
	object_class->get_property = gnome_db_graphviz_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_graphviz_init (GnomeDbGraphviz *gnome_db_graphviz)
{
	gnome_db_graphviz->priv = g_new0 (GnomeDbGraphvizPrivate, 1);
	gnome_db_graphviz->priv->graphed_objects = NULL;
}

/**
 * gnome_db_graphviz_new
 * @dict: a #GnomeDbDict object
 *
 * Creates a new #GnomeDbGraphviz object
 *
 * Returns: the new object
 */
GObject*
gnome_db_graphviz_new (GnomeDbDict *dict)
{
	GObject *obj;
	GnomeDbGraphviz *gnome_db_graphviz;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	obj = g_object_new (GNOME_DB_TYPE_GRAPHVIZ, "dict", ASSERT_DICT (dict), NULL);
	gnome_db_graphviz = GNOME_DB_GRAPHVIZ (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_graphviz), 0);

	return obj;
}

static void
gnome_db_graphviz_dispose (GObject *object)
{
	GnomeDbGraphviz *gnome_db_graphviz;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRAPHVIZ (object));

	gnome_db_graphviz = GNOME_DB_GRAPHVIZ (object);
	if (gnome_db_graphviz->priv) {
		while (gnome_db_graphviz->priv->graphed_objects) {
			g_object_weak_unref (gnome_db_graphviz->priv->graphed_objects->data, 
					     (GWeakNotify) weak_obj_notify, gnome_db_graphviz);
			weak_obj_notify (gnome_db_graphviz, gnome_db_graphviz->priv->graphed_objects->data);
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_graphviz_finalize (GObject   * object)
{
	GnomeDbGraphviz *gnome_db_graphviz;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRAPHVIZ (object));

	gnome_db_graphviz = GNOME_DB_GRAPHVIZ (object);
	if (gnome_db_graphviz->priv) {
		g_free (gnome_db_graphviz->priv);
		gnome_db_graphviz->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_graphviz_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbGraphviz *gnome_db_graphviz;

	gnome_db_graphviz = GNOME_DB_GRAPHVIZ (object);
	if (gnome_db_graphviz->priv) {
		switch (param_id) {
		case PROP:
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_graphviz_get_property (GObject              *object,
			  guint                 param_id,
			  GValue               *value,
			  GParamSpec           *pspec)
{
	GnomeDbGraphviz *gnome_db_graphviz;
	gnome_db_graphviz = GNOME_DB_GRAPHVIZ (object);
	
	if (gnome_db_graphviz->priv) {
		switch (param_id) {
		case PROP:
			break;
		}	
	}
}

/**
 * gnome_db_graphviz_add_to_graph
 * @graph: a #GnomeDbGraphviz object
 * @obj: a #GObject object to be graphed
 *
 * Adds @obj to be graphed by @graph
 */
void
gnome_db_graphviz_add_to_graph (GnomeDbGraphviz *graph, GObject *obj)
{
	g_return_if_fail (graph && IS_GNOME_DB_GRAPHVIZ (graph));
	g_return_if_fail (graph->priv);

	if (!g_slist_find (graph->priv->graphed_objects, obj)) {
		graph->priv->graphed_objects = g_slist_append (graph->priv->graphed_objects, obj);
		g_object_weak_ref (obj, (GWeakNotify) weak_obj_notify, graph);
	}
}

static void
weak_obj_notify (GnomeDbGraphviz *graph, GObject *obj)
{
	graph->priv->graphed_objects = g_slist_remove (graph->priv->graphed_objects, obj);
}

/*
 * Graphing functions
 */
static GSList *prepare_queries (GnomeDbGraphviz *graph);
static void    do_graph_query (GnomeDbGraphviz *graph, GString *string, GnomeDbQuery *query, gint taboffset);
static void    do_graph_context (GnomeDbGraphviz *graph, GString *string, GnomeDbDataSet *context, gint numcontext, gint taboffset);

/**
 * gnome_db_graphviz_save_file
 * @graph: a #GnomeDbGraphviz object
 * @filename:
 * @error:
 *
 * Saves a dot representation of the @graph object to @filename
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gnome_db_graphviz_save_file (GnomeDbGraphviz *graph, const gchar *filename, GError **error)
{
	gboolean allok = TRUE;
	GString *dot;
	FILE *file;
	size_t size;
	GSList *list, *tmplist;
	gint ncontext = 0;

	g_return_val_if_fail (graph && IS_GNOME_DB_GRAPHVIZ (graph), FALSE);
	g_return_val_if_fail (graph->priv, FALSE);

	/* File handling */
	file = fopen (filename, "w");
	if (!file) {
		TO_IMPLEMENT;
		return FALSE;
	}

	/* actual graph rendering */
	dot = g_string_new ("digraph G {\n"
			    "\tnode [shape=box];\n"
			    "\tnodesep = 0.5;\n");
	list = prepare_queries (graph);
	tmplist = list;
	while (list) {
		if (IS_GNOME_DB_QUERY (list->data)) 
			do_graph_query (graph, dot, GNOME_DB_QUERY (list->data), 1);
		list = g_slist_next (list);
	}
	g_slist_free (tmplist);

	list = graph->priv->graphed_objects;
	while (list) {
		if (IS_GNOME_DB_DATA_SET (list->data))
			do_graph_context (graph, dot, GNOME_DB_DATA_SET (list->data), ncontext++, 1);
		list = g_slist_next (list);
	}

	g_string_append (dot, "}\n");

	/* writing to filename */
	size = fwrite (dot->str, sizeof (gchar), strlen (dot->str), file);
	if (size != strlen (dot->str)) {
		TO_IMPLEMENT;
		allok = FALSE;
	}

	/* Finish */
	fclose (file);
	g_string_free (dot, TRUE);

	return allok;
}


/*
 * Queries
 */

static void    get_depend_query (GnomeDbQuery *query, GSList **deplist, GSList **alldisplayed);

static void
prepare_single_query (GnomeDbQuery *query, GSList **top_queries, GSList **all_queries)
{
	GSList *list;
	GSList *deplist, *alllist;

	get_depend_query (query, &deplist, &alllist);
	list = deplist;
	while (list) {
		if (!g_slist_find (*all_queries, list->data))
			prepare_single_query (GNOME_DB_QUERY (list->data), top_queries, all_queries);
		list = g_slist_next (list);
	}
	g_slist_free (deplist);

	if (alllist)
		*all_queries = g_slist_concat (*all_queries, alllist);

	if (!g_slist_find (*top_queries, query))
		*top_queries = g_slist_append (*top_queries, query);
	if (!g_slist_find (*all_queries, query))
		*all_queries = g_slist_append (*all_queries, query);
}

static void
get_depend_query (GnomeDbQuery *query, GSList **deplist, GSList **alldisplayed)
{
	GSList *list, *tmplist;

	*deplist = NULL;
	*alldisplayed = NULL;

	list = gnome_db_query_get_sub_queries (query);
	*alldisplayed = g_slist_concat (*alldisplayed, list);

	list = g_slist_copy (gnome_db_query_get_param_sources (query));
	*alldisplayed = g_slist_concat (*alldisplayed, list);

	list = gnome_db_query_get_all_fields (query);
	tmplist = list;
	while (list) {
		GnomeDbBase *base;
		if (g_object_class_find_property (G_OBJECT_GET_CLASS (list->data), "value_provider")) {
			g_object_get (G_OBJECT (list->data), "value_provider", &base, NULL);
			if (base) {
				GnomeDbQuery *dquery = GNOME_DB_QUERY (gnome_db_field_get_entity (GNOME_DB_FIELD (base)));
				if (!g_slist_find (*alldisplayed, dquery) && (dquery != query))
					*deplist = g_slist_append (*deplist, dquery);
			}
		}
		list = g_slist_next (list);
	}
	g_slist_free (tmplist);
}

static GSList *
prepare_queries (GnomeDbGraphviz *graph)
{
	GSList *top_queries = NULL, *all_queries = NULL, *list;

	list = graph->priv->graphed_objects;
	while (list) {
		if (IS_GNOME_DB_QUERY (list->data) && !g_slist_find (all_queries, list->data)) 
			prepare_single_query (GNOME_DB_QUERY (list->data), &top_queries, &all_queries);
		list = g_slist_next (list);
	}	

	g_slist_free (all_queries);

	return top_queries;
}

static gchar *render_qf_all_label (GnomeDbGraphviz *graph, GnomeDbQfAll *field);
static gchar *render_qf_field_label (GnomeDbGraphviz *graph, GnomeDbQfField *field);
static gchar *render_qf_value_label (GnomeDbGraphviz *graph, GnomeDbQfValue *field);
static gchar *render_qf_func_label (GnomeDbGraphviz *graph, GnomeDbQfFunc *field);
static void
do_graph_query (GnomeDbGraphviz *graph, GString *string, GnomeDbQuery *query, gint taboffset)
{
	gchar *str, *qid, *qtype, *sql;
	gint i;
	GSList *list;
	const GSList *clist;
	GnomeDbCondition *cond;

	/* offset string */
	str = g_new0 (gchar, taboffset+1);
	for (i=0; i<taboffset; i++)
		str [i] = '\t';
	
	qid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (query));
	g_string_append_printf (string, "%ssubgraph cluster%s {\n", str, qid);

	/* param sources */
	clist = gnome_db_query_get_param_sources (query);
	if (clist) {
		g_string_append_printf (string, "%s\tsubgraph cluster%sps {\n", str, qid);
		g_string_append_printf (string, "%s\t\tnode [style=filled, color=black, fillcolor=white, fontsize=10];\n", str);
		while (clist) {
			do_graph_query (graph, string, GNOME_DB_QUERY (clist->data), taboffset+2);
			clist = g_slist_next (clist);
		}
		g_string_append_printf (string, "\n%s\t\tlabel = \"Parameter sources\";\n", str);
		g_string_append_printf (string, "%s\t\tstyle = filled;\n", str);
		g_string_append_printf (string, "%s\t\tcolor = white;\n", str);

		g_string_append_printf (string, "%s\t}\n", str);
	}
	
	/* sub queries */
	list = gnome_db_query_get_sub_queries (query);
	if (list) {
		GSList *tmplist = list;
		g_string_append_printf (string, "%s\tsubgraph cluster%ssub {\n", str, qid);
		g_string_append_printf (string, "%s\t\tnode [style=filled, color=black, fillcolor=white, fontsize=10];\n", str);
		while (list) {
			do_graph_query (graph, string, GNOME_DB_QUERY (list->data), taboffset+2);
			list = g_slist_next (list);
		}
		g_string_append_printf (string, "\n%s\t\tlabel = \"Sub queries\";\n", str);
		g_string_append_printf (string, "%s\t\tstyle = filled;\n", str);
		g_string_append_printf (string, "%s\t\tcolor = white;\n", str);

		g_string_append_printf (string, "%s\t}\n", str);
		g_slist_free (tmplist);
	}
	

	/* targets */
	list = gnome_db_query_get_targets (query);
	if (list) {
		GSList *tmplist = list;
		GnomeDbEntity *ent;

		while (list) {
			gchar *tid;

			tid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
			g_string_append_printf (string, "%s\t\"%s\" ", str, tid);
			ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (list->data));
			g_string_append_printf (string, "[label=\"%s (%s)\" style=filled fillcolor=orange];\n",
						gnome_db_base_get_name (GNOME_DB_BASE (ent)), gnome_db_target_get_alias (GNOME_DB_TARGET (list->data)));
			g_free (tid);
			list = g_slist_next (list);
		}
		g_slist_free (tmplist);

		list = gnome_db_query_get_joins (query);
		tmplist = list;
		while (list) {
			GnomeDbTarget *target;
			gchar *tid;
			gchar *jtype = "none";

			target = gnome_db_join_get_target_1 (GNOME_DB_JOIN (list->data));
			tid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (target));
			g_string_append_printf (string, "%s\t\"%s\" -> ", str, tid);
			g_free (tid);
			target = gnome_db_join_get_target_2 (GNOME_DB_JOIN (list->data));
			tid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (target));
			g_string_append_printf (string, "\"%s\" ", tid);
			g_free (tid);

			switch (gnome_db_join_get_join_type (GNOME_DB_JOIN (list->data))) {
			case GNOME_DB_JOIN_TYPE_LEFT_OUTER:
				jtype = "back";
				break;
			case GNOME_DB_JOIN_TYPE_RIGHT_OUTER:
				jtype = "forward";
				break;
			case GNOME_DB_JOIN_TYPE_FULL_OUTER:
				jtype = "both";
				break;
			default:
				jtype = NULL;
			}
			if (jtype)
				g_string_append_printf (string, "[dir=%s, style=filled arrowhead=odot];\n", jtype);
			else
				g_string_append (string, "[dir=none, style=filled];\n");
			list = g_slist_next (list);
		}
		g_slist_free (tmplist);
	}

	/* fields */
	g_object_get (G_OBJECT (query), "really_all_fields", &list, NULL);
	if (list) {
		GSList *tmplist = list;

		while (list) {
			gchar *fid;
			gchar *label = NULL;

			if (IS_GNOME_DB_QF_ALL (list->data))
				label = render_qf_all_label (graph, GNOME_DB_QF_ALL (list->data));
			if (IS_GNOME_DB_QF_FIELD (list->data))
				label = render_qf_field_label (graph, GNOME_DB_QF_FIELD (list->data));
			if (IS_GNOME_DB_QF_VALUE (list->data))
				label = render_qf_value_label (graph, GNOME_DB_QF_VALUE (list->data));
			if (IS_GNOME_DB_QF_FUNC (list->data))
				label = render_qf_func_label (graph, GNOME_DB_QF_FUNC (list->data));
			if (!label)
				label = g_strdup ("{??? | {?|?|?|?}}");

			fid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
			g_string_append_printf (string, "%s\t\"%s\" ", str, fid);
			if (gnome_db_referer_is_active (GNOME_DB_REFERER (list->data)))
				g_string_append_printf (string, "[shape=record, label=\"%s\", style=filled, fillcolor=light_blue];\n",
							label);
			else
				g_string_append_printf (string, "[shape=record, label=\"%s\", style=filled, fillcolor=indianred];\n",
							label);
			g_free (label);
			g_free (fid);
			list = g_slist_next (list);
		}

		list = tmplist;
		while (list) {
			GnomeDbTarget *target = NULL;
			
			/* target link */
			if (IS_GNOME_DB_QF_ALL (list->data))
				target = gnome_db_qf_all_get_target (GNOME_DB_QF_ALL (list->data));
			if (IS_GNOME_DB_QF_FIELD (list->data))
				target = gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (list->data));

			if (target) {
				gchar *fid, *tid = NULL;
				fid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
				tid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (target));
				g_string_append_printf (string, "%s\t\"%s\" -> \"%s\";\n", str, fid, tid);
				g_free (fid);
				g_free (tid);
			}

			/* value provider link */
			if (g_object_class_find_property (G_OBJECT_GET_CLASS (list->data), "value_provider")) {
				GnomeDbBase *base;

				g_object_get (G_OBJECT (list->data), "value_provider", &base, NULL);
				if (base) {
					gchar *fid, *bid;

					fid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
					bid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
					g_string_append_printf (string, "%s\t\"%s\" -> \"%s\" [style=dotted];\n", str, fid, bid);
					g_free (fid);
					g_free (bid);
				}
			}
			
			/* arguments for functions */
			if (IS_GNOME_DB_QF_FUNC (list->data)) {
				GSList *args;

				args = gnome_db_qf_func_get_args (GNOME_DB_QF_FUNC (list->data));
				if (args) {
					GSList *tmplist = args;
					gchar *fid;

					fid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
					while (args) {
						if (args->data) {
							gchar *did;

							did = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (args->data));
							g_string_append_printf (string, "%s\t\"%s\" -> \"%s\" [style=filled];\n", str, fid, did);
							g_free (did);
						}
						args = g_slist_next (args);
					}
					g_free (fid);
					g_slist_free (tmplist);
				}
			}

			list = g_slist_next (list);
		}
	}

	switch (gnome_db_query_get_query_type (query)) {
	case GNOME_DB_QUERY_TYPE_SELECT:
		qtype = "SELECT";
		break;
	case GNOME_DB_QUERY_TYPE_INSERT:
		qtype = "INSERT";
		break;
	case GNOME_DB_QUERY_TYPE_UPDATE:
		qtype = "UPDATE";
		break;
	case GNOME_DB_QUERY_TYPE_DELETE:
		qtype = "DELETE";
		break;
	case GNOME_DB_QUERY_TYPE_UNION:
		qtype = "UNION";
		break;
	case GNOME_DB_QUERY_TYPE_INTERSECT:
		qtype = "INTERSECT";
		break;
	case GNOME_DB_QUERY_TYPE_EXCEPT:
		qtype = "EXCEPT";
		break;
	case GNOME_DB_QUERY_TYPE_NON_PARSED_SQL:
		qtype = "SQL";
		break;
	default:
		qtype = "???";
		break;
	}

	/* condition */
	cond = gnome_db_query_get_condition (query);
	if (cond) {
		GSList *tmplist, *list;
		gchar *strcond;
		GError *error = NULL;

		strcond = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (cond), NULL, 0, &error);
		if (error) {
			g_string_append_printf (string, "%s\t\"%s:Cond\" [label=\"%s\" style=filled fillcolor=yellow];\n", str, qid,
						error->message);
			g_error_free (error);
		}
		else {
			g_string_append_printf (string, "%s\t\"%s:Cond\" [label=\"%s\" style=filled fillcolor=yellow];\n", str, qid, strcond);
			g_free (strcond);
		}

		list = gnome_db_condition_get_ref_objects_all (cond);
		tmplist = list;
		while (list) {	
			gchar *fid;

			fid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
			g_string_append_printf (string, "%s\t\"%s:Cond\" -> \"%s\";\n", str, qid, fid);
		
			list = g_slist_next (list);
		}
	}


	/* query attributes */
	sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, 0, NULL);
	g_string_append_printf (string, "\n%s\tlabel = \"%s Query %d: %s\";\n", str, qtype,
				gnome_db_base_get_id (GNOME_DB_BASE (query)), gnome_db_base_get_name (GNOME_DB_BASE (query)));
	g_free (sql);
	g_string_append_printf (string, "%s\tstyle = filled;\n", str);
	g_string_append_printf (string, "%s\tcolor = lightgrey;\n", str);
	g_string_append_printf (string, "%s}\n", str);
	g_free (qid);
	g_free (str);
}

static gchar *
render_qf_all_label (GnomeDbGraphviz *graph, GnomeDbQfAll *field)
{
	GString *retval;
	gchar *str;
	const gchar *cstr;

	retval = g_string_new ("{");
	cstr = gnome_db_base_get_name (GNOME_DB_BASE (field));
	if (cstr)
		g_string_append (retval, cstr);

	g_string_append (retval, " | {*");

	if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
		g_string_append (retval, " |V");
	else
		g_string_append (retval, " |-");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (field)))
		g_string_append (retval, " |I");
	else
		g_string_append (retval, " |-");
	g_string_append (retval, " |-}}");

	str = retval->str;
	g_string_free (retval, FALSE);
	return str;
}

static gchar *
render_qf_field_label (GnomeDbGraphviz *graph, GnomeDbQfField *field)
{
	GString *retval;
	gchar *str;
	const gchar *cstr;
	GnomeDbField *ref;

	retval = g_string_new ("{");
	cstr = gnome_db_base_get_name (GNOME_DB_BASE (field));
	if (cstr)
		g_string_append (retval, cstr);
	ref = gnome_db_qf_field_get_ref_field (field);
	if (ref)
		g_string_append_printf (retval, " (%s)", gnome_db_base_get_name (GNOME_DB_BASE (ref)));

	g_string_append (retval, " | {Field");

	if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
		g_string_append (retval, " |V");
	else
		g_string_append (retval, " |-");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (field)))
		g_string_append (retval, " |I");
	else
		g_string_append (retval, " |-");
	g_string_append (retval, " |-}}");

	str = retval->str;
	g_string_free (retval, FALSE);
	return str;
}

static gchar *
render_qf_value_label (GnomeDbGraphviz *graph, GnomeDbQfValue *field)
{
	GString *retval;
	gchar *str;
	const gchar *cstr;
	const GdaValue *value;

	retval = g_string_new ("{");
	cstr = gnome_db_base_get_name (GNOME_DB_BASE (field));
	if (cstr)
		g_string_append (retval, cstr);
	value = gnome_db_qf_value_get_value (field);
	if (value) {
		str = gda_value_stringify (value);
		g_string_append_printf (retval, " (%s)", str);
		g_free (str);
	}

	g_string_append (retval, " | {Value");

	if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
		g_string_append (retval, " |V");
	else
		g_string_append (retval, " |-");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (field)))
		g_string_append (retval, " |I");
	else
		g_string_append (retval, " |-");
	if (gnome_db_qf_value_is_parameter (field))
		g_string_append (retval, " |P}}");
	else
		g_string_append (retval, " |-}}");

	str = retval->str;
	g_string_free (retval, FALSE);
	return str;
}

static gchar *
render_qf_func_label (GnomeDbGraphviz *graph, GnomeDbQfFunc *field)
{
	GString *retval;
	gchar *str;
	const gchar *cstr;
	GnomeDbServerFunction *func;

	retval = g_string_new ("{");
	cstr = gnome_db_base_get_name (GNOME_DB_BASE (field));
	if (cstr)
		g_string_append (retval, cstr);
	func = gnome_db_qf_func_get_ref_func (field);
	if (func)
		g_string_append_printf (retval, " (%s)", gnome_db_base_get_name (GNOME_DB_BASE (func)));

	g_string_append (retval, " | {Func()");

	if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
		g_string_append (retval, " |V");
	else
		g_string_append (retval, " |-");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (field)))
		g_string_append (retval, " |I");
	else
		g_string_append (retval, " |-");
	g_string_append (retval, " |-}}");

	str = retval->str;
	g_string_free (retval, FALSE);
	return str;
}



/*
 * Context rendering
 */
static void
do_graph_context (GnomeDbGraphviz *graph, GString *string, GnomeDbDataSet *context, gint numcontext, gint taboffset)
{
	gchar *str;
	gint i, j;
	GSList *list;

	/* offset string */
	str = g_new0 (gchar, taboffset+1);
	for (i=0; i<taboffset; i++)
		str [i] = '\t';

	/* parameters */
	list = context->parameters;
	while (list) {
		GSList *dest;
		gchar *fid;

		g_string_append_printf (string, "%sParameter%p [label=\"%s (%d)\", shape=ellipse, style=filled, fillcolor=linen];\n", 
					str, list->data, gnome_db_base_get_name (GNOME_DB_BASE (list->data)), numcontext);
		dest = gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (list->data));
		while (dest) {
			fid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (dest->data));
			g_string_append_printf (string, "%sParameter%p -> \"%s\";\n", str, list->data, fid);
			g_free (fid);
			dest = g_slist_next (dest);
		}
		list = g_slist_next (list);
	}

	/* context nodes */
	j = 0;
	g_string_append_printf (string, "%ssubgraph clustercontext%d {\n", str, numcontext);	
	list = context->nodes;
	while (list) {
		GnomeDbDataSetNode *node = GNOME_DB_DATA_SET_NODE (list->data);
		g_string_append_printf (string, "%s\tNode%p [label=\"Node%d\", shape=octagon];\n", str, list->data, j);
		
		if (node->param)
			g_string_append_printf (string, "%s\tNode%p -> Parameter%p [constraint=false];\n", 
						str, list->data, node->param);
		else {
			GSList *params = node->params;
			while (params) {
				g_string_append_printf (string, "%s\tNode%p -> Parameter%p;\n", str, list->data, params->data);
				params = g_slist_next (params);
			}
		}
		list = g_slist_next (list);
		j++;
	}
	g_string_append_printf (string, "%s\tlabel = \"Context %d\";\n", str, numcontext);
	g_string_append_printf (string, "%s}\n", str);
	

	g_free (str);
}
