/* utility.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <glib/gprintf.h>
#include "gnome-db-decl.h"
#include "utility.h"
#include <libgda/libgda.h>
#include "gnome-db-data-entry.h"
#include "gnome-db-result-set.h"
#include "gnome-db-parameter.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-server.h"
#include "gnome-db-query.h"
#include "gnome-db-renderer.h"
#include "gnome-db-entity.h"
#include "gnome-db-field.h"
#include "gnome-db-qfield.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-data-model.h"
#include "gnome-db-data-proxy.h"
#include "gnome-db-table-field.h"


/**
 * utility_entry_build_actions_menu
 * @obj_data:
 * @attrs:
 * @function:
 *
 * Creates a GtkMenu widget which contains the possible actions on a data entry which 
 * attributes are @attrs. After the menu has been displayed, and when an action is selected,
 * the @function callback is called with the 'user_data' being @obj_data.
 *
 * The menu item which emits the signal has a "action" attribute which describes what action the
 * user has requested.
 *
 * Returns: the new menu
 */
GtkWidget *
utility_entry_build_actions_menu (GObject *obj_data, guint attrs, GCallback function)
{
	GtkWidget *menu, *mitem;
	gchar *str;
	gboolean nullact = FALSE;
	gboolean defact = FALSE;
	gboolean reset = FALSE;

	gboolean value_is_null;
	gboolean value_is_modified;
	gboolean value_is_default;

	menu = gtk_menu_new ();

	/* which menu items to make sensitive ? */
	value_is_null = attrs & GNOME_DB_VALUE_IS_NULL;
	value_is_modified = ! (attrs & GNOME_DB_VALUE_IS_UNCHANGED);
	value_is_default = attrs & GNOME_DB_VALUE_IS_DEFAULT;

	if ((attrs & GNOME_DB_VALUE_CAN_BE_NULL) && 
	    !(attrs & GNOME_DB_VALUE_IS_NULL))
		nullact = TRUE;
	if ((attrs & GNOME_DB_VALUE_CAN_BE_DEFAULT) && 
	    !(attrs & GNOME_DB_VALUE_IS_DEFAULT))
		defact = TRUE;
	if (!(attrs & GNOME_DB_VALUE_IS_UNCHANGED)) {
		if (attrs & GNOME_DB_VALUE_HAS_VALUE_ORIG) 
			reset = TRUE;
	}

	/* set to NULL item */
	str = g_strdup (_("Unset"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem),
					value_is_null);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GNOME_DB_VALUE_IS_NULL));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, nullact);

	/* default value item */
	str = g_strdup (_("Set to default value"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), 
					value_is_default);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GNOME_DB_VALUE_IS_DEFAULT));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, defact);
		
	/* reset to original value item */
	str = g_strdup (_("Reset to original value"));
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), 
					!value_is_modified);
	gtk_widget_show (mitem);
	g_object_set_data (G_OBJECT (mitem), "action", GUINT_TO_POINTER (GNOME_DB_VALUE_IS_UNCHANGED));
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (function), obj_data);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, reset);

	return menu;
}


/**
 * utility_entry_build_info_colors_array
 * 
 * Creates an array of colors for the different states of an entry:
 *    Valid   <-> No special color
 *    Null    <-> Green
 *    Default <-> Blue
 *    Invalid <-> Red
 * Each status (except Valid) is represented by two colors for GTK_STATE_NORMAL and
 * GTK_STATE_PRELIGHT.
 *
 * Returns: a new array of 6 colors
 */
GdkColor **utility_entry_build_info_colors_array ()
{
	GdkColor **colors;
	GdkColor *color;
	
	colors = g_new0 (GdkColor *, 6);
	
	/* Green color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_NULL, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[0] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_NULL, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[1] = color;
	
	
	/* Blue color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_DEFAULT, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[2] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_DEFAULT, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[3] = color;
	
	
	/* Red color */
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_NORMAL_INVALID, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[4] = color;
	
	color = g_new0 (GdkColor, 1);
	gdk_color_parse (GNOME_DB_COLOR_PRELIGHT_INVALID, color);
	if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
		g_free (color);
		color = NULL;
	}
	colors[5] = color;

	return colors;
}


/**
 * utility_query_execute_modif
 * @query: the #GnomeDbQuery to be executed
 * @context: a #GnomeDbDataSet object
 * @ask_confirm_insert:
 * @ask_confirm_update:
 * @ask_confirm_delete:
 * @parent_window: a #GtkWindow object, or a #GtkWidget and the parent window will be found automatically
 * @user_cancelled: a place to store if the user cancelled the query if the choice was given, or %NULL
 * @query_error: a place to store if there was an error, or %NULL
 *
 * Executes @query and displays question and information dialogs if necessary.
 * If a user confirmation was required and the user cancelled the execution, then 
 * the returned value is FALSE.
 *
 * Returns: TRUE if the query was executed.
 */
gboolean
utility_query_execute_modif (GnomeDbQuery *query, GnomeDbDataSet *context,
			     gboolean ask_confirm_insert,
			     gboolean ask_confirm_update,
			     gboolean ask_confirm_delete,
			     GtkWidget *parent_window,
			     gboolean *user_cancelled,
			     gboolean *query_error)
{
	gchar *sql = NULL;
	GnomeDbQueryType qtype;
	gchar *confirm = NULL;
	gboolean do_execute = TRUE;
	gboolean allok = TRUE;
	GError *error = NULL;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), FALSE);
	
	/* find the GtkWindow object for @parent_window */
	while (parent_window && !GTK_IS_WINDOW (parent_window)) 
		parent_window = gtk_widget_get_parent (parent_window);

	sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), context, 0, &error);
	qtype = gnome_db_query_get_query_type (query);

	switch (qtype) {
	case GNOME_DB_QUERY_TYPE_INSERT:
		if (ask_confirm_insert)
			confirm = _("Execute the following insertion query ?");
		break;
	case GNOME_DB_QUERY_TYPE_UPDATE:
		if (ask_confirm_update)
			confirm = _("Execute the following update query ?");
		break;
	case GNOME_DB_QUERY_TYPE_DELETE:
		if (ask_confirm_delete)
			confirm = _("Execute the following deletion query ?");
		break;
	default:
		g_assert_not_reached ();
	}

	if (user_cancelled)
		*user_cancelled = FALSE;
	if (query_error)
		*query_error = FALSE;

	if (sql) {
		if (confirm) {
			GtkWidget *dlg;
			gint result;
			gchar *msg;
			
			msg = g_strdup_printf (_("<b><big>%s</big></b>\n"
						 "<small>The preferences require a confirmation for the "
						 "following query</small>\n\n%s"), confirm, sql);
			dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
						      GTK_MESSAGE_QUESTION,
						      GTK_BUTTONS_YES_NO, msg);
			g_free (msg);
			gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dlg)->label), TRUE);
			result = gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
			do_execute = (result == GTK_RESPONSE_YES);
			if (user_cancelled)
				*user_cancelled = !do_execute;
		}
			
		if (do_execute) {
			GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
				
#ifdef debug
			g_print ("MODIF SQL: %s\n", sql);
#endif
			gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (dict), sql, 
								GNOME_DB_SERVER_QUERY_SQL, &error);
			if (error) {
				GtkWidget *dlg;
				gchar *message;
				
				message = g_strdup (error->message);
				g_error_free (error);
				dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
							      GTK_MESSAGE_ERROR,
							      GTK_BUTTONS_CLOSE,
							      message);
				g_free (message);
				gtk_dialog_run (GTK_DIALOG (dlg));
				gtk_widget_destroy (dlg);
				allok = FALSE;

				if (query_error)
					*query_error = TRUE;
			}
		}
		else
			allok = FALSE;
		
		g_free (sql);
	}
	else {
		GtkWidget *dlg;
		gchar *message;
		
		if (error) {
			message = g_strdup_printf (_("The following error occurred while preparing the query:\n%s"),
						   error->message);
			g_error_free (error);
		}
		else
			message = g_strdup_printf (_("An unknown error occurred while preparing the query."));
		
		dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
					      GTK_MESSAGE_ERROR,
					      GTK_BUTTONS_CLOSE,
					      message);
		g_free (message);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
		allok = FALSE;

		if (query_error)
			*query_error = TRUE;
	}

	return allok;
}




static void compute_shown_columns_index (DataSetNodeInfo *info);
static void compute_ref_columns_index (DataSetNodeInfo *info);

/**
 * utility_data_set_node_info_create
 * @data_set:
 * @node:
 *
 * Creates a new DataSetNodeInfo structure and initializes it.
 *
 * Returns: the new structure (to be freed using utility_data_set_node_info_destroy())
 */
DataSetNodeInfo *
utility_data_set_node_info_create (GnomeDbDataSet *data_set, GnomeDbDataSetNode *node)
{
	DataSetNodeInfo *info;

	g_assert (data_set && IS_GNOME_DB_DATA_SET (data_set));
	g_assert (node && g_slist_find (data_set->nodes, node));
	g_assert (node->params);
	g_assert (node->data_for_param && IS_GNOME_DB_DATA_MODEL (node->data_for_param));

	info = g_new0 (DataSetNodeInfo, 1);

	g_object_ref (data_set);
	info->data_set = data_set;
	info->node = node;
	info->data_model = node->data_for_param;
	
	compute_shown_columns_index (info);
	compute_ref_columns_index (info);

	return info;
}

static void
compute_shown_columns_index (DataSetNodeInfo *info)
{
        gint ncols, nparams;
        gint *mask = NULL, masksize = 0;

        nparams = g_slist_length (info->node->params);
        g_return_if_fail (nparams > 0);
        ncols = gda_data_model_get_n_columns (GDA_DATA_MODEL (info->data_model));
        g_return_if_fail (ncols > 0);

        if (ncols > nparams) {
                /* we only want columns which are not parameter values */
                gint i, current = 0;

                masksize = ncols - nparams;
                mask = g_new0 (gint, masksize);
                for (i = 0; i < ncols ; i++) {
                        GSList *list = info->node->params;
                        gboolean found = FALSE;
                        while (list && !found) {
                                if (GPOINTER_TO_INT (g_hash_table_lookup (info->node->pos_for_param, list->data)) == i)
                                        found = TRUE;
                                else
                                        list = g_slist_next (list);
                        }
                        if (!found) {
                                mask[current] = i;
                                current ++;
                        }
                }
                masksize = current;
        }
        else {
                /* we want all the columns */
                gint i;

                masksize = ncols;
                mask = g_new0 (gint, masksize);
                for (i=0; i<ncols; i++) {
                        mask[i] = i;
                }
        }

        info->shown_n_cols = masksize;
        info->shown_cols_index = mask;
}

static void
compute_ref_columns_index (DataSetNodeInfo *info)
{
        gint ncols, nparams;
        gint *mask = NULL, masksize = 0;

        nparams = g_slist_length (info->node->params);
        g_return_if_fail (nparams > 0);
        ncols = gda_data_model_get_n_columns (GDA_DATA_MODEL (info->data_model));
        g_return_if_fail (ncols > 0);

        if (ncols > nparams) {
                /* we only want columns which are parameters values */
                gint i, current = 0;

                masksize = ncols - nparams;
                mask = g_new0 (gint, masksize);
                for (i=0; i<ncols ; i++) {
                        GSList *list = info->node->params;
                        gboolean found = FALSE;
                        while (list && !found) {
                                if (GPOINTER_TO_INT (g_hash_table_lookup (info->node->pos_for_param, list->data)) == i)
                                        found = TRUE;
                                else
                                        list = g_slist_next (list);
                        }
                        if (found) {
                                mask[current] = i;
                                current ++;
                        }
                }
                masksize = current;
        }
        else {
                /* we want all the columns */
                gint i;

                masksize = ncols;
                mask = g_new0 (gint, masksize);
                for (i=0; i<ncols; i++) {
                        mask[i] = i;
                }
        }

        info->ref_n_cols = masksize;
        info->ref_cols_index = mask;
}

/**
 * utility_data_set_node_info_destroy
 * @info:
 *
 * Destroys @info and cleans-up all its internals
 */
void
utility_data_set_node_info_destroy (DataSetNodeInfo *info)
{
	g_object_unref (info->data_set);

	if (info->shown_cols_index) 
		g_free (info->shown_cols_index);
	if (info->ref_cols_index) 
		g_free (info->ref_cols_index);

	g_free (info);
}

/**
 * utility_data_set_node_info_compute_values
 * @info:
 * @proxy:
 * @iter:
 *
 * Makes a list of the values in @proxy, at row @iter, which correspond to the parameters in the @info
 * structure.
 *
 * Returns: a new list which must be freed using g_list_free() (don't change or free the values in the list)
 */
GList *
utility_data_set_node_info_compute_values (DataSetNodeInfo *info, GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	GList *values = NULL;
	GSList *list;
	GnomeDbDataModel *proxy_model;
	gint col;
	GdaValue *value;

	g_return_val_if_fail (info, NULL);
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), NULL);
	proxy_model = gnome_db_data_proxy_get_model (proxy);
	g_return_val_if_fail (proxy_model, NULL);

	/* list the values in proxy_model for each param in info->node->params */
	list = info->node->params;
	while (list) {
		col = gnome_db_data_model_get_column_at_param (proxy_model, info->data_set, list->data);
		gtk_tree_model_get (GTK_TREE_MODEL (proxy), iter, col, &value, -1);
		values = g_list_append (values, value);
		
		list = g_slist_next (list);
	}
	
	return values;
}

/**
 * utility_data_set_node_info_compute_values_ext
 * @info:
 * @proxy:
 * @iter:
 * 
 * Makes a list of the values in @proxy, at row @iter, which correspond to the parameters in the @info
 * structure, but also adding values to make it a complete valid row of node->data_for_param. Note: this feature
 * is not available on all the GnomeDbDataModel present in @proxy, so if it fails, then try 
 * utility_data_set_node_info_compute_values().
 *
 * Returns: a new list which must be freed using g_list_free() (don't change or free the values in the list)
 */
GList *
utility_data_set_node_info_compute_values_ext (DataSetNodeInfo *info, GnomeDbDataProxy *proxy, GtkTreeIter *iter)
{
	gint i, max, col, nb_cols;
	GList *values = NULL;
	GnomeDbDataModel *proxy_model;
	GdaValue *value;
	gboolean modified = FALSE;
	guint attributes;

	g_return_val_if_fail (info, NULL);
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), NULL);
	proxy_model = gnome_db_data_proxy_get_model (proxy);
	g_return_val_if_fail (proxy_model, NULL);
	
	if (!IS_GNOME_DB_RESULT_SET (proxy_model))
		return NULL;

	/* compute values */
	max = gda_data_model_get_n_columns (GDA_DATA_MODEL (info->node->data_for_param));
	nb_cols = gnome_db_data_proxy_get_n_columns (proxy);

	for (i = 0 ; (i < max) && !modified; i++) {
		col = gnome_db_result_set_find_column_ext (GNOME_DB_RESULT_SET (proxy_model),
							   info->node, i);
		g_assert (col >= 0);
		gtk_tree_model_get (GTK_TREE_MODEL (proxy), iter, col, &value, 
				    nb_cols + col, &attributes, -1);
		modified = ! (attributes & GNOME_DB_VALUE_IS_UNCHANGED);
		values = g_list_append (values, value);
	}

	/* if the proxy has been modified for the concerned columns, then this function can't be used */
	if (modified) {
		g_list_free (values);
		values = NULL;
	}

	return values;
}

/**
 * utility_data_set_node_info_compute_attributes
 * @info:
 * @proxy:
 * @iter:
 * @to_be_deleted: a place to store if the row is to be deleted, or %NULL
 *
 * Get the attributes of the values in @proxy, at row @iter, which correspond to the parameters in the @info
 *
 * Returns: the attributes
 */
guint
utility_data_set_node_info_compute_attributes (DataSetNodeInfo *info, GnomeDbDataProxy *proxy, GtkTreeIter *iter, 
					       gboolean *to_be_deleted)
{
	guint attributes = GNOME_DB_VALUE_IS_NULL | GNOME_DB_VALUE_CAN_BE_NULL |
		GNOME_DB_VALUE_IS_DEFAULT | GNOME_DB_VALUE_CAN_BE_DEFAULT |
		GNOME_DB_VALUE_IS_UNCHANGED | GNOME_DB_VALUE_HAS_VALUE_ORIG;
	gboolean to_del = TRUE, local_to_del;
	GSList *list;
	GnomeDbDataModel *proxy_model;
	gint col;
	gint offset;
	guint localattr;

	g_return_val_if_fail (info, 0);
	g_return_val_if_fail (proxy && IS_GNOME_DB_DATA_PROXY (proxy), 0);
	proxy_model = gnome_db_data_proxy_get_model (proxy);
	g_return_val_if_fail (proxy_model, 0);
	offset = gnome_db_data_proxy_get_n_columns (proxy);

	/* list the values in proxy_model for each param in info->node->params */
	list = info->node->params;
	while (list) {
		col = gnome_db_data_model_get_column_at_param (proxy_model, info->data_set, list->data);
		gtk_tree_model_get (GTK_TREE_MODEL (proxy), iter, 
				    PROXY_COL_TO_DELETE, &local_to_del,
				    offset + col, &localattr, -1);
		attributes &= localattr;
		to_del = to_del && local_to_del;
		
		list = g_slist_next (list);
	}

	if (to_be_deleted)
		*to_be_deleted = to_del;
	
	return attributes;
}

/**
 * utility_table_field_attrs_stringify
 */
gchar *
utility_table_field_attrs_stringify (guint attributes)
{
	if (attributes & FIELD_AUTO_INCREMENT)
		return g_strdup ("AUTO_INCREMENT");

	return NULL;
}

/**
 * utility_table_field_attrs_parse
 */
guint
utility_table_field_attrs_parse (const gchar *str)
{
	if (!strcmp (str, "AUTO_INCREMENT"))
		return FIELD_AUTO_INCREMENT;

	return 0;
}


/**
 * utility_build_encoded_id
 *
 * Creates a BASE64 kind encoded string. It's not really a BASE64 because:
 * - the characters + and / of BASE64 are replaced with - and _
 * - no padding is done using the = character
 *
 * The created string is a valid NCName XML token.
 */
static inline unsigned char
to_uchar (gchar ch)
{
	return ch;
}

gchar *
utility_build_encoded_id (const gchar *prefix, const gchar *id)
{
	const gchar conv[64] = 
		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";
	gchar *str, *out;
	const gchar *in;
	int ln = 0, pln = 0;
	int offset;

	/* size computation */
	if (prefix) {
		pln = strlen (prefix);
		ln = pln;
	}

	ln += (strlen (id) * 4 + 2) / 3 + 1;
	str = g_new0 (char, ln);

	/* copy prefix */
	out = str;
	if (prefix) {
		strcpy (str, prefix);
		out += pln;
	}

	/* create data */
	offset = 4;
	for (in = id; offset == 4; in += 3) {
		offset = 0;
		
		if (in[0]) {
			offset = 2;
			
			out[0] = conv [to_uchar (in[0]) >> 2];

			if (in[1]) {
				offset = 3;
				out[1] = conv [((to_uchar (in[0]) << 4) + 
						(to_uchar (in[1]) >> 4)) & 0x3f];

				if (in[2]) {
					offset = 4;

					out[2] = conv [((to_uchar (in[1]) << 2) + 
							(to_uchar (in[2]) >> 6)) & 0x3f];
					out[3] = conv [to_uchar (in[2]) & 0x3f];
				}
				else
					out[2] = conv [(to_uchar (in[1]) << 2) & 0x3f];
			}
			else
				out[1] = conv [(to_uchar (in[0]) << 4) & 0x3f];


			out += offset;
		}
        }

	return str;
}

#define B64(x)	     \
  ((x) == 'A' ? 0    \
   : (x) == 'B' ? 1  \
   : (x) == 'C' ? 2  \
   : (x) == 'D' ? 3  \
   : (x) == 'E' ? 4  \
   : (x) == 'F' ? 5  \
   : (x) == 'G' ? 6  \
   : (x) == 'H' ? 7  \
   : (x) == 'I' ? 8  \
   : (x) == 'J' ? 9  \
   : (x) == 'K' ? 10 \
   : (x) == 'L' ? 11 \
   : (x) == 'M' ? 12 \
   : (x) == 'N' ? 13 \
   : (x) == 'O' ? 14 \
   : (x) == 'P' ? 15 \
   : (x) == 'Q' ? 16 \
   : (x) == 'R' ? 17 \
   : (x) == 'S' ? 18 \
   : (x) == 'T' ? 19 \
   : (x) == 'U' ? 20 \
   : (x) == 'V' ? 21 \
   : (x) == 'W' ? 22 \
   : (x) == 'X' ? 23 \
   : (x) == 'Y' ? 24 \
   : (x) == 'Z' ? 25 \
   : (x) == 'a' ? 26 \
   : (x) == 'b' ? 27 \
   : (x) == 'c' ? 28 \
   : (x) == 'd' ? 29 \
   : (x) == 'e' ? 30 \
   : (x) == 'f' ? 31 \
   : (x) == 'g' ? 32 \
   : (x) == 'h' ? 33 \
   : (x) == 'i' ? 34 \
   : (x) == 'j' ? 35 \
   : (x) == 'k' ? 36 \
   : (x) == 'l' ? 37 \
   : (x) == 'm' ? 38 \
   : (x) == 'n' ? 39 \
   : (x) == 'o' ? 40 \
   : (x) == 'p' ? 41 \
   : (x) == 'q' ? 42 \
   : (x) == 'r' ? 43 \
   : (x) == 's' ? 44 \
   : (x) == 't' ? 45 \
   : (x) == 'u' ? 46 \
   : (x) == 'v' ? 47 \
   : (x) == 'w' ? 48 \
   : (x) == 'x' ? 49 \
   : (x) == 'y' ? 50 \
   : (x) == 'z' ? 51 \
   : (x) == '0' ? 52 \
   : (x) == '1' ? 53 \
   : (x) == '2' ? 54 \
   : (x) == '3' ? 55 \
   : (x) == '4' ? 56 \
   : (x) == '5' ? 57 \
   : (x) == '6' ? 58 \
   : (x) == '7' ? 59 \
   : (x) == '8' ? 60 \
   : (x) == '9' ? 61 \
   : (x) == '-' ? 62 \
   : (x) == '_' ? 63 \
   : -1)

static const signed char b64[0x100] = {
  B64 (0), B64 (1), B64 (2), B64 (3),
  B64 (4), B64 (5), B64 (6), B64 (7),
  B64 (8), B64 (9), B64 (10), B64 (11),
  B64 (12), B64 (13), B64 (14), B64 (15),
  B64 (16), B64 (17), B64 (18), B64 (19),
  B64 (20), B64 (21), B64 (22), B64 (23),
  B64 (24), B64 (25), B64 (26), B64 (27),
  B64 (28), B64 (29), B64 (30), B64 (31),
  B64 (32), B64 (33), B64 (34), B64 (35),
  B64 (36), B64 (37), B64 (38), B64 (39),
  B64 (40), B64 (41), B64 (42), B64 (43),
  B64 (44), B64 (45), B64 (46), B64 (47),
  B64 (48), B64 (49), B64 (50), B64 (51),
  B64 (52), B64 (53), B64 (54), B64 (55),
  B64 (56), B64 (57), B64 (58), B64 (59),
  B64 (60), B64 (61), B64 (62), B64 (63),
  B64 (64), B64 (65), B64 (66), B64 (67),
  B64 (68), B64 (69), B64 (70), B64 (71),
  B64 (72), B64 (73), B64 (74), B64 (75),
  B64 (76), B64 (77), B64 (78), B64 (79),
  B64 (80), B64 (81), B64 (82), B64 (83),
  B64 (84), B64 (85), B64 (86), B64 (87),
  B64 (88), B64 (89), B64 (90), B64 (91),
  B64 (92), B64 (93), B64 (94), B64 (95),
  B64 (96), B64 (97), B64 (98), B64 (99),
  B64 (100), B64 (101), B64 (102), B64 (103),
  B64 (104), B64 (105), B64 (106), B64 (107),
  B64 (108), B64 (109), B64 (110), B64 (111),
  B64 (112), B64 (113), B64 (114), B64 (115),
  B64 (116), B64 (117), B64 (118), B64 (119),
  B64 (120), B64 (121), B64 (122), B64 (123),
  B64 (124), B64 (125), B64 (126), B64 (127),
  B64 (128), B64 (129), B64 (130), B64 (131),
  B64 (132), B64 (133), B64 (134), B64 (135),
  B64 (136), B64 (137), B64 (138), B64 (139),
  B64 (140), B64 (141), B64 (142), B64 (143),
  B64 (144), B64 (145), B64 (146), B64 (147),
  B64 (148), B64 (149), B64 (150), B64 (151),
  B64 (152), B64 (153), B64 (154), B64 (155),
  B64 (156), B64 (157), B64 (158), B64 (159),
  B64 (160), B64 (161), B64 (162), B64 (163),
  B64 (164), B64 (165), B64 (166), B64 (167),
  B64 (168), B64 (169), B64 (170), B64 (171),
  B64 (172), B64 (173), B64 (174), B64 (175),
  B64 (176), B64 (177), B64 (178), B64 (179),
  B64 (180), B64 (181), B64 (182), B64 (183),
  B64 (184), B64 (185), B64 (186), B64 (187),
  B64 (188), B64 (189), B64 (190), B64 (191),
  B64 (192), B64 (193), B64 (194), B64 (195),
  B64 (196), B64 (197), B64 (198), B64 (199),
  B64 (200), B64 (201), B64 (202), B64 (203),
  B64 (204), B64 (205), B64 (206), B64 (207),
  B64 (208), B64 (209), B64 (210), B64 (211),
  B64 (212), B64 (213), B64 (214), B64 (215),
  B64 (216), B64 (217), B64 (218), B64 (219),
  B64 (220), B64 (221), B64 (222), B64 (223),
  B64 (224), B64 (225), B64 (226), B64 (227),
  B64 (228), B64 (229), B64 (230), B64 (231),
  B64 (232), B64 (233), B64 (234), B64 (235),
  B64 (236), B64 (237), B64 (238), B64 (239),
  B64 (240), B64 (241), B64 (242), B64 (243),
  B64 (244), B64 (245), B64 (246), B64 (247),
  B64 (248), B64 (249), B64 (250), B64 (251),
  B64 (252), B64 (253), B64 (254), B64 (255)
};

#define isbase64(x) ((to_uchar (x) <= 255) && (0 <= b64[to_uchar (x)]))


/**
 * utility_build_decoded_id
 *
 * Reverse of utility_build_encoded_id()
 */
gchar *
utility_build_decoded_id (const gchar *prefix, const gchar *id)
{
	gchar *str;
	gchar *out;
	const gchar *in;
	int ln = 0;
	int offset;

	/* go to the first byte of actual data */
	in = id;
	if (prefix) {
		const gchar *tmp;
		tmp = prefix;
		while (*tmp) {
			in++;
			tmp++;
		}
	}

	ln = (strlen (in) * 3) / 4 + 3;
	str = g_new0 (char, ln);
	out = str;

	/* create data */
	offset = 3;
	for (; offset == 3; in += 4) {
		offset = 0;
		
		if (isbase64 (in[0]) && isbase64 (in[1])) {
			out[0] = ((b64 [to_uchar (in[0])] << 2) | 
				  (b64 [to_uchar (in[1])] >> 4));
			
			if (isbase64 (in[2])) {
				out[1] = (((b64 [to_uchar (in[1])] << 4) & 0xf0) |
					  (b64 [to_uchar (in[2])] >> 2));

				if (isbase64 (in[3])) {
					out[2] = (((b64 [to_uchar (in[2])] << 6) & 0xc0) |
						  b64 [to_uchar (in[3])]);
					offset = 3;
				}
			}
		}

		out += offset;
        }

	return str;
}
