/* gnome-db-target.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-target.h"
#include "gnome-db-entity.h"
#include <string.h>
#include "gnome-db-ref-base.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-referer.h"
#include "gnome-db-renderer.h"
#include "gnome-db-query.h"
#include "gnome-db-table.h"
#include "gnome-db-server.h"

/* 
 * Main static functions 
 */
static void gnome_db_target_class_init (GnomeDbTargetClass * class);
static void gnome_db_target_init (GnomeDbTarget * srv);
static void gnome_db_target_dispose (GObject   * object);
static void gnome_db_target_finalize (GObject   * object);

static void gnome_db_target_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_target_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_target_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_target_get_xml_id       (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_target_save_to_xml      (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_target_load_from_xml    (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Referer interface */
static void        gnome_db_target_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_target_activate            (GnomeDbReferer *iface);
static void        gnome_db_target_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_target_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_target_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_target_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

/* Renderer interface */
static void        gnome_db_target_renderer_init   (GnomeDbRendererIface *iface);
static GdaXqlItem *gnome_db_target_render_as_xql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error);
static gchar      *gnome_db_target_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar      *gnome_db_target_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);

/* Alias interface */
/* static void        gnome_db_target_alias_init        (GnomeDbAlias *iface); */


/* When the GnomeDbQuery is nullified */
static void        nullified_object_cb (GObject *obj, GnomeDbTarget *target);

#ifdef debug
static void        gnome_db_target_dump (GnomeDbTarget *table, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;


/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbTargetPrivate
{
	GnomeDbQuery    *query;
	GnomeDbRefBase  *entity_ref;
	gchar           *alias;
};


/* module error */
GQuark gnome_db_target_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_target_error");
	return quark;
}


GType
gnome_db_target_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbTargetClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_target_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbTarget),
			0,
			(GInstanceInitFunc) gnome_db_target_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_target_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_target_referer_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo renderer_info = {
			(GInterfaceInitFunc) gnome_db_target_renderer_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbTarget", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_REFERER, &referer_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_RENDERER, &renderer_info);
	}
	return type;
}

static void 
gnome_db_target_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_target_get_xml_id;
	iface->save_to_xml = gnome_db_target_save_to_xml;
	iface->load_from_xml = gnome_db_target_load_from_xml;
}

static void
gnome_db_target_referer_init (GnomeDbRefererIface *iface)
{
	iface->activate = gnome_db_target_activate;
	iface->deactivate = gnome_db_target_deactivate;
	iface->is_active = gnome_db_target_is_active;
	iface->get_ref_objects = gnome_db_target_get_ref_objects;
	iface->replace_refs = gnome_db_target_replace_refs;
}

static void
gnome_db_target_renderer_init (GnomeDbRendererIface *iface)
{
	iface->render_as_xql = gnome_db_target_render_as_xql;
	iface->render_as_sql = gnome_db_target_render_as_sql;
	iface->render_as_str = gnome_db_target_render_as_str;
	iface->is_valid = NULL;
}

static void
gnome_db_target_class_init (GnomeDbTargetClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_target_dispose;
	object_class->finalize = gnome_db_target_finalize;

	/* Properties */
	object_class->set_property = gnome_db_target_set_property;
	object_class->get_property = gnome_db_target_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_target_dump;
#endif

}

static void
gnome_db_target_init (GnomeDbTarget *gnome_db_target)
{
	gnome_db_target->priv = g_new0 (GnomeDbTargetPrivate, 1);
	gnome_db_target->priv->query = NULL;
	gnome_db_target->priv->entity_ref = NULL;
	gnome_db_target->priv->alias = NULL;
}

/**
 * gnome_db_target_new_with_entity
 * @query: a #GnomeDbQuery object
 * @entity: an object implementing the #GnomeDbEntity interface
 *
 * Creates a new #GnomeDbTarget object, specifying the #GnomeDbEntity to represent
 *
 * Returns: the new object
 */
GObject*
gnome_db_target_new_with_entity (GnomeDbQuery *query, GnomeDbEntity *entity)
{
	GObject *obj;
	GnomeDbTarget *gnome_db_target;
	GnomeDbDict *dict;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (entity && IS_GNOME_DB_ENTITY (entity), NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_TYPE_TARGET, "dict", dict, NULL);
	gnome_db_target = GNOME_DB_TARGET (obj);
	g_object_get (G_OBJECT (query), "target_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_target), id);

	gnome_db_target->priv->query = query;
	gnome_db_base_connect_nullify (query,
				 G_CALLBACK (nullified_object_cb), gnome_db_target);

	gnome_db_target->priv->entity_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_object (gnome_db_target->priv->entity_ref, GNOME_DB_BASE (entity));

	return obj;
}

/**
 * gnome_db_target_new_with_xml_id
 * @query: a #GnomeDbQuery object
 * @entity_xml_id: the XML Id of an object implementing the #GnomeDbEntity interface
 *
 * Creates a new #GnomeDbTarget object, specifying the XML id of the #GnomeDbEntity to represent
 *
 * Returns: the new object
 */
GObject *
gnome_db_target_new_with_xml_id (GnomeDbQuery *query, const gchar *entity_xml_id)
{
	GObject   *obj;
	GnomeDbTarget *gnome_db_target;
	GnomeDbDict *dict;
	GType target_ref;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (entity_xml_id && *entity_xml_id, NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_TYPE_TARGET, "dict", dict, NULL);
	gnome_db_target = GNOME_DB_TARGET (obj);
	g_object_get (G_OBJECT (query), "target_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_target), id);

	gnome_db_target->priv->query = query;
	gnome_db_base_connect_nullify (query,
				 G_CALLBACK (nullified_object_cb), gnome_db_target);

	gnome_db_target->priv->entity_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	if (*entity_xml_id == 'T') 
		target_ref = GNOME_DB_TYPE_TABLE;
	else
		target_ref = GNOME_DB_TYPE_QUERY;
	gnome_db_ref_base_set_ref_name (gnome_db_target->priv->entity_ref, target_ref, REFERENCE_BY_XML_ID, entity_xml_id);

	return obj;
}


/**
 * gnome_db_target_new_copy
 * @orig: a #GnomeDbTarget object to copy
 *
 * Makes a copy of an existing object (copy constructor)
 *
 * Returns: the new object
 */
GObject *
gnome_db_target_new_copy (GnomeDbTarget *orig)
{
	GObject   *obj;
	GnomeDbTarget *gnome_db_target;
	GnomeDbDict *dict;

	g_return_val_if_fail (orig && IS_GNOME_DB_TARGET (orig), NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (orig));
	obj = g_object_new (GNOME_DB_TYPE_TARGET, "dict", dict, NULL);
	gnome_db_target = GNOME_DB_TARGET (obj);

	gnome_db_target->priv->query = orig->priv->query;
	gnome_db_base_connect_nullify (orig->priv->query,
				 G_CALLBACK (nullified_object_cb), gnome_db_target);

	gnome_db_target->priv->entity_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new_copy (orig->priv->entity_ref));

	return obj;	
}


static void
nullified_object_cb (GObject *obj, GnomeDbTarget *target)
{
	gnome_db_base_nullify (GNOME_DB_BASE (target));
}


static void
gnome_db_target_dispose (GObject *object)
{
	GnomeDbTarget *gnome_db_target;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_TARGET (object));

	gnome_db_target = GNOME_DB_TARGET (object);
	if (gnome_db_target->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));
		
		if (gnome_db_target->priv->query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_target->priv->query),
							      G_CALLBACK (nullified_object_cb), gnome_db_target);
			gnome_db_target->priv->query = NULL;
		}
		if (gnome_db_target->priv->entity_ref) {
			g_object_unref (G_OBJECT (gnome_db_target->priv->entity_ref));
			gnome_db_target->priv->entity_ref = NULL;
		}

		if (gnome_db_target->priv->alias) {
			g_free (gnome_db_target->priv->alias);
			gnome_db_target->priv->alias = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_target_finalize (GObject   * object)
{
	GnomeDbTarget *gnome_db_target;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_TARGET (object));

	gnome_db_target = GNOME_DB_TARGET (object);
	if (gnome_db_target->priv) {
		g_free (gnome_db_target->priv);
		gnome_db_target->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_target_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbTarget *gnome_db_target;

	gnome_db_target = GNOME_DB_TARGET (object);
	if (gnome_db_target->priv) {
		switch (param_id) {
		case PROP:
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_target_get_property (GObject              *object,
			  guint                 param_id,
			  GValue               *value,
			  GParamSpec           *pspec)
{
	GnomeDbTarget *gnome_db_target;
	gnome_db_target = GNOME_DB_TARGET (object);
	
	if (gnome_db_target->priv) {
		switch (param_id) {
		case PROP:
			break;
		}	
	}
}


/**
 * gnome_db_target_get_query
 * @target: a #GnomeDbTarget object
 *
 * Get the #GnomeDbQuery in which @target is
 *
 * Returns: the #GnomeDbQuery object
 */
GnomeDbQuery *
gnome_db_target_get_query (GnomeDbTarget *target)
{
	g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), NULL);
	g_return_val_if_fail (target->priv, NULL);

	return target->priv->query;
}

/**
 * gnome_db_target_get_represented_entity
 * @target: a #GnomeDbTarget object
 *
 * Get the #GnomeDbEntity object which is represented by @target
 *
 * Returns: the #GnomeDbEntity object or NULL if @target is not active
 */
GnomeDbEntity *
gnome_db_target_get_represented_entity (GnomeDbTarget *target)
{
	GnomeDbBase *ent;
	GnomeDbEntity *entity = NULL;

	g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), NULL);
	g_return_val_if_fail (target->priv, NULL);

	ent = gnome_db_ref_base_get_ref_object (target->priv->entity_ref);
	if (ent)
		entity = GNOME_DB_ENTITY (ent);

	return entity;
}

/**
 * gnome_db_target_set_alias
 * @target: a #GnomeDbTarget object
 * @alias: the alias
 *
 * Sets @target's alias to @alias
 */
void
gnome_db_target_set_alias (GnomeDbTarget *target, const gchar *alias)
{
	g_return_if_fail (target && IS_GNOME_DB_TARGET (target));
	g_return_if_fail (target->priv);

	if (target->priv->alias) {
		g_free (target->priv->alias);
		target->priv->alias = NULL;
	}
	
	if (alias)
		target->priv->alias = g_strdup (alias);
}

/**
 * gnome_db_target_get_alias
 * @target: a #GnomeDbTarget object
 *
 * Get @target's alias
 *
 * Returns: the alias
 */
const gchar *
gnome_db_target_get_alias (GnomeDbTarget *target)
{
	g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), NULL);
	g_return_val_if_fail (target->priv, NULL);

	
	if (!target->priv->alias)
		target->priv->alias = g_strdup_printf ("t%d", gnome_db_base_get_id (GNOME_DB_BASE (target)));

	return target->priv->alias;
}

/**
 * gnome_db_target_get_complete_name
 * @target: a #GnomeDbTarget object
 *
 * Get a complete name for target in the form of "&lt;entity name&gt; AS &lt;target alias&gt;"
 *
 * Returns: a new string
 */
gchar *
gnome_db_target_get_complete_name (GnomeDbTarget *target)
{
	const gchar *cstr, *cstr2;
	GnomeDbEntity *ent;
	gchar *tmpstr = NULL;

	g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), NULL);
	g_return_val_if_fail (target->priv, NULL);

	ent = gnome_db_target_get_represented_entity (target);
	if (!IS_GNOME_DB_QUERY (ent)) {
		cstr = gnome_db_base_get_name (GNOME_DB_BASE (target));
		if (!cstr || !(*cstr))
			cstr = gnome_db_base_get_name (GNOME_DB_BASE (ent));
		if (cstr && *cstr)
			tmpstr = g_strdup (cstr);
		
		cstr2 = gnome_db_target_get_alias (target);
		if (cstr2 && *cstr2) {
			if (tmpstr) {
				gchar *str2 = g_strdup_printf ("%s AS %s", tmpstr, cstr2);
			g_free (tmpstr);
			tmpstr = str2;
			}
			else 
				tmpstr = g_strdup (cstr2);
		}
		
		if (!tmpstr)
			tmpstr = g_strdup (_("No name"));
	}
	else {
		/* ent is a query => print only the alias */
		cstr2 = gnome_db_target_get_alias (target);
		if (cstr2 && *cstr2) 
			tmpstr = g_strdup (cstr2);
		else
			tmpstr = g_strdup (_("No name"));
	}

	return tmpstr;
}

#ifdef debug
static void
gnome_db_target_dump (GnomeDbTarget *target, guint offset)
{
	gchar *str;
        guint i;
	
	g_return_if_fail (target && IS_GNOME_DB_TARGET (target));

        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (target->priv) {
                g_print ("%s" D_COL_H1 "GnomeDbTarget" D_COL_NOR " %p (id=%d) ",
                         str, target, gnome_db_base_get_id (GNOME_DB_BASE (target)));
		if (gnome_db_target_is_active (GNOME_DB_REFERER (target)))
			g_print ("Active, references %p ", gnome_db_ref_base_get_ref_object (target->priv->entity_ref));
		else
			g_print (D_COL_ERR "Non active" D_COL_NOR ", ");
		g_print ("requested name: %s\n", gnome_db_ref_base_get_ref_name (target->priv->entity_ref, NULL, NULL));
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, target);
}
#endif



/* 
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_target_activate (GnomeDbReferer *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, FALSE);

	return gnome_db_ref_base_activate (GNOME_DB_TARGET (iface)->priv->entity_ref);
}

static void
gnome_db_target_deactivate (GnomeDbReferer *iface)
{
	g_return_if_fail (iface && IS_GNOME_DB_TARGET (iface));
	g_return_if_fail (GNOME_DB_TARGET (iface)->priv);

	gnome_db_ref_base_deactivate (GNOME_DB_TARGET (iface)->priv->entity_ref);
}

static gboolean
gnome_db_target_is_active (GnomeDbReferer *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, FALSE);

	return gnome_db_ref_base_is_active (GNOME_DB_TARGET (iface)->priv->entity_ref);
}

static GSList *
gnome_db_target_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;
	GnomeDbBase *base;
	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, NULL);

	base = gnome_db_ref_base_get_ref_object (GNOME_DB_TARGET (iface)->priv->entity_ref);
	if (base)
		list = g_slist_append (list, base);

	return list;
}

static void
gnome_db_target_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GnomeDbTarget *target;

	g_return_if_fail (iface && IS_GNOME_DB_TARGET (iface));
	g_return_if_fail (GNOME_DB_TARGET (iface)->priv);

	target = GNOME_DB_TARGET (iface);
	if (target->priv->query) {
		GnomeDbQuery *query = g_hash_table_lookup (replacements, target->priv->query);
		if (query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (target->priv->query),
							      G_CALLBACK (nullified_object_cb), target);
			target->priv->query = query;
			gnome_db_base_connect_nullify (query,
						 G_CALLBACK (nullified_object_cb), target);
		}
	}

	gnome_db_ref_base_replace_ref_object (target->priv->entity_ref, replacements);
}



/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_target_get_xml_id (GnomeDbXmlStorage *iface)
{
	gchar *str, *retval;
	GnomeDbTarget *target;

	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, NULL);
	target = GNOME_DB_TARGET (iface);

	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (target->priv->query));
	retval = g_strdup_printf ("%s:T%d", str, gnome_db_base_get_id (GNOME_DB_BASE (target)));
	g_free (str);

	return retval;
}

static xmlNodePtr
gnome_db_target_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbTarget *target;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, NULL);

	target = GNOME_DB_TARGET (iface);

	node = xmlNewNode (NULL, "GNOME_DB_TARGET");
	
	str = gnome_db_target_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);
	
	if (target->priv->entity_ref) {
		str = NULL;
		if (gnome_db_ref_base_is_active (target->priv->entity_ref)) {
			GnomeDbBase *base = gnome_db_ref_base_get_ref_object (target->priv->entity_ref);
			g_assert (base);
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
		}
		else 
			str = g_strdup (gnome_db_ref_base_get_ref_name (target->priv->entity_ref, NULL, NULL));
		
		if (str) {
			xmlSetProp (node, "entity_ref", str);
			g_free (str);
		}
	}

	return node;
}

static gboolean
gnome_db_target_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbTarget *target;
	gboolean has_name=FALSE, id=FALSE;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	target = GNOME_DB_TARGET (iface);
	if (strcmp (node->name, "GNOME_DB_TARGET")) {
		g_set_error (error,
			     GNOME_DB_TARGET_ERROR,
			     GNOME_DB_TARGET_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_TARGET>"));
		return FALSE;
	}

	str = xmlGetProp (node, "id");
	if (str) {
		gchar *tok, *ptr;

		ptr = strtok_r (str, ":", &tok);
		ptr = strtok_r (NULL, ":", &tok);
		if (*ptr && (*ptr == 'T')) {
			guint bid;
			
			bid = atoi (ptr + 1);
			gnome_db_base_set_id (GNOME_DB_BASE (target), bid);
			id = TRUE;
		}
		g_free (str);
	}

	str = xmlGetProp (node, "entity_ref");
	if (str) {
		if (target->priv->entity_ref) {
			GType target_ref;
			const gchar *name;
			GnomeDbBase *base;

			if (*str == 'T') 
				target_ref = GNOME_DB_TYPE_TABLE;
			else
				target_ref = GNOME_DB_TYPE_QUERY;
			gnome_db_ref_base_set_ref_name (target->priv->entity_ref, target_ref, 
						  REFERENCE_BY_XML_ID, str);
			
			base = gnome_db_ref_base_get_ref_object (target->priv->entity_ref);
			if (base) {
				name = gnome_db_base_get_name (base);
				if (name && *name)
					gnome_db_base_set_name (GNOME_DB_BASE (target), name);
			}
					  
			has_name = TRUE;
		}
		g_free (str);
	}

	if (has_name)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_TARGET_ERROR,
			     GNOME_DB_TARGET_XML_LOAD_ERROR,
			     _("Error loading data from <GNOME_DB_TARGET> node"));
		return FALSE;
	}
}

/* 
 * GnomeDbRenderer interface implementation
 */
static GdaXqlItem *
gnome_db_target_render_as_xql (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, NULL);

	TO_IMPLEMENT;
	return NULL;
}

static gchar *
gnome_db_target_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
	gchar *str;
	GString *string = NULL;
	GnomeDbTarget *target;
	GnomeDbEntity *entity;
	gboolean done = FALSE;
	gboolean err = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, NULL);
	g_return_val_if_fail (gnome_db_referer_activate (GNOME_DB_REFERER (iface)), NULL);
	target = GNOME_DB_TARGET (iface);

	entity = gnome_db_target_get_represented_entity (target);
	if (IS_GNOME_DB_TABLE (entity)) {
		string = g_string_new (gnome_db_base_get_name (GNOME_DB_BASE (entity)));
		done = TRUE;
	}

	if (IS_GNOME_DB_QUERY (entity)) {
		string = g_string_new ("(");
		str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (entity), context, options, error);
		if (str) {
			g_string_append (string, str);
			g_free (str);
		}
		else
			err = TRUE;

		g_string_append (string, ")");
		done = TRUE;
	}

	g_assert (done);

	if (string) {
		if (!err) {
			/* adding alias */
			GnomeDbServerInfo *info;

			info = gnome_db_server_get_server_info (gnome_db_dict_get_server 
								(gnome_db_base_get_dict (GNOME_DB_BASE (target))));
			if (!info || (info->alias_needs_as_keyword))
				g_string_append (string, " AS ");
			else
				g_string_append_c (string, ' ');
			g_string_append (string, gnome_db_target_get_alias (target));
			
			str = string->str;
		}
		else
			str = NULL;
		g_string_free (string, err);
	}
	else
		str = NULL;

	return str;
}

static gchar *
gnome_db_target_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TARGET (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TARGET (iface)->priv, NULL);

	TO_IMPLEMENT;
	return NULL;
}
