/* gnome-db-server-data-type.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-server.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-data-handler.h"
#include "marshal.h"
#include <libgda/libgda.h>
#include <string.h>
#include "utility.h"

/* 
 * Main static functions 
 */
static void gnome_db_server_data_type_class_init (GnomeDbServerDataTypeClass * class);
static void gnome_db_server_data_type_init (GnomeDbServerDataType * srv);
static void gnome_db_server_data_type_dispose (GObject   * object);
static void gnome_db_server_data_type_finalize (GObject   * object);

static void gnome_db_server_data_type_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_server_data_type_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

static void        gnome_db_data_type_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_data_type_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_data_type_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_data_type_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_server_data_type_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbServerDataTypePrivate
{
	GnomeDbServer     *srv;
	guint              numparams;
	GdaValueType       gda_type;
	GSList            *synonyms;
};


/* module error */
GQuark gnome_db_server_data_type_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_server_data_type_error");
	return quark;
}


GType
gnome_db_server_data_type_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbServerDataTypeClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_server_data_type_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbServerDataType),
			0,
			(GInstanceInitFunc) gnome_db_server_data_type_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_data_type_xml_storage_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbServerDataType", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
	}
	return type;
}

static void 
gnome_db_data_type_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_data_type_get_xml_id;
	iface->save_to_xml = gnome_db_data_type_save_to_xml;
	iface->load_from_xml = gnome_db_data_type_load_from_xml;
}


static void
gnome_db_server_data_type_class_init (GnomeDbServerDataTypeClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_server_data_type_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbServerDataTypeClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_server_data_type_dispose;
	object_class->finalize = gnome_db_server_data_type_finalize;

	/* Properties */
	object_class->set_property = gnome_db_server_data_type_set_property;
	object_class->get_property = gnome_db_server_data_type_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_server_data_type_init (GnomeDbServerDataType * gnome_db_server_data_type)
{
	gnome_db_server_data_type->priv = g_new0 (GnomeDbServerDataTypePrivate, 1);
	gnome_db_server_data_type->priv->srv = NULL;
	gnome_db_server_data_type->priv->numparams = -1;
	gnome_db_server_data_type->priv->gda_type = 0;
	gnome_db_server_data_type->priv->synonyms = NULL;
}

/**
 * gnome_db_server_data_type_new
 * @srv: a #GnomeDbServer object
 *
 * Creates a new GnomeDbServerDataType object
 *
 * Returns: the new object
 */
GObject*
gnome_db_server_data_type_new (GnomeDbServer *srv)
{
	GObject   *obj;
	GnomeDbServerDataType *gnome_db_server_data_type;

	g_return_val_if_fail (srv && GNOME_DB_SERVER (srv), NULL);

	obj = g_object_new (GNOME_DB_TYPE_SERVER_DATA_TYPE, "dict",
			    gnome_db_server_get_dict (srv), NULL);
	gnome_db_server_data_type = GNOME_DB_SERVER_DATA_TYPE (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_server_data_type), 0);
	g_object_add_weak_pointer (G_OBJECT (srv), (gpointer) &(gnome_db_server_data_type->priv->srv));
	gnome_db_server_data_type->priv->srv = srv;
	
	return obj;
}


static void
gnome_db_server_data_type_dispose (GObject *object)
{
	GnomeDbServerDataType *gnome_db_server_data_type;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SERVER_DATA_TYPE (object));

	gnome_db_server_data_type = GNOME_DB_SERVER_DATA_TYPE (object);
	if (gnome_db_server_data_type->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		g_object_remove_weak_pointer (G_OBJECT (gnome_db_server_data_type->priv->srv), 
					      (gpointer) &(gnome_db_server_data_type->priv->srv));
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_server_data_type_finalize (GObject   * object)
{
	GnomeDbServerDataType *gnome_db_server_data_type;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SERVER_DATA_TYPE (object));

	gnome_db_server_data_type = GNOME_DB_SERVER_DATA_TYPE (object);
	if (gnome_db_server_data_type->priv) {
		if (gnome_db_server_data_type->priv->synonyms) {
			g_slist_foreach (gnome_db_server_data_type->priv->synonyms, (GFunc) g_free, NULL);
			g_slist_free (gnome_db_server_data_type->priv->synonyms);
		}
		g_free (gnome_db_server_data_type->priv);
		gnome_db_server_data_type->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_server_data_type_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbServerDataType *gnome_db_server_data_type;

	gnome_db_server_data_type = GNOME_DB_SERVER_DATA_TYPE (object);
	if (gnome_db_server_data_type->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_server_data_type_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbServerDataType *gnome_db_server_data_type;
	gnome_db_server_data_type = GNOME_DB_SERVER_DATA_TYPE (object);
	
	if (gnome_db_server_data_type->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			g_value_set_pointer (value, NULL);
			break;
		}	
	}
}

/* GnomeDbXmlStorage interface implementation */
static gchar *
gnome_db_data_type_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_DATA_TYPE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_SERVER_DATA_TYPE (iface)->priv, NULL);

	return utility_build_encoded_id ("DT", gnome_db_base_get_name (GNOME_DB_BASE (iface)));
}

static xmlNodePtr
gnome_db_data_type_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbServerDataType *dt;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_DATA_TYPE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_SERVER_DATA_TYPE (iface)->priv, NULL);
	dt = GNOME_DB_SERVER_DATA_TYPE (iface);

	node = xmlNewNode (NULL, "GNOME_DB_DATATYPE");
	
	str = gnome_db_data_type_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (dt)));
	xmlSetProp (node, "owner", gnome_db_base_get_owner (GNOME_DB_BASE (dt)));
	xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (dt)));
	str = g_strdup_printf ("%d", dt->priv->numparams);
	xmlSetProp (node, "nparam", str);
	g_free (str);
	xmlSetProp (node, "gdatype", gda_type_to_string (dt->priv->gda_type));

	if (dt->priv->synonyms) {
		GSList *list = dt->priv->synonyms;
		GString *string;

		string = g_string_new ((gchar *) dt->priv->synonyms->data);
		list = g_slist_next (list);
		while (list) {
			g_string_append_c (string, ',');
			g_string_append (string, (gchar *) list->data);
			list = g_slist_next (list);			
		}
		xmlSetProp (node, "synonyms", string->str);
		g_string_free (string, TRUE);
	}

	return node;
}

static gboolean
gnome_db_data_type_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbServerDataType *dt;
	gchar *prop;
	gboolean pname = FALSE, pnparam = FALSE, pgdatype = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_DATA_TYPE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_SERVER_DATA_TYPE (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	dt = GNOME_DB_SERVER_DATA_TYPE (iface);
	if (strcmp (node->name, "GNOME_DB_DATATYPE")) {
		g_set_error (error,
			     GNOME_DB_SERVER_DATA_TYPE_ERROR,
			     GNOME_DB_SERVER_DATA_TYPE_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_DATATYPE>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		pname = TRUE;
		gnome_db_base_set_name (GNOME_DB_BASE (dt), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (dt), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "owner");
	if (prop) {
		gnome_db_base_set_owner (GNOME_DB_BASE (dt), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "nparam");
	if (prop) {
		pnparam = TRUE;
		dt->priv->numparams = atoi (prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "gdatype");
	if (prop) {
		pgdatype = TRUE;
		dt->priv->gda_type = gda_type_from_string (prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "synonyms");
	if (prop) {
		gchar *tok;
		gchar *buf;
		GSList *list = NULL;

		tok = strtok_r (prop, ",", &buf);
		if (tok) {
			list = g_slist_append (list, g_strdup (tok));
			tok = strtok_r (NULL, ",", &buf);
			while (tok) {
				list = g_slist_append (list, g_strdup (tok));
				tok = strtok_r (NULL, ",", &buf);				
			}
		}
		g_free (prop);
		dt->priv->synonyms = list;
	}

	if (pname && pnparam && pgdatype)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_SERVER_DATA_TYPE_ERROR,
			     GNOME_DB_SERVER_DATA_TYPE_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_DATATYPE>"));
		return FALSE;
	}
}







/**
 * gnome_db_server_data_type_set_sqlname
 * @dt: a #GnomeDbServerDataType object
 * @sqlname: 
 *
 * Set the SQL name of the data type.
 */
void
gnome_db_server_data_type_set_sqlname (GnomeDbServerDataType *dt, const gchar *sqlname)
{
	g_return_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt));
	g_return_if_fail (dt->priv);

	gnome_db_base_set_name (GNOME_DB_BASE (dt), sqlname);
}


/**
 * gnome_db_server_data_type_get_sqlname
 * @dt: a #GnomeDbServerDataType object
 *
 * Get the DBMS's name of a data type.
 *
 * Returns: the name of the data type
 */
const gchar *
gnome_db_server_data_type_get_sqlname (GnomeDbServerDataType *dt)
{
	g_return_val_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt), NULL);
	g_return_val_if_fail (dt->priv, NULL);

	return gnome_db_base_get_name (GNOME_DB_BASE (dt));
}

/**
 * gnome_db_server_data_type_set_gda_type
 * @dt: a #GnomeDbServerDataType object
 * @gda_type: 
 *
 * Set the gda type for a data type
 */
void
gnome_db_server_data_type_set_gda_type (GnomeDbServerDataType *dt, GdaValueType gda_type)
{
	g_return_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt));
	g_return_if_fail (dt->priv);

	dt->priv->gda_type = gda_type;
}

/**
 * gnome_db_server_data_type_get_gda_type
 * @dt: a #GnomeDbServerDataType object
 *
 * Get the gda type of a data type
 *
 * Returns: the gda type
 */
GdaValueType
gnome_db_server_data_type_get_gda_type (GnomeDbServerDataType *dt)
{
	g_return_val_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt), GDA_VALUE_TYPE_UNKNOWN);
	g_return_val_if_fail (dt->priv, GDA_VALUE_TYPE_UNKNOWN);

	return dt->priv->gda_type;
}

/**
 * gnome_db_server_data_type_add_synonym
 * @dt: a #GnomeDbServerDataType object
 * @synonym:
 *
 * Sets a new synonym to the @dt data type.
 */
void
gnome_db_server_data_type_add_synonym (GnomeDbServerDataType *dt, const gchar *synonym)
{
	gboolean found = FALSE;
	GSList *list;

	g_return_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt));
	g_return_if_fail (dt->priv);
	g_return_if_fail (synonym && *synonym);

	list = dt->priv->synonyms;
	while (list && !found) {
		if (!strcmp (synonym, (gchar *) list->data))
			found = TRUE;
		list = g_slist_next (list);
	}
	if (!found)
		dt->priv->synonyms = g_slist_prepend (dt->priv->synonyms, g_strdup (synonym));
}

/**
 * gnome_db_server_data_type_get_synonyms
 * @dt: a #GnomeDbServerDataType object
 *
 * Get a list of @dt's synonyms
 *
 * Returns: a list of strings which must not be modified
 */
const GSList *
gnome_db_server_data_type_get_synonyms (GnomeDbServerDataType *dt)
{
	g_return_val_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt), NULL);
	g_return_val_if_fail (dt->priv, NULL);

	return dt->priv->synonyms;
}

/**
 * gnome_db_server_data_type_clear_synonyms
 * @dt: a #GnomeDbServerDataType object
 *
 * Removes any synonym attached to @dt
 */
void
gnome_db_server_data_type_clear_synonyms (GnomeDbServerDataType *dt)
{
	g_return_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt));
	g_return_if_fail (dt->priv);

	if (dt->priv->synonyms) {
		g_slist_foreach (dt->priv->synonyms, (GFunc) g_free, NULL);
		g_slist_free (dt->priv->synonyms);
		dt->priv->synonyms = NULL;
	}
}

/**
 * gnome_db_server_data_type_set_handler
 * @dt: a #GnomeDbServerDataType object
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Forces the GnomeDbDataHandler associated with the data type.
 */
void
gnome_db_server_data_type_set_handler (GnomeDbServerDataType *dt, GnomeDbDataHandler *dh)
{
	g_return_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt));
	g_return_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh));
	g_return_if_fail (dt->priv);

	gnome_db_server_set_object_handler (dt->priv->srv, G_OBJECT (dt), dh);
}


/**
 * gnome_db_server_data_type_get_handler
 * @dt: a #GnomeDbServerDataType object
 *
 * Get the GnomeDbDataHandler associated with the data type.
 *
 * Returns: the GnomeDbDataHandler
 */
GnomeDbDataHandler *
gnome_db_server_data_type_get_handler (GnomeDbServerDataType *dt)
{
	g_return_val_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt), NULL);
	g_return_val_if_fail (dt->priv, NULL);

	return gnome_db_server_get_object_handler (dt->priv->srv, G_OBJECT (dt));
}
