/* gnome-db-basic-form.c
 *
 * Copyright (C) 2002 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <gtk/gtk.h>
#include "gnome-db-basic-form.h"
#include "gnome-db-server.h"
#include "marshal.h"
#include "gnome-db-parameter.h"
#include "gnome-db-data-handler.h"
#include "gnome-db-data-entry.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-field.h"
#include "gnome-db-data-set.h"
#include <libgnomedb/handlers/gnome-db-entry-combo.h>

static void gnome_db_basic_form_class_init (GnomeDbBasicFormClass * class);
static void gnome_db_basic_form_init (GnomeDbBasicForm * wid);
static void gnome_db_basic_form_dispose (GObject   * object);

static void entry_contents_modified (GnomeDbDataEntry *entry, GnomeDbBasicForm *form);
static void parameter_changed_cb (GnomeDbParameter *param, GnomeDbDataEntry *entry);

static void mark_not_null_entry_labels (GnomeDbBasicForm *form, gboolean show_mark);
enum
{
	PARAM_CHANGED,
	LAST_SIGNAL
};


struct _GnomeDbBasicFormPriv
{
	GnomeDbDict            *dict;
	GnomeDbDataSet         *data_set;/* parameters */
	GSList                 *entries;/* list of GnomeDbDataEntry widgets */
	GSList                 *not_null_labels;/* list of GtkLabel widgets corresponding to NOT NULL entries */
	gulong                 *signal_ids; /* array of signal ids */
	GtkWidget              *entries_table;
	GSList                 *hidden_entries;

	gboolean                forward_param_updates; /* forward them to the GnomeDbDataEntry widgets ? */
	GtkTooltips            *tooltips;
};


static gint gnome_db_basic_form_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
gnome_db_basic_form_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbBasicFormClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_basic_form_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbBasicForm),
			0,
			(GInstanceInitFunc) gnome_db_basic_form_init
		};		
		
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbBasicForm", &info, 0);
	}

	return type;
}

static void
gnome_db_basic_form_class_init (GnomeDbBasicFormClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	gnome_db_basic_form_signals[PARAM_CHANGED] =
		g_signal_new ("param_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbBasicFormClass, param_changed),
			      NULL, NULL,
			      marshal_VOID__OBJECT_BOOLEAN, G_TYPE_NONE, 2,
			      G_TYPE_OBJECT, G_TYPE_BOOLEAN);

	class->param_changed = NULL;
	object_class->dispose = gnome_db_basic_form_dispose;
}

static void
gnome_db_basic_form_init (GnomeDbBasicForm * wid)
{
	wid->priv = g_new0 (GnomeDbBasicFormPriv, 1);
	wid->priv->dict = NULL;
	wid->priv->data_set = NULL;
	wid->priv->entries = NULL;
	wid->priv->not_null_labels = NULL;
	wid->priv->entries_table = NULL;
	wid->priv->hidden_entries = NULL;
	wid->priv->signal_ids = NULL;

	wid->priv->forward_param_updates = TRUE;
	wid->priv->tooltips = NULL;
}

static void gnome_db_basic_form_initialize (GnomeDbBasicForm *form, GnomeDbDict *dict, 
					    GtkWidget *layout, GHashTable *box_widgets);
static void widget_shown_cb (GtkWidget *wid, GnomeDbBasicForm *form);
static void gnome_db_dict_weak_notify (GnomeDbBasicForm *form, GnomeDbDict *dict);
/**
 * gnome_db_basic_form_new
 * @dict: a #GnomeDbDict object
 * @data_set: a #GnomeDbDataSet structure
 *
 * Creates a new #GnomeDbBasicForm widget using all the parameters provided in @data_set.
 * @data_set is copied in the process.
 *
 * The global layout is rendered using a table (a #GtkTable), and an entry is created for each
 * node of @data_set.
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_basic_form_new (GnomeDbDict *dict, GnomeDbDataSet *data_set)
{
	return gnome_db_basic_form_new_in_layout (dict, data_set, NULL, NULL);
}

/**
 * gnome_db_basic_form_new_in_layout
 * @dict: a #GnomeDbDict object
 * @data_set: a #GnomeDbDataSet structure
 * @layout: a #GtkWidget container of all the widgets in @box_widgets
 *
 * Creates a new #GnomeDbBasicForm widget using all the parameters provided in @data_set.
 * @data_set is unchanged in the process.
 *
 * This function is identical to gnome_db_basic_form_new() except that the layout is not done using a table, but
 * using the @layout widget; and each entry is packed into one of the #GtkBox widgets of @box_widgets
 * (specifically each entry corresponds to a #GnomeDbDataSetNode of @data_set, and that data_set node
 * is used a the key to @box_widgets to find the box to pack the entry in).
 *
 * If any of @layout or @box_widgets is %NULL, then this function is equivalent to gnome_db_basic_form_new().
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_basic_form_new_in_layout (GnomeDbDict *dict, GnomeDbDataSet *data_set, GtkWidget *layout, GHashTable *box_widgets)
{
	GObject *obj;
	GnomeDbBasicForm *form;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);
	if (data_set) {
		GError *error = NULL;
		if (! gnome_db_data_set_is_coherent (data_set, &error)) {
			g_warning ("gnome_db_data_set_is_coherent() returned FALSE: %s", error->message);
			g_error_free (error);
			return NULL;
		}
	}

	obj = g_object_new (GNOME_DB_BASIC_FORM_TYPE, NULL);
	form = GNOME_DB_BASIC_FORM (obj);

	form->priv->dict = ASSERT_DICT (dict);
	
	/* data_set */
	if (data_set) {
		form->priv->data_set = data_set;
		g_object_ref (data_set);
	}
	else
		form->priv->data_set = GNOME_DB_DATA_SET (gnome_db_data_set_new (ASSERT_DICT (dict), NULL));

	g_object_weak_ref (G_OBJECT (form->priv->dict),
			   (GWeakNotify) gnome_db_dict_weak_notify, form);

	gnome_db_basic_form_initialize (form, ASSERT_DICT (dict), layout, box_widgets);

	return GTK_WIDGET (obj);
}

static void
widget_shown_cb (GtkWidget *wid, GnomeDbBasicForm *form)
{
	if (g_slist_find (form->priv->hidden_entries, wid)) {
		if (form->priv->entries_table && g_slist_find (form->priv->entries, wid)) {
			gint row = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "row_no"));
			gtk_table_set_row_spacing (GTK_TABLE (form->priv->entries_table), row, 0);
		}
		
		gtk_widget_hide (wid);
	}
}

static void
gnome_db_dict_weak_notify (GnomeDbBasicForm *form, GnomeDbDict *dict)
{
	GSList *list;

	/* render all the entries non sensitive */
	list = form->priv->entries;
	while (list) {
		gtk_widget_set_sensitive (GTK_WIDGET (list->data), FALSE);
		list = g_slist_next (list);
	}

	/* Tell that we don't need to weak unref the GnomeDbDict */
	form->priv->dict = NULL;
}

static void
gnome_db_basic_form_dispose (GObject *object)
{
	GnomeDbBasicForm *form;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_BASIC_FORM (object));
	form = GNOME_DB_BASIC_FORM (object);

	if (form->priv) {
		GSList *list;

		/* data_set */
		if (form->priv->data_set) {
			gint i = 0;
			list = form->priv->data_set->parameters;;
			while (list) {
				g_signal_handler_disconnect (G_OBJECT (list->data), form->priv->signal_ids[i]);

				list = g_slist_next (list);
				i++;
			}

			g_object_unref (G_OBJECT (form->priv->data_set));
			form->priv->data_set = NULL;
			g_free (form->priv->signal_ids);
		}

		/* list of entries */
		list = form->priv->hidden_entries;
		if (list) {
			while (list) { 
				g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), 
								      G_CALLBACK (widget_shown_cb), form);
				list = g_slist_next (list);
			}
			g_slist_free (form->priv->hidden_entries);
			form->priv->hidden_entries = NULL;
		}

		list = form->priv->entries;
		if (list) {
			while (list) { 
				g_object_set_data (G_OBJECT (list->data), "param", NULL);
				list = g_slist_next (list);
			}
			g_slist_free (form->priv->entries);
			form->priv->entries = NULL;
		}

		if (form->priv->not_null_labels) {
			g_slist_free (form->priv->not_null_labels);
			form->priv->not_null_labels = NULL;
		}

		/* Weak unref the GnomeDbDict is necessary */
		if (form->priv->dict)
			g_object_weak_unref (G_OBJECT (form->priv->dict),
					     (GWeakNotify) gnome_db_dict_weak_notify, form);
		
		/* tooltips */
		if (form->priv->tooltips) {
			gtk_object_destroy (GTK_OBJECT (form->priv->tooltips));
			form->priv->tooltips = NULL;
		}


		/* the private area itself */
		g_free (form->priv);
		form->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void entry_destroyed_cb (GtkWidget *entry, GnomeDbBasicForm *form);

/*
 * create the entries in the widget
 */
static void 
gnome_db_basic_form_initialize (GnomeDbBasicForm *form, GnomeDbDict *dict, GtkWidget *layout, GHashTable *box_widgets)
{
	GSList *list;
	GnomeDbParameter *param;
	GnomeDbDataSetNode *node;
	gint i;
	
	/* tooltips */
	form->priv->tooltips = gtk_tooltips_new ();

	/* data set management */
	if (form->priv->data_set && !form->priv->data_set->nodes) {
		GnomeDbDataSet *ndata_set = GNOME_DB_DATA_SET (gnome_db_data_set_new (dict, 
										      form->priv->data_set->parameters));
		g_object_unref (G_OBJECT (form->priv->data_set));
		form->priv->data_set = ndata_set;
	}

	/* allocating space for the signal ids and connect to the parameter's changes */
	form->priv->signal_ids = g_new0 (gulong, g_slist_length (form->priv->data_set->parameters));
	i = 0;

	/* creating all the entries */
	list = form->priv->data_set->nodes;
	while (list) {
		GtkWidget *entry = NULL;

		node = GNOME_DB_DATA_SET_NODE (list->data);
		if ((param = node->param)) { /* single direct parameter */
			GnomeDbServerDataType *type;
			const GdaValue *val, *default_val, *value;
			gboolean nnul;
			GnomeDbDataHandler *dh = NULL;
			gchar *plugin = NULL;

			type = gnome_db_parameter_get_data_type (param);
			g_assert (type);

			g_object_get (G_OBJECT (param), "handler_plugin", &plugin, NULL);
			if (plugin) {
				dh = gnome_db_server_get_handler_by_name (gnome_db_dict_get_server (dict), plugin);

				/* test if plugin can handle the parameter's data type */
				if (dh && 
				    !gnome_db_data_handler_accepts_gda_type (dh, gnome_db_server_data_type_get_gda_type (type)))
					dh = NULL;
			}
			
			if (!dh)
				dh = gnome_db_server_data_type_get_handler (type);
			val = gnome_db_parameter_get_value (param);
			default_val = gnome_db_parameter_get_default_value (param);
			nnul = gnome_db_parameter_get_not_null (param);

			/* determine initial value */
			value = val;
			if (!value && default_val && 
			    (gda_value_get_type (default_val) == gnome_db_server_data_type_get_gda_type (type)))
				value = default_val;
			
			/* create entry */
			entry = GTK_WIDGET (gnome_db_data_handler_get_entry_from_value (dh, value,
											gnome_db_server_data_type_get_gda_type (type)));

			/* set current value */
			gnome_db_data_entry_set_value (GNOME_DB_DATA_ENTRY (entry), val);

			if (!nnul ||
			    (nnul && value && (gda_value_get_type (value) != GDA_VALUE_TYPE_NULL)))
				gnome_db_data_entry_set_value_orig (GNOME_DB_DATA_ENTRY (entry), value);
			
			if (default_val) {
				gnome_db_data_entry_set_value_default (GNOME_DB_DATA_ENTRY (entry), default_val);
				gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entry),
								    GNOME_DB_VALUE_CAN_BE_DEFAULT,
								    GNOME_DB_VALUE_CAN_BE_DEFAULT);
			}

			gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entry),
							    nnul ? 0 : GNOME_DB_VALUE_CAN_BE_NULL,
							    GNOME_DB_VALUE_CAN_BE_NULL);
			    
			g_object_set_data (G_OBJECT (entry), "param", param);
			g_object_set_data (G_OBJECT (entry), "form", form);
			form->priv->entries = g_slist_append (form->priv->entries, entry);
			g_signal_connect (entry, "destroy", G_CALLBACK (entry_destroyed_cb), form);

			/* connect to the parameter's changes */
                        form->priv->signal_ids[i] = g_signal_connect (G_OBJECT (param), "changed",
                                                                      G_CALLBACK (parameter_changed_cb), entry);
                        i++;
		}
		else { /* parameter depending on the values of a GnomeDbDataModel object */
			GSList *plist;
			gboolean nnul = TRUE;

			entry = gnome_db_entry_combo_new (form->priv->data_set, node);
			g_object_set_data (G_OBJECT (entry), "node", node);
			g_object_set_data (G_OBJECT (entry), "form", form);
			form->priv->entries = g_slist_append (form->priv->entries, entry);
			g_signal_connect (entry, "destroy", G_CALLBACK (entry_destroyed_cb), form);

			/* connect to the parameter's changes */
			plist = node->params;
			while (plist) {
				if (!gnome_db_parameter_get_not_null (plist->data))
					nnul = FALSE;
				form->priv->signal_ids[i] = g_signal_connect (G_OBJECT (plist->data), "changed",
                                                                              G_CALLBACK (parameter_changed_cb), entry);
                                i++;
				plist = g_slist_next (plist);
			}
			gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entry),
							    nnul ? 0 : GNOME_DB_VALUE_CAN_BE_NULL,
							    GNOME_DB_VALUE_CAN_BE_NULL);
		}

		/* connect the entry's changes */
		g_signal_connect (G_OBJECT (entry), "contents_modified",
				  G_CALLBACK (entry_contents_modified), form);
		list = g_slist_next (list);
	}

	if (layout) {
		/* there is actually a layout, test it */
		/* tests: GtkWindow? already parented widget? */
		/* set layout to NULL on error */
		
		if (!box_widgets)
			layout = NULL;
	}

	if (!layout) {
		GtkWidget *table, *label;

		/* creating a table for all the entries */
		table = gtk_table_new (g_slist_length (form->priv->entries), 2, FALSE);
		gtk_table_set_row_spacings (GTK_TABLE (table), 5);
		gtk_table_set_col_spacings (GTK_TABLE (table), 5);
		form->priv->entries_table = table;
		gtk_box_pack_start (GTK_BOX (form), table,  FALSE, TRUE, 0);
		list = form->priv->entries;
		i = 0;
		while (list) {
			gboolean expand;
			GtkWidget *entry_label;
			
			/* label for the entry */
			param = g_object_get_data (G_OBJECT (list->data), "param");
			if (param) {
				const gchar *str;
				gchar *str2;
				GtkWidget *evbox;
				
				str = gnome_db_base_get_name (GNOME_DB_BASE (param));
				if (!str)
					str = _("Value");
				str2 = g_strdup_printf ("%s:", str);
				evbox = gtk_event_box_new ();
				label = gtk_label_new (str2);
				if (gnome_db_parameter_get_not_null (param)) {
					gpointer data = NULL;
					form->priv->not_null_labels = g_slist_prepend (form->priv->not_null_labels, label);
					data = form->priv->not_null_labels->data;
				}
				
				gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
				g_free (str2);
				gtk_container_add (GTK_CONTAINER (evbox), label);
				
				gtk_table_attach (GTK_TABLE (table), evbox, 0, 1, i, i+1,
						  GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
				gtk_widget_show (evbox);
				gtk_widget_show (label);
				entry_label = evbox;
				
				str = gnome_db_base_get_description (GNOME_DB_BASE (param));
				if (str && *str)
					gtk_tooltips_set_tip (form->priv->tooltips, evbox, str, NULL);
			}
			else {
				/* FIXME: find a better label and tooltip and improve data entry attributes */
				const gchar *str = NULL;
				gchar *str2;
				GtkWidget *evbox;
				gboolean nullok = TRUE;
				GSList *params;
				
				node = g_object_get_data (G_OBJECT (list->data), "node");
				params = node->params;
				while (params && nullok) {
					if (gnome_db_parameter_get_not_null (GNOME_DB_PARAMETER (params->data)))
						nullok = FALSE;
					params = g_slist_next (params);
				}
				
				str = gnome_db_base_get_name (GNOME_DB_BASE (node->data_for_param));
				if (!str)
					str = _("Value");
				str2 = g_strdup_printf ("%s:", str);
				evbox = gtk_event_box_new ();
				label = gtk_label_new (str2);
				if (!nullok) 
					form->priv->not_null_labels = g_slist_prepend (form->priv->not_null_labels, label);
				gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
				g_free (str2);
				gtk_container_add (GTK_CONTAINER (evbox), label);
				
				gtk_table_attach (GTK_TABLE (table), evbox, 0, 1, i, i+1,
						  GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
				gtk_widget_show (evbox);
				gtk_widget_show (label);
				entry_label = evbox;
				
				str = gnome_db_base_get_description (GNOME_DB_BASE (node->data_for_param));
				if (str && *str)
					gtk_tooltips_set_tip (form->priv->tooltips, evbox, str, NULL);
			}

			/* add the entry itself to the table */
			expand = gnome_db_data_entry_expand_in_layout (GNOME_DB_DATA_ENTRY (list->data));
			gtk_table_attach (GTK_TABLE (table), GTK_WIDGET (list->data), 1, 2, i, i+1,
					  GTK_FILL | GTK_SHRINK | GTK_EXPAND, 
					  GTK_FILL | GTK_SHRINK | expand? GTK_EXPAND : 0, 0, 0);
			gtk_widget_show (GTK_WIDGET (list->data));
			g_object_set_data (G_OBJECT (list->data), "entry_label", entry_label);
			g_object_set_data (G_OBJECT (list->data), "row_no", GINT_TO_POINTER (i));
			
			list = g_slist_next (list);
			i++;
		}
		mark_not_null_entry_labels (form, TRUE);
		gtk_widget_show (table);
	}
	else {
		/* really use the provided layout */
		GtkWidget *box;
		GSList *nodes;

		gtk_box_pack_start (GTK_BOX (form), layout,  FALSE, TRUE, 0);
		list = form->priv->entries;
		nodes = form->priv->data_set->nodes;
		while (nodes && list) {
			box = g_hash_table_lookup (box_widgets, nodes->data);
			if (box) {
				gtk_box_pack_start (GTK_BOX (box), GTK_WIDGET (list->data),
						    gnome_db_data_entry_expand_in_layout (GNOME_DB_DATA_ENTRY (list->data)),
						    TRUE, 0);
				gtk_widget_show (GTK_WIDGET (list->data));
				if (! g_object_get_data (G_OBJECT (box), "show_actions")) 
					gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (list->data),
								      0, GNOME_DB_VALUE_ACTIONS_SHOWN);
			}
			list = g_slist_next (list);
			nodes = g_slist_next (nodes);
		}
		g_assert (!nodes && !list);
		gtk_widget_show (layout);
	}
}

static void
entry_destroyed_cb (GtkWidget *entry, GnomeDbBasicForm *form)
{
	/* debug only function */
}

/*
 * if @show_mark is TRUE, display the label as bold 
 */
static void
mark_not_null_entry_labels (GnomeDbBasicForm *form, gboolean show_mark)
{
	PangoAttrList *attrs = NULL;
	PangoAttribute *att;
	GSList *list;

	if (show_mark) {
		attrs = pango_attr_list_new ();
		att = pango_attr_weight_new (PANGO_WEIGHT_BOLD);
		att->start_index = 0;
		att->end_index = G_MAXUINT;
		pango_attr_list_insert (attrs, att);
	}

	list = form->priv->not_null_labels;
	while (list) {
		gtk_label_set_attributes (GTK_LABEL (list->data), attrs);
		list = g_slist_next (list);
	}

	if (show_mark)
		pango_attr_list_unref (attrs);
}

static void
entry_contents_modified (GnomeDbDataEntry *entry, GnomeDbBasicForm *form)
{
	GnomeDbParameter *param;
	GnomeDbDataSetNode *node;
	guint attr;

	attr = gnome_db_data_entry_get_attributes (entry);
	param = g_object_get_data (G_OBJECT (entry), "param");
	if (param) { /* single parameter */
		GdaValue *value;
		
		form->priv->forward_param_updates = FALSE;

		/* parameter's value */
		value = gnome_db_data_entry_get_value (entry);
		if ((!value || gda_value_is_null (value)) &&
		    (attr & GNOME_DB_VALUE_IS_DEFAULT))
			g_object_set (G_OBJECT (param), "use_default_value", TRUE, NULL);
		else
			g_object_set (G_OBJECT (param), "use_default_value", FALSE, NULL);
		gnome_db_parameter_set_value (param, value);
#ifdef debug_signal
		g_print (">> 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (form), gnome_db_basic_form_signals[PARAM_CHANGED], 0, param, TRUE);
#ifdef debug_signal
		g_print ("<< 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
		form->priv->forward_param_updates = TRUE;
		gda_value_free (value);
	}
	else { /* multiple parameters */
		GSList *params;
		GSList *values, *list;
		node = g_object_get_data (G_OBJECT (entry), "node");
		params = node->params;
		values = gnome_db_entry_combo_get_values (GNOME_DB_ENTRY_COMBO (entry));
		g_assert (g_slist_length (params) == g_slist_length (values));

		list = values;
		while (list) {
			/* REM: if there is more than one value in 'params', then a signal is emitted for each
			   param that is changed, and there is no way for the listener of that signal to know if it
			   the end of the "param_changed" sequence. What could be done is:
			   - adding another boolean to tell if that signal is the last one in the "param_changed" sequence, or
			   - modify the signal to add a list of parameters which are changed and emit only one signal.
			*/
			form->priv->forward_param_updates = FALSE;

			/* parameter's value */
			gnome_db_parameter_set_value (GNOME_DB_PARAMETER (params->data), (GdaValue *)(list->data));
#ifdef debug_signal
			g_print (">> 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
			g_signal_emit (G_OBJECT (form), gnome_db_basic_form_signals[PARAM_CHANGED], 0, params->data, TRUE);
#ifdef debug_signal
			g_print ("<< 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
			form->priv->forward_param_updates = TRUE;;
			gda_value_free (list->data);

			list = g_slist_next (list);
			params = g_slist_next (params);
		}
		g_slist_free (values);
	}
}


/*
 * Called when a parameter changes
 * We emit a "param_changed" signal only if the 'form->priv->forward_param_updates' is TRUE, which means
 * the param change does not come from a GnomeDbDataEntry change.
 */ 
static void
parameter_changed_cb (GnomeDbParameter *param, GnomeDbDataEntry *entry)
{
	GnomeDbBasicForm *form = g_object_get_data (G_OBJECT (entry), "form");
	GnomeDbDataSetNode *node = g_object_get_data (G_OBJECT (entry), "node");
	const GdaValue *value = gnome_db_parameter_get_value (param);

	if (form->priv->forward_param_updates) {
		gboolean param_valid;
		gboolean default_if_invalid = FALSE;

		/* There can be a feedback from the entry if the param is invalid and "set_default_if_invalid"
		   exists and is TRUE */
		param_valid = gnome_db_parameter_is_valid (param);
		if (!param_valid) 
			if (g_object_class_find_property (G_OBJECT_GET_CLASS (entry), "set_default_if_invalid"))
				g_object_get (G_OBJECT (entry), "set_default_if_invalid", &default_if_invalid, NULL);

		/* updating the corresponding entry */
		if (! default_if_invalid)
			g_signal_handlers_block_by_func (G_OBJECT (entry),
							 G_CALLBACK (entry_contents_modified), form);
		if (node) {
			GSList *values = NULL;
			GSList *list = node->params;
			gboolean allnull = TRUE;

			while (list) {
				const GdaValue *pvalue = gnome_db_parameter_get_value (GNOME_DB_PARAMETER (list->data));
				values = g_slist_append (values, pvalue);
				if (allnull && pvalue && (gda_value_get_type (pvalue) != GDA_VALUE_TYPE_NULL))
					allnull = FALSE;

				list = g_slist_next (list);
			}
			
			if (!allnull) 
				gnome_db_entry_combo_set_values (GNOME_DB_ENTRY_COMBO (entry), values);
			else 
				gnome_db_entry_combo_set_values (GNOME_DB_ENTRY_COMBO (entry), NULL);

			g_slist_free (values);
		}
		else
			gnome_db_data_entry_set_value (entry, value);

		if (! default_if_invalid)
			g_signal_handlers_unblock_by_func (G_OBJECT (entry),
							   G_CALLBACK (entry_contents_modified), form);

#ifdef debug_signal
		g_print (">> 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (form), gnome_db_basic_form_signals[PARAM_CHANGED], 0, param, FALSE);
#ifdef debug_signal
		g_print ("<< 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
	}
}

/**
 * gnome_db_basic_form_get_data_set
 * @form: a #GnomeDbBasicForm widget
 *
 * Get a pointer to the #GnomeDbDataSet used internally by @form to store
 * values
 *
 * Returns:
 */
GnomeDbDataSet *
gnome_db_basic_form_get_data_set (GnomeDbBasicForm *form)
{
	g_return_val_if_fail (form && IS_GNOME_DB_BASIC_FORM (form), NULL);
	g_return_val_if_fail (form->priv, NULL);

	return form->priv->data_set;
}

/**
 * gnome_db_basic_form_set_current_as_orig
 * @form: a #GnomeDbBasicForm widget
 *
 * Tells @form that the current values in the different entries are
 * to be considered as the original values for all the entries; the immediate
 * consequence is that any sub-sequent call to gnome_db_basic_form_has_been_changed()
 * will return FALSE (of course until any entry is changed).
 */
void
gnome_db_basic_form_set_current_as_orig (GnomeDbBasicForm *form)
{
	GSList *list;
	GnomeDbParameter *param;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	list = form->priv->entries;
	while (list) {
		GnomeDbDataSetNode *node = g_object_get_data (G_OBJECT (list->data), "node");

		if (node) {
			/* Combo entry */
			GSList *values = NULL;
			GSList *params = node->params;
			gboolean allnull = TRUE;
			
			while (params) {
				const GdaValue *pvalue = gnome_db_parameter_get_value (GNOME_DB_PARAMETER (params->data));
				values = g_slist_append (values, pvalue);
				if (allnull && pvalue && (gda_value_get_type (pvalue) != GDA_VALUE_TYPE_NULL))
					allnull = FALSE;
				
				params = g_slist_next (params);
			}
			
			if (!allnull) 
				gnome_db_entry_combo_set_values_orig (GNOME_DB_ENTRY_COMBO (list->data), values);
			else 
				gnome_db_entry_combo_set_values_orig (GNOME_DB_ENTRY_COMBO (list->data), NULL);
			
			g_slist_free (values);
		}
		else {
			/* non combo entry */
			param = g_object_get_data (G_OBJECT (list->data), "param");
			gnome_db_data_entry_set_value_orig (GNOME_DB_DATA_ENTRY (list->data), gnome_db_parameter_get_value (param));
		}
		list = g_slist_next (list);
	}
}

/**
 * gnome_db_basic_form_is_valid 
 * @form: a #GnomeDbBasicForm widget
 *
 * Tells if the form can be used as-is (if all the parameters do have some valid values)
 *
 * Returns: TRUE if the form is valid
 */
gboolean
gnome_db_basic_form_is_valid (GnomeDbBasicForm *form)
{
	g_return_val_if_fail (form && IS_GNOME_DB_BASIC_FORM (form), FALSE);
	g_return_val_if_fail (form->priv, FALSE);

	return gnome_db_data_set_is_valid (form->priv->data_set);
}

/**
 * gnome_db_basic_form_has_been_changed
 * @form: a #GnomeDbBasicForm widget
 *
 * Tells if the form has had at least on entry changed, or not
 *
 * Returns:
 */
gboolean
gnome_db_basic_form_has_been_changed (GnomeDbBasicForm *form)
{
	gboolean changed = FALSE;
	GSList *list;

	g_return_val_if_fail (form && IS_GNOME_DB_BASIC_FORM (form), FALSE);
	g_return_val_if_fail (form->priv, FALSE);
	
	list = form->priv->entries;
	while (list && !changed) {
		if (! (gnome_db_data_entry_get_attributes (GNOME_DB_DATA_ENTRY (list->data)) & GNOME_DB_VALUE_IS_UNCHANGED))
			changed = TRUE;
		list = g_slist_next (list);
	}

	return changed;
}

/**
 * gnome_db_basic_form_show_entries_actions
 * @form: a #GnomeDbBasicForm widget
 * @show_actions: a boolean
 *
 * Show or hide the actions button available at the end of each data entry
 * in the form
 */
void
gnome_db_basic_form_show_entries_actions (GnomeDbBasicForm *form, gboolean show_actions)
{
	GSList *entries;
	guint show;
	
	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	show = show_actions ? GNOME_DB_VALUE_ACTIONS_SHOWN : 0;

	entries = form->priv->entries;
	while (entries) {
		gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entries->data), show, 
						    GNOME_DB_VALUE_ACTIONS_SHOWN);
		entries = g_slist_next (entries);
	}

	/* mark_not_null_entry_labels (form, show_actions); */
}

/**
 * gnome_db_basic_form_reset
 * @form: a #GnomeDbBasicForm widget
 *
 * Resets all the entries in the form to their
 * original values
 */
void
gnome_db_basic_form_reset (GnomeDbBasicForm *form)
{
	GSList *list;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);
	
	list = form->priv->entries;
	while (list) {
		GnomeDbDataSetNode *node = g_object_get_data (G_OBJECT (list->data), "node");

		if (node) {
			/* Combo entry */
			GSList *values = NULL;

			values = gnome_db_entry_combo_get_values_orig (GNOME_DB_ENTRY_COMBO (list->data));
			gnome_db_entry_combo_set_values (GNOME_DB_ENTRY_COMBO (list->data), values);
			g_slist_free (values);
		}
		else {
			/* non combo entry */
			const GdaValue *value;

			value = gnome_db_data_entry_get_value_orig (GNOME_DB_DATA_ENTRY (list->data));
			gnome_db_data_entry_set_value (GNOME_DB_DATA_ENTRY (list->data), value);
		}
		list = g_slist_next (list);
	}
}


/**
 * gnome_db_basic_form_entry_show
 * @form: a #GnomeDbBasicForm widget
 * @param: a #GnomeDbParameter object
 * @show:
 *
 * Shows or hides the #GnomeDbDataEntry in @form which corresponds to the
 * @param parameter
 */
void
gnome_db_basic_form_entry_show (GnomeDbBasicForm *form, GnomeDbParameter *param, gboolean show)
{
	GSList *entries;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	entries = form->priv->entries;
	while (entries) {
		GtkWidget *entry = NULL;
		GnomeDbParameter *thisparam = g_object_get_data (G_OBJECT (entries->data), "param");

		if (thisparam) {
			if (thisparam == param)
				entry = GTK_WIDGET (entries->data);
		}
		else {
			/* multiple parameters */
			GSList *params;
			GnomeDbDataSetNode *node;

			node = g_object_get_data (G_OBJECT (entries->data), "node");
			params = node->params;
			while (params && !entry) {
				if (params->data == (gpointer) param)
					entry = GTK_WIDGET (entries->data);
				params = g_slist_next (params);
			}
		}

		if (entry) {
			gint row = -1;
			GtkWidget *entry_label = g_object_get_data (G_OBJECT (entry), "entry_label");

			if (form->priv->entries_table)
				row = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (entry), "row_no"));

			if (show) {
				if (g_slist_find (form->priv->hidden_entries, entry)) {
					form->priv->hidden_entries = g_slist_remove (form->priv->hidden_entries, entry);
					g_signal_handlers_disconnect_by_func (G_OBJECT (entry), 
									      G_CALLBACK (widget_shown_cb), form);
				}
				gtk_widget_show (entry);

				if (entry_label) {
					if (g_slist_find (form->priv->hidden_entries, entry_label)) {
						form->priv->hidden_entries = g_slist_remove (form->priv->hidden_entries, 
											     entry_label);
						g_signal_handlers_disconnect_by_func (G_OBJECT (entry_label), 
										      G_CALLBACK (widget_shown_cb), form);
					}
					gtk_widget_show (entry_label);
				}
				if (row > -1) 
					gtk_table_set_row_spacing (GTK_TABLE (form->priv->entries_table), row, 5);
			}
			else {
				if (!g_slist_find (form->priv->hidden_entries, entry)) {
					form->priv->hidden_entries = g_slist_append (form->priv->hidden_entries, entry);
					g_signal_connect_after (G_OBJECT (entry), "show", 
								G_CALLBACK (widget_shown_cb), form);
				}
				gtk_widget_hide (entry);

				if (entry_label) {
					if (!g_slist_find (form->priv->hidden_entries, entry_label)) {
						form->priv->hidden_entries = g_slist_append (form->priv->hidden_entries, 
											     entry_label);
						g_signal_connect_after (G_OBJECT (entry_label), "show", 
									G_CALLBACK (widget_shown_cb), form);
					}
					gtk_widget_hide (entry_label);
				}
				if (row > -1)
					gtk_table_set_row_spacing (GTK_TABLE (form->priv->entries_table), row, 0);
			}
		}

		entries = g_slist_next (entries);
	}
}

/**
 * gnome_db_basic_form_entry_set_sensitive
 * @form: a #GnomeDbBasicForm widget
 * @param: a #GnomeDbParameter object
 * @sensitive:
 *
 * Shows or hides the #GnomeDbDataEntry in @form which corresponds to the
 * @param parameter
 */
void
gnome_db_basic_form_entry_set_sensitive (GnomeDbBasicForm *form, GnomeDbParameter *param, gboolean sensitive)
{
	GSList *entries;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	entries = form->priv->entries;
	while (entries) {
		GtkWidget *entry = NULL;
		GnomeDbParameter *thisparam = g_object_get_data (G_OBJECT (entries->data), "param");

		if (thisparam) {
			if (thisparam == param)
				entry = GTK_WIDGET (entries->data);
		}
		else {
			/* multiple parameters */
			GSList *params;
			GnomeDbDataSetNode *node;

			node = g_object_get_data (G_OBJECT (entries->data), "node");
			params = node->params;
			while (params && !entry) {
				if (params->data == (gpointer) param)
					entry = GTK_WIDGET (entries->data);
				params = g_slist_next (params);
			}
		}

		if (entry) {
			GtkWidget *entry_label = g_object_get_data (G_OBJECT (entry), "entry_label");

			gtk_widget_set_sensitive (entry, sensitive);
			if (entry_label)
				gtk_widget_set_sensitive (entry_label, sensitive);
		}

		entries = g_slist_next (entries);
	}
}


/**
 * gnome_db_basic_form_set_entries_auto_default
 * @form: a #GnomeDbBasicForm widget
 * @auto_default:
 *
 * Sets weather all the #GnomeDbDataEntry entries in the form must default
 * to a default value if they are assigned a non valid value.
 * Depending on the real type of entry, it will provide a default value
 * which the user does not need to modify if it is OK.
 *
 * For example a date entry can by default display the current date.
 */
void
gnome_db_basic_form_set_entries_auto_default (GnomeDbBasicForm *form, gboolean auto_default)
{
	GSList *entries;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	entries = form->priv->entries;
	while (entries) {
		if (g_object_class_find_property (G_OBJECT_GET_CLASS (entries->data), "set_default_if_invalid"))
			g_object_set (G_OBJECT (entries->data), "set_default_if_invalid", auto_default, NULL);
		entries = g_slist_next (entries);
	}	
}

/**
 * gnome_db_basic_form_set_entries_default
 * @form: a #GnomeDbBasicForm widget
 *
 * For each entry in the form, sets it to a default value if it is possible to do so.
 */
void
gnome_db_basic_form_set_entries_default (GnomeDbBasicForm *form)
{
	GSList *entries;
	guint attrs;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	entries = form->priv->entries;
	while (entries) {
		attrs = gnome_db_data_entry_get_attributes (GNOME_DB_DATA_ENTRY (entries->data));
		if (attrs & GNOME_DB_VALUE_CAN_BE_DEFAULT)
			gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entries->data), 
						      GNOME_DB_VALUE_IS_DEFAULT, GNOME_DB_VALUE_IS_DEFAULT);
		entries = g_slist_next (entries);
	}
}

static void form_param_changed (GnomeDbBasicForm *form, GnomeDbParameter *param, gboolean is_user_modif, GtkDialog *dlg);

/**
 * gnome_db_basic_form_new_in_dialog
 * @dict: a #GnomeDbDict object
 * @data_set: a #GnomeDbDataSet structure
 * @parent: the parent window for the new dialog, or %NULL
 * @title: the title of the dialog window, or %NULL
 * @header: a helper text displayed at the top of the dialog, or %NULL
 *
 * Creates a new #GnomeDbBasicForm widget in the same way as gnome_db_basic_form_new()
 * and puts it into a #GtkDialog widget. The returned dialog has the "Ok" and "Cancel" buttons
 * which respectively return GTK_RESPONSE_ACCEPT and GTK_RESPONSE_REJECT.
 *
 * The #GnomeDbBasicForm widget is attached to the dialog using the user property
 * "form".
 *
 * Returns: the new #GtkDialog widget
 */
GtkWidget *
gnome_db_basic_form_new_in_dialog (GnomeDbDict *dict, GnomeDbDataSet *data_set, GtkWindow *parent,
				   const gchar *title, const gchar *header)
{
	GtkWidget *form;
	GtkWidget *dlg;
	const gchar *rtitle;

	form = gnome_db_basic_form_new (dict, data_set);
 
	rtitle = title;
	if (!rtitle)
		rtitle = _("Values to be filled");
		
	dlg = gtk_dialog_new_with_buttons (rtitle, parent,
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK,
					   GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL,
					   GTK_RESPONSE_REJECT,
					   NULL);
	if (header && *header) {
		GtkWidget *label;

		label = gtk_label_new (NULL);
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
		gtk_label_set_markup (GTK_LABEL (label), header);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), label, FALSE, FALSE, 5);
		gtk_widget_show (label);
	}

	gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG (dlg)->vbox), 4);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), form, TRUE, TRUE, 10);

	g_signal_connect (G_OBJECT (form), "param_changed",
			  G_CALLBACK (form_param_changed), dlg);
	g_object_set_data (G_OBJECT (dlg), "form", form);

	gtk_widget_show_all (form);
	form_param_changed (GNOME_DB_BASIC_FORM (form), NULL, FALSE, GTK_DIALOG (dlg));

	return dlg;
}

static void
form_param_changed (GnomeDbBasicForm *form, GnomeDbParameter *param, gboolean is_user_modif, GtkDialog *dlg)
{
	gboolean valid;

	valid = gnome_db_basic_form_is_valid (form);

	gtk_dialog_set_response_sensitive (dlg, GTK_RESPONSE_ACCEPT, valid);
}
