/* gnome-db-result-set.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-result-set.h"
#include "marshal.h"
#include "gnome-db-server.h"
#include "gnome-db-data-model.h"
#include "gnome-db-parameter.h"
#include "gnome-db-data-set.h"

#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include "gnome-db-join.h"
#include "gnome-db-table.h"
#include "gnome-db-table-field.h"
#include "gnome-db-entity.h"
#include "gnome-db-qfield.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-qf-value.h"
#include "gnome-db-qf-all.h"
#include "gnome-db-field.h"
#include "gnome-db-data-handler.h"
#include "gnome-db-condition.h"
#include "gnome-db-constraint.h"

#include "gnome-db-referer.h"
#include "gnome-db-renderer.h"
#include "gnome-db-wrapper-field.h"

/* 
 * Main static functions 
 */
static void gnome_db_result_set_class_init (GnomeDbResultSetClass * class);
static void gnome_db_result_set_init (GnomeDbResultSet * srv);
static void gnome_db_result_set_dispose (GObject   * object);
static void gnome_db_result_set_finalize (GObject   * object);

static void gnome_db_result_set_set_property (GObject              *object,
					     guint                 param_id,
					     const GValue         *value,
					     GParamSpec           *pspec);
static void gnome_db_result_set_get_property (GObject              *object,
					     guint                 param_id,
					     GValue               *value,
					     GParamSpec           *pspec);
static void make_entity_fields (GnomeDbResultSet *rs);

#ifdef debug
static void gnome_db_result_set_dump (GnomeDbResultSet *rs, guint offset);
#endif

/* GnomeDbDataModel interface */
static void              gnome_db_result_set_model_init             (GnomeDbDataModelIface *iface);
static GObject          *gnome_db_result_set_model_copy             (GnomeDbDataModel *iface, GHashTable *replacements);
static gboolean          gnome_db_result_set_model_refresh          (GnomeDbDataModel *iface, GError **error);
static GnomeDbDataSet   *gnome_db_result_set_model_get_params       (GnomeDbDataModel *iface);
static GnomeDbDataSet   *gnome_db_result_set_model_get_new_data_set (GnomeDbDataModel *iface);
static gboolean          gnome_db_result_set_model_sync_data_set    (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint row);
static GnomeDbParameter *gnome_db_result_set_model_get_param_at_col (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint col);
static gint              gnome_db_result_set_model_get_col_at_param (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, 
								     GnomeDbParameter *param);
static GSList           *gnome_db_result_set_model_get_key_columns  (GnomeDbDataModel *iface, GnomeDbDataSet *data_set);
static guint             gnome_db_result_set_model_get_status       (GnomeDbDataModel *iface);


/* GdaDataModel interface */
static void                 gnome_db_result_set_data_model_init (GdaDataModelIface *iface);
static gint                 gnome_db_result_set_get_n_rows      (GdaDataModel *model);
static gint                 gnome_db_result_set_get_n_columns   (GdaDataModel *model);
static GdaDataModelColumnAttributes  *gnome_db_result_set_describe_column (GdaDataModel *model, gint col);
static const gchar         *gnome_db_result_set_get_column_title(GdaDataModel *model, gint col);
static void                 gnome_db_result_set_set_column_title(GdaDataModel *model, gint col, const gchar *title);
static gint                 gnome_db_result_set_get_column_pos  (GdaDataModel *model, const gchar *title);
static const GdaRow        *gnome_db_result_set_get_row         (GdaDataModel *model, gint row);
static const GdaValue      *gnome_db_result_set_get_value_at    (GdaDataModel *model, gint col, gint row);
static gboolean             gnome_db_result_set_is_updatable    (GdaDataModel *model);
static gboolean             gnome_db_result_set_has_changed     (GdaDataModel *model);
static void                 gnome_db_result_set_begin_changes   (GdaDataModel *model);
static gboolean             gnome_db_result_set_commit_changes  (GdaDataModel *model);
static gboolean             gnome_db_result_set_cancel_changes  (GdaDataModel *model);
static gboolean             gnome_db_result_set_append_row      (GdaDataModel *model, GdaRow *row);
static gboolean             gnome_db_result_set_remove_row      (GdaDataModel *model, const GdaRow *row);
static gboolean             gnome_db_result_set_update_row      (GdaDataModel *model, const GdaRow *row);
static void                 gnome_db_result_set_set_notify      (GdaDataModel *model, gboolean do_notify_changes);
static gboolean             gnome_db_result_set_get_notify      (GdaDataModel *model);
static gboolean             gnome_db_result_set_set_command     (GdaDataModel *model, const gchar *txt, GdaCommandType type);
static const gchar         *gnome_db_result_set_get_command     (GdaDataModel *model, GdaCommandType *type);

/* Entity interface */
static void             gnome_db_result_set_entity_init         (GnomeDbEntityIface *iface);
static gboolean         gnome_db_result_set_has_field           (GnomeDbEntity *iface, GnomeDbField *field);
static GSList          *gnome_db_result_set_get_fields          (GnomeDbEntity *iface);
static GnomeDbField    *gnome_db_result_set_get_field_by_name   (GnomeDbEntity *iface, const gchar *name);
static GnomeDbField    *gnome_db_result_set_get_field_by_index  (GnomeDbEntity *iface, gint index);
static gint             gnome_db_result_set_get_field_index     (GnomeDbEntity *iface, GnomeDbField *field);
static gboolean         gnome_db_result_set_is_writable         (GnomeDbEntity *iface);
static GSList          *gnome_db_result_set_get_parameters      (GnomeDbEntity *iface);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* properties */
enum
{
	PROP_0,
	PROP_REPLACEMENTS
};

typedef struct {
        GnomeDbParameter *param;    /* param in the 'data_set' set */
        gint              position; /* corresponding field position in query_select (and in the 'model' GdaDataModel) */
} RsNode;
#define RS_NODE(x) ((RsNode *)x)


/* private structure */
struct _GnomeDbResultSetPrivate
{
	GnomeDbServer          *srv;
	GdaDataModel           *model; /* may be NULL if created with new_with_query() and not yet refreshed  */
	GSList                 *fields; /* list of GnomeDbWrapperField objects for the GnomeDbEntity interface */
	gboolean                auto_refresh; /* tells if a refresh is performed after each row modification */
	gboolean                refresh_pending; /* tells if a refresh will have to be performed when auto_refresh becomes TRUE */

	/*
	 * fields when created with new_with_query()
	 */
	GnomeDbQuery           *query_select; /* NULL if created with new_with_data () */
	GnomeDbTarget          *modif_target; /* may be NULL */
	GnomeDbDataSet         *params_set;   /* for the parameters required to run the SELECT query */
        GnomeDbDataSet         *data_set;     /* for the parameters required to run the modif queries */

        GnomeDbQuery           *query_select_hybrid; /* copy of query_select to which various fields have been added */
        GnomeDbQuery           *query_update;
        GnomeDbQuery           *query_delete;
        GnomeDbQuery           *query_insert;

        GSList                 *nodes; /* params which appear in 'model' (RsNode structs) and their pos. */

	/* Translation from original query to query_select, and the other way around, 
	 * and when this entity's fields are made (make_entity_fields), from original query to this->fields and
	 * still from query_select to original query */
        GHashTable             *replacements_query_select;

	/* list of GnomeDbQfValue fields added to 'query_select_hybrid' to express a condition in the modif queries */
	GSList                 *for_cond_value_qfields;

	/* list of ('query_select_hybrid') fields which have been added when using FK constraints */
	GSList                 *ref_integ_fields;

	/* RsNodes lists for extra data sets */
	GHashTable             *extra_data_sets_nodes;

	/* temporary information used for initialization */
        GHashTable             *replacements_query_select_hybrid; /* from the 'query_select' to 'query_select_hybrid' */

	/* key = a GnomeDbQfield in a GnomeDbDataSetNode->data_for_param, value = the GnomeDbQfield in query_select_hybrid */
	GHashTable             *data_set_data_for_params_source;
};


/* module error */
GQuark gnome_db_result_set_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_result_set_error");
	return quark;
}


guint
gnome_db_result_set_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbResultSetClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_result_set_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbResultSet),
			0,
			(GInstanceInitFunc) gnome_db_result_set_init
		};

		static const GInterfaceInfo gda_model_info = {
                        (GInterfaceInitFunc) gnome_db_result_set_data_model_init,
                        NULL,
                        NULL
                };
		

		static const GInterfaceInfo model_info = {
                        (GInterfaceInitFunc) gnome_db_result_set_model_init,
                        NULL,
                        NULL
                };

		static const GInterfaceInfo entity_info = {
			(GInterfaceInitFunc) gnome_db_result_set_entity_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbResultSet", &info, 0);
		g_type_add_interface_static (type, GDA_TYPE_DATA_MODEL, &gda_model_info);
		g_type_add_interface_static (type, GNOME_DB_ENTITY_TYPE, &entity_info);
		g_type_add_interface_static (type, GNOME_DB_DATA_MODEL_TYPE, &model_info);
	}
	return type;
}

static void
gnome_db_result_set_model_init (GnomeDbDataModelIface *iface)
{
	iface->copy = gnome_db_result_set_model_copy;
	iface->refresh = gnome_db_result_set_model_refresh;
	iface->get_params = gnome_db_result_set_model_get_params;
	iface->get_new_data_set = gnome_db_result_set_model_get_new_data_set;
	iface->sync_data_set = gnome_db_result_set_model_sync_data_set;
	iface->get_param_at_col = gnome_db_result_set_model_get_param_at_col;
	iface->get_col_at_param = gnome_db_result_set_model_get_col_at_param ;
	iface->get_key_columns = gnome_db_result_set_model_get_key_columns;
	iface->get_status = gnome_db_result_set_model_get_status;
}

static void
gnome_db_result_set_data_model_init (GdaDataModelIface *iface)
{
        iface->i_get_n_rows = gnome_db_result_set_get_n_rows;
        iface->i_get_n_columns = gnome_db_result_set_get_n_columns;
        iface->i_describe_column = gnome_db_result_set_describe_column;
        iface->i_get_column_title = gnome_db_result_set_get_column_title;
        iface->i_set_column_title = gnome_db_result_set_set_column_title;
        iface->i_get_column_pos = gnome_db_result_set_get_column_pos;
        iface->i_get_row = gnome_db_result_set_get_row;
        iface->i_get_value_at = gnome_db_result_set_get_value_at;
        iface->i_is_updatable = gnome_db_result_set_is_updatable;
        iface->i_has_changed = gnome_db_result_set_has_changed;
        iface->i_begin_changes = gnome_db_result_set_begin_changes;
        iface->i_cancel_changes = gnome_db_result_set_cancel_changes;
        iface->i_commit_changes = gnome_db_result_set_commit_changes;
        iface->i_append_values = NULL; /* not implemented */
        iface->i_append_row = gnome_db_result_set_append_row;
        iface->i_remove_row = gnome_db_result_set_remove_row;
        iface->i_update_row = gnome_db_result_set_update_row;
        iface->i_append_column = NULL;
        iface->i_update_column = NULL;
        iface->i_remove_column = NULL;
        iface->i_set_notify = gnome_db_result_set_set_notify;
        iface->i_get_notify = gnome_db_result_set_get_notify;
        iface->i_set_command = gnome_db_result_set_set_command;
        iface->i_get_command = gnome_db_result_set_get_command;
}

static void
gnome_db_result_set_entity_init (GnomeDbEntityIface *iface)
{
	iface->has_field = gnome_db_result_set_has_field;
	iface->get_fields = gnome_db_result_set_get_fields;
	iface->get_field_by_name = gnome_db_result_set_get_field_by_name;
	iface->get_field_by_xml_id = NULL;
	iface->get_field_by_index = gnome_db_result_set_get_field_by_index;
	iface->get_field_index = gnome_db_result_set_get_field_index;
	iface->add_field = NULL;
	iface->add_field_before = NULL;
	iface->swap_fields = NULL;
	iface->remove_field = NULL;
	iface->is_writable = gnome_db_result_set_is_writable;
	iface->get_parameters = gnome_db_result_set_get_parameters;
}


static void
gnome_db_result_set_class_init (GnomeDbResultSetClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_result_set_dispose;
	object_class->finalize = gnome_db_result_set_finalize;

	/* Properties */
	object_class->set_property = gnome_db_result_set_set_property;
	object_class->get_property = gnome_db_result_set_get_property;
	g_object_class_install_property (object_class, PROP_REPLACEMENTS,
					 g_param_spec_pointer ("init_query_replacements", "Fields replacements", 
							       "translate from fields of the query used for construction to "
							       "the fields of the GnomeDbResultset", G_PARAM_READABLE));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_result_set_dump;
#endif
}

static void
gnome_db_result_set_init (GnomeDbResultSet * rs)
{
	rs->priv = g_new0 (GnomeDbResultSetPrivate, 1);
	rs->priv->srv = NULL;
	rs->priv->model = NULL;
	rs->priv->auto_refresh = TRUE;
	rs->priv->data_set_data_for_params_source = g_hash_table_new (NULL, NULL);	
}

static void rs_init_with_query (GnomeDbResultSet *rs);
static void conn_closed_cb (GnomeDbServer *srv, GnomeDbResultSet *rs);
static void nullified_query_cb (GnomeDbQuery *query, GnomeDbResultSet *rs);
static void nullified_target_cb (GnomeDbTarget *target, GnomeDbResultSet *rs);
static void nullified_context_cb (GnomeDbDataSet *context, GnomeDbResultSet *rs);

/**
 * gnome_db_result_set_new
 * @query: a #GnomeDbQuery (a SELECT query)
 * @modif_target: the target representing the modified entity, or %NULL
 *
 * Creates a new #GnomeDbResultSet object which will contain the data extracted from the
 * @query SELECT query. If @modif_target is not %NULL, then the resulting data model will
 * have modifiable values for fields belonging to the entity represented by @modif_target.
 *
 * Note: to actually have some data in the new #GnomeDbResultSet object, the
 * gnome_db_data_model_refresh() function will have to be called.
 *
 * Returns: the new object
 */
GObject *
gnome_db_result_set_new (GnomeDbQuery *query, GnomeDbTarget *modif_target)
{
	GObject   *obj;
	GnomeDbResultSet *rs;
	GnomeDbDict *dict;
	GHashTable *replacements;
	GSList *list;
	
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (gnome_db_query_is_select_query (query), NULL);
	if (modif_target) {
		g_return_val_if_fail (IS_GNOME_DB_TARGET (modif_target), NULL);
		g_return_val_if_fail (gnome_db_target_get_query (modif_target) == query, NULL);
		g_return_val_if_fail (gnome_db_entity_is_writable (gnome_db_target_get_represented_entity (modif_target)), NULL);
	}

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_RESULT_SET_TYPE, "dict", dict, NULL);
	rs = GNOME_DB_RESULT_SET (obj);

	rs->priv->srv = gnome_db_dict_get_server (dict);
	rs->priv->model = NULL;

	replacements = g_hash_table_new (NULL, NULL);
	rs->priv->query_select = GNOME_DB_QUERY (gnome_db_query_new_copy (query, replacements));
	g_object_set (rs->priv->query_select, "auto_clean", FALSE, NULL);
	gnome_db_query_order_fields_using_join_conds (rs->priv->query_select);
	gnome_db_base_connect_nullify (rs->priv->query_select,
				       G_CALLBACK (nullified_query_cb), rs);

	if (modif_target) {
		rs->priv->modif_target = g_hash_table_lookup (replacements, modif_target);
		gnome_db_base_connect_nullify (rs->priv->modif_target,
					       G_CALLBACK (nullified_target_cb), rs);
	}

	/* the 'replacement' contains transtation from 'query' to 'rs->priv->query_select'; we want to 
	 * add the reverse translation at this point */
	g_object_get (query, "really_all_fields", &list, NULL);
	while (list) {
		g_hash_table_insert (replacements, g_hash_table_lookup (replacements, list->data), list->data);
		list = g_slist_next (list);
	}
	rs->priv->replacements_query_select = replacements;

	rs_init_with_query (rs);

	g_signal_connect (G_OBJECT (rs->priv->srv), "conn_closed",
			  G_CALLBACK (conn_closed_cb), rs);

	return obj;
}

static void
nullified_query_cb (GnomeDbQuery *query, GnomeDbResultSet *rs)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (nullified_query_cb), rs);

	if (query == rs->priv->query_select) 
		rs->priv->query_select = NULL;

	if (rs->priv && (query == rs->priv->query_select_hybrid)) 
		rs->priv->query_select_hybrid = NULL;

	if (query == rs->priv->query_update) 
		rs->priv->query_update = NULL;
	
	if (query == rs->priv->query_delete) 
		rs->priv->query_delete = NULL;

	if (query == rs->priv->query_insert) 
		rs->priv->query_insert = NULL;
	
	g_object_unref (G_OBJECT (query));
}

static void
nullified_target_cb (GnomeDbTarget *target, GnomeDbResultSet *rs)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (target),
					      G_CALLBACK (nullified_target_cb), rs);
	rs->priv->modif_target = NULL;
}

static void
nullified_context_cb (GnomeDbDataSet *context, GnomeDbResultSet *rs)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (context),
					      G_CALLBACK (nullified_context_cb), rs);
       
	if (context == rs->priv->params_set) 
		rs->priv->params_set = NULL;

	if (context == rs->priv->data_set) 
		rs->priv->data_set = NULL;

	g_object_unref (G_OBJECT (context));
}


static void
conn_closed_cb (GnomeDbServer *srv, GnomeDbResultSet *rs)
{
	gnome_db_base_nullify (GNOME_DB_BASE (rs));
}


static void
gnome_db_result_set_dispose (GObject *object)
{
	GnomeDbResultSet *rs;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_RESULT_SET (object));

	rs = GNOME_DB_RESULT_SET (object);
	if (rs->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		g_signal_handlers_disconnect_by_func (G_OBJECT (rs->priv->srv),
						      G_CALLBACK (conn_closed_cb), rs);

		/* data */
		if (rs->priv->model) {
			g_object_unref (G_OBJECT (rs->priv->model));
			rs->priv->model = NULL;
		}
		
		/* hash tables */
		if (rs->priv->query_select_hybrid)
			nullified_query_cb (rs->priv->query_select_hybrid, rs);

		if (rs->priv->replacements_query_select) {
			g_hash_table_destroy (rs->priv->replacements_query_select);
			rs->priv->replacements_query_select = NULL;
		}
		
		g_assert (!rs->priv->replacements_query_select_hybrid);

		if (rs->priv->for_cond_value_qfields) {
			g_slist_free (rs->priv->for_cond_value_qfields);
			rs->priv->for_cond_value_qfields = NULL;
		}

		if (rs->priv->ref_integ_fields) {
			g_slist_free (rs->priv->ref_integ_fields);
			rs->priv->ref_integ_fields = NULL;
		}

		if (rs->priv->data_set_data_for_params_source) {
			g_hash_table_destroy (rs->priv->data_set_data_for_params_source);
			rs->priv->data_set_data_for_params_source = NULL;
		}

		/* nodes */
		if (rs->priv->nodes) {
			GSList *list = rs->priv->nodes;
			while (list) {
				g_free (list->data);
				list = g_slist_next (list);
			}
			g_slist_free (rs->priv->nodes);
			rs->priv->nodes = NULL;
		}
		
		/* contexts */
		if (rs->priv->params_set)
			nullified_context_cb (rs->priv->params_set, rs);
	
		if (rs->priv->data_set) 
			nullified_context_cb (rs->priv->data_set, rs);
	
		/* queries */
		if (rs->priv->query_select)
			nullified_query_cb (rs->priv->query_select, rs);
		if (rs->priv->query_update)
			nullified_query_cb (rs->priv->query_update, rs);
		if (rs->priv->query_delete)
			nullified_query_cb (rs->priv->query_delete, rs);
		if (rs->priv->query_insert)
			nullified_query_cb (rs->priv->query_insert, rs);
	
		/* mofifiable target */
		if (rs->priv->modif_target)
			nullified_target_cb (rs->priv->modif_target, rs);

		/* RsNodes for extra data sets */
		if (rs->priv->extra_data_sets_nodes) {
			g_hash_table_destroy (rs->priv->extra_data_sets_nodes);
			rs->priv->extra_data_sets_nodes = NULL;
		}

		/* fields */
		while (rs->priv->fields) 
			gnome_db_base_nullify (GNOME_DB_BASE (rs->priv->fields->data));
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_result_set_finalize (GObject   * object)
{
	GnomeDbResultSet *rs;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_RESULT_SET (object));

	rs = GNOME_DB_RESULT_SET (object);
	if (rs->priv) {
		g_free (rs->priv);
		rs->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_result_set_set_property (GObject              *object,
				  guint                 param_id,
				  const GValue         *value,
				  GParamSpec           *pspec)
{
	GnomeDbResultSet *rs;

	rs = GNOME_DB_RESULT_SET (object);
	if (rs->priv) {
		switch (param_id) {
		default:
			break;
		}
	}
}

static void
gnome_db_result_set_get_property (GObject              *object,
				  guint                 param_id,
				  GValue               *value,
				  GParamSpec           *pspec)
{
	GnomeDbResultSet *rs;
	rs = GNOME_DB_RESULT_SET (object);
	
	if (rs->priv) {
		switch (param_id) {
		case PROP_REPLACEMENTS:
			make_entity_fields (rs);
			g_value_set_pointer (value, rs->priv->replacements_query_select);
			break;
		}	
	}
}

#ifdef debug
static void
gnome_db_result_set_dump (GnomeDbResultSet *rs, guint offset)
{
	gchar *offstr, *str;
	gint n_cols, n_rows;
	gint *cols_size;
	gchar *sep_col  = " | ";
	gchar *sep_row  = "-+-";
	gchar sep_fill = '-';
	gint i, j;
	GdaDataModel *model;
	const GdaValue *value;
	
	g_return_if_fail (rs && IS_GNOME_DB_RESULT_SET (rs));
	g_return_if_fail (rs->priv);

        /* string for the offset */
        offstr = g_new0 (gchar, offset+1);
	memset (offstr, ' ', offset);

	g_print ("%s" D_COL_H1 "GnomeDbResultSet" D_COL_NOR " %p\n", offstr, rs);

	/* display fields information */
	if (rs->priv->fields) {
		GSList *list = rs->priv->fields;
		g_print ("%sFields: ", offstr);

		while (list) {
			g_print (" %p", list->data);
			list = g_slist_next (list);
		}
		g_print ("\n");
	}
	else
		g_print ("%sHas no field (yet)\n", offstr);
	

	model = rs->priv->model;
	if (model) {
		/* compute the columns widths: using column titles... */
		n_cols = gda_data_model_get_n_columns (model);
		n_rows = gda_data_model_get_n_rows (model);
		cols_size = g_new0 (gint, n_cols);
	
		for (i = 0; i < n_cols; i++) {
			str = gda_data_model_get_column_title (model, i);
			cols_size [i] = strlen (str);
		}

		/* ... and using column data */
		for (j = 0; j < n_rows; j++) {
			for (i = 0; i < n_cols; i++) {
				value = gda_data_model_get_value_at (model, i, j);
				str = value ? gda_value_stringify (value) : g_strdup ("_null_");
				cols_size [i] = MAX (cols_size [i], strlen (str));
				g_free (str);
			}
		}
	
		/* actual dumping of the contents: column titles...*/
		for (i = 0; i < n_cols; i++) {
			str = gda_data_model_get_column_title (model, i);
			if (i != 0)
				g_print ("%s", sep_col);
			g_print ("%*s", cols_size [i], str);
		}
		g_print ("\n");
		
		/* ... separation line ... */
		for (i = 0; i < n_cols; i++) {
			if (i != 0)
				g_print ("%s", sep_row);
			for (j = 0; j < cols_size [i]; j++)
				g_print ("%c", sep_fill);
		}
		g_print ("\n");

		/* ... and data */
		for (j = 0; j < n_rows; j++) {
			for (i = 0; i < n_cols; i++) {
				value = gda_data_model_get_value_at (model, i, j);
				str = value ? gda_value_stringify (value) : g_strdup ("_null_");
				if (i != 0)
					g_print ("%s", sep_col);
				g_print ("%*s", cols_size [i], str);
				g_free (str);
			}
			g_print ("\n");
		}
		g_free (cols_size);
	}
	else
		g_print ("%sNo data model present\n", offstr);

	g_free (offstr);
}
#endif

/**
 * gnome_db_result_set_check_data_model
 * @model: a #GdaDataModel object
 * @nbcols: the requested number of columns
 * @Varargs: @nbcols arguments of type GdaValueType or -1 (if any data type is accepted)
 *
 * Check the column types of a GdaDataModel.
 *
 * Returns: TRUE if the data model's columns match the provided data types and number
 */
gboolean
gnome_db_result_set_check_data_model (GdaDataModel *model, gint nbcols, ...)
{
	gboolean retval = TRUE;
	gint i;

	g_return_val_if_fail (model && GDA_IS_DATA_MODEL (model), FALSE);
	
	/* number of columns */
	if (gda_data_model_get_n_columns (model) != nbcols)
		return FALSE;

	/* type of each column */
	if (nbcols > 0) {
		GdaDataModelColumnAttributes *att;
		GdaValueType mtype, rtype;
		gint argtype;
		va_list ap;

		va_start  (ap, nbcols);
		i = 0;
		while ((i<nbcols) && retval) {
			att = gda_data_model_describe_column (model, i);
			mtype = gda_data_model_column_attributes_get_gdatype (att);
			gda_data_model_column_attributes_free (att);
			
			argtype = va_arg (ap, GdaValueType);
			if (argtype >= 0) {
				rtype = (GdaValueType) argtype;
				if (mtype != rtype) {
					retval = FALSE;
#ifdef debug
					g_print ("Position %d: Expected %d, got %d\n",
						 i, rtype, mtype);
#endif
				}
			}
			
			i++;
		}
		va_end (ap);
	}

	return retval;

}

/** 
 * gnome_db_result_set_check_model
 * @rs: a #GnomeDbResultSet object
 * @nbcols: the requested number of columns
 * @Varargs: @nbcols arguments of type GdaValueType or -1 (if any data type is accepted)
 * 
 * Check the column types of a resultset. 
 *
 * Returns: TRUE if the resultset's columns match the provided data types and number
 */
gboolean
gnome_db_result_set_check_model (GnomeDbResultSet * rs, gint nbcols, ...)
{
	va_list args;
	gboolean retval;

	g_return_val_if_fail (rs && IS_GNOME_DB_RESULT_SET (rs), FALSE);
	g_return_val_if_fail (rs->priv, FALSE);
	g_return_val_if_fail (rs->priv->model, FALSE);

	va_start (args, nbcols);
	retval = gnome_db_result_set_check_data_model (rs->priv->model, nbcols, args);	
	va_end (args);

	return retval;
}

/**
 * gnome_db_result_set_get_row_as_string
 * @model: a #GdaDataModel object
 * @query: a #GnomeDbQuery object
 * @row:
 * @sep:
 *
 * Creates a string representing, for the specified @row, the visible fields @query,
 * separated by @sep. @query is usually the query which has been used to create the
 * resultset.
 *
 * Returns: a new string, or %NULL if an error occured.
 */
gchar *
gnome_db_result_set_get_row_as_string (GdaDataModel *model, GnomeDbQuery *query, gint row, gchar *sep)
{
	GSList *fields, *list;
	GString *string;
	gchar *str;
	gboolean first = TRUE;
	GnomeDbServer *srv;

	g_return_val_if_fail (model && GDA_IS_DATA_MODEL (model), NULL);
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	
	srv = gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (query)));
	string = g_string_new ("");
	fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (query));
	list = fields;
	while (list) {
		if (!gnome_db_qfield_is_internal (GNOME_DB_QFIELD (list->data))) {
			GnomeDbDataHandler *dh;
			const GdaValue *val;
			gchar *str;

			dh = gnome_db_server_get_handler_by_type (srv,
								  gnome_db_field_get_data_type (GNOME_DB_FIELD (list->data)));
			val = gda_data_model_get_value_at (model,
							   gnome_db_entity_get_field_index (GNOME_DB_ENTITY (query),
											    GNOME_DB_FIELD (list->data)),
							   row);
			if (first)
				first = FALSE;
			else
				if (sep && *sep) g_string_append (string, sep);
				
			str = gnome_db_data_handler_get_str_from_value (dh, val);
			g_string_append (string, str);
			g_free (str);
		}
		list = g_slist_next (list);
	}
	g_slist_free (fields);
	str = string->str;
	g_string_free (string, FALSE);

	return str;
}

static void
foreach_hash_copy (gpointer key, gpointer value, GHashTable *dest)
{
	g_hash_table_insert (dest, key, value);
}

/**
 * gnome_db_result_set_find_column_ext
 */
gint
gnome_db_result_set_find_column_ext (GnomeDbResultSet *rs, GnomeDbDataSetNode *node, gint col)
{
	gint retval = -1;
	GnomeDbField *field, *field2;

	if (!node->data_for_param) 
		return -1;

	field = gnome_db_entity_get_field_by_index (GNOME_DB_ENTITY (node->data_for_param), col);
	if (!field) 
		return -1;

	field2 = g_hash_table_lookup (rs->priv->data_set_data_for_params_source, field);
	if (!field2) {
		g_warning ("Field %p is not in rs->priv->data_set_data_for_params_source!", field);
		return -1;
	}
	
	retval = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (rs->priv->query_select), field2);

	return retval;
}

/*
 * GnomeDbDataModel interface implementation
 */
static GObject *
gnome_db_result_set_model_copy (GnomeDbDataModel *iface, GHashTable *replacements)
{
	GObject *obj;
	GnomeDbResultSet *rs, *orig;
	GHashTable *repl;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), FALSE);
	orig = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (orig->priv, NULL);

	obj = g_object_new (GNOME_DB_RESULT_SET_TYPE, "dict", gnome_db_server_get_dict (orig->priv->srv), NULL);
	gnome_db_base_set_name (GNOME_DB_BASE (obj),
				gnome_db_base_get_name (GNOME_DB_BASE (iface)));
	rs = GNOME_DB_RESULT_SET (obj);

	rs->priv->srv = orig->priv->srv;
	if (orig->priv->model) {
		rs->priv->model = orig->priv->model;
		g_object_ref (rs->priv->model);
	}

	/* attributes copy */
	if (replacements)
		repl = replacements;
	else
		repl = g_hash_table_new (NULL, NULL);

	if (orig->priv->query_select) {
		rs->priv->query_select = orig->priv->query_select;
		g_object_ref (rs->priv->query_select);
	}

	if (orig->priv->modif_target) {
		rs->priv->modif_target = orig->priv->modif_target;
		g_object_ref (rs->priv->modif_target);
	}

	if (orig->priv->query_select) {
		rs->priv->query_select = orig->priv->query_select;
		g_object_ref (rs->priv->query_select);
	}

	if (orig->priv->params_set) 
		rs->priv->params_set = GNOME_DB_DATA_SET (gnome_db_data_set_new_copy (orig->priv->params_set, repl));
	if (orig->priv->data_set) 
		rs->priv->data_set = GNOME_DB_DATA_SET (gnome_db_data_set_new_copy (orig->priv->data_set, repl));
	
	if (orig->priv->query_update) {
		rs->priv->query_update = orig->priv->query_update;
		g_object_ref (rs->priv->query_update);
	}

	if (orig->priv->query_delete) {
		rs->priv->query_delete = orig->priv->query_delete;
		g_object_ref (rs->priv->query_delete);
	}

	if (orig->priv->query_insert) {
		rs->priv->query_insert = orig->priv->query_insert;
		g_object_ref (rs->priv->query_insert);
	}

	rs->priv->replacements_query_select = g_hash_table_new (NULL, NULL);
	g_hash_table_foreach (orig->priv->replacements_query_select, 
			      (GHFunc) foreach_hash_copy, rs->priv->replacements_query_select);

	if (orig->priv->fields) {
		make_entity_fields (rs); /* rem: 'replacements_query_select' is also updated */

		if (replacements) {
			/* insert into @replacements the translation from orig->priv->fields to rs->priv->fields */
			GSList *olist, *nlist;

			olist = orig->priv->fields;
			nlist = rs->priv->fields;
			while (olist && nlist) {
				g_hash_table_insert (replacements, olist->data, nlist->data);
				olist = g_slist_next (olist);
				nlist = g_slist_next (nlist);
			}
			g_assert (!olist && !nlist);
		}
	}

	g_signal_connect (G_OBJECT (rs->priv->srv), "conn_closed",
			  G_CALLBACK (conn_closed_cb), rs);
	
	/* rem: other attributes do not need to be changed:
	   - for_cond_value_qfields, ref_integ_fields, replacements_query_select_hybrid because the queries don't change
	   - extra_data_sets_nodes because it can stay NULL
	*/

	if (!replacements)
		g_hash_table_destroy (repl);

	return obj;
}

static gboolean
gnome_db_result_set_model_refresh (GnomeDbDataModel *iface, GError **error)
{
	GnomeDbResultSet *rs;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), FALSE);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, FALSE);

	/* if there was a refresh pending then consider it done */
	rs->priv->refresh_pending = FALSE;

	/* get rid of old data */
	if (rs->priv->model) {
		g_object_unref (G_OBJECT (rs->priv->model));
		rs->priv->model = NULL;
	}

	if (rs->priv->query_select) {
		gchar *sql;

		/* Actual running query */
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (rs->priv->query_select), 
						       rs->priv->params_set, 0, error);
		if (sql) {
			GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_select));
			
#ifdef debug
			g_print ("RS %p refresh SQL: %s ", rs, sql);
#endif
			rs->priv->model = gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (dict), sql, 
										  GNOME_DB_SERVER_QUERY_SQL, NULL);

			/* Setting columns names */
			if (rs->priv->model) {
				GSList *list, *fields;
				gint i=0;

				fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (rs->priv->query_select));
				list = fields;
				while (list) {
					gda_data_model_set_column_title (rs->priv->model, i,
									 gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
					list = g_slist_next (list);
					i++;
				}
				g_slist_free (fields);
#ifdef debug
				g_print ("=> %d row(s)\n", gda_data_model_get_n_rows (rs->priv->model));
#endif

				g_signal_emit_by_name (iface, "data_refreshed");
			}
#ifdef debug
			else 
				g_print ("=> Error!\n");	
#endif
			g_free (sql);
		}
	}

	return rs->priv->model ? TRUE : FALSE;
}

static GnomeDbDataSet *
gnome_db_result_set_model_get_params (GnomeDbDataModel *iface)
{
	GnomeDbResultSet *rs;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), NULL);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, NULL);

	return rs->priv->params_set;
}

static void
nodes_list_destroy_notify (GSList *list)
{
	GSList *tmp = list;

	while (tmp) {
		g_free (tmp->data);
		tmp = g_slist_next (tmp);
	}
	g_slist_free (list);
}

static void
foreach_hash_new_dataset (gpointer key, gpointer value, GHashTable *to_modif)
{
	gpointer myvalue;

	myvalue = g_hash_table_lookup (to_modif, key);
	if (myvalue) 
		g_hash_table_insert (to_modif, value, myvalue);
}

static GnomeDbDataSet *
gnome_db_result_set_model_get_new_data_set (GnomeDbDataModel *iface)
{
	GnomeDbResultSet *rs;
	GHashTable *repl;
	GnomeDbDataSet *retset;
	GSList *nodes = NULL, *list;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), NULL);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, NULL);

	repl = g_hash_table_new (NULL, NULL);

	/* new data set */
	retset = GNOME_DB_DATA_SET (gnome_db_data_set_new_copy (rs->priv->data_set, repl));

	/* new set of RsNode structures for the new data set */
	list = rs->priv->nodes;
	while (list) {
		RsNode *node;

		node = g_new0 (RsNode, 1);
		node->param = g_hash_table_lookup (repl, RS_NODE (list->data)->param);
		node->position = RS_NODE (list->data)->position;
		nodes = g_slist_append (nodes, node);

		list = g_slist_next (list);
	}

	if (!rs->priv->extra_data_sets_nodes)
		rs->priv->extra_data_sets_nodes = 
			g_hash_table_new_full (NULL, NULL, NULL, (GDestroyNotify) nodes_list_destroy_notify);
	g_hash_table_insert (rs->priv->extra_data_sets_nodes, retset, nodes);

	/* update data_set_data_for_params_source */
	g_hash_table_foreach (repl, (GHFunc) foreach_hash_new_dataset, rs->priv->data_set_data_for_params_source);
	g_hash_table_destroy (repl);

	return retset;
}

static gboolean
gnome_db_result_set_model_sync_data_set (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint row)
{
	GnomeDbResultSet *rs;
	GSList *list;
	const GdaValue *value;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), FALSE);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, FALSE);

	if (data_set && (data_set != rs->priv->data_set)) {
		g_return_val_if_fail (IS_GNOME_DB_DATA_SET (data_set), FALSE);
		list = g_hash_table_lookup (rs->priv->extra_data_sets_nodes, data_set);
		if (!list) {
			g_warning ("data_set was not created with gnome_db_result_set_model_get_new_data_set()");
			return FALSE;
		}
	}
	else
		list = rs->priv->nodes;

	while (list) {
		value = gnome_db_result_set_get_gdavalue (rs->priv->model, row, 
							  RS_NODE (list->data)->position);
		gnome_db_parameter_set_value (RS_NODE (list->data)->param, value);
		list = g_slist_next (list);
	}

	return TRUE;
}


static GnomeDbParameter *
gnome_db_result_set_model_get_param_at_col (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint col)
{
	GnomeDbResultSet *rs;
	GSList *nodes, *list;
	GnomeDbParameter *param = NULL;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), NULL);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, NULL);

	nodes = g_hash_table_lookup (rs->priv->extra_data_sets_nodes, data_set);
	if (!nodes) {
		g_warning ("GnomeDbDataSet was not created using the gnome_db_data_model_get_new_data_set() method");
		return NULL;
	}
	list = nodes;
	while (list && !param) {
		if (RS_NODE (list->data)->position == col)
			param = RS_NODE (list->data)->param;
		list = g_slist_next (list);      
	}

	return param;
}

static gint
gnome_db_result_set_model_get_col_at_param (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, GnomeDbParameter *param)
{
	GnomeDbResultSet *rs;
	GSList *nodes, *list;
	gint col = -1;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), -1);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, -1);

	nodes = g_hash_table_lookup (rs->priv->extra_data_sets_nodes, data_set);
	if (!nodes) {
		g_warning ("GnomeDbDataSet was not created using the gnome_db_data_model_get_new_data_set() method");
		return -1;
	}
	list = nodes;
	while (list && (col == -1)) {
		if (RS_NODE (list->data)->param == param)
			col = RS_NODE (list->data)->position;
		list = g_slist_next (list);      
	}

	return col;
}

static GSList *
gnome_db_result_set_model_get_key_columns (GnomeDbDataModel *iface, GnomeDbDataSet *data_set)
{
	GnomeDbResultSet *rs;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), NULL);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, NULL);

	TO_IMPLEMENT;
	return NULL;
}

static guint
gnome_db_result_set_model_get_status (GnomeDbDataModel *iface)
{
	GnomeDbResultSet *rs;
	guint status = 0;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (iface), 0);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, 0);

	if (rs->priv->query_select) {
		status = status | GNOME_DB_DATA_MODEL_CAN_BE_REFRESHED;
		if (! rs->priv->model)
			status = status | GNOME_DB_DATA_MODEL_NEEDS_INIT_REFRESH;
	}

	return status;
}



/*
 * GdaDataModel interface implementation
 */
static gint
gnome_db_result_set_get_n_rows (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), 0);

	gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) 
		return 0;

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_n_rows)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_n_rows) (gdamodel);
	else
		return 0;
}

static gint
gnome_db_result_set_get_n_columns (GdaDataModel *model)
{
	GnomeDbResultSet *rs;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), -1);
	rs = GNOME_DB_RESULT_SET (model);
	g_return_val_if_fail (rs->priv, -1);

	make_entity_fields (rs);
	return g_slist_length (rs->priv->fields);
}

static GdaDataModelColumnAttributes *
gnome_db_result_set_describe_column (GdaDataModel *model, gint col)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), NULL);
	gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) {
		g_warning ("GnomeDbResultSet contains no data, call gnome_db_data_model_refresh() first");
		return NULL;
	}	

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_describe_column)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_describe_column) (gdamodel, col);
	else
		return NULL;
}

static void
gnome_db_result_set_set_column_title (GdaDataModel *model, gint col, const gchar *title)
{
	GdaDataModel *gdamodel;
	g_return_if_fail (IS_GNOME_DB_RESULT_SET (model));
	gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) {
		g_warning ("GnomeDbResultSet contains no data, call gnome_db_data_model_refresh() first");
		return;
	}

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_column_title)
		(GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_column_title) (gdamodel, col, title);
}

static const gchar *
gnome_db_result_set_get_column_title (GdaDataModel *model, gint col)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), NULL);
	gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) {
		g_warning ("GnomeDbResultSet contains no data, call gnome_db_data_model_refresh() first");
		return NULL;
	}

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_column_title)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_column_title) (gdamodel, col);
	else
		return NULL;
}

static gint
gnome_db_result_set_get_column_pos (GdaDataModel *model, const gchar *title)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), -1);
	gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) {
		g_warning ("GnomeDbResultSet contains no data, call gnome_db_data_model_refresh() first");
		return -1;
	}

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_column_pos)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_column_pos) (gdamodel, title);
	else
		return -1;
}

static const GdaRow *
gnome_db_result_set_get_row (GdaDataModel *model, gint row)
{
	GdaDataModel *gdamodel;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), NULL);
	gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) {
		g_warning ("GnomeDbResultSet contains no data, call gnome_db_data_model_refresh() first");
		return NULL;
	}

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_row) {
		GdaRow *gdarow;

		gdarow = (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_row) (gdamodel, row);
		gda_row_set_number (gdarow, row);
		return gdarow;
	}
	else
		return NULL;
}

static const GdaValue *
gnome_db_result_set_get_value_at (GdaDataModel *model, gint col, gint row)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), NULL);
		gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) {
		g_warning ("GnomeDbResultSet contains no data, call gnome_db_data_model_refresh() first");
		return NULL;
	}

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_value_at)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_value_at) (gdamodel, col, row);
	else
		return NULL;
}

static gboolean
gnome_db_result_set_is_updatable (GdaDataModel *model)
{
	GnomeDbResultSet *rs;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);
	rs = GNOME_DB_RESULT_SET (model);
	g_return_val_if_fail (rs->priv, FALSE);

	return rs->priv->query_update ? TRUE : FALSE;
}

static gboolean
gnome_db_result_set_has_changed (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);

	TO_IMPLEMENT;
	return FALSE;
}

static void
gnome_db_result_set_begin_changes (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_if_fail (IS_GNOME_DB_RESULT_SET (model));

	TO_IMPLEMENT;
}

static gboolean
gnome_db_result_set_commit_changes (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);
	
	TO_IMPLEMENT;
	return FALSE;
}

static gboolean
gnome_db_result_set_cancel_changes (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);

	TO_IMPLEMENT;
	return FALSE;
}

static gboolean
sync_data_set_with_gda_row (GnomeDbResultSet *rs, const GdaRow *row)
{
	GSList *list;
	const GdaValue *value;
	gint length;

	length = gda_row_get_length (row);	
	list = rs->priv->nodes;
	while (list) {
		if (RS_NODE (list->data)->position > length)
			return FALSE;

		value = gda_row_get_value (row, RS_NODE (list->data)->position);
		gnome_db_parameter_set_value (RS_NODE (list->data)->param, value);
		list = g_slist_next (list);
	}
	
	return TRUE;
}

static gboolean
gnome_db_result_set_append_row (GdaDataModel *model, GdaRow *row)
{
	GnomeDbResultSet *rs;
	gboolean retval = FALSE;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);
	rs = GNOME_DB_RESULT_SET (model);
	g_return_val_if_fail (rs->priv, FALSE);
	g_return_val_if_fail (rs->priv->query_insert, FALSE);
	g_return_val_if_fail (rs->priv->model, FALSE);
	g_return_val_if_fail (row, FALSE);
	
	if (sync_data_set_with_gda_row (GNOME_DB_RESULT_SET (model), row)) {
		gchar *sql;
		
		/* Actual running query */
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (rs->priv->query_insert), 
						       rs->priv->data_set, 0, NULL);
		if (sql) {
			GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_update));
			GError *error = NULL;
#ifdef debug
			g_print ("RS %p _INSERT_ SQL: %s\n", rs, sql);
#endif
			gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (dict), sql, 
								GNOME_DB_SERVER_QUERY_SQL, &error);
			g_free (sql);
			if (!error) {
				retval = TRUE;
				if (rs->priv->auto_refresh)
					gnome_db_result_set_model_refresh (GNOME_DB_DATA_MODEL (rs), NULL);
				else
					rs->priv->refresh_pending = TRUE;
			}
			else
				g_error_free (error);
		}
	}

	return retval;
}

static gboolean
gnome_db_result_set_remove_row (GdaDataModel *model, const GdaRow *row)
{
	GnomeDbResultSet *rs;
	gboolean retval = FALSE;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);
	rs = GNOME_DB_RESULT_SET (model);
	g_return_val_if_fail (rs->priv, FALSE);
	g_return_val_if_fail (rs->priv->query_delete, FALSE);
	
	if (sync_data_set_with_gda_row (GNOME_DB_RESULT_SET (model), row)) {
		gchar *sql;
		
		/* Actual running query */
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (rs->priv->query_delete), 
						       rs->priv->data_set, 0, NULL);
		if (sql) {
			GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_update));
			GError *error = NULL;
#ifdef debug
			g_print ("RS %p _DELETE_ SQL: %s\n", rs, sql);
#endif
			gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (dict), sql, 
								GNOME_DB_SERVER_QUERY_SQL, &error);
			g_free (sql);
			if (!error) {
				retval = TRUE;
				if (rs->priv->auto_refresh)
					gnome_db_result_set_model_refresh (GNOME_DB_DATA_MODEL (rs), NULL);
				else
					rs->priv->refresh_pending = TRUE;
			}
			else
				g_error_free (error);
		}
	}

	return retval;
}

static gboolean
gnome_db_result_set_update_row (GdaDataModel *model, const GdaRow *row)
{
	GnomeDbResultSet *rs;
	gboolean retval = FALSE;

	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);
	rs = GNOME_DB_RESULT_SET (model);
	g_return_val_if_fail (rs->priv, FALSE);
	g_return_val_if_fail (rs->priv->query_update, FALSE);
	
	if (sync_data_set_with_gda_row (GNOME_DB_RESULT_SET (model), row)) {
		gchar *sql;
		
		/* Actual running query */
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (rs->priv->query_update), 
						       rs->priv->data_set, 0, NULL);
		if (sql) {
			GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_update));
			GError *error = NULL;
#ifdef debug
			g_print ("RS %p _UPDATE_ SQL: %s\n", rs, sql);
#endif
			gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (dict), sql, 
								GNOME_DB_SERVER_QUERY_SQL, &error);
			g_free (sql);
			if (!error) {
				retval = TRUE;
				if (rs->priv->auto_refresh)
					gnome_db_result_set_model_refresh (GNOME_DB_DATA_MODEL (rs), NULL);
				else
					rs->priv->refresh_pending = TRUE;
			}
			else
				g_error_free (error);
		}
	}

	return retval;
}

static void
gnome_db_result_set_set_notify (GdaDataModel *model, gboolean do_notify_changes)
{
	GnomeDbResultSet *rs;
	g_return_if_fail (IS_GNOME_DB_RESULT_SET (model));
	rs = GNOME_DB_RESULT_SET (model);
	g_return_if_fail (rs->priv);

	if (rs->priv->auto_refresh != do_notify_changes) {
		rs->priv->auto_refresh = do_notify_changes;
		if (rs->priv->auto_refresh && rs->priv->refresh_pending)
			gnome_db_result_set_model_refresh (GNOME_DB_DATA_MODEL (rs), NULL);
	}
}

static gboolean
gnome_db_result_set_get_notify (GdaDataModel *model)
{
	GnomeDbResultSet *rs;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);
	rs = GNOME_DB_RESULT_SET (model);
	g_return_val_if_fail (rs->priv, FALSE);

	return rs->priv->auto_refresh;
}

static gboolean
gnome_db_result_set_set_command (GdaDataModel *model, const gchar *txt, GdaCommandType type)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), FALSE);
	gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) {
		g_warning ("GnomeDbResultSet contains no data, call gnome_db_data_model_refresh() first");
		return FALSE;
	}

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_command)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_command) (gdamodel, txt, type);
	else
		return FALSE;
}

static const gchar *
gnome_db_result_set_get_command (GdaDataModel *model, GdaCommandType *type)

{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_RESULT_SET (model), NULL);
	gdamodel = GNOME_DB_RESULT_SET (model)->priv->model;
	if (!gdamodel) {
		g_warning ("GnomeDbResultSet contains no data, call gnome_db_data_model_refresh() first");
		return NULL;
	}

	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_command)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_command) (gdamodel, type);
	else
		return FALSE;
}



/* 
 * GnomeDbEntity interface implementation
 */
static void
nullified_field_cb (GnomeDbField *field, GnomeDbResultSet *rs)
{
	rs->priv->fields = g_slist_remove (rs->priv->fields, field);
        g_signal_handlers_disconnect_by_func (G_OBJECT (field),
                                              G_CALLBACK (nullified_field_cb), rs);
#ifdef debug_signal
        g_print (">> 'FIELD_REMOVED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (rs), "field_removed", field);
#ifdef debug_signal
        g_print ("<< 'FIELD_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_object_unref (field);
}

static void
make_entity_fields (GnomeDbResultSet *rs)
{
	if (rs->priv->fields)
		return;

	if (rs->priv->query_select) {
		GSList *fields, *list;
		GnomeDbField *field;
		GHashTable *hash = rs->priv->replacements_query_select;

		fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (rs->priv->query_select));
		list = fields;
		while (list) {
			field = GNOME_DB_FIELD (gnome_db_wrapper_field_new_in_entity (GNOME_DB_ENTITY (rs),
										      GNOME_DB_FIELD (list->data)));
			rs->priv->fields = g_slist_prepend (rs->priv->fields, field);
			g_hash_table_insert (hash, g_hash_table_lookup (hash, list->data), field);
			gnome_db_base_connect_nullify (field, G_CALLBACK (nullified_field_cb), rs);
			list = g_slist_next (list);
		}
		g_slist_free (fields);
		rs->priv->fields = g_slist_reverse (rs->priv->fields);
	}
	else {
		TO_IMPLEMENT;
	}

#ifdef debug_NO
	gnome_db_base_dump (GNOME_DB_BASE (rs), 5);
#endif
}

static gboolean
gnome_db_result_set_has_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	GnomeDbResultSet *rs;

	g_return_val_if_fail (iface && IS_GNOME_DB_RESULT_SET (iface), FALSE);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, FALSE);
	
	make_entity_fields (rs);
	return g_slist_find (rs->priv->fields, field) ? TRUE : FALSE;
}

static GSList *
gnome_db_result_set_get_fields (GnomeDbEntity *iface)
{
	GnomeDbResultSet *rs;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_RESULT_SET (iface), NULL);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, NULL);

	make_entity_fields (rs);
	return g_slist_copy (rs->priv->fields);
}

static GnomeDbField *
gnome_db_result_set_get_field_by_name (GnomeDbEntity *iface, const gchar *name)
{
	GnomeDbResultSet *rs;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_RESULT_SET (iface), NULL);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, NULL);

	make_entity_fields (rs);
	TO_IMPLEMENT;
	return NULL;
}

static GnomeDbField *
gnome_db_result_set_get_field_by_index (GnomeDbEntity *iface, gint index)
{
	GnomeDbResultSet *rs;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_RESULT_SET (iface), NULL);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, NULL);

	make_entity_fields (rs);
	return g_slist_nth_data (rs->priv->fields, index);
}

static gint
gnome_db_result_set_get_field_index (GnomeDbEntity *iface, GnomeDbField *field)
{
	GnomeDbResultSet *rs;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_RESULT_SET (iface), -1);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, -1);

	make_entity_fields (rs);
	return g_slist_index (rs->priv->fields, field);
}


static gboolean
gnome_db_result_set_is_writable (GnomeDbEntity *iface)
{
	GnomeDbResultSet *rs;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_RESULT_SET (iface), FALSE);
	rs = GNOME_DB_RESULT_SET (iface);
	g_return_val_if_fail (rs->priv, FALSE);
	
	return rs->priv->query_update ? TRUE : FALSE;
}

static GSList *
gnome_db_result_set_get_parameters (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_RESULT_SET (iface), NULL);
	g_return_val_if_fail (GNOME_DB_RESULT_SET (iface)->priv, NULL);
	
	TO_IMPLEMENT;
	return NULL;
}




/*
 * Complex initialization of the object is below
 */

/* 
 * this structure represents a dependency resulting from a join between 
 * 'this->dependant->target' (FK) and 'this->target' (Ref'ed PK),
 * the GnomeDbJoin for this dependency being 'this->join'
 */
typedef struct _TargetDep{
	GnomeDbTarget     *target;
	GSList            *depend_on; /* list of TargetDep structures */

	struct _TargetDep *dependant;
	GnomeDbJoin       *join;      /* join between 'this->dependant->target' and 'this->target' */ 
} TargetDep;
#define TARGET_DEP(x) ((TargetDep *) x)

GnomeDbDataSet   *make_data_set_when_target    (GnomeDbResultSet *rs);
GnomeDbDataSet   *make_data_set_when_no_target (GnomeDbResultSet *rs);

static TargetDep *make_target_deps        (GnomeDbResultSet *rs);
static void       make_target_deps_free   (TargetDep *dep);
static TargetDep *make_target_deps_recurs (GnomeDbResultSet *rs, GnomeDbTarget *on_target, 
				    const GSList *joins, GHashTable *joinsdep);

static gboolean   make_target_select_queries_improved (GnomeDbResultSet *rs, TargetDep *dep, GHashTable *target_query_h, 
						GHashTable *replacements, GError **error);
static void       make_query_update (GnomeDbResultSet *rs);
static void       make_query_delete (GnomeDbResultSet *rs);
static void       make_query_insert (GnomeDbResultSet *rs);

static void params_set_changed_cb (GnomeDbDataSet *data_set, GnomeDbResultSet *rs);

static void
rs_init_with_query (GnomeDbResultSet *rs)
{
	GnomeDbQuery *top_query = NULL;
	GSList *list;
	GSList *fields;

	/*
	 * Making modifications to the SELECT query to:
	 * -> expand any '*' into all the corresponding fields and order by those fields
	 * -> add PK fields for the entity which will be modified
	 */
	/* expand any '*' into all the corresponding fields */
	fields = gnome_db_query_expand_all_field (rs->priv->query_select, NULL);
	list = fields;
	while (list) {
		gnome_db_query_set_order_by_field (rs->priv->query_select, GNOME_DB_QFIELD (list->data), G_MAXINT, TRUE);
		list = g_slist_next (list);
	}
	g_slist_free (fields);

	/* 
	 * add PK fields for the entity which will be modified (and add ordering on them) 
	 */
	if (rs->priv->modif_target) {
		GSList *pk_fields = gnome_db_query_get_target_pkfields (rs->priv->query_select, rs->priv->modif_target);
		if (! pk_fields) {
			GnomeDbConstraint *pkcons = NULL;
			GnomeDbEntity *entity;

			entity = gnome_db_target_get_represented_entity (rs->priv->modif_target);
			pkcons = gnome_db_table_get_pk_constraint (GNOME_DB_TABLE (entity));
			if (pkcons) {
				GSList *fields, *list;
				GnomeDbQfield *pkf;

				fields = gnome_db_constraint_pkey_get_fields (pkcons);
				list = fields;
				while (list) {
					if (! (pkf =  gnome_db_query_get_field_by_ref_field (rs->priv->query_select, 
										       NULL, GNOME_DB_FIELD (list->data),
										       GNOME_DB_FIELD_ANY))) {
						pkf = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (rs->priv->query_select, 
											       rs->priv->modif_target,
											       GNOME_DB_FIELD (list->data)));
						gnome_db_entity_add_field (GNOME_DB_ENTITY (rs->priv->query_select), GNOME_DB_FIELD (pkf));
						gnome_db_qfield_set_visible (GNOME_DB_QFIELD (pkf), TRUE);
						gnome_db_qfield_set_internal (GNOME_DB_QFIELD (pkf), TRUE);
						gnome_db_base_set_name (GNOME_DB_BASE (pkf), "PK field");
						g_object_unref (G_OBJECT (pkf));
					}
					else {
						if (! gnome_db_qfield_is_visible (GNOME_DB_QFIELD (pkf))) {
							gnome_db_qfield_set_visible (GNOME_DB_QFIELD (pkf), TRUE);
							gnome_db_qfield_set_internal (GNOME_DB_QFIELD (pkf), TRUE);
						}
					}

					if (gnome_db_query_get_order_by_field (rs->priv->query_select, pkf, NULL) < 0) 
						gnome_db_query_set_order_by_field (rs->priv->query_select, pkf, G_MAXINT, TRUE);
					list = g_slist_next (list);
				}
				g_slist_free (fields);
			}
		}
		else {
			GSList *flist = pk_fields;
			while (flist) {
				if (!gnome_db_qfield_is_visible (GNOME_DB_QFIELD (flist->data))) {
					gnome_db_qfield_set_visible (GNOME_DB_QFIELD (flist->data), TRUE);
					gnome_db_qfield_set_internal (GNOME_DB_QFIELD (flist->data), TRUE);
					/* FIXME: move the field to the end? */
				}

#ifdef debug_NO
				DEBUG_HEADER;
				gnome_db_base_dump (rs->priv->query_select, 2);
#endif
				if (gnome_db_query_get_order_by_field (rs->priv->query_select, 
								       GNOME_DB_QFIELD (flist->data), NULL) < 0) 
					gnome_db_query_set_order_by_field (rs->priv->query_select, 
									   GNOME_DB_QFIELD (flist->data), 
									   G_MAXINT, TRUE);

				flist = g_slist_next (flist);
			}
			g_slist_free (pk_fields);
		}
	}
 
	/*
	 * args context : parameters required for the 'select query' to be executed
	 */
	if (!rs->priv->params_set) {
		rs->priv->params_set = gnome_db_entity_get_exec_dataset (GNOME_DB_ENTITY (rs->priv->query_select));
		gnome_db_base_connect_nullify (rs->priv->params_set,
					       G_CALLBACK (nullified_context_cb), rs);
		g_signal_connect (G_OBJECT (rs->priv->params_set), "changed",
				  G_CALLBACK (params_set_changed_cb), rs);
	}

	/*
	 * work context : parameters required to execute the derived INSERT, UPDATE and DELETE queries
	 */
	if (rs->priv->modif_target) {
		rs->priv->data_set = make_data_set_when_target (rs);
		top_query = rs->priv->query_select_hybrid;
	}
	else 
		rs->priv->data_set = make_data_set_when_no_target (rs);
	gnome_db_base_connect_nullify (rs->priv->data_set,
				 G_CALLBACK (nullified_context_cb), rs);
	
	/*
	 * making modification queries
	 */
	if (rs->priv->modif_target) {
		make_query_update (rs);
		make_query_delete (rs);
		make_query_insert (rs);

#ifdef debug_NO
		if (1) {
			/*
			 * Graph output
			 */

			GnomeDbGraphviz *graph;
			graph = GNOME_DB_GRAPHVIZ (gnome_db_graphviz_new (gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_select))));
			if (top_query)
				gnome_db_graphviz_add_to_graph (graph, G_OBJECT (top_query));
			gnome_db_graphviz_add_to_graph (graph, G_OBJECT (rs->priv->query_update));
			gnome_db_graphviz_add_to_graph (graph, G_OBJECT (rs->priv->query_delete));
			gnome_db_graphviz_add_to_graph (graph, G_OBJECT (rs->priv->query_insert));
			gnome_db_graphviz_add_to_graph (graph, G_OBJECT (rs->priv->data_set));
			gnome_db_graphviz_save_file (graph, "_data_set.dot", NULL);
			g_object_unref (G_OBJECT (graph));
		}
#endif
	}

#ifdef debug_NO
	g_print ("---------------------------------- >> ARGS CONTEXT ----------------------------------\n");
	gnome_db_base_dump (GNOME_DB_BASE (rs->priv->params_set), 0);
	g_print ("---------------------------------- << ARGS CONTEXT ----------------------------------\n");
	g_print ("---------------------------------- >> WORK CONTEXT ----------------------------------\n");
	gnome_db_base_dump (GNOME_DB_BASE (rs->priv->data_set), 0);
	g_print ("---------------------------------- << WORK CONTEXT ----------------------------------\n");
	g_print ("---------------------------------- >> QUERY SELECT ----------------------------------\n");
	gnome_db_base_dump (GNOME_DB_BASE (rs->priv->query_select), 0);
	g_print ("---------------------------------- << QUERY SELECT ----------------------------------\n");
	g_print ("---------------------------------- >> QUERY SELECT HYBRID ----------------------------------\n");
	gnome_db_base_dump (GNOME_DB_BASE (rs->priv->query_select_hybrid), 0);
	g_print ("---------------------------------- << QUERY SELECT HYBRID ----------------------------------\n");
#endif	

	/* trying to store some 'proposed' values or parameters's values to be used by the parameters
	 * in data_set. For example is the select query contains "WHERE dict_id=8" then the
	 * data_set's parameter for dict_id should have a 'proposed' value of the data_set's
	 * parameter corresponding to the '8' value */
	list = gnome_db_query_get_main_conditions (rs->priv->query_select);
	if (list && rs->priv->modif_target) {
		GSList *ptr = list;
		GnomeDbCondition *cond;
		
		while (ptr) {
			cond = GNOME_DB_CONDITION (ptr->data);
			if (gnome_db_condition_get_cond_type (cond) == GNOME_DB_CONDITION_LEAF_EQUAL) {
				GnomeDbQfield *field1, *field2;

				field1 = gnome_db_condition_leaf_get_operator (cond, GNOME_DB_CONDITION_OP_LEFT);
				field2 = gnome_db_condition_leaf_get_operator (cond, GNOME_DB_CONDITION_OP_RIGHT);

				if (IS_GNOME_DB_QF_FIELD (field2) && IS_GNOME_DB_QF_VALUE (field1)) {
					GnomeDbQfield *tmp = field1;
					field1 = field2;
					field2 = tmp;
				}

				if (IS_GNOME_DB_QF_FIELD (field1) && IS_GNOME_DB_QF_VALUE (field2)) {
					GnomeDbQfield *rfield1, *rfield2, *value_prov;	

					rfield1 = g_hash_table_lookup (rs->priv->replacements_query_select_hybrid, field1);
					rfield2 = g_hash_table_lookup (rs->priv->replacements_query_select_hybrid, field2);
					g_object_get (G_OBJECT (rfield1), "value_provider", &value_prov, NULL);

					if (value_prov && rfield2) {
						GnomeDbParameter *param1, *param2;

						param1 = gnome_db_data_set_find_parameter_for_field (rs->priv->data_set, 
												     GNOME_DB_FIELD (value_prov));
						param2 = gnome_db_data_set_find_parameter_for_field (rs->priv->data_set, 
												     GNOME_DB_FIELD (rfield2));

						if (param1 && param2) 
							gnome_db_data_set_set_param_default_alias (rs->priv->data_set, 
												   param1, param2);
#ifdef debug_NO
						g_print ("Context DEFAULT ALIAS Param1: %p (%s), Param2: %p (%s)\n", 
							 param1, gnome_db_base_get_name (param1), 
							 param2, gnome_db_base_get_name (param2));
#endif
					}
				}
			}
			ptr = g_slist_next (ptr);
		}
		g_slist_free (list);
	}
	

	/*
	 * get rid of rs->priv->replacements_query_select_hybrid as we won't need it anymore 
	 */
	if (rs->priv->replacements_query_select_hybrid) {
		g_hash_table_destroy (rs->priv->replacements_query_select_hybrid);
		rs->priv->replacements_query_select_hybrid = NULL;
	}
}

/*
 * this function (re)computes the SELECT query and (re)runs it, resulting
 * in a change in the data the resultset holds
 */
static void
params_set_changed_cb (GnomeDbDataSet *data_set, GnomeDbResultSet *rs)
{
	gnome_db_result_set_model_refresh (GNOME_DB_DATA_MODEL (rs), NULL);
}

static void
make_query_update (GnomeDbResultSet *rs)
{
	GSList *list;
	GnomeDbTarget *target;
	GnomeDbQuery *query;
	GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_select));
	GnomeDbCondition *cond = NULL;
	GHashTable *replacements;
	GnomeDbQuery *top_query = rs->priv->query_select_hybrid;

	/* Query object */
	query = GNOME_DB_QUERY (gnome_db_query_new (dict));
	g_object_set (query, "auto_clean", FALSE, NULL);
	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_UPDATE);
	target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, gnome_db_target_get_represented_entity (rs->priv->modif_target)));
	gnome_db_query_add_target (query, target, NULL);
	replacements = g_hash_table_new (NULL, NULL);

	/* Fields */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_FIELD (list->data)) {
			if (gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (list->data)) == 
			    g_object_get_data (G_OBJECT (top_query), "for_target")) {
				GnomeDbField *field, *value_prov, *new_value_prov;
				GnomeDbParameter *param;
				
				field = GNOME_DB_FIELD (gnome_db_qf_field_new_with_objects (query, target,
						  gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data))));
				gnome_db_base_set_name (GNOME_DB_BASE (field), gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), field);
				if (!gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data)))
					gnome_db_qfield_set_visible (GNOME_DB_QFIELD (field), FALSE);

				g_hash_table_insert (replacements, list->data, field);
				g_object_unref (G_OBJECT (field));
				g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);

				if (value_prov) {
					g_assert (IS_GNOME_DB_QF_VALUE (value_prov));

					param = gnome_db_data_set_find_parameter_for_field (rs->priv->data_set, 
											    value_prov);
					new_value_prov = GNOME_DB_FIELD (gnome_db_qf_value_new (query, 
							 gnome_db_qf_value_get_server_data_type (GNOME_DB_QF_VALUE (value_prov))));
					gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (new_value_prov), TRUE);
					gnome_db_base_set_name (GNOME_DB_BASE (new_value_prov), 
							  gnome_db_base_get_name (GNOME_DB_BASE (value_prov)));
					gnome_db_entity_add_field (GNOME_DB_ENTITY (query), new_value_prov);
					gnome_db_qfield_set_visible (GNOME_DB_QFIELD (new_value_prov), FALSE);
					g_object_unref (G_OBJECT (new_value_prov));
					g_object_set (G_OBJECT (field), "value_provider", new_value_prov, NULL);
				
					gnome_db_parameter_add_dest_field (param, GNOME_DB_FIELD (new_value_prov));
				}
			}
		}

		list = g_slist_next (list);
	}

	/* Condition */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_VALUE (list->data)) {
			if (g_slist_find (rs->priv->for_cond_value_qfields, list->data)) {
				GnomeDbQfield *field, *value;
				GnomeDbCondition *subcond;
				GnomeDbParameter *param;

				if (!cond) {
					cond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_AND));
					gnome_db_query_set_condition (query, cond);
					g_object_unref (G_OBJECT (cond));
				}

				field = g_hash_table_lookup (replacements, 
							     g_object_get_data (G_OBJECT (list->data), "for_cond"));
				g_assert (field);
				g_assert (GNOME_DB_QF_FIELD (field));
				
				param = gnome_db_data_set_find_parameter_for_field (rs->priv->data_set, GNOME_DB_FIELD (list->data));
				g_assert (param);
				value = GNOME_DB_QFIELD (gnome_db_qf_value_new (query, 
						   gnome_db_field_get_data_type (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field)))));
				gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (value), TRUE);
				gnome_db_base_set_name (GNOME_DB_BASE (value), gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (value));
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (value), FALSE);
				g_object_unref (G_OBJECT (value));
				gnome_db_parameter_add_dest_field (param, GNOME_DB_FIELD (value));

				subcond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_LEAF_EQUAL));
				gnome_db_condition_leaf_set_operator (subcond, GNOME_DB_CONDITION_OP_LEFT, field);
				gnome_db_condition_leaf_set_operator (subcond, GNOME_DB_CONDITION_OP_RIGHT, value);
				gnome_db_condition_node_add_child (cond, subcond, NULL);
				g_object_unref (G_OBJECT (subcond));
			}
		}
		list = g_slist_next (list);
	}

	
	rs->priv->query_update = query;
	gnome_db_base_connect_nullify (rs->priv->query_update,
				       G_CALLBACK (nullified_query_cb), rs);

#ifdef debug_NO
	{
		gchar *sql;
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, 0, NULL);
		g_print ("Rs UPDATE: %s\n", sql);
		g_free (sql);
	}
#endif

	/* Free memory */
	g_hash_table_destroy (replacements);

}

static void
make_query_delete (GnomeDbResultSet *rs)
{
	GSList *list;
	GnomeDbTarget *target;
	GnomeDbQuery *query;
	GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_select));
	GnomeDbCondition *cond = NULL;
	GnomeDbQuery *top_query = rs->priv->query_select_hybrid;

	/* Query object */
	query = GNOME_DB_QUERY (gnome_db_query_new (dict));
	g_object_set (query, "auto_clean", FALSE, NULL);
	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_DELETE);
	target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, gnome_db_target_get_represented_entity (rs->priv->modif_target)));
	gnome_db_query_add_target (query, target, NULL);
	
	/* Condition */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_VALUE (list->data)) {
			if (g_slist_find (rs->priv->for_cond_value_qfields, list->data)) {
				GnomeDbQfield *fieldv, *fieldf;
				GnomeDbCondition *subcond;
				GnomeDbParameter *param;

				if (!cond) {
					cond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_AND));
					gnome_db_query_set_condition (query, cond);
					g_object_unref (G_OBJECT (cond));
				}

				fieldf = g_object_get_data (G_OBJECT (list->data), "for_cond");
				fieldf = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (query, target,
										 gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (fieldf))));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (fieldf));
				g_object_unref (G_OBJECT (fieldf));
				gnome_db_qfield_set_visible (fieldf, FALSE);

				param = gnome_db_data_set_find_parameter_for_field (rs->priv->data_set, GNOME_DB_FIELD (list->data));
				g_assert (param);
				fieldv = GNOME_DB_QFIELD (gnome_db_qf_value_new (query, 
							   gnome_db_field_get_data_type (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (fieldf)))));
				gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (fieldv), TRUE);
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (fieldv));
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (fieldv), FALSE);
				g_object_unref (G_OBJECT (fieldv));
				gnome_db_parameter_add_dest_field (param, GNOME_DB_FIELD (fieldv));

				subcond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_LEAF_EQUAL));
				gnome_db_condition_leaf_set_operator (subcond, GNOME_DB_CONDITION_OP_LEFT, fieldf);
				gnome_db_condition_leaf_set_operator (subcond, GNOME_DB_CONDITION_OP_RIGHT, fieldv);
				gnome_db_condition_node_add_child (cond, subcond, NULL);
				g_object_unref (G_OBJECT (subcond));
			}
		}
		list = g_slist_next (list);
	}

#ifdef debug_NO
	{
		gchar *sql;
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, 0, NULL);
		g_print ("Rs DELETE: %s\n", sql);
		g_free (sql);
	}
#endif
	
	rs->priv->query_delete = query;
	gnome_db_base_connect_nullify (rs->priv->query_delete,
				       G_CALLBACK (nullified_query_cb), rs);

}

static void
make_query_insert (GnomeDbResultSet *rs)
{
	GSList *list;
	GnomeDbTarget *target;
	GnomeDbQuery *query;
	GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_select));
	GnomeDbQuery *top_query = rs->priv->query_select_hybrid;

	/* Query object */
	query = GNOME_DB_QUERY (gnome_db_query_new (dict));
	g_object_set (query, "auto_clean", FALSE, NULL);
	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_INSERT);
	target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, gnome_db_target_get_represented_entity (rs->priv->modif_target)));
	gnome_db_query_add_target (query, target, NULL);

	/* Fields */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_FIELD (list->data)) {
			if (gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (list->data)) == 
			    g_object_get_data (G_OBJECT (top_query), "for_target")) {
				GnomeDbField *field, *value_prov;
				GnomeDbParameter *param;
				
				field = GNOME_DB_FIELD (gnome_db_qf_field_new_with_objects (query, target,
						  gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data))));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), field);
				if (!gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data)))
					gnome_db_qfield_set_visible (GNOME_DB_QFIELD (field), FALSE);

				g_object_unref (G_OBJECT (field));
				g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
				if (value_prov) {
					g_assert (IS_GNOME_DB_QF_VALUE (value_prov));

					param = gnome_db_data_set_find_parameter_for_field (rs->priv->data_set, value_prov);
					g_assert (param);
					value_prov = GNOME_DB_FIELD (gnome_db_qf_value_new (query, 
							       gnome_db_qf_value_get_server_data_type (GNOME_DB_QF_VALUE (value_prov))));
					gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (value_prov), TRUE);
					gnome_db_entity_add_field (GNOME_DB_ENTITY (query), value_prov);
					gnome_db_qfield_set_visible (GNOME_DB_QFIELD (value_prov), FALSE);
					g_object_unref (G_OBJECT (value_prov));
					g_object_set (G_OBJECT (field), "value_provider", value_prov, NULL);
				
					gnome_db_parameter_add_dest_field (param, GNOME_DB_FIELD (value_prov));
				}
			}
		}

		list = g_slist_next (list);
	}

#ifdef debug_NO
	{
		gchar *sql;
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, 0, NULL);
		g_print ("Rs INSERT: %s\n", sql);
		g_free (sql);
	}
#endif

	rs->priv->query_insert = query;
	gnome_db_base_connect_nullify (rs->priv->query_insert,
				       G_CALLBACK (nullified_query_cb), rs);

}

/*
 * Computing the GnomeDbDataSet and the RsNode
 * when there is no target for the modifications
 */
GnomeDbDataSet *
make_data_set_when_no_target (GnomeDbResultSet *rs)
{
	GSList *fields, *list;
	GSList *params = NULL;
	GnomeDbDataSet *data_set;
	GnomeDbDict *dict;

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_select));
        fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (rs->priv->query_select));
        list = fields;
        while (list) {
                GnomeDbQfield *field = GNOME_DB_QFIELD (list->data);
		
		if (! IS_GNOME_DB_QF_ALL (field)) {
			RsNode *node = g_new0 (RsNode, 1);

			/* node->param = GNOME_DB_PARAMETER (gnome_db_parameter_new_with_dest_field (GNOME_DB_FIELD (field),  */
/* 							  gnome_db_field_get_data_type (GNOME_DB_FIELD (field)))); */
			node->param = GNOME_DB_PARAMETER (gnome_db_parameter_new (dict,
							  gnome_db_field_get_data_type (GNOME_DB_FIELD (field))));
			node->position = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (rs->priv->query_select), 
									  GNOME_DB_FIELD (field));
			rs->priv->nodes = g_slist_append (rs->priv->nodes, node);
			params = g_slist_append (params, node->param);
			
			/* set the parameter properties */
			gnome_db_base_set_name (GNOME_DB_BASE (node->param), gnome_db_base_get_name (GNOME_DB_BASE (field)));
			gnome_db_base_set_description (GNOME_DB_BASE (node->param), 
						       gnome_db_base_get_description (GNOME_DB_BASE (field)));
			if (G_OBJECT_TYPE (field) == GNOME_DB_QF_FIELD_TYPE) {
				GnomeDbField *realfield = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field));
				if (G_OBJECT_TYPE (realfield) == GNOME_DB_TABLE_FIELD_TYPE)
					gnome_db_parameter_set_not_null (node->param,
						 !gnome_db_table_field_is_null_allowed (GNOME_DB_TABLE_FIELD (realfield)));
			}
		}

                list = g_slist_next (list);
        }

	data_set = GNOME_DB_DATA_SET (gnome_db_data_set_new (dict, params));

	/* get rid of the params list since we don't use them anymore */
	list = params;
        while (list) { 
                g_object_unref (G_OBJECT (list->data));
                list = g_slist_next (list);
        }
        g_slist_free (params);

	return data_set;
}

static void
foreach_hash_make_when_target (gpointer key, gpointer value, GHashTable *to_modif)
{
	gpointer myvalue;

	myvalue = g_hash_table_lookup (to_modif, key);
	if (myvalue) 
		g_hash_table_insert (to_modif, value, myvalue);
}

/*
 * Computing the GnomeDbDataSet and the RsNode s
 * when rs->priv->modif_target is NOT NULL
 */
GnomeDbDataSet *
make_data_set_when_target (GnomeDbResultSet *rs)
{
	GHashTable *targets_h; /* (key, value) = (target, SELECT query for that target) */
	TargetDep *targets_d;
	GSList *fields, *list, *tmplist;
	GSList *sel_queries = NULL;
	GnomeDbQuery *top_query;
	GnomeDbDataSet *data_set;
	GHashTable *replacements; /* objects replacements in the copy of 'rs->priv->query_select' */
	GSList *cond_fields;
	GSList *main_conds;
	GnomeDbDict *dict;

	g_return_val_if_fail (rs->priv->query_select, NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (rs->priv->query_select));

	/*
	 * Create a tree of all the target dependencies
	 */
	targets_d = make_target_deps (rs);
	targets_h = g_hash_table_new (NULL, NULL);

	/*
	 * make a copy of 'rs->priv->query_select' into 'top_query'
	 */
	replacements = g_hash_table_new (NULL, NULL);
	top_query = GNOME_DB_QUERY (gnome_db_query_new_copy (rs->priv->query_select, replacements));
	g_object_set (top_query, "auto_clean", FALSE, NULL);
	g_object_set_data (G_OBJECT (top_query), "for_target", g_hash_table_lookup (replacements, rs->priv->modif_target));
	sel_queries = g_slist_append (sel_queries, top_query);
	g_hash_table_insert (targets_h, rs->priv->modif_target, top_query);


	/*
	 * POSSIBLE IMPROVEMENT:
	 * Parse top_query's joins which have one target corresponding to the modif target,
	 * and if one of the fields used by the join condition is not visible, then move it
	 * close to the one which is visible.
	 *
	 * The problem is that the parameter for "fk_field" of such a join is set close to the "fk_field" and usually
	 * we want it at the same place as what gets displayed in the SELECT (which is the "ref_pk_field").
	 */


	/*
	 * For each GnomeDbQfValue in 'top_query', we need to remove the "value_provider" property if it exists because
	 * they would give parameters which would only be aliases for parameters we have in the 'rs->priv->params_set'
	 */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_VALUE (list->data))
			g_object_set (G_OBJECT (list->data), "value_provider", NULL, NULL);
		list = g_slist_next (list);
	}
	
	/*
	 * make a list ('sel_queries') of SELECT queries: one for each target in the 'rs->priv->query_select'
	 * query. The queries are then populated with fields which are the copies of the fields related to each
	 * target: only the GnomeDbQfield of type GNOME_DB_QF_FIELD are inserted into the new queries.
	 *
	 * For the 'rs->priv->modif_target', we don't start from scratch, but from the 'top_query' query created just above
	 */
	fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (rs->priv->query_select));
	list = fields;
	while (list) {
		GnomeDbTarget *target;
		GnomeDbQuery *query;
		GnomeDbField *realfield;
		GnomeDbTarget *newtarget;

		if (! IS_GNOME_DB_QF_FIELD (list->data))
			continue;

		target = gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (list->data));
		if (target != rs->priv->modif_target) {
			query = g_hash_table_lookup (targets_h, target);
			if (query) {
				/* add a field */
				realfield = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data));
				newtarget = g_object_get_data (G_OBJECT (query), "for_target");
				realfield = GNOME_DB_FIELD (gnome_db_qf_field_new_with_objects (query, newtarget, realfield));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), realfield);
				g_object_unref (G_OBJECT (realfield));
				
				/* set it to appear in the ORDER BY list */
				gnome_db_query_set_order_by_field (query, GNOME_DB_QFIELD (realfield), G_MAXINT, TRUE);
			}
			else {
				/* create a new query and insert in into targets_h, and add a field */
				query = GNOME_DB_QUERY (gnome_db_query_new (dict));
				g_object_set (query, "auto_clean", FALSE, NULL);
				gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_SELECT);
				
				gnome_db_base_set_name (GNOME_DB_BASE (query), gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
				gnome_db_base_set_description (GNOME_DB_BASE (query), 
							       gnome_db_base_get_description (GNOME_DB_BASE (list->data)));
				
				realfield = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data));
				newtarget = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, 
											      gnome_db_field_get_entity (realfield)));
				g_assert (gnome_db_query_add_target (query, newtarget, NULL));
				g_object_set_data (G_OBJECT (query), "for_target", newtarget);
				
				realfield = GNOME_DB_FIELD (gnome_db_qf_field_new_with_objects (query, newtarget, realfield));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), realfield);
				g_object_unref (G_OBJECT (newtarget));
				g_object_unref (G_OBJECT (realfield));
				
				g_hash_table_insert (targets_h, target, query);
				sel_queries = g_slist_append (sel_queries, query);
			}
			gnome_db_base_set_name (GNOME_DB_BASE (realfield), gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
			gnome_db_base_set_description (GNOME_DB_BASE (realfield), 
						       gnome_db_base_get_description (GNOME_DB_BASE (list->data)));

			/* store for later use */
			g_hash_table_insert (rs->priv->data_set_data_for_params_source, realfield, list->data);

		}
		else {
			/* Add a GnomeDbQfValue for that field */
			GnomeDbQfValue *vfield;
			const gchar *plugin;
			GnomeDbDataHandler *default_dh = NULL;
			
			query = top_query;
			realfield = g_hash_table_lookup (replacements, list->data);
			
			vfield = GNOME_DB_QF_VALUE (gnome_db_qf_value_new (query, gnome_db_field_get_data_type (realfield)));
			gnome_db_entity_add_field_before (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (vfield), GNOME_DB_FIELD (realfield));
			gnome_db_base_set_name (GNOME_DB_BASE (vfield), gnome_db_base_get_name (GNOME_DB_BASE (realfield)));
			gnome_db_base_set_description (GNOME_DB_BASE (vfield), 
						       gnome_db_base_get_description (GNOME_DB_BASE (realfield)));
			g_object_unref (G_OBJECT (vfield));
			gnome_db_qf_value_set_is_parameter (vfield, TRUE);
			gnome_db_qfield_set_visible (GNOME_DB_QFIELD (vfield), FALSE);
			g_object_set (G_OBJECT (realfield), "value_provider", vfield, NULL);
			gnome_db_qfield_set_internal (GNOME_DB_QFIELD (vfield), TRUE);
			
			/* GnomeDbQfValue properties */
			if (IS_GNOME_DB_QF_FIELD (realfield) && 
			    IS_GNOME_DB_TABLE_FIELD (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (realfield)))) {
				GnomeDbTableField *dbfield;
				GnomeDbServer *srv;

				dbfield = GNOME_DB_TABLE_FIELD (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (realfield)));
				gnome_db_qf_value_set_not_null (vfield, !gnome_db_table_field_is_null_allowed (dbfield));
				if (gnome_db_table_field_get_default_value (dbfield))
					gnome_db_qf_value_set_default_value (vfield, 
									     gnome_db_table_field_get_default_value (dbfield));
				
				srv = gnome_db_dict_get_server (dict);
				default_dh = gnome_db_server_get_object_handler (srv, G_OBJECT (dbfield));
			}
			else {
				TO_IMPLEMENT;
			}
			
			/* Entry plugin if available */
			g_object_get (G_OBJECT (list->data), "handler_plugin", &plugin, NULL);
			if (plugin) 
				g_object_set (G_OBJECT (vfield), "handler_plugin", plugin, NULL);
			else {
				if (default_dh) {
					plugin = gnome_db_base_get_name (GNOME_DB_BASE (default_dh));
					g_object_set (G_OBJECT (vfield), "handler_plugin", plugin, NULL);
				}
			}
			
			/* keep a reference to the field in 'rs->priv->query_select' which we are now treating */
			g_object_set_data (G_OBJECT (vfield), "position", list->data);
		}

		list = g_slist_next (list);
	}

	g_slist_free (fields);

	/* For each target dependency, from the corresponding join, add the missing fields participating in the
	 * join to the SELECT queries associated with each target;
	 *
	 * And create some GnomeDbParameter objects on the 'FK' side and make them have a value provider on the 'PK' side.
	 */
	make_target_select_queries_improved (rs, targets_d, targets_h, replacements, NULL);

	/* 
	 * Analyse all the main conditions (= the conditions from the query_select whih are always evaluated)
	 * and copy some of them to the other SELECT queries created in the previous step. The idea is to
	 * copy some of the restrictions that are in the original query into the other SELECT queries from which
	 * some values must be chosen.
	 *
	 * Example: if we have the main query as 'SELECT a.name, b.name FROM a INNER JOIN b WHERE b.id=12'
	 * then we want the a.fkfield to be selected from 'SELECT b.pkfield, b.name FROM b WHERE b.id=12' and
	 * not from 'SELECT b.pkfield, b.name FROM'.
	 *
	 * FIXME: this implementation is not correct, better do it _after_ the data_set has been created
	 * so we don't add some useless parameters in the select queries.
	 *
	 * At the moment only conditions whith non parameter values are accepted.
	 */
	main_conds = gnome_db_query_get_main_conditions (rs->priv->query_select);
	list = main_conds;
	while (list) {
		gboolean fields_ok = TRUE;
		GnomeDbTarget *sel_target = NULL;

		/* see if this condition is OK to be processed further */
		cond_fields = gnome_db_condition_get_ref_objects_all (GNOME_DB_CONDITION (list->data));
		fields = cond_fields;
		while (fields && fields_ok) {
			if (!IS_GNOME_DB_QF_FIELD (fields->data) &&
			    !IS_GNOME_DB_QF_VALUE (fields->data))
				fields_ok = FALSE;

			if (IS_GNOME_DB_QF_FIELD (fields->data)) {
				if (!sel_target) 
					sel_target = gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (fields->data));
				else {
					if (sel_target != gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (fields->data)))
						fields_ok = FALSE;
				}
			}

			if (IS_GNOME_DB_QF_VALUE (fields->data) && gnome_db_qf_value_is_parameter (GNOME_DB_QF_VALUE (fields->data)))
				fields_ok = FALSE;

			fields = g_slist_next (fields);
		}
		
		/* process that condition if fields_ok && sel_target */
		if (fields_ok && sel_target && (sel_target != rs->priv->modif_target)) {
			GHashTable *tmprepl;
			GnomeDbQuery *sel_query = g_hash_table_lookup (targets_h, sel_target);
			GnomeDbCondition *cond = GNOME_DB_CONDITION (list->data), *newcond, *andcond;
			GnomeDbQfield *cfield;
			
			/* initialize a temporary hash table to be used by for replacements in condition */
			tmprepl = g_hash_table_new (NULL, NULL);
			g_hash_table_insert (tmprepl, rs->priv->query_select, sel_query);

			/* copy the fields taking part of the condition if they don't already exist */
			fields = cond_fields;
			while (fields) {
				gboolean field_done = FALSE;

				if (IS_GNOME_DB_QF_FIELD (fields->data)) {
					GnomeDbField *ref_field = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (fields->data));
					GnomeDbField *realfield = 
						(GnomeDbField*) gnome_db_query_get_field_by_ref_field (sel_query, 
												       NULL, ref_field, 
												       GNOME_DB_FIELD_ANY);

					if (!realfield) {
						gchar *str;
						GnomeDbTarget *newtarget = g_object_get_data (G_OBJECT (sel_query), 
											      "for_target");
						cfield = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (sel_query, 
													      newtarget, 
													      ref_field));
						gnome_db_entity_add_field (GNOME_DB_ENTITY (sel_query), GNOME_DB_FIELD (cfield));
						str = g_strdup_printf ("_%s", gnome_db_base_get_name (GNOME_DB_BASE (fields->data)));
						gnome_db_base_set_name (GNOME_DB_BASE (cfield), str);
						gnome_db_qfield_set_visible (cfield, FALSE);
						g_free (str);
						g_hash_table_insert (tmprepl, fields->data, cfield);
						g_object_unref (G_OBJECT (cfield));
					}
					else
						g_hash_table_insert (tmprepl, fields->data, realfield);
					field_done = TRUE;
				}
				
				if (!field_done && IS_GNOME_DB_QF_VALUE (fields->data)) {
					/* make a copy of that field since GnomeDbQfValue fields are not copied above */
					cfield = GNOME_DB_QFIELD (gnome_db_qfield_new_copy (GNOME_DB_QFIELD (fields->data)));
					g_object_set (G_OBJECT (cfield), "query", sel_query, NULL);
					gnome_db_entity_add_field (GNOME_DB_ENTITY (sel_query), GNOME_DB_FIELD (cfield));
					gnome_db_base_set_name (GNOME_DB_BASE (cfield), "_main cond copy");
					gnome_db_qfield_set_visible (cfield, FALSE);
					g_hash_table_insert (tmprepl, fields->data, cfield);

					gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (cfield), 
								      gnome_db_qf_value_is_parameter (GNOME_DB_QF_VALUE (fields->data)));
					g_object_unref (G_OBJECT (cfield));
					field_done = TRUE;
				}
				g_assert (field_done);

				fields = g_slist_next (fields);
			}

			/* copy of the condition */
			andcond = gnome_db_query_get_condition (sel_query);
			if (!andcond) {
				andcond = GNOME_DB_CONDITION (gnome_db_condition_new (sel_query, GNOME_DB_CONDITION_NODE_AND));
				gnome_db_query_set_condition (sel_query, andcond);
				gnome_db_base_set_name (GNOME_DB_BASE (andcond), "_: AND");
				g_object_unref (G_OBJECT (andcond));
			}
			else {
				if (gnome_db_condition_is_leaf (andcond)) {
					g_object_ref (G_OBJECT (andcond));
					newcond = GNOME_DB_CONDITION (gnome_db_condition_new (sel_query, 
											      GNOME_DB_CONDITION_NODE_AND));
					gnome_db_base_set_name (GNOME_DB_BASE (newcond), "_: AND");
					gnome_db_query_set_condition (sel_query, newcond);
					g_assert (gnome_db_condition_node_add_child (newcond, andcond, NULL));
					g_object_unref (G_OBJECT (andcond));
					g_object_unref (G_OBJECT (newcond));
					andcond = newcond;
				}
			}
			
			newcond = GNOME_DB_CONDITION (gnome_db_condition_new_copy (cond, tmprepl));
			gnome_db_referer_replace_refs (GNOME_DB_REFERER (newcond), tmprepl);
			g_hash_table_destroy (tmprepl);
			g_assert (gnome_db_condition_node_add_child (andcond, newcond, NULL));
			g_object_unref (G_OBJECT (newcond));
		}

		g_slist_free (cond_fields);

		list = g_slist_next (list);
	}
	g_slist_free (main_conds);


	/* 
	 * Preparing the WHERE condition values for modification queries (UPDATE and DELETE) 
	 */
	cond_fields = gnome_db_query_get_target_pkfields (rs->priv->query_select, rs->priv->modif_target);
	if (!cond_fields) {
		/* we don't have a primary key in the list of fields, so we just add all the fields
		 * and hope we don't modify too many tuples in the database.
		 * (there is no JOIN in the UPDATE and DELETE, so we only have fields visible
		 * from the original SELECT query to make criteria of).
		 */
		GSList *tmplist = gnome_db_entity_get_fields (GNOME_DB_ENTITY (rs->priv->query_select));
		list = tmplist;
		while (list) {
			if (IS_GNOME_DB_QF_FIELD (list->data) && 
			    (gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (list->data)) == rs->priv->modif_target)) {
				cond_fields = g_slist_append (cond_fields, list->data);
			}
			list = g_slist_next (list);
		}
		g_slist_free (tmplist);
	}
	list = cond_fields;
	while (list) {
		GnomeDbQfValue *vfield;
		
		vfield = GNOME_DB_QF_VALUE (gnome_db_qf_value_new (top_query, 
								   gnome_db_field_get_data_type (GNOME_DB_FIELD (list->data))));
		gnome_db_base_set_name (GNOME_DB_BASE (vfield), "_Condition Value");
		gnome_db_entity_add_field (GNOME_DB_ENTITY (top_query), GNOME_DB_FIELD (vfield));
		g_object_unref (G_OBJECT (vfield));
		gnome_db_qf_value_set_is_parameter (vfield, TRUE);
		gnome_db_qfield_set_visible (GNOME_DB_QFIELD (vfield), FALSE);
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (vfield), TRUE);
		
		/* GnomeDbQfValue Properties */
		if (IS_GNOME_DB_QF_FIELD (list->data) && 
		    IS_GNOME_DB_TABLE_FIELD (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data)))) {
			GnomeDbTableField *dbfield;

			dbfield = GNOME_DB_TABLE_FIELD (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data)));
			gnome_db_qf_value_set_not_null (vfield, !gnome_db_table_field_is_null_allowed (dbfield));
			if (gnome_db_table_field_get_default_value (dbfield))
				gnome_db_qf_value_set_default_value (vfield, gnome_db_table_field_get_default_value (dbfield));
		}
		else {
			TO_IMPLEMENT;
		}

		/* keep a reference to the field in 'rs->priv->query_select' which we are now treating */
		g_object_set_data (G_OBJECT (vfield), "position", list->data);
		g_object_set_data (G_OBJECT (vfield), "for_cond", g_hash_table_lookup (replacements, list->data));
		rs->priv->for_cond_value_qfields = g_slist_append (rs->priv->for_cond_value_qfields, vfield);
		
		list = g_slist_next (list);
	}
	g_slist_free (cond_fields);

	/*
	 * create data_set
	 */
	data_set = gnome_db_entity_get_exec_dataset (GNOME_DB_ENTITY (top_query));

	/*
	 * for each parameter in data_set
	 * treat it if it's an alias for a parameter in 'rs->priv->params_set'.
	 */
	if (rs->priv->params_set)
		list = rs->priv->params_set->parameters;
	else
		list = NULL;
	while (list) {
		GSList *ff = gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (list->data));
		GnomeDbQfield *alias_qf = NULL;

		while (ff && !alias_qf) {
			alias_qf = g_hash_table_lookup (replacements, ff->data);
			ff = g_slist_next (ff);
		}

		if (alias_qf) { /* find the param. in 'data_set' which is for 'alias_qf' */
			GnomeDbParameter *alias = gnome_db_data_set_find_parameter_for_field (data_set, 
											      GNOME_DB_FIELD (alias_qf));
			
			if (alias) 
				g_object_set (G_OBJECT (alias), "full_bind", list->data, NULL);
		}
			
		list = g_slist_next (list);
	}
	rs->priv->replacements_query_select_hybrid = replacements;

	/*
	 * translate keys of rs->priv->data_set_data_for_params_source to the data_for_param in each
	 * GnomeDbResultSetNode.
	 */
	if (data_set)
		list = data_set->nodes;
	else
		list = NULL;
	while (list) {
		if (GNOME_DB_DATA_SET_NODE (list->data)->data_for_param) {
			GHashTable *repl;

			g_object_get (G_OBJECT (GNOME_DB_DATA_SET_NODE (list->data)->data_for_param),
				      "init_query_replacements", &repl, NULL);
			if (repl) 
				/* repl is a GHashTable which translates from the GnomeDbQuery from which the
				 * node->data_for_param GnomeDbDataModel was created to GnomeDbField fields of the object
				 * which implements that GnomeDbDataModel */
				g_hash_table_foreach (repl, (GHFunc) foreach_hash_make_when_target, 
						      rs->priv->data_set_data_for_params_source);
		}
		list = g_slist_next (list);
	}
	

	/*
	 * for each parameter in data_set, set the hints
	 */
	/* condition parameters */
	list = rs->priv->for_cond_value_qfields;
	while (list) {
		GnomeDbParameter *tmp = gnome_db_data_set_find_parameter_for_field (data_set, GNOME_DB_FIELD (list->data));
		g_hash_table_insert (data_set->hints, tmp, 
				     GUINT_TO_POINTER (GNOME_DB_DATA_SET_PARAM_READ_ONLY | GNOME_DB_DATA_SET_PARAM_HIDE));
		list = g_slist_next (list);
	}

	/* alias parameters */
	list = data_set->parameters;
	while (list) {
		GnomeDbParameter *tmp;
		
		g_object_get (G_OBJECT (list->data), "full_bind", &tmp, NULL);
		if (tmp) 
			g_hash_table_insert (data_set->hints, list->data,
					     GUINT_TO_POINTER (GNOME_DB_DATA_SET_PARAM_READ_ONLY | 
							       GNOME_DB_DATA_SET_PARAM_HIDE));
		list = g_slist_next (list);
	}

	/* params for internal fields */
	tmplist = gnome_db_entity_get_fields (GNOME_DB_ENTITY (top_query));
	list = tmplist;
	while (list) {
		if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (list->data)) &&
		    !g_slist_find (rs->priv->ref_integ_fields, list->data)) {
			GnomeDbParameter *param;
			GnomeDbQfield *value_prov;
			
			g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
			if (value_prov) {
				param = gnome_db_data_set_find_parameter_for_field (data_set, 
										    GNOME_DB_FIELD (value_prov));
				if (param) 
					g_hash_table_insert (data_set->hints, param,
							     GUINT_TO_POINTER (GNOME_DB_DATA_SET_PARAM_READ_ONLY | 
									       GNOME_DB_DATA_SET_PARAM_HIDE));
			}
		}
		list = g_slist_next (list);
	}
	g_slist_free (tmplist);
	

	/*
	 * create RsNode nodes 
	 */
	list = data_set->parameters;
	while (list) {
		gint pos = -1;
		GSList *dest = gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (list->data));
		GnomeDbField *field;

		while (dest && (pos==-1)) {
			field = g_object_get_data (G_OBJECT (dest->data), "position");
			if (field)
				pos = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (rs->priv->query_select), field);
			dest = g_slist_next (dest);
		}

		if (pos > -1) {
			RsNode *node = g_new0 (RsNode, 1);
			node->param = GNOME_DB_PARAMETER (list->data);
			node->position = pos;
			rs->priv->nodes = g_slist_append (rs->priv->nodes, node);
		}
#ifdef debug_NO
		else {
			g_print ("Don't create a code node for param %p\n", list->data);
			dest = gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (list->data));
			g_print ("For %d field(s):\n", g_slist_length (dest));
			while (dest) {
				gnome_db_base_dump (dest->data, 10);
				dest = g_slist_next (dest);
			}
		}
#endif
		
		list = g_slist_next (list);
	}

	/*
	 * Add parameters to data_set to make sure all the columns of the GnomeDbResultSet are matched
	 * with a parameter in data_set and all the parameters in data_set correspond to one column
	 */
	{
		gint *positions;
		gint nb_cols, i;
		
		nb_cols = gnome_db_entity_get_n_fields (GDA_DATA_MODEL (rs));
		positions = g_new0 (gint, nb_cols);

		list = rs->priv->nodes;
		while (list) {
			g_assert (RS_NODE (list->data)->position < nb_cols);
			positions [RS_NODE (list->data)->position] = 1;
			list = g_slist_next (list);
		}
		
		/* add missing RsNode structures, and integrate each new parameter into data_set */
		for (i = 0; i < nb_cols; i++) {
			if (positions [i] == 0) {
				RsNode *node = g_new0 (RsNode, 1);
				GnomeDbField *field;

				field = gnome_db_entity_get_field_by_index (GNOME_DB_ENTITY (rs), i);
				node->param = GNOME_DB_PARAMETER (gnome_db_parameter_new (dict, gnome_db_field_get_data_type (field)));
				g_hash_table_insert (data_set->hints, node->param, 
						     GUINT_TO_POINTER (GNOME_DB_DATA_SET_PARAM_READ_ONLY |
								       GNOME_DB_DATA_SET_PARAM_HIDE));
				/* set the parameter properties */
				gnome_db_base_set_name (GNOME_DB_BASE (node->param), gnome_db_base_get_name (GNOME_DB_BASE (field)));
				gnome_db_base_set_description (GNOME_DB_BASE (node->param), 
							       gnome_db_base_get_description (GNOME_DB_BASE (field)));
				
				/* add the param to data_set */
				gnome_db_data_set_add_param (data_set, node->param);
				g_object_unref (node->param);
				node->position = i;
				rs->priv->nodes = g_slist_append (rs->priv->nodes, node);
			}
		}

		g_free (positions);
	}
	

	/*
	 * Free allocated memory
	 */
#ifdef debug_NO
	list = sel_queries;
	while (list) {
		gchar *gname = g_strdup_printf ("_sel_query%d.dot", g_slist_position (sel_queries, list));
		GnomeDbGraphviz *graph;

		graph = GNOME_DB_GRAPHVIZ (gnome_db_graphviz_new (dict));
		gnome_db_graphviz_add_to_graph (graph, G_OBJECT (top_query));
		gnome_db_graphviz_save_file (graph, gname, NULL);
		g_object_unref (G_OBJECT (graph));
		g_print ("Written file %s\n", gname);
		g_free (gname);

		g_print ("------------------------------------------- SEL QUERY ---------------------\n");
		gnome_db_base_dump (list->data, 10);

		list = g_slist_next (list);
	}
#endif
	list = sel_queries;
	while (list) {
		if (list->data != top_query)
			g_object_unref (G_OBJECT (list->data));
		else {
			rs->priv->query_select_hybrid = top_query;
			gnome_db_base_connect_nullify (rs->priv->query_select_hybrid,
						       G_CALLBACK (nullified_query_cb), rs);
		}
		list = g_slist_next (list);
	}
	g_slist_free (sel_queries);
	make_target_deps_free (targets_d);
	g_hash_table_destroy (targets_h);

	return data_set;
}

static void            improve_queries_with_fk_constraint (GnomeDbResultSet *rs, TargetDep *dep, 
							   GnomeDbQuery *query_fk, GnomeDbQuery *query_ref_pk, 
							   GnomeDbConstraint *fkcons,
							   GHashTable *replacements);
static void            improve_queries_with_join_condition (GnomeDbResultSet *rs, TargetDep *dep, 
							    GnomeDbQuery *query_fk, GnomeDbQuery *query_ref_pk, 
							    GnomeDbTarget *fk_target, GnomeDbTarget *ref_pk_target,
							    GnomeDbCondition *cond, GHashTable *replacements);


static gboolean
make_target_select_queries_improved (GnomeDbResultSet *rs, TargetDep *dep, GHashTable *target_query_h,
				     GHashTable *replacements, GError **error)
{
	GnomeDbQuery *sel_query1, *sel_query2;
	GnomeDbJoin *join = dep->join;
	GnomeDbTarget *target1, *target2;
	GnomeDbEntity *ent1, *ent2;
	GnomeDbCondition *cond;
	GSList *list;

	/* recursive behaviour */
	list = dep->depend_on;
	while (list) {
		if (!make_target_select_queries_improved (rs, TARGET_DEP (list->data), target_query_h, 
							  replacements, error)) {
			TO_IMPLEMENT; /* set error msg */
			return FALSE;
		}
		list = g_slist_next (list);
	}

	/* this TargetDep */
	if (!join)
		return TRUE;

	cond = gnome_db_join_get_condition (join);

	target1 = dep->dependant->target;
	target2 = dep->target;
	ent1 = gnome_db_target_get_represented_entity (target1);
	ent2 = gnome_db_target_get_represented_entity (target2);

	sel_query1 = g_hash_table_lookup (target_query_h, target1); /* query with FK */
	sel_query2 = g_hash_table_lookup (target_query_h, target2); /* query with ref'ed PK */
	if (!sel_query1 || !sel_query2)
		return TRUE;

	if (cond) {
		GnomeDbTarget *t1, *t2;
		gboolean is_equi_join;
		if (gnome_db_condition_represents_join (cond, &t1, &t2, &is_equi_join)) {
			if (is_equi_join &&
			    (((t1 == target1) && (t2 == target2)) ||
			     ((t1 == target2) && (t2 == target1)))) {
				/* here we are sure that we have a condition which is in the form 
				 * t1.a=t2.b AND t1.c=t2.d...
				 */
				improve_queries_with_join_condition (rs, dep, sel_query1, sel_query2,
								     target1, target2,
								     cond, replacements);
			}
			else {
				TO_IMPLEMENT; /* set error msg */
				g_print ("Equi join: %d\n", is_equi_join );
				return FALSE;
			}
		}
		else {
			TO_IMPLEMENT; /* set error msg */
			return FALSE;
		}
	}
	else {
		GSList *fklist = gnome_db_dict_get_entities_fk_constraints (gnome_db_base_get_dict (GNOME_DB_BASE (rs)),
									    ent1, ent2, TRUE);
		if (fklist) {
			improve_queries_with_fk_constraint (rs, dep, sel_query1, sel_query2, 
							    GNOME_DB_CONSTRAINT (fklist->data), replacements);
			g_slist_free (fklist);
		}
	}

	return TRUE;
}

static void
improve_queries_with_db_fields (GnomeDbResultSet *rs, TargetDep *dep, 
				GnomeDbQuery *query_fk, GnomeDbQuery *query_ref_pk,
				const gchar *fk_entity_name, const gchar *ref_pk_entity_name,
				GnomeDbField *fk_field, GnomeDbField *ref_pk_field,
				GHashTable *replacements)
{
	GnomeDbQfield *tmp;
	GnomeDbQfField *qfield, *qfield_pk, *query_sel_field;
	GnomeDbQfValue *vfield;
	GnomeDbCondition *cond, *newcond;
	gchar *str;

	/* adding a field to rs->priv->query_select to make sure the value for that field is actually present
	 * in the rs->priv->query_select query. This means that we need to add _either_ one of @fk_field or
	 * @ref_pk_field
	 *
	 * This field is pointed at by 'query_sel_field' */
	tmp = gnome_db_query_get_field_by_ref_field (rs->priv->query_select, NULL, ref_pk_field, GNOME_DB_FIELD_ANY);
	if (!tmp) {
		tmp = gnome_db_query_get_field_by_ref_field (rs->priv->query_select, NULL, fk_field, GNOME_DB_FIELD_ANY);
		if (!tmp) {
			gchar *str;
			GnomeDbTarget *newtarget;
			newtarget = dep->dependant->target;
			qfield = GNOME_DB_QF_FIELD (gnome_db_qf_field_new_with_objects (rs->priv->query_select,
											newtarget, fk_field));
			query_sel_field = qfield;
			gnome_db_entity_add_field (GNOME_DB_ENTITY (rs->priv->query_select), GNOME_DB_FIELD (qfield));
			g_object_unref (G_OBJECT (qfield));
			gnome_db_qfield_set_internal (GNOME_DB_QFIELD (query_sel_field), TRUE);
			str = g_strdup_printf ("_Target: %s", gnome_db_base_get_name (GNOME_DB_BASE (newtarget)));
			gnome_db_base_set_name (GNOME_DB_BASE (query_sel_field), str);
			g_free (str);
			rs->priv->ref_integ_fields = g_slist_prepend (rs->priv->ref_integ_fields, query_sel_field);
		}
		else
			query_sel_field = GNOME_DB_QF_FIELD (tmp);
	}
	else
		query_sel_field = GNOME_DB_QF_FIELD (tmp);	

	if (! gnome_db_qfield_is_visible (GNOME_DB_QFIELD (query_sel_field))) {
		/* rename the field by prepending an underscore to its name */
		gchar *str;
		str = g_strdup_printf ("_%s", gnome_db_base_get_name (GNOME_DB_BASE (query_sel_field)));
		gnome_db_base_set_name (GNOME_DB_BASE (query_sel_field), str);
		g_free (str);
	}
	gnome_db_qfield_set_visible (GNOME_DB_QFIELD (query_sel_field), TRUE);
	gnome_db_qfield_set_internal (GNOME_DB_QFIELD (query_sel_field), TRUE);

	/* adding PK fields if necessary to query_ref_pk */
	tmp = gnome_db_query_get_field_by_ref_field (query_ref_pk, NULL, ref_pk_field, GNOME_DB_FIELD_ANY);
	if (!tmp) {
		GnomeDbTarget *newtarget;
		newtarget = g_object_get_data (G_OBJECT (query_ref_pk), "for_target");
		qfield = GNOME_DB_QF_FIELD (gnome_db_qf_field_new_with_objects (query_ref_pk,
										newtarget, ref_pk_field));
		gnome_db_entity_add_field (GNOME_DB_ENTITY (query_ref_pk), GNOME_DB_FIELD (qfield));
		g_object_unref (G_OBJECT (qfield));
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (qfield), TRUE);
		gnome_db_base_set_name (GNOME_DB_BASE (qfield), "_IndAdded: PK field");
		rs->priv->ref_integ_fields = g_slist_prepend (rs->priv->ref_integ_fields, qfield);
	}
	else {
		qfield = GNOME_DB_QF_FIELD (tmp);
		if (! gnome_db_qfield_is_visible (GNOME_DB_QFIELD (tmp))) {
			gnome_db_qfield_set_visible (GNOME_DB_QFIELD (tmp), TRUE);
			gnome_db_qfield_set_internal (GNOME_DB_QFIELD (tmp), TRUE);
		}
	}
	gnome_db_qfield_set_visible (GNOME_DB_QFIELD (qfield), TRUE);
	qfield_pk = qfield;

	/* store for later use */
	g_hash_table_insert (rs->priv->data_set_data_for_params_source, qfield_pk, query_sel_field);

	/* adding FK fields if necessary to query_fk */
	tmp = gnome_db_query_get_field_by_ref_field (query_fk, NULL, fk_field, GNOME_DB_FIELD_ANY);
	if (!tmp) {
		GnomeDbTarget *newtarget;
		newtarget = g_object_get_data (G_OBJECT (query_fk), "for_target");
		qfield = GNOME_DB_QF_FIELD (gnome_db_qf_field_new_with_objects (query_fk,
								    newtarget, fk_field));
		gnome_db_entity_add_field (GNOME_DB_ENTITY (query_fk), GNOME_DB_FIELD (qfield));
		g_object_unref (G_OBJECT (qfield));
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (qfield), TRUE);
		gnome_db_base_set_name (GNOME_DB_BASE (qfield), "_IndAdded: FK field");
		rs->priv->ref_integ_fields = g_slist_prepend (rs->priv->ref_integ_fields, qfield);
	}
	else {
		/* We may need to remove a GnomeDbQfValue value provider which has been attached to this field
		 * because otherwise we have an GnomeDbParameter in the data_set which corresponds to nothing */
		GnomeDbQfield *vfield;
		
		qfield = GNOME_DB_QF_FIELD (tmp);
		g_object_get (G_OBJECT (qfield), "value_provider", &vfield, NULL);
		if (vfield) 
			gnome_db_entity_remove_field (GNOME_DB_ENTITY (query_fk), GNOME_DB_FIELD (vfield));
		
	}
	gnome_db_qfield_set_visible (GNOME_DB_QFIELD (qfield), TRUE);
	
	if (replacements && !g_hash_table_lookup (replacements, query_sel_field))
		g_hash_table_insert (replacements, query_sel_field, qfield);

	/* store for later use */
	g_hash_table_insert (rs->priv->data_set_data_for_params_source, qfield, query_sel_field);

	/* add GnomeDbQfValue to query_fk for this field */
	vfield = GNOME_DB_QF_VALUE (gnome_db_qf_value_new (query_fk, gnome_db_field_get_data_type (fk_field)));
	gnome_db_entity_add_field_before (GNOME_DB_ENTITY (query_fk), GNOME_DB_FIELD (vfield), GNOME_DB_FIELD (qfield));
	g_object_unref (G_OBJECT (vfield));
	gnome_db_qf_value_set_is_parameter (vfield, TRUE);
	gnome_db_qfield_set_visible (GNOME_DB_QFIELD (vfield), FALSE);
	gnome_db_qfield_set_internal (GNOME_DB_QFIELD (vfield), TRUE);
	g_object_set (G_OBJECT (vfield), "value_provider", qfield_pk, NULL); /* when PK changes, so does this GnomeDbQfValue */
	g_object_set (G_OBJECT (qfield), "value_provider", vfield, NULL); /* FK is linked to the GnomeDbQfValue */

	/* condition */
	cond = gnome_db_query_get_condition (query_fk);
	if (!cond) {
		cond = GNOME_DB_CONDITION (gnome_db_condition_new (query_fk, GNOME_DB_CONDITION_NODE_AND));
		gnome_db_query_set_condition (query_fk, cond);
		gnome_db_base_set_name (GNOME_DB_BASE (cond), "_IndAdded: AND");
		g_object_unref (G_OBJECT (cond));
	}
	else {
		if (gnome_db_condition_is_leaf (cond)) {
			g_object_ref (G_OBJECT (cond));
			newcond = GNOME_DB_CONDITION (gnome_db_condition_new (query_fk, GNOME_DB_CONDITION_NODE_AND));
			gnome_db_base_set_name (GNOME_DB_BASE (newcond), "_IndAdded: AND");
			gnome_db_query_set_condition (query_fk, newcond);
			g_assert (gnome_db_condition_node_add_child (newcond, cond, NULL));
			g_object_unref (G_OBJECT (cond));
			g_object_unref (G_OBJECT (newcond));
			cond = newcond;
		}
	}
	
	newcond = GNOME_DB_CONDITION (gnome_db_condition_new (query_fk, GNOME_DB_CONDITION_LEAF_EQUAL));
	g_assert (gnome_db_condition_node_add_child (cond, newcond, NULL));
	gnome_db_condition_leaf_set_operator (newcond, GNOME_DB_CONDITION_OP_LEFT, GNOME_DB_QFIELD (qfield));
	gnome_db_condition_leaf_set_operator (newcond, GNOME_DB_CONDITION_OP_RIGHT, GNOME_DB_QFIELD (vfield));
	str = g_strdup_printf ("_Fkcons: %s->%s", fk_entity_name, ref_pk_entity_name);
	gnome_db_base_set_name (GNOME_DB_BASE (newcond), str);
	g_free (str);
	g_object_unref (G_OBJECT (newcond));

	/* GnomeDbQfValue properties */
	str = g_strdup_printf ("_IndAdded: Val %s", gnome_db_base_get_name (GNOME_DB_BASE (fk_field)));
	gnome_db_base_set_name (GNOME_DB_BASE (vfield), str);
	g_free (str);
	gnome_db_base_set_description (GNOME_DB_BASE (vfield), gnome_db_base_get_description (GNOME_DB_BASE (fk_field)));
	gnome_db_qf_value_set_not_null (vfield, !gnome_db_table_field_is_null_allowed (GNOME_DB_TABLE_FIELD (fk_field)));
	if (gnome_db_table_field_get_default_value (GNOME_DB_TABLE_FIELD (fk_field)))
		gnome_db_qf_value_set_default_value (vfield, gnome_db_table_field_get_default_value (GNOME_DB_TABLE_FIELD (fk_field)));
	
	/* keep a reference to the field in 'rs->priv->query_select' which we 
	   are now treating */
	g_object_set_data (G_OBJECT (vfield), "position", query_sel_field);
}

static void
improve_queries_with_fk_constraint (GnomeDbResultSet *rs, TargetDep *dep, GnomeDbQuery *query_fk, 
				    GnomeDbQuery *query_ref_pk, GnomeDbConstraint *fkcons,
				    GHashTable *replacements)
{
	GSList *pairs, *list;
	const GSList *clist;
	
	pairs = gnome_db_constraint_fkey_get_fields (fkcons);
	list = pairs;
	while (list) {
		improve_queries_with_db_fields (rs, dep, query_fk, query_ref_pk,
						gnome_db_base_get_name (GNOME_DB_BASE (gnome_db_constraint_get_table (fkcons))),
						gnome_db_base_get_name (GNOME_DB_BASE (gnome_db_constraint_fkey_get_ref_table (fkcons))),
						GNOME_DB_FIELD (GNOME_DB_CONSTRAINT_FK_PAIR (list->data)->fkey),
						GNOME_DB_FIELD (GNOME_DB_CONSTRAINT_FK_PAIR (list->data)->ref_pkey),
						replacements);
		
		list = g_slist_next (list);
	}
	g_slist_free (pairs);


	/* setting query_ref_pk to be a 'param_source' for query_fk */
	clist = gnome_db_query_get_param_sources (query_fk);
	if (!g_slist_find (clist, query_ref_pk))
		gnome_db_query_add_param_source (query_fk, query_ref_pk);
}


static void
improve_queries_with_join_condition (GnomeDbResultSet *rs, TargetDep *dep, 
				     GnomeDbQuery *query_fk, GnomeDbQuery *query_ref_pk, 
				     GnomeDbTarget *fk_target, GnomeDbTarget *ref_pk_target,
				     GnomeDbCondition *cond, GHashTable *replacements)
{
	const GSList *list, *mainconds;
	
	
	mainconds = gnome_db_condition_get_main_conditions (cond);
	list = mainconds;
	while (list) {
		GnomeDbCondition *cond = GNOME_DB_CONDITION (list->data);
		GnomeDbQfield *field;
		GnomeDbField *fk_field = NULL, *ref_pk_field = NULL;

		g_return_if_fail (gnome_db_condition_get_cond_type (cond) == GNOME_DB_CONDITION_LEAF_EQUAL);
		
		field = gnome_db_condition_leaf_get_operator (cond, GNOME_DB_CONDITION_OP_LEFT);
		g_return_if_fail (field && IS_GNOME_DB_QF_FIELD (field));
		if (gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (field)) == fk_target)
			fk_field = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field));
		else
			ref_pk_field = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field));
		
		field = gnome_db_condition_leaf_get_operator (cond, GNOME_DB_CONDITION_OP_RIGHT);
		g_return_if_fail (field && IS_GNOME_DB_QF_FIELD (field));
		if (gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (field)) == fk_target)
			fk_field = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field));
		else
			ref_pk_field = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field));
		
		g_return_if_fail (fk_field);
		g_return_if_fail (ref_pk_field);

		improve_queries_with_db_fields (rs, dep, query_fk, query_ref_pk,
						gnome_db_base_get_name (GNOME_DB_BASE (gnome_db_field_get_entity (fk_field))),
						gnome_db_base_get_name (GNOME_DB_BASE (gnome_db_field_get_entity (ref_pk_field))),
						fk_field, ref_pk_field, 
						replacements);
		
		list = g_slist_next (list);
	}
	

	/* setting query_ref_pk to be a 'param_source' for query_fk */
	list = gnome_db_query_get_param_sources (query_fk);
	if (!g_slist_find (list, query_ref_pk))
		gnome_db_query_add_param_source (query_fk, query_ref_pk);
}

#ifdef debug
static void
target_dep_dump (TargetDep *dep, gint offset)
{
	gchar *str;
        guint i;
	GSList *list;

	/* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

	g_print ("%sDEP target=%p (%s) dependant:%p\n", str, dep->target, gnome_db_base_get_name (GNOME_DB_BASE (dep->target)), 
		 dep->dependant ? dep->dependant->target : NULL);
	list = dep->depend_on;
	while (list) {
		target_dep_dump (TARGET_DEP (list->data), offset+5);
		list = g_slist_next (list);
	}
	g_free (str);
}
#endif


/*
 * make all the required work to call make_target_deps_recurs
 */
static TargetDep *
make_target_deps (GnomeDbResultSet *rs)
{
	TargetDep *dep;
	GSList *joins;
	GHashTable *hash;
	
	g_return_val_if_fail (rs->priv->query_select, NULL);
	g_return_val_if_fail (rs->priv->modif_target, NULL);

	joins = gnome_db_query_get_joins (rs->priv->query_select);
	hash = g_hash_table_new (NULL, NULL);

	dep = make_target_deps_recurs (rs, rs->priv->modif_target, joins, hash);

	g_slist_free (joins);
	g_hash_table_destroy (hash);

#ifdef debug_NO
	g_print ("###################################### target deps\n");
	target_dep_dump (dep, 0);
#endif

	return dep;
}

/*
 * Creates a TargetDep structure to hold all the GnomeDbTarget on which 'on_target' depends.
 */
static TargetDep *
make_target_deps_recurs (GnomeDbResultSet *rs, GnomeDbTarget *on_target, const GSList *joins, GHashTable *joinsdep)
{
	TargetDep *dep = NULL;
	const GSList *list;

	dep = g_new0 (TargetDep, 1);
	dep->target = on_target;
	dep->depend_on = NULL;
	dep->join = NULL;

	list = joins;
	while (list) {
		if (!g_hash_table_lookup (joinsdep, list->data)) {
			/* This join has not yet been treaded */
			GnomeDbTarget *t1, *t2, *tmp = NULL;
			GnomeDbJoinType type;
			gboolean useit = FALSE;

			t1 = gnome_db_join_get_target_1 (GNOME_DB_JOIN (list->data));
			t2 = gnome_db_join_get_target_2 (GNOME_DB_JOIN (list->data));
			type = gnome_db_join_get_join_type (GNOME_DB_JOIN (list->data));
			switch (type) {
			case GNOME_DB_JOIN_TYPE_INNER:
				if ((t1 == on_target) || (t2 == on_target)) {
					useit = TRUE;
					if (t1 == on_target)
						tmp = t2;
					else
						tmp = t1;
				}
				break;
			case GNOME_DB_JOIN_TYPE_RIGHT_OUTER:
				tmp = t1;
				t1 = t2;
				t2 = tmp;
			case GNOME_DB_JOIN_TYPE_LEFT_OUTER:
				if (t1 == on_target) {
					useit = TRUE;
					tmp = t2;
				}
			default:
				/* we can't do anything here */
				break;
			}
			
			if (useit) {
				TargetDep *tdep;

				g_hash_table_insert (joinsdep, list->data, tmp);
				tdep = make_target_deps_recurs (rs, tmp, joins, joinsdep);
				dep->depend_on = g_slist_append (dep->depend_on, tdep);
				tdep->join = GNOME_DB_JOIN (list->data);
				tdep->dependant = dep;
			}
		}
		list = g_slist_next (list);
	}

	return dep;
}

static void
make_target_deps_free (TargetDep *dep)
{
	GSList *list = dep->depend_on;
	
	while (list) {
		make_target_deps_free (TARGET_DEP (list->data));
		list = g_slist_next (list);
	}

	if (dep->depend_on)
		g_slist_free (dep->depend_on);
	g_free (dep);
}
