#include <stdio.h>
#include <string.h>
#include "../libgnomedb/libgnomedb.h"

#define MAKE_DUMPS 1

#define STEP_SEPARATOR "------------------"
#define xmlfile "SQL_tests.xml"

void make_query_test (GnomeDbDict *dict, FILE *stream, const gchar *sql, gboolean expected_result);

gint main (int argc, char **argv) {
	GnomeDbDict *dict = NULL;
	GError *error = NULL;
	xmlDocPtr doc;
	xmlNodePtr node, subnode;
	FILE *out = stdout;
	gboolean in_file = FALSE;

	gtk_init (&argc, &argv);

	g_print ("# This test creates\n");
	g_print ("# Queries FROM SQL statements\n\n");


	if (! g_file_test (xmlfile, G_FILE_TEST_EXISTS)) {
		g_print ("cant' find file '%s'\n", xmlfile);
		exit (1);
	}

	doc = xmlParseFile (xmlfile);
	if (!doc) {
		g_print ("Cant' load XML file '%s'\n", xmlfile);
		exit (1);
	}

	node = xmlDocGetRootElement (doc);
	if (strcmp (node->name, "test_scenario")) {
		g_print ("XML file top node is not <test_scenario>\n");
		exit (1);
	}

	subnode = node->children;
	while (subnode) {
		/* new dictionnary */
		if (!strcmp (subnode->name, "dictionnary")) {
			gchar *filename = xmlGetProp (subnode, "name");
			if (dict) {
				g_object_unref (G_OBJECT (dict));
				dict = NULL;
			}

			if (filename) {
				dict = GNOME_DB_DICT (gnome_db_dict_new ());
				g_print ("Loading dictionnary %s\n", filename);
				if (!gnome_db_dict_load_xml_file (dict, filename, &error)) {
					g_print ("Error occurred:\n\t%s\n", error->message);
					g_error_free (error);
					exit (1);
				}
				g_free (filename);
			}
		}
		
		/* new output file */
		if (!strcmp (subnode->name, "output_file")) {
			gchar *filename = xmlGetProp (subnode, "name");
			if (in_file) {
				/* HTML end */
				g_fprintf (out, "</table>\n");
				g_fprintf (out, "</body>\n");
				g_fprintf (out, "</html>\n");
				fclose (out);
			}
			out = stdout;

			if (filename) {
				g_print ("Writing to %s\n", filename);
				out = fopen (filename, "w");
				if (!out) {
					g_print ("Can't open '%s' to write\n", filename);
					exit (1);
				}
				g_free (filename);
			}

			/* HTML init */
			g_fprintf (out, "<html>\n");
			g_fprintf (out, "<body>\n");
			g_fprintf (out, "<h2>Note:</h2>"
				   "This test reads the " xmlfile " file and for each query listed in the file "
				   "creates a GnomeDbQuery object, parses the SQL statement a first time (from an empty query), "
				   "and parses a second time, and makes the comparison.\n");
			g_fprintf (out, "<table cellspacing=\"3\" cellpadding=\"3\" border=\"1\" width=\"100%\">\n");

			in_file = TRUE;
		}
		
		/* new test group */
		if (!strcmp (subnode->name, "test_group")) {
			xmlNodePtr test = subnode->children;
			gchar *descr = xmlGetProp (subnode, "descr");

			g_fprintf (out, "<tr><th colspan=\"4\" bgcolor=\"green4\">%s</th></tr>\n", descr);
			g_fprintf (out, "<tr><th>SQL to parse</th><th>Result SQL</th><th>Re-parsed result SQL</th><th>Error</th></tr>\n", descr);
			g_free (descr);

			while (test) {
				if (!strcmp (test->name, "test")) {
					gchar *sql = xmlGetProp (test, "sql");
					gchar *expected = xmlGetProp (test, "expect");

					if (sql) {
						make_query_test (dict, out, sql, 
								 expected && (*expected=='N') ? FALSE : TRUE);
						g_free (sql);
					}
					if (expected)
						g_free (expected);
				}
				test = test->next;
			}
		}
		subnode = subnode->next;
	}
	xmlFreeDoc (doc);

	if (in_file) {
		/* HTML end */
		g_fprintf (out, "</table>\n");
		g_fprintf (out, "</body>\n");
		g_fprintf (out, "</html>\n");
		fclose (out);
	}

	if (dict)
		g_object_unref (G_OBJECT (dict));
	return 0;
}

static void write_empty (FILE *stream, gint nb_cols);
static void write_error (FILE *stream, GError *error, gboolean render_error, gboolean highlight);

void
make_query_test (GnomeDbDict *dict, FILE *stream, const gchar *sql, gboolean expected_result)
{
	GnomeDbQuery *query;
	GError *error = NULL;
	gint err = 0;

	g_fprintf (stream, "\t<tr><td>%s</td>", sql);
	query = (GnomeDbQuery *) gnome_db_query_new_from_sql (dict, sql, &error);
	if (gnome_db_query_get_query_type (query) != GNOME_DB_QUERY_TYPE_NON_PARSED_SQL) {
		gchar *sql2;
		sql2 = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, GNOME_DB_RENDERER_EXTRA_VAL_ATTRS, &error);
		if (sql2) {
			g_fprintf (stream, "<td>%s</td>", sql2);
			gnome_db_query_set_sql_text (query, sql, &error);
			if (gnome_db_query_get_query_type (query) != GNOME_DB_QUERY_TYPE_NON_PARSED_SQL) {
				gchar *sql3, *extra = "";
				sql3 = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, GNOME_DB_RENDERER_EXTRA_VAL_ATTRS, &error);
				if (sql3) {
					if (strcmp (sql2, sql3))
						extra = "bgcolor=\"yellow\"";
					g_fprintf (stream, "<td %s>%s</td>", extra, sql3);
					write_empty (stream, 1);
					g_free (sql3);
				}
				else {
					/* SQL render pb */
					write_empty (stream, 1);
					write_error (stream, error, TRUE, FALSE);
					err = 4;
				}
			}
			else {
				/* SQL parsing pb */
				write_empty (stream, 1);
				write_error (stream, error, FALSE, expected_result);
				err = 3;
			}
			g_free (sql2);
		}
		else {
			/* SQL render pb */
			write_empty (stream, 2);
			write_error (stream, error, TRUE, FALSE);
			err = 2;
		}
	}
	else {
		/* SQL parsing pb */
		write_empty (stream, 2);
		write_error (stream, error, FALSE, expected_result);
		err = 1;
	}

	if (err) {
		switch (err) {
		case 1:
		case 2:
			g_print ("ERROR PARSING, ");
			g_print ("SQL: %s\n", sql);
			break;
		case 3:
		case 4:
			g_print ("ERROR RENDERING\n");
			break;
		}
	}

	g_object_unref (query);
	g_fprintf (stream, "</tr>\n");
}

static void
write_empty (FILE *stream, gint nb_cols)
{
	gint i;
	for (i=0; i<nb_cols; i++)
		g_fprintf (stream, "<td>---</td>");
}

static void
write_error (FILE *stream, GError *error, gboolean render_error, gboolean highlight)
{
	gchar *extra = "";
	if (render_error)
		extra = "bgcolor=\"orange\"";
	else {
		if (highlight)
			extra = "bgcolor=\"red\"";
	}
	
	if (error && error->message)
		g_fprintf (stream, "<td %s>ERROR: %s</td>", extra, error->message);
	else
		g_fprintf (stream, "<td %s>NO ERROR MSG</td>", extra);
	if (error)
		g_error_free (error);
}
