/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgda/gda-data-model-list.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-list.h>
#include <libgnomedb/gnome-db-util.h>

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbListPrivate {
	GdaDataModel *data_model;
	GtkWidget *grid;
};

static void gnome_db_list_class_init   (GnomeDbListClass *klass);
static void gnome_db_list_init         (GnomeDbList *list, GnomeDbListClass *klass);
static void gnome_db_list_set_property (GObject *object,
					guint paramid,
					const GValue *value,
					GParamSpec *pspec);
static void gnome_db_list_get_property (GObject *object,
					guint param_id,
					GValue *value,
					GParamSpec *pspec);
static void gnome_db_list_finalize     (GObject *object);

enum {
	ROW_SELECTED,
	SELECTION_CLEARED,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_MODEL
};

static gint list_signals[LAST_SIGNAL];
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
grid_row_selected_cb (GnomeDbGrid *grid, gint row, gpointer user_data)
{
	g_return_if_fail (GNOME_DB_IS_LIST (user_data));
	g_signal_emit (G_OBJECT (user_data), list_signals[ROW_SELECTED], 0, row);
}

static void
grid_selection_cleared_cb (GnomeDbGrid *grid, gpointer user_data)
{
	g_return_if_fail (GNOME_DB_IS_LIST (user_data));
	g_signal_emit (G_OBJECT (user_data), list_signals[SELECTION_CLEARED], 0);
}

/*
 * GnomeDbList class implementation
 */

static void
gnome_db_list_class_init (GnomeDbListClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_list_set_property;
	object_class->get_property = gnome_db_list_get_property;
	object_class->finalize = gnome_db_list_finalize;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_MODEL,
		g_param_spec_object ("model", NULL, NULL,
				     GDA_TYPE_DATA_MODEL,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* add class signals */
	list_signals[ROW_SELECTED] =
		g_signal_new ("row_selected",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, row_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);
	list_signals[SELECTION_CLEARED] =
		g_signal_new ("selection_cleared",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, row_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
}

static void
gnome_db_list_init (GnomeDbList *list, GnomeDbListClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_LIST (list));

	/* allocate private structure */
	list->priv = g_new0 (GnomeDbListPrivate, 1);
	list->priv->data_model = NULL;

	/* set up widgets */
	list->priv->grid = gnome_db_new_grid_widget (NULL);
	gnome_db_grid_set_selection_mode (GNOME_DB_GRID (list->priv->grid),
					  GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (list->priv->grid), "row_selected",
			  G_CALLBACK (grid_row_selected_cb), list);
	g_signal_connect (G_OBJECT (list->priv->grid), "selection_cleared",
			  G_CALLBACK (grid_selection_cleared_cb), list);
	gnome_db_grid_hide_column_titles (GNOME_DB_GRID (list->priv->grid));
	gtk_box_pack_start (GTK_BOX (list), list->priv->grid, TRUE, TRUE, 0);
}

static void
gnome_db_list_set_property (GObject *object,
			    guint param_id,
			    const GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbList *list = (GnomeDbList *) object;

	g_return_if_fail (GNOME_DB_IS_LIST (list));

	switch (param_id) {
	case PROP_MODEL :
		gnome_db_list_set_model (list,
					 GDA_DATA_MODEL (g_value_get_object (value)),
					 0);
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_list_get_property (GObject *object,
			    guint param_id,
			    GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbList *list = (GnomeDbList *) object;

	g_return_if_fail (GNOME_DB_IS_LIST (list));

	switch (param_id) {
	case PROP_MODEL :
		g_value_set_object (value, G_OBJECT (list->priv->data_model));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_list_finalize (GObject *object)
{
	GnomeDbList *list = (GnomeDbList *) object;

	g_return_if_fail (GNOME_DB_IS_LIST (list));

	/* free memory */
	if (GDA_IS_DATA_MODEL (list->priv->data_model)) {
		g_object_unref (G_OBJECT (list->priv->data_model));
		list->priv->data_model = NULL;
	}

	g_free (list->priv);
	list->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_list_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbListClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_list_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbList),
			0,
			(GInstanceInitFunc) gnome_db_list_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbList", &info, 0);
	}
	return type;
}

/**
 * gnome_db_list_new
 *
 * Create a new #GnomeDbList widget, which lets you display lists
 * of columns.
 *
 * Returns: the newly created object
 */
GtkWidget *
gnome_db_list_new (void)
{
	GnomeDbList *list;

	list = g_object_new (GNOME_DB_TYPE_LIST, NULL);
	return GTK_WIDGET (list);
}

/**
 * gnome_db_list_new_with_model
 * @model:
 * @col:
 *
 *
 *
 * Returns:
 */
GtkWidget *
gnome_db_list_new_with_model (GdaDataModel *model, gint col)
{
	GtkWidget *list;

	list = gnome_db_list_new ();
	gnome_db_list_set_model (GNOME_DB_LIST (list), model, col);

	return list;
}

/**
 * gnome_db_list_get_model
 * @list:
 *
 *
 *
 * Returns:
 */
GdaDataModel *
gnome_db_list_get_model (GnomeDbList *list)
{
	g_return_val_if_fail (GNOME_DB_IS_LIST (list), NULL);
	return list->priv->data_model;
}

/**
 * gnome_db_list_set_model
 * @list: a #GnomeDbList widget.
 * @model: a #GdaDataModel object.
 * @col: column to be shown from the data model.
 *
 * Set the data model to be displayed by the #GnomeDbList widget.
 * As this widget just shows single-columns data models, you have
 * to specify, along with the #GdaDataModel object, the column position
 * to be shown.
 */
void
gnome_db_list_set_model (GnomeDbList *list, GdaDataModel *model, gint col)
{
	gint n;
	gint row_count;

	g_return_if_fail (GNOME_DB_IS_LIST (list));

	/* free previous data model */
	if (GDA_IS_DATA_MODEL (list->priv->data_model)) {
		g_object_unref (G_OBJECT (list->priv->data_model));
		list->priv->data_model = NULL;
	}

	/* fill in our private data model */
	list->priv->data_model = gda_data_model_list_new ();

	if (GDA_IS_DATA_MODEL (model)) {
		row_count = gda_data_model_get_n_rows (model);
		for (n = 0; n < row_count; n++) {
			const GdaValue *value;

			value = gda_data_model_get_value_at (model, col, n);
			if (value != NULL) {
				gda_data_model_list_append_value (
					GDA_DATA_MODEL_LIST (list->priv->data_model),
					value);
			}
		}
	}

	gnome_db_grid_set_model (GNOME_DB_GRID (list->priv->grid), list->priv->data_model);
	gnome_db_grid_hide_column_titles (GNOME_DB_GRID (list->priv->grid));
}

/**
 * gnome_db_list_get_selection:
 * @list: a #GnomeDbList widget.
 *
 * Gets the list of selected rows in the given %GnomeDbList widget.
 *
 * Returns: a GList of integers representing the numbers of the selected
 * rows. This list should be freed (by calling * #g_list_free) when no
 * longer needed.
 */
GList *
gnome_db_list_get_selection (GnomeDbList *list)
{
	g_return_val_if_fail (GNOME_DB_IS_LIST (list), NULL);
	return gnome_db_grid_get_selection (GNOME_DB_GRID (list->priv->grid));
}
