/* GNOME DB library
 * Copyright (C) 1999-2001 The Free Software Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <bonobo/bonobo-i18n.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkpaned.h>
#include <libgnomedb/gnome-db-browser.h>
#include <libgnomedb/gnome-db-list.h>
#include <libgnomedb/gnome-db-util.h>

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbBrowserPrivate {
	GdaConnection *cnc;

	/* widgets */
	GtkWidget *paned;
	GtkWidget *object_notebook;
	GtkWidget *detail_notebook;

	/* objects management */
	struct {
		GdaConnectionSchema schema;
		GtkWidget *object_list;
		GtkWidget *object_detail;
		gboolean data_retrieved;
	} objects_info [1];
};

GtkWidget  *gnome_db_browser_tables_new (void);

static void gnome_db_browser_class_init (GnomeDbBrowserClass *klass);
static void gnome_db_browser_init       (GnomeDbBrowser *browser, GnomeDbBrowserClass *klass);
static void gnome_db_browser_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
initialize_object_widgets (GnomeDbBrowser *browser)
{
	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* initialize tables widgets */
	browser->priv->objects_info[0].schema = GDA_CONNECTION_SCHEMA_TABLES;
	browser->priv->objects_info[0].data_retrieved = FALSE;

	browser->priv->objects_info[0].object_list = gnome_db_list_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->object_notebook),
				  browser->priv->objects_info[0].object_list,
				  gtk_label_new (_("Tables")));
	gtk_widget_show (browser->priv->objects_info[0].object_list);

	browser->priv->objects_info[0].object_detail = gnome_db_browser_tables_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
				  browser->priv->objects_info[0].object_detail,
				  gtk_label_new (_("Tables")));
}

/*
 * GnomeDbBrowser class implementation
 */

static void
gnome_db_browser_class_init (GnomeDbBrowserClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_browser_finalize;
}

static void
gnome_db_browser_init (GnomeDbBrowser *browser, GnomeDbBrowserClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* allocate private structure */
	browser->priv = g_new0 (GnomeDbBrowserPrivate, 1);
	browser->priv->cnc = NULL;

	/* set up widgets */
	browser->priv->paned = gnome_db_new_hpaned_widget ();
	gtk_box_pack_start (GTK_BOX (browser), browser->priv->paned, TRUE, TRUE, 0);

	browser->priv->object_notebook = gnome_db_new_notebook_widget ();
	gtk_paned_add1 (GTK_PANED (browser->priv->paned), browser->priv->object_notebook);

	browser->priv->detail_notebook = gnome_db_new_notebook_widget ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (browser->priv->detail_notebook), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (browser->priv->detail_notebook), FALSE);
	gtk_paned_add2 (GTK_PANED (browser->priv->paned), browser->priv->detail_notebook);

	initialize_object_widgets (browser);
}

static void
gnome_db_browser_finalize (GObject *object)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* free memory */
	if (GDA_IS_CONNECTION (browser->priv->cnc)) {
		g_object_unref (G_OBJECT (browser->priv->cnc));
		browser->priv->cnc = NULL;
	}

	g_free (browser->priv);
	browser->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_browser_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbBrowserClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_browser_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbBrowser),
			0,
			(GInstanceInitFunc) gnome_db_browser_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbBrowser", &info, 0);
	}
	return type;
}

/**
 * gnome_db_browser_new
 *
 * Create a new #GnomeDbBrowser widget.
 *
 * Returns: the newly created object.
 */
GtkWidget *
gnome_db_browser_new (void)
{
	GnomeDbBrowser *browser;

	browser = g_object_new (GNOME_DB_TYPE_BROWSER, NULL);
	return GTK_WIDGET (browser);
}

/**
 * gnome_db_browser_get_connection
 */
GdaConnection *
gnome_db_browser_get_connection (GnomeDbBrowser *browser)
{
	g_return_val_if_fail (GNOME_DB_IS_BROWSER (browser), NULL);
	return browser->priv->cnc;
}

/*
 * gnome_db_browser_set_connection
 */
void
gnome_db_browser_set_connection (GnomeDbBrowser *browser, GdaConnection *cnc)
{
	gint n;
	gint object_count;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* clear the current setup */
	if (GDA_IS_CONNECTION (browser->priv->cnc)) {
		g_object_unref (G_OBJECT (browser->priv->cnc));
		browser->priv->cnc = NULL;
	}

	object_count = sizeof (browser->priv->objects_info) / sizeof (browser->priv->objects_info[0]);
	for (n = 0; n < object_count; n++) {
		gnome_db_list_set_model (
		     GNOME_DB_LIST (browser->priv->objects_info[n].object_list),
		     NULL, 0);
		browser->priv->objects_info[n].data_retrieved = FALSE;
	}

	/* refresh views */
	if (GDA_IS_CONNECTION (cnc))
		g_object_ref (G_OBJECT (cnc));
	browser->priv->cnc = cnc;

	/* refresh current view */
	for (n = 0; n < object_count; n++) {
		GdaDataModel *recset;

		recset = gda_connection_get_schema (
			browser->priv->cnc,
			browser->priv->objects_info[n].schema,
			NULL);
		gnome_db_list_set_model (
		     GNOME_DB_LIST (browser->priv->objects_info[n].object_list),
		     recset, 0);

		g_object_unref (G_OBJECT (recset));
	}
}
