/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 * 	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgnomedb/gnome-db-stock.h>

/**
 * gnome_db_stock_get_icon_pixbuf
 * @stock_id: ID of icon to get.
 *
 * Get the given stock icon as a GdkPixbuf.
 *
 * Returns: a GdkPixbuf representing the loaded icon.
 */
GdkPixbuf *
gnome_db_stock_get_icon_pixbuf (const gchar *stock_id)
{
	gchar *path;
	GdkPixbuf *pixbuf;

	g_return_val_if_fail (stock_id != NULL, NULL);

	path = gnome_db_stock_get_icon_path (stock_id);
	if (!path)
		return NULL;

	pixbuf = gdk_pixbuf_new_from_file (path, NULL);

	g_free (path);

	return pixbuf;
}

/**
 * gnome_db_stock_get_icon_pixbuf_file
 * @pixmapfile: the real filename of the icon to get.
 *
 * Get the given icon as a GdkPixbuf.
 *
 * Returns: a GdkPixbuf representing the loaded icon.
 */
GdkPixbuf *
gnome_db_stock_get_icon_pixbuf_file (const gchar *pixmapfile)
{
	gchar *path;
	GdkPixbuf *pixbuf;

	g_return_val_if_fail (pixmapfile != NULL, NULL);

	path = g_strdup_printf ("%s/%s", LIBGNOMEDB_ICONSDIR, pixmapfile);
	if (!path)
		return NULL;

	pixbuf = gdk_pixbuf_new_from_file (path, NULL);

	g_free (path);

	return pixbuf;
}

/**
 * gnome_db_stock_get_icon_path
 * @stock_id: Icon id to get path.
 *
 * Return the full path of the given stock icon.
 *
 * Returns: the full path of the given stock icon. The returned
 * value must be freed by the caller when no longer needed.
 */
gchar *
gnome_db_stock_get_icon_path (const gchar *stock_id)
{
	gchar *path;

	g_return_val_if_fail (stock_id != NULL, NULL);

	path = g_strdup_printf ("%s/%s_24x24.png", LIBGNOMEDB_ICONSDIR, stock_id);
	return path;
}
