/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 * 	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <libgnomedb/gnome-db-model.h>
#include <libgnomedb/gnome-db-util.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrenderertoggle.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreeviewcolumn.h>
#include "libgnomedb-private.h"

#define DATA_MODEL_INFO_TYPE data_model_info_get_type ()

/*
 * Private functions
 */

static void
data_model_info_free (DataModelInfo *info)
{
	g_object_unref (G_OBJECT (info->model));
	g_free (info);
}

static DataModelInfo *
data_model_info_copy (DataModelInfo *src)
{
	DataModelInfo *info;

	g_return_val_if_fail (src != NULL, NULL);

	info = g_new0 (DataModelInfo, 1);
	info->row = src->row;
	info->model = src->model;
	if (info->model)
		g_object_ref (G_OBJECT (info->model));

	return info;
}

static GType
data_model_info_get_type (void)
{
	static GType type = 0;

	if (!type) {
		type = g_boxed_type_register_static (
			"GNOME_DB_DataModelInfo",
			(GBoxedCopyFunc) data_model_info_copy,
			(GBoxedFreeFunc) data_model_info_free);
	}
	return type;
}

static gint
sort_values (GtkTreeModel *model, GtkTreeIter *itera, GtkTreeIter *iterb, gpointer user_data)
{
	DataModelInfo *infoa, *infob;
	const GdaValue *valuea, *valueb;
	gint retval = 1;

	gtk_tree_model_get (model, itera, 0, &infoa, -1);
	gtk_tree_model_get (model, iterb, 0, &infob, -1);
	if (!infoa || !infob)
		return 0;

	valuea = gda_data_model_get_value_at (infoa->model,
					      GPOINTER_TO_INT (user_data),
					      infoa->row);
	valueb = gda_data_model_get_value_at (infob->model,
					      GPOINTER_TO_INT (user_data),
					      infob->row);
	if (!valuea || !valueb)
		return 0;

	/* compare values */
	if (gda_value_isa ((GdaValue *) valuea, GDA_VALUE_TYPE_NULL) &&
	    gda_value_isa ((GdaValue *) valueb, GDA_VALUE_TYPE_NULL))
		retval = 0;
	else if (gda_value_isa ((GdaValue *) valuea, GDA_VALUE_TYPE_BIGINT) &&
		 gda_value_isa ((GdaValue *) valueb, GDA_VALUE_TYPE_BIGINT)) {
		if (gda_value_get_bigint ((GdaValue *) valuea) >
		    gda_value_get_bigint ((GdaValue *) valueb))
			retval = -1;
	}
	else {
		gchar *stra, *strb;

		/* data types don't match, so use strings to sort */
		stra = gda_value_stringify ((GdaValue *) valuea);
		strb = gda_value_stringify ((GdaValue *) valueb);
		retval = strcmp (stra, strb);

		g_free (stra);
		g_free (strb);
	}

	return retval;

}

/*
 * Callbacks
 */

static void
default_value_set_func (GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *model,
			GtkTreeIter *iter,
			gpointer user_data)
{
	DataModelInfo *info = NULL;
	const GdaValue *value;
	gchar *txt;
	gchar *str;

	gtk_tree_model_get (model, iter, 0, &info, -1);
	if (!info)
		return;

	value = gda_data_model_get_value_at (info->model,
					     GPOINTER_TO_INT (user_data),
					     info->row);
	if (!value)
		return;

	txt = gda_value_stringify ((GdaValue *) value);
	if (gda_value_isa (value, GDA_VALUE_TYPE_BIGINT)
	    || gda_value_isa (value, GDA_VALUE_TYPE_DOUBLE)
	    || gda_value_isa (value, GDA_VALUE_TYPE_INTEGER)
	    || gda_value_isa (value, GDA_VALUE_TYPE_NUMERIC)
	    || gda_value_isa (value, GDA_VALUE_TYPE_SINGLE)
	    || gda_value_isa (value, GDA_VALUE_TYPE_SMALLINT)
	    || gda_value_isa (value, GDA_VALUE_TYPE_TINYINT)) {
		/* FIXME: get color for numbers from configuration */
		str = g_strdup_printf ("<span foreground=\"#0000FF\">%s</span>", txt);
		g_object_set (G_OBJECT (cell), "markup", str,
			      "xalign", 1.0, NULL);
		g_free (str);
	}
	else
		g_object_set (G_OBJECT (cell), "text", txt, NULL);

	g_free (txt);
}

static void
toggle_value_set_func (GtkTreeViewColumn *tree_column,
		       GtkCellRenderer *cell,
		       GtkTreeModel *model,
		       GtkTreeIter *iter,
		       gpointer user_data)
{
	DataModelInfo *info = NULL;
	const GdaValue *value;

	gtk_tree_model_get (model, iter, 0, &info, -1);
	if (!info)
		return;

	value = gda_data_model_get_value_at (info->model,
					     GPOINTER_TO_INT (user_data),
					     info->row);
	if (!value)
		return;

	gtk_cell_renderer_toggle_set_active (GTK_CELL_RENDERER_TOGGLE (cell),
					     gda_value_get_boolean ((GdaValue *) value));
}

static gchar *double_underscores (const gchar *str)
{
	gchar **arr;
	gchar *ret;

	arr = g_strsplit (str, "_", 0);
	ret = g_strjoinv ("__", arr);
	g_strfreev (arr);

	return ret;
}

/**
 * gnome_db_model_to_gtk_tree_model
 */
GtkTreeView *
gnome_db_model_to_gtk_tree_view (GdaDataModel *model)
{
	gint i;
	gint row_count;
	gint col_count;
	GtkTreeView *tree_view;
	GtkListStore *tree_model = NULL;

	tree_model = gtk_list_store_new (1, DATA_MODEL_INFO_TYPE);
	row_count = gda_data_model_get_n_rows (model);
	for (i = 0; i < row_count; i++) {
		DataModelInfo info;
		GtkTreeIter iter;

		info.row = i;
		info.model = model;
		gtk_list_store_append (tree_model, &iter);
		gtk_list_store_set (tree_model, &iter, 0, &info, -1);
	}

	/* create the tree view widget */
	tree_view = GTK_TREE_VIEW (gnome_db_new_tree_view_widget (
					   GTK_TREE_MODEL (tree_model)));
	g_object_unref (G_OBJECT (tree_model));

	col_count = gda_data_model_get_n_columns (model);
	for (i = 0; i < col_count; i++) {
		GtkTreeViewColumn *column;
		GtkCellRenderer *renderer;
		GdaFieldAttributes *attrs;
		gchar *title;
		gboolean alloc_title;

		alloc_title = FALSE;
		title = (gchar *) gda_data_model_get_column_title (model, i);
		if (title && strchr (title, '_')) {
			title = double_underscores (title);
			alloc_title = TRUE;
		}
		attrs = gda_data_model_describe_column (model, i);
		if (attrs && attrs->gdaType == GDA_TYPE_BOOLEAN) {
			renderer = gtk_cell_renderer_toggle_new ();
			gtk_tree_view_insert_column_with_data_func (
				GTK_TREE_VIEW (tree_view),
				i,
				title,
				renderer,
				toggle_value_set_func,
				GINT_TO_POINTER (i),
				NULL);
			gda_field_attributes_free (attrs);
		}
		else {
			renderer = gtk_cell_renderer_text_new ();
			gtk_tree_view_insert_column_with_data_func (
				GTK_TREE_VIEW (tree_view),
				i,
				title,
				renderer,
				default_value_set_func,
				GINT_TO_POINTER (i),
				NULL);
		}

		if (alloc_title)
			g_free (title);

		column = gtk_tree_view_get_column (GTK_TREE_VIEW (tree_view), i);
		gtk_tree_view_column_set_resizable (column, TRUE);
		gtk_tree_view_column_set_clickable (column, TRUE);
		gtk_tree_view_column_set_reorderable (column, TRUE);

		gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (tree_model),
						 i, sort_values,
						 GINT_TO_POINTER (i),
						 NULL);
	}

	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (tree_model),
					      0, GTK_SORT_ASCENDING);
	gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (tree_view), TRUE);

	return tree_view;
}
