/* testhashmap.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <tests/testhashmap.h>
#include <gee/hashmap.h>
#include <stdlib.h>
#include <string.h>
#include <gee/map.h>
#include <gee/set.h>
#include <gee/collection.h>




#define CODE_NOT_REACHABLE "*code should not be reached*"
static void test_hashmap_get (void);
static void test_hashmap_set (void);
static void test_hashmap_remove (void);
static void test_hashmap_contains (void);
static void test_hashmap_size (void);
static void test_hashmap_get_keys (void);
static void test_hashmap_get_values (void);
static void test_hashmap_clear (void);
static void _test_hashmap_get_gcallback (void);
static void _test_hashmap_set_gcallback (void);
static void _test_hashmap_remove_gcallback (void);
static void _test_hashmap_contains_gcallback (void);
static void _test_hashmap_size_gcallback (void);
static void _test_hashmap_get_keys_gcallback (void);
static void _test_hashmap_get_values_gcallback (void);
static void _test_hashmap_clear_gcallback (void);
static void _main (char** args, int args_length1);
static int _vala_strcmp0 (const char * str1, const char * str2);



static void test_hashmap_get (void) {
	GeeHashMap* hashmap;
	char* _tmp0;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	char* _tmp4;
	char* _tmp5;
	hashmap = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check get from empty map*/
	_tmp0 = NULL;
	g_assert ((_tmp0 = ((char*) (gee_map_get (GEE_MAP (hashmap), "foo")))) == NULL);
	_tmp0 = (g_free (_tmp0), NULL);
	/* Check get from map with one items*/
	gee_map_set (GEE_MAP (hashmap), "key", "value");
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 ((_tmp1 = ((char*) (gee_map_get (GEE_MAP (hashmap), "key")))), "value") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	/* Check get from non-existing key*/
	_tmp2 = NULL;
	g_assert ((_tmp2 = ((char*) (gee_map_get (GEE_MAP (hashmap), "foo")))) == NULL);
	_tmp2 = (g_free (_tmp2), NULL);
	/* Check get from map with multiple items*/
	gee_map_set (GEE_MAP (hashmap), "key2", "value2");
	gee_map_set (GEE_MAP (hashmap), "key3", "value3");
	_tmp3 = NULL;
	g_assert (_vala_strcmp0 ((_tmp3 = ((char*) (gee_map_get (GEE_MAP (hashmap), "key")))), "value") == 0);
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp4 = NULL;
	g_assert (_vala_strcmp0 ((_tmp4 = ((char*) (gee_map_get (GEE_MAP (hashmap), "key2")))), "value2") == 0);
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp5 = NULL;
	g_assert (_vala_strcmp0 ((_tmp5 = ((char*) (gee_map_get (GEE_MAP (hashmap), "key3")))), "value3") == 0);
	_tmp5 = (g_free (_tmp5), NULL);
	(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
}


static void test_hashmap_set (void) {
	GeeHashMap* hashmap;
	char* _tmp0;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	char* _tmp4;
	hashmap = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_str_equal);
	/* check map is empty*/
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	/* check set an item to map*/
	gee_map_set (GEE_MAP (hashmap), "abc", "one");
	g_assert (gee_map_contains (GEE_MAP (hashmap), "abc"));
	_tmp0 = NULL;
	g_assert (_vala_strcmp0 ((_tmp0 = ((char*) (gee_map_get (GEE_MAP (hashmap), "abc")))), "one") == 0);
	_tmp0 = (g_free (_tmp0), NULL);
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 1);
	/* check set an item to map with same value*/
	gee_map_set (GEE_MAP (hashmap), "def", "one");
	g_assert (gee_map_contains (GEE_MAP (hashmap), "def"));
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 ((_tmp1 = ((char*) (gee_map_get (GEE_MAP (hashmap), "abc")))), "one") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	_tmp2 = NULL;
	g_assert (_vala_strcmp0 ((_tmp2 = ((char*) (gee_map_get (GEE_MAP (hashmap), "def")))), "one") == 0);
	_tmp2 = (g_free (_tmp2), NULL);
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 2);
	/* check set with same key*/
	gee_map_set (GEE_MAP (hashmap), "def", "two");
	g_assert (gee_map_contains (GEE_MAP (hashmap), "def"));
	_tmp3 = NULL;
	g_assert (_vala_strcmp0 ((_tmp3 = ((char*) (gee_map_get (GEE_MAP (hashmap), "abc")))), "one") == 0);
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp4 = NULL;
	g_assert (_vala_strcmp0 ((_tmp4 = ((char*) (gee_map_get (GEE_MAP (hashmap), "def")))), "two") == 0);
	_tmp4 = (g_free (_tmp4), NULL);
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 2);
	(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
}


static void test_hashmap_remove (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_str_equal);
	/* check removing when map is empty*/
	gee_map_remove (GEE_MAP (hashmap), "foo");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	/* add items*/
	gee_map_set (GEE_MAP (hashmap), "aaa", "111");
	gee_map_set (GEE_MAP (hashmap), "bbb", "222");
	gee_map_set (GEE_MAP (hashmap), "ccc", "333");
	gee_map_set (GEE_MAP (hashmap), "ddd", "444");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 4);
	/* check remove on first place*/
	gee_map_remove (GEE_MAP (hashmap), "aaa");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 3);
	/* check remove in between */
	gee_map_remove (GEE_MAP (hashmap), "ccc");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 2);
	/* check remove in last place*/
	gee_map_remove (GEE_MAP (hashmap), "ddd");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 1);
	/* check remove invalid key*/
	gee_map_remove (GEE_MAP (hashmap), "bar");
	/* check remove last in map*/
	gee_map_remove (GEE_MAP (hashmap), "bbb");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
}


static void test_hashmap_contains (void) {
	GeeHashMap* hashmap;
	char* _tmp0;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	char* _tmp4;
	char* _tmp5;
	GeeHashMap* hashmapOfInt;
	hashmap = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check on empty map*/
	g_assert (!gee_map_contains (GEE_MAP (hashmap), "111"));
	/* Check items*/
	gee_map_set (GEE_MAP (hashmap), "10", "111");
	g_assert (gee_map_contains (GEE_MAP (hashmap), "10"));
	g_assert (!gee_map_contains (GEE_MAP (hashmap), "20"));
	g_assert (!gee_map_contains (GEE_MAP (hashmap), "30"));
	_tmp0 = NULL;
	g_assert (_vala_strcmp0 ((_tmp0 = ((char*) (gee_map_get (GEE_MAP (hashmap), "10")))), "111") == 0);
	_tmp0 = (g_free (_tmp0), NULL);
	gee_map_set (GEE_MAP (hashmap), "20", "222");
	g_assert (gee_map_contains (GEE_MAP (hashmap), "10"));
	g_assert (gee_map_contains (GEE_MAP (hashmap), "20"));
	g_assert (!gee_map_contains (GEE_MAP (hashmap), "30"));
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 ((_tmp1 = ((char*) (gee_map_get (GEE_MAP (hashmap), "10")))), "111") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	_tmp2 = NULL;
	g_assert (_vala_strcmp0 ((_tmp2 = ((char*) (gee_map_get (GEE_MAP (hashmap), "20")))), "222") == 0);
	_tmp2 = (g_free (_tmp2), NULL);
	gee_map_set (GEE_MAP (hashmap), "30", "333");
	g_assert (gee_map_contains (GEE_MAP (hashmap), "10"));
	g_assert (gee_map_contains (GEE_MAP (hashmap), "20"));
	g_assert (gee_map_contains (GEE_MAP (hashmap), "30"));
	_tmp3 = NULL;
	g_assert (_vala_strcmp0 ((_tmp3 = ((char*) (gee_map_get (GEE_MAP (hashmap), "10")))), "111") == 0);
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp4 = NULL;
	g_assert (_vala_strcmp0 ((_tmp4 = ((char*) (gee_map_get (GEE_MAP (hashmap), "20")))), "222") == 0);
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp5 = NULL;
	g_assert (_vala_strcmp0 ((_tmp5 = ((char*) (gee_map_get (GEE_MAP (hashmap), "30")))), "333") == 0);
	_tmp5 = (g_free (_tmp5), NULL);
	/* Clear and recheck*/
	gee_map_clear (GEE_MAP (hashmap));
	g_assert (!gee_map_contains (GEE_MAP (hashmap), "10"));
	g_assert (!gee_map_contains (GEE_MAP (hashmap), "20"));
	g_assert (!gee_map_contains (GEE_MAP (hashmap), "30"));
	hashmapOfInt = gee_hash_map_new (G_TYPE_INT, NULL, NULL, G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal, g_direct_equal);
	/* Check items*/
	gee_map_set (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (10), GINT_TO_POINTER (111));
	g_assert (gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (10)));
	g_assert (!gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (20)));
	g_assert (!gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_map_get (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (10))) == 111);
	gee_map_set (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (20), GINT_TO_POINTER (222));
	g_assert (gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (10)));
	g_assert (gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (20)));
	g_assert (!gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_map_get (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (10))) == 111);
	g_assert (GPOINTER_TO_INT (gee_map_get (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (20))) == 222);
	gee_map_set (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (30), GINT_TO_POINTER (333));
	g_assert (gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (10)));
	g_assert (gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (20)));
	g_assert (gee_map_contains (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_map_get (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (10))) == 111);
	g_assert (GPOINTER_TO_INT (gee_map_get (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (20))) == 222);
	g_assert (GPOINTER_TO_INT (gee_map_get (GEE_MAP (hashmapOfInt), GINT_TO_POINTER (30))) == 333);
	(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
	(hashmapOfInt == NULL ? NULL : (hashmapOfInt = (g_object_unref (hashmapOfInt), NULL)));
}


static void test_hashmap_size (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check empty map*/
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	/* Check when one item*/
	gee_map_set (GEE_MAP (hashmap), "1", "1");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 1);
	/* Check when more items*/
	gee_map_set (GEE_MAP (hashmap), "2", "2");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 2);
	/* Check when items cleared*/
	gee_map_clear (GEE_MAP (hashmap));
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
}


static void test_hashmap_get_keys (void) {
	GeeHashMap* hashmap;
	GeeSet* keySet;
	GeeSet* _tmp0;
	GeeSet* _tmp1;
	GeeSet* _tmp2;
	hashmap = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check keys on empty map*/
	keySet = gee_map_get_keys (GEE_MAP (hashmap));
	g_assert (gee_collection_get_size (GEE_COLLECTION (keySet)) == 0);
	/* Check keys on map with one item*/
	gee_map_set (GEE_MAP (hashmap), "aaa", "111");
	g_assert (gee_collection_get_size (GEE_COLLECTION (keySet)) == 1);
	g_assert (gee_collection_contains (GEE_COLLECTION (keySet), "aaa"));
	_tmp0 = NULL;
	keySet = (_tmp0 = gee_map_get_keys (GEE_MAP (hashmap)), (keySet == NULL ? NULL : (keySet = (g_object_unref (keySet), NULL))), _tmp0);
	g_assert (gee_collection_get_size (GEE_COLLECTION (keySet)) == 1);
	g_assert (gee_collection_contains (GEE_COLLECTION (keySet), "aaa"));
	/* Check modify key set directly*/
	if (g_test_trap_fork (((guint64) (0)), G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_collection_add (GEE_COLLECTION (keySet), "ccc");
		(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
		(keySet == NULL ? NULL : (keySet = (g_object_unref (keySet), NULL)));
		return;
	}
	g_test_trap_assert_failed ();
	g_test_trap_assert_stderr (CODE_NOT_REACHABLE);
	/* Check keys on map with multiple items*/
	gee_map_set (GEE_MAP (hashmap), "bbb", "222");
	g_assert (gee_collection_get_size (GEE_COLLECTION (keySet)) == 2);
	g_assert (gee_collection_contains (GEE_COLLECTION (keySet), "aaa"));
	g_assert (gee_collection_contains (GEE_COLLECTION (keySet), "bbb"));
	_tmp1 = NULL;
	keySet = (_tmp1 = gee_map_get_keys (GEE_MAP (hashmap)), (keySet == NULL ? NULL : (keySet = (g_object_unref (keySet), NULL))), _tmp1);
	g_assert (gee_collection_get_size (GEE_COLLECTION (keySet)) == 2);
	g_assert (gee_collection_contains (GEE_COLLECTION (keySet), "aaa"));
	g_assert (gee_collection_contains (GEE_COLLECTION (keySet), "bbb"));
	/* Check keys on map clear*/
	gee_map_clear (GEE_MAP (hashmap));
	g_assert (gee_collection_get_size (GEE_COLLECTION (keySet)) == 0);
	_tmp2 = NULL;
	keySet = (_tmp2 = gee_map_get_keys (GEE_MAP (hashmap)), (keySet == NULL ? NULL : (keySet = (g_object_unref (keySet), NULL))), _tmp2);
	g_assert (gee_collection_get_size (GEE_COLLECTION (keySet)) == 0);
	(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
	(keySet == NULL ? NULL : (keySet = (g_object_unref (keySet), NULL)));
}


static void test_hashmap_get_values (void) {
	GeeHashMap* hashmap;
	GeeCollection* valueCollection;
	GeeCollection* _tmp0;
	GeeCollection* _tmp1;
	GeeCollection* _tmp2;
	hashmap = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check keys on empty map*/
	valueCollection = gee_map_get_values (GEE_MAP (hashmap));
	g_assert (gee_collection_get_size (valueCollection) == 0);
	/* Check keys on map with one item*/
	gee_map_set (GEE_MAP (hashmap), "aaa", "111");
	g_assert (gee_collection_get_size (valueCollection) == 1);
	g_assert (gee_collection_contains (valueCollection, "111"));
	_tmp0 = NULL;
	valueCollection = (_tmp0 = gee_map_get_values (GEE_MAP (hashmap)), (valueCollection == NULL ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL))), _tmp0);
	g_assert (gee_collection_get_size (valueCollection) == 1);
	g_assert (gee_collection_contains (valueCollection, "111"));
	/* Check modify key set directly*/
	if (g_test_trap_fork (((guint64) (0)), G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_collection_add (valueCollection, "ccc");
		(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
		(valueCollection == NULL ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL)));
		return;
	}
	g_test_trap_assert_failed ();
	g_test_trap_assert_stderr (CODE_NOT_REACHABLE);
	/* Check keys on map with multiple items*/
	gee_map_set (GEE_MAP (hashmap), "bbb", "222");
	g_assert (gee_collection_get_size (valueCollection) == 2);
	g_assert (gee_collection_contains (valueCollection, "111"));
	g_assert (gee_collection_contains (valueCollection, "222"));
	_tmp1 = NULL;
	valueCollection = (_tmp1 = gee_map_get_values (GEE_MAP (hashmap)), (valueCollection == NULL ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL))), _tmp1);
	g_assert (gee_collection_get_size (valueCollection) == 2);
	g_assert (gee_collection_contains (valueCollection, "111"));
	g_assert (gee_collection_contains (valueCollection, "222"));
	/* Check keys on map clear*/
	gee_map_clear (GEE_MAP (hashmap));
	g_assert (gee_collection_get_size (valueCollection) == 0);
	_tmp2 = NULL;
	valueCollection = (_tmp2 = gee_map_get_values (GEE_MAP (hashmap)), (valueCollection == NULL ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL))), _tmp2);
	g_assert (gee_collection_get_size (valueCollection) == 0);
	(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
	(valueCollection == NULL ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL)));
}


static void test_hashmap_clear (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_str_equal);
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	/* Check clear on empty map*/
	gee_map_clear (GEE_MAP (hashmap));
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	/* Check clear one item*/
	gee_map_set (GEE_MAP (hashmap), "1", "1");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 1);
	gee_map_clear (GEE_MAP (hashmap));
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	/* Check clear multiple items*/
	gee_map_set (GEE_MAP (hashmap), "1", "1");
	gee_map_set (GEE_MAP (hashmap), "2", "2");
	gee_map_set (GEE_MAP (hashmap), "3", "3");
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 3);
	gee_map_clear (GEE_MAP (hashmap));
	g_assert (gee_map_get_size (GEE_MAP (hashmap)) == 0);
	(hashmap == NULL ? NULL : (hashmap = (g_object_unref (hashmap), NULL)));
}


static void _test_hashmap_get_gcallback (void) {
	test_hashmap_get ();
}


static void _test_hashmap_set_gcallback (void) {
	test_hashmap_set ();
}


static void _test_hashmap_remove_gcallback (void) {
	test_hashmap_remove ();
}


static void _test_hashmap_contains_gcallback (void) {
	test_hashmap_contains ();
}


static void _test_hashmap_size_gcallback (void) {
	test_hashmap_size ();
}


static void _test_hashmap_get_keys_gcallback (void) {
	test_hashmap_get_keys ();
}


static void _test_hashmap_get_values_gcallback (void) {
	test_hashmap_get_values ();
}


static void _test_hashmap_clear_gcallback (void) {
	test_hashmap_clear ();
}


static void _main (char** args, int args_length1) {
	g_test_init (&args_length1, &args, NULL);
	g_test_add_func ("/HashMap/Map/get", _test_hashmap_get_gcallback);
	g_test_add_func ("/HashMap/Map/set", _test_hashmap_set_gcallback);
	g_test_add_func ("/HashMap/Map/remove", _test_hashmap_remove_gcallback);
	g_test_add_func ("/HashMap/Map/contains", _test_hashmap_contains_gcallback);
	g_test_add_func ("/HashMap/Map/size", _test_hashmap_size_gcallback);
	g_test_add_func ("/HashMap/Map/get_keys", _test_hashmap_get_keys_gcallback);
	g_test_add_func ("/HashMap/Map/get_values", _test_hashmap_get_values_gcallback);
	g_test_add_func ("/HashMap/Map/clear", _test_hashmap_clear_gcallback);
	g_test_run ();
}


int main (int argc, char ** argv) {
	g_type_init ();
	_main (argv, argc);
	return 0;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




