/* GDA - SQL console
 * Copyright (C) 2007 The GNOME Foundation.
 *
 * AUTHORS:
 * 	Vivien Malerba <malerba@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include <glib/gprintf.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <glib/gstdio.h>
#include "tools-input.h"
#include "command-exec.h"
#include <unistd.h>
#include <sys/types.h>

#ifndef G_OS_WIN32
#include <signal.h>
#include <pwd.h>
#endif

/* options */
gchar *pass = NULL;
gchar *user = NULL;

gchar *dsn = NULL;
gchar *direct = NULL;
gchar *prov = NULL;

gchar *single_command = NULL;
gchar *commandsfile = NULL;

gboolean list_configs = FALSE;
gboolean list_providers = FALSE;

gchar *outfile = NULL;


static GOptionEntry entries[] = {
        { "cnc", 'c', 0, G_OPTION_ARG_STRING, &direct, "Direct connection string", NULL},
        { "provider", 'p', 0, G_OPTION_ARG_STRING, &prov, "Provider name", NULL},
        { "dsn", 's', 0, G_OPTION_ARG_STRING, &dsn, "Data source", NULL},
        { "user", 'U', 0, G_OPTION_ARG_STRING, &user, "Username", "username" },
        { "password", 'P', 0, G_OPTION_ARG_STRING, &pass, "Password", "password" },

        { "output-file", 'o', 0, G_OPTION_ARG_STRING, &outfile, "Output file", "output file"},
        { "command", 'C', 0, G_OPTION_ARG_STRING, &single_command, "Run only single command (SQL or internal) and exit", "command" },
        { "commands-file", 'f', 0, G_OPTION_ARG_STRING, &commandsfile, "Execute commands from file, then exit", "filename" },
        { "list-dsn", 'l', 0, G_OPTION_ARG_NONE, &list_configs, "List configured data sources and exit", NULL },
        { "list-providers", 'L', 0, G_OPTION_ARG_NONE, &list_providers, "List installed database providers and exit", NULL },
        { NULL }
};

/* interruption handling */
#ifndef G_OS_WIN32
struct sigaction old_sigint_handler; 
#endif
static void sigint_handler (int sig_num);
static void setup_sigint_handler (void);
typedef enum {
	SIGINT_HANDLER_DISABLED = 0,
	SIGINT_HANDLER_PARTIAL_COMMAND,
} SigintHandlerCode;
static SigintHandlerCode sigint_handler_status = SIGINT_HANDLER_DISABLED;

typedef enum {
	OUTPUT_FORMAT_DEFAULT = 0,
	OUTPUT_FORMAT_HTML,
	OUTPUT_FORMAT_XML
} OutputFormat;

/* structure to hold program's data */
typedef struct {
	GdaClient *client;
	GdaConnection *cnc;
	GdaDict *dict;
	GdaInternalCommandsList *internal_commands;

	FILE *input_stream;
	FILE *output_stream;
	gboolean output_is_pipe;
	OutputFormat output_format;

	GString *partial_command;
	GString *query_buffer;

	GHashTable *parameters; /* key = name, value = G_TYPE_STRING GdaParameter */
} MainData;
MainData *main_data;
GString *prompt = NULL;

static gchar   *read_a_line (MainData *data);
static void     compute_prompt (MainData *data, GString *string, gboolean in_command);
static gboolean set_output_file (MainData *data, const gchar *file, GError **error);
static gboolean set_input_file (MainData *data, const gchar *file, GError **error);
static void     output_data_model (MainData *data, GdaDataModel *model);
static gboolean open_connection (MainData *data, const gchar *dsn,
				 const gchar *provider,  const gchar *direct, 
				 const gchar *user, const gchar *pass, GError **error);
static GdaDataModel *list_all_sections (MainData *data);
static GdaDataModel *list_all_providers (MainData *data);

/* commands manipulation */
static GdaInternalCommandsList  *build_internal_commands_list (MainData *data);
static gboolean                  command_is_complete (const gchar *command);
static GdaInternalCommandResult *command_execute (MainData *data, gchar *command, GError **error);

int
main (int argc, char *argv[])
{
	GOptionContext *context;
	GError *error = NULL;
	MainData *data;
	int exit_status = EXIT_SUCCESS;
	prompt = g_string_new ("");

	context = g_option_context_new (_("Gda SQL console"));        g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);
        if (!g_option_context_parse (context, &argc, &argv, &error)) {
                g_warning ("Can't parse arguments: %s", error->message);
		exit_status = EXIT_FAILURE;
		goto cleanup;
        }
        g_option_context_free (context);
        gda_init ("Gda SQL console", PACKAGE_VERSION, argc, argv);
	data = g_new0 (MainData, 1);
	data->parameters = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
	main_data = data;

	/* output file */
	if (outfile) {
		if (! set_output_file (data, outfile, &error)) {
			g_warning ("Can't set output file as '%s': %s", outfile,
				   error->message);
			exit_status = EXIT_FAILURE;
			goto cleanup;
		}
	}

	/* treat here lists of providers and defined DSN */
	if (list_providers) {
		GdaDataModel *model = list_all_providers (data);
		output_data_model (data, model);
		g_object_unref (model);
		goto cleanup;
	}
	if (list_configs) {
		GdaDataModel *model = list_all_sections (data);
		output_data_model (data, model);
		g_object_unref (model);
		goto cleanup;
	}

	/* commands file */
	if (commandsfile) {
		if (! set_input_file (data, commandsfile, &error)) {
			g_warning ("Can't read file '%s': %s", commandsfile,
				   error->message);
			exit_status = EXIT_FAILURE;
			goto cleanup;
		}
	}
	else {
		/* check if stdin is a term */
		if (!isatty (fileno (stdin))) 
			data->input_stream = stdin;
	}

	/* check connection parameters coherence */
	if (direct && dsn) {
                g_fprintf (stderr, _("DSN and connection string are exclusive\n"));
		exit_status = EXIT_FAILURE;
                goto cleanup;
        }

        if (!direct && !dsn) {
                g_fprintf (stderr, 
			   _("You must specify a connection to open either as a DSN or a connection string\n"));
                exit_status = EXIT_FAILURE;
                goto cleanup;
        }

        if (direct && !prov) {
                g_fprintf (stderr, _("You must specify a provider when using a connection string\n"));
                exit_status = EXIT_FAILURE;
                goto cleanup;
        }

	/* welcome message */
	if (!data->output_stream) {
		g_print (_("Welcome to the GDA SQL console, version " PACKAGE_VERSION));
		g_print ("\n\n");
		g_print (_("Type: \\copyright to show usage and distribution terms\n"
			   "      \\? for help with internal commands\n"
			   "      \\q (or CTRL-D) to quit\n"
			   "      or any query terminated by a semicolon\n\n"));
	}

	/* open connection */
	if (!open_connection (data, dsn, prov, direct, user, pass, &error)) {
		g_warning (_("Can't open connection: %s"), error && error->message ? error->message : _("No detail"));
		exit_status = EXIT_FAILURE;
                goto cleanup;
	}

	/* build internal command s list */
	data->internal_commands = build_internal_commands_list (data);

	/* loop over commands */
	setup_sigint_handler ();
	init_input ();
	init_history ();
	for (;;) {
		gchar *cmde = read_a_line (data);

		if (!cmde) {
			save_history (NULL, NULL);
			if (!data->output_stream)
				g_print ("\n");
			goto cleanup;
		}

		g_strchug (cmde);
		if (*cmde) {
			if (!data->partial_command) {
				/* enable SIGINT handling */
				sigint_handler_status = SIGINT_HANDLER_PARTIAL_COMMAND;
				data->partial_command = g_string_new (cmde);
			}
			else {
				g_string_append_c (data->partial_command, ' ');
				g_string_append (data->partial_command, cmde);
			}
			if (command_is_complete (data->partial_command->str)) {
				/* execute command */
				GdaInternalCommandResult *res;
				FILE *to_stream;

				if (*data->partial_command->str != '\\') {
					if (!data->query_buffer)
						data->query_buffer = g_string_new ("");
					g_string_assign (data->query_buffer, data->partial_command->str);
				}

				if (data && data->output_stream)
					to_stream = data->output_stream;
				else
					to_stream = stdout;
				res = command_execute (data, data->partial_command->str, &error);
				
				if (!res) {
					g_fprintf (to_stream,
						   "ERROR: %s\n", 
						   error && error->message ? error->message : _("No detail"));
					if (error) {
						g_error_free (error);
						error = NULL;
					}
				}
				else {
					switch (res->type) {
					case GDA_INTERNAL_COMMAND_RESULT_DATA_MODEL:
						output_data_model (data, res->u.model);
						break;
					case GDA_INTERNAL_COMMAND_RESULT_PLIST: {
						GSList *list;
						GString *string;
						xmlNodePtr node;
						xmlBufferPtr buffer;
						switch (data->output_format) {
						case OUTPUT_FORMAT_DEFAULT:
							string = g_string_new ("");
							for (list = res->u.plist->parameters; list; list = list->next) {
								gchar *str;
								const GValue *value;
								value = gda_parameter_get_value (GDA_PARAMETER (list->data));
								str = gda_value_stringify ((GValue *) value);
								g_string_append_printf (string, "%s => %s\n",
											gda_object_get_name (GDA_OBJECT (list->data)), str);
								g_free (str);
							}
							g_fprintf (to_stream, "%s", string->str);
							g_string_free (string, TRUE);
							break;
						case OUTPUT_FORMAT_XML:
							buffer = xmlBufferCreate ();
							node = xmlNewNode (NULL, BAD_CAST "parameters");
							for (list = res->u.plist->parameters; list; list = list->next) {
								const GValue *value;
								xmlNodePtr pnode, vnode;
								
								pnode = xmlNewNode (NULL, BAD_CAST "parameter");
								xmlAddChild (node, pnode);
								xmlSetProp (pnode, BAD_CAST "name", 
									    gda_object_get_name (GDA_OBJECT (list->data)));
								value = gda_parameter_get_value (GDA_PARAMETER (list->data));
								vnode = gda_value_to_xml (value);
								xmlAddChild (pnode, vnode);
							}
							xmlNodeDump (buffer, NULL, node, 0, 1);
							xmlBufferDump (to_stream, buffer);
							xmlBufferFree (buffer);
							g_fprintf (to_stream, "\n");
							break;
						case OUTPUT_FORMAT_HTML:
						default:
							TO_IMPLEMENT;
							break;
						}
						break;
					}
					case GDA_INTERNAL_COMMAND_RESULT_TXT: {
						xmlNodePtr node;
						xmlBufferPtr buffer;
						switch (data->output_format) {
						case OUTPUT_FORMAT_DEFAULT:
							g_fprintf (to_stream, "%s", res->u.txt->str);
							break;
						case OUTPUT_FORMAT_XML:
							buffer = xmlBufferCreate ();
							node = xmlNewNode (NULL, BAD_CAST "txt");
							xmlNodeSetContent (node, res->u.txt->str);
							xmlNodeDump (buffer, NULL, node, 0, 1);
							xmlBufferDump (to_stream, buffer);
							xmlBufferFree (buffer);
							break;
						case OUTPUT_FORMAT_HTML:
						default:
							TO_IMPLEMENT;
							break;
						}
						break;
					}
					case GDA_INTERNAL_COMMAND_RESULT_TXT_STDOUT: 
						g_print ("%s", res->u.txt->str);
						break;
					case GDA_INTERNAL_COMMAND_RESULT_EMPTY:
						break;
					case GDA_INTERNAL_COMMAND_RESULT_EXIT:
						goto cleanup;
					default:
						TO_IMPLEMENT;
						break;
					}
					gda_internal_command_exec_result_free (res);
				}
				g_string_free (data->partial_command, TRUE);
				data->partial_command = NULL;
				
				/* disable SIGINT handling */
				sigint_handler_status = SIGINT_HANDLER_DISABLED;
			}
		}
		g_free (cmde);
	}
	
	/* cleanups */
 cleanup:
	if (data->cnc)
		g_object_unref (data->cnc);
	if (data->client)
		g_object_unref (data->client);
	set_input_file (data, NULL, NULL); 
	set_output_file (data, NULL, NULL); 

	g_free (data);

	return EXIT_SUCCESS;
}

/*
 * SIGINT handling
 */

static void 
setup_sigint_handler (void) 
{
#ifndef G_OS_WIN32
	struct sigaction sac;
	memset (&sac, 0, sizeof (sac));
	sigemptyset (&sac.sa_mask);
	sac.sa_handler = sigint_handler;
	sac.sa_flags = SA_RESTART;
	sigaction (SIGINT, &sac, &old_sigint_handler);
#endif
}

#ifndef G_OS_WIN32
static void
sigint_handler (int sig_num)
{
	if (sigint_handler_status == SIGINT_HANDLER_PARTIAL_COMMAND) {
		if (main_data->partial_command) {
			g_string_free (main_data->partial_command, TRUE);
			main_data->partial_command = NULL;
		}
		/* show a new prompt */
		compute_prompt (main_data, prompt, main_data->partial_command == NULL ? FALSE : TRUE);
		g_print ("\n%s", prompt->str);
	}
	else {
		g_print ("\n");
		exit (EXIT_SUCCESS);
	}

	if (old_sigint_handler.sa_handler)
		old_sigint_handler.sa_handler (sig_num);
}
#endif

/*
 * read_a_line
 *
 * Read a line to be processed
 */
static gchar *read_a_line (MainData *data)
{
	gchar *cmde;

	compute_prompt (data, prompt, data->partial_command == NULL ? FALSE : TRUE);
	if (data->input_stream) {
		cmde = input_from_stream (data->input_stream);
		if (!cmde && !commandsfile && isatty (fileno (stdin))) {
			/* go back to console after file is over */
			set_input_file (data, NULL, NULL);
			cmde = input_from_console (prompt->str);
		}
	}
	else
		cmde = input_from_console (prompt->str);

	return cmde;
}

/*
 * command_is_complete
 *
 * Checks if @command can be executed, or if more data is required
 */
static gboolean
command_is_complete (const gchar *command)
{
	if (!command || !(*command))
		return FALSE;
	if (*command == '\\') {
		/* internal command */
		return TRUE;
	}
	else {
		if (command [strlen (command) - 1] == ';')
			return TRUE;
		else
			return FALSE;
	}
}


/*
 * command_execute
 */
static GdaInternalCommandResult *execute_external_command (MainData *data, const gchar *command, GError **error);
static GdaInternalCommandResult *
command_execute (MainData *data, gchar *command, GError **error)
{
	if (!command || !(*command))
		return NULL;
	if (*command == '\\') 
		return gda_internal_command_execute (data->internal_commands, 
						     data->cnc, data->dict, command, error);
	else {
		if (!data->cnc) {
			g_set_error (error, 0, 0, 
				     _("No connection specified"));
			return NULL;
		}
		if (!gda_connection_is_opened (data->cnc)) {
			g_set_error (error, 0, 0, 
				     _("Connection closed"));
			return NULL;
		}
			
		return execute_external_command (data, command, error);
	}
}

/*
 * execute_external_command
 *
 * Executes an SQL statement as understood by the DBMS
 */
static GdaInternalCommandResult *
execute_external_command (MainData *data, const gchar *command, GError **error)
{
	GdaInternalCommandResult *res = NULL;
	GdaQuery *query;
	GdaParameterList *plist;
	GdaObject *obj;

	res = g_new0 (GdaInternalCommandResult, 1);
	query = gda_query_new_from_sql (data->dict, command, NULL);
	plist = gda_query_get_parameter_list (query);
	if (plist && plist->parameters) {
		GSList *params;
		/* fill parameters with some defined parameters */
		for (params = plist->parameters; params; params = params->next) {
			GdaParameter *p_to_set = GDA_PARAMETER (params->data);
			const gchar *pname = gda_object_get_name (GDA_OBJECT (p_to_set));
			GdaParameter *p_in_data = g_hash_table_lookup (data->parameters, pname);
			if (p_in_data) {
				gchar *str;
				str = gda_parameter_get_value_str (p_in_data);
				gda_parameter_set_value_str (p_to_set, str);
				g_free (str);
			}
			else {
				if (! gda_parameter_is_valid (p_to_set)) {
					g_set_error (error, 0, 0,
						     _("No internal parameter named '%s' required by query"), pname);
					g_free (res);
					res = NULL;
					goto cleanup;
				}
			}
		}
	}
	obj = gda_query_execute (query, plist, FALSE, error);
	if (!obj) {
		g_free (res);
		res = NULL;
	}
	else {
		if (GDA_IS_DATA_MODEL (obj)) {
			res->type = GDA_INTERNAL_COMMAND_RESULT_DATA_MODEL;
			res->u.model = GDA_DATA_MODEL (obj);
		}
		else if (GDA_IS_PARAMETER_LIST (obj)) {
			res->type = GDA_INTERNAL_COMMAND_RESULT_PLIST;
			res->u.plist = GDA_PARAMETER_LIST (obj);
		}
		else
			g_assert_not_reached ();
	}

 cleanup:
	g_object_unref (query);
	if (plist)
		g_object_unref (plist);

	return res;
}



static void
compute_prompt (MainData *data, GString *string, gboolean in_command)
{
	g_assert (string);
	if (in_command)
		g_string_assign (string, "   > ");
	else
		g_string_assign (string, "gda> ");
}

/*
 * Change the output file, set to %NULL to be back on stdout
 */
static gboolean
set_output_file (MainData *data, const gchar *file, GError **error)
{
	if (data->output_stream) {
		if (data->output_is_pipe) {
			pclose (data->output_stream);
#ifndef G_OS_WIN32
			signal (SIGPIPE, SIG_DFL);
#endif
		}
		else
			fclose (data->output_stream);
		data->output_stream = NULL;
		data->output_is_pipe = FALSE;
	}

	if (file) {
		gchar *copy = g_strdup (file);
		g_strchug (copy);

		if (*copy != '|') {
			/* output to a file */
			data->output_stream = g_fopen (copy, "w");
			if (!data->output_stream) {
				g_set_error (error, 0, 0,
					     _("Can't open file '%s' for writing: %s\n"), 
					     copy,
					     strerror (errno));
				g_free (copy);
				return FALSE;
			}
			data->output_is_pipe = FALSE;
		}
		else {
			/* output to a pipe */
			data->output_stream = popen (copy+1, "w");
			if (!data->output_stream) {
				g_set_error (error, 0, 0,
					     _("Can't open pipe '%s': %s\n"), 
					     copy,
					     strerror (errno));
				g_free (copy);
				return FALSE;
			}
#ifndef G_OS_WIN32
			signal (SIGPIPE, SIG_IGN);
#endif
			data->output_is_pipe = TRUE;
		}
		g_free (copy);
	}

	return TRUE;
}

/*
 * Change the input file, set to %NULL to be back on stdin
 */
static gboolean
set_input_file (MainData *data, const gchar *file, GError **error)
{
	if (data->input_stream) {
		fclose (data->input_stream);
		data->input_stream = NULL;
	}

	if (file) {
		data->input_stream = g_fopen (file, "r");
		if (!data->input_stream) {
			g_set_error (error, 0, 0,
				     _("Can't open file '%s' for reading: %s\n"), 
				     file,
				     strerror (errno));
			return FALSE;
		}
	}

	return TRUE;
}


/*
 * Opens connection
 */
static gboolean
open_connection (MainData *data, const gchar *dsn, const gchar *provider, const gchar *direct, 
		 const gchar *user, const gchar *pass,
		 GError **error)
{
	GdaConnection *newcnc = NULL;

	if (!data->client)
		data->client = gda_client_new ();
	
	if (dsn) {
                GdaDataSourceInfo *info = NULL;
                info = gda_config_find_data_source (dsn);
                if (!info)
                        g_set_error (error, 0, 0,
				     _("DSN '%s' is not declared"), dsn);
                else {
                        newcnc = gda_client_open_connection (data->client, info->name,
							     user ? user : info->username,
							     pass ? pass : ((info->password) ? info->password : ""),
							     0, error);
                        gda_data_source_info_free (info);
                }
        }
        else {
		newcnc = gda_client_open_connection_from_string (data->client, provider, direct,
								 user, pass, 0, error);
        }

	/* if there is a new connection, then get rid of the older connection, and replace dictionary
	 * as well
	 */
	if (newcnc) {
		if (data->dict) {
			gda_dict_save (data->dict, NULL);
			g_object_unref (data->dict);
		}
		if (data->cnc)
			g_object_unref (data->cnc);
		data->cnc = newcnc;

		/* compute new GdaDict object */
		GError *lerror = NULL;

		data->dict = gda_dict_new ();
		gda_dict_set_connection (data->dict, data->cnc);
		if (dsn) {
			gchar *filename;
			filename = gda_dict_compute_xml_filename (data->dict, dsn, NULL, &lerror);
			if (!filename)
				g_fprintf (stderr, 
					   _("Could not compute output XML file name: %s\n"), 
					   lerror && lerror->message ? lerror->message: _("No detail"));
			else {
				gda_dict_set_xml_filename (data->dict, filename);
				g_free (filename);
				if (!gda_dict_load (data->dict, NULL)) {
					if (!data->output_stream) {
						g_print (_("Synchronizing internal dictionary with database, "
							   "this may take some time..."));
						fflush (stdout);
					}
					if (!gda_dict_update_dbms_meta_data (data->dict, 0, NULL, &lerror))
						g_fprintf (stderr, _("Could synchronize dictionary with DBMS: %s\n"),
							   lerror && lerror->message ? lerror->message: _("No detail"));
					else
						if (!data->output_stream) 
							g_print (_(" Done.\n"));
				}
			}
		}
		if (lerror) {
			g_error_free (lerror);
			lerror = NULL;
		}
	}

	return newcnc ? TRUE : FALSE;
}

/* 
 * Dumps the data model contents onto @data->output
 */
static void
output_data_model (MainData *data, GdaDataModel *model)
{
	gchar *str;
	FILE *to_stream;
	gint rows, cols, model_rows;

	model_rows = gda_data_model_get_n_rows (model);

	if (data && data->output_stream)
		to_stream = data->output_stream;
	else
		to_stream = stdout;
	input_get_size (&cols, &rows);

	if (isatty (fileno (to_stream))) {
		/* use pager */
		FILE *pipe;
		const char *pager;
		
		pager = getenv ("PAGER");
		if (!pager)
			pager = "more";
		pipe = popen (pager, "w");
#ifndef G_OS_WIN32
		signal (SIGPIPE, SIG_IGN);
#endif
		switch (data->output_format) {
		case OUTPUT_FORMAT_DEFAULT:
			str = gda_data_model_dump_as_string (model);
			break;
		case OUTPUT_FORMAT_XML:
			str = gda_data_model_export_to_string (model, GDA_DATA_MODEL_IO_DATA_ARRAY_XML,
							       NULL, 0,
							       NULL, 0, NULL);
			break;
		case OUTPUT_FORMAT_HTML:
		default:
			str = g_strdup ("");
			TO_IMPLEMENT;
			break;
		}
		g_fprintf (pipe, "%s", str);
		g_free (str);
		pclose (pipe);
#ifndef G_OS_WIN32
		signal(SIGPIPE, SIG_DFL);
#endif
	}
	else {
		switch (data->output_format) {
		case OUTPUT_FORMAT_DEFAULT:
			str = gda_data_model_dump_as_string (model);
			break;
		case OUTPUT_FORMAT_XML:
			str = gda_data_model_export_to_string (model, GDA_DATA_MODEL_IO_DATA_ARRAY_XML,
							       NULL, 0,
							       NULL, 0, NULL);
			break;
		case OUTPUT_FORMAT_HTML:
		default:
			str = g_strdup ("");
			TO_IMPLEMENT;
			break;
		}
		g_fprintf (to_stream, "%s", str);
		g_free (str);
	}
}

/*
 * Lists all the sections in the config files (local to the user and global) in the index page
 */
static GdaDataModel *
list_all_sections (MainData *data)
{
        GList *sections;
        GList *l;
	GdaDataModel *model;

	model = gda_data_model_array_new_with_g_types (5,
						       G_TYPE_STRING,
						       G_TYPE_STRING,
						       G_TYPE_STRING,
						       G_TYPE_STRING,
						       G_TYPE_STRING);
	gda_data_model_set_column_title (model, 0, _("DSN"));
	gda_data_model_set_column_title (model, 1, _("Provider"));
	gda_data_model_set_column_title (model, 2, _("Description"));
	gda_data_model_set_column_title (model, 3, _("Connection string"));
	gda_data_model_set_column_title (model, 4, _("Username"));

        sections = gda_config_list_sections ("/apps/libgda/Datasources");
        for (l = sections; l; l = l->next) {
                gchar *section = (gchar *) l->data;
		GValue *value;
		gint row;

		row = gda_data_model_append_row (model, NULL);
		value = gda_value_new_from_string (section, G_TYPE_STRING);
		gda_data_model_set_value_at (model, 0, row, value, NULL);
		gda_value_free (value);

		GList *keys;
		gchar *root;
		
		/* list keys in dsn */ 
		root = g_strdup_printf ("/apps/libgda/Datasources/%s", section);
		keys = gda_config_list_keys (root);
		if (keys) {
			GList *l;
			gint col;

			for (l = keys; l ; l = l->next) {
				gchar *str, *tmp;
				gchar *key = (gchar *) l->data;
				
				tmp = g_strdup_printf ("%s/%s", root, key);
				str = gda_config_get_string (tmp);
				value =  gda_value_new_from_string (str, G_TYPE_STRING);
				g_free (tmp);
				
				col = -1;
				if (!strcmp (key, "DSN"))
					col = 3;
				else if (!strcmp (key, "Description"))
					col = 2;
				else if (!strcmp (key, "Username"))
					col = 4;
				else if (!strcmp (key, "Provider"))
					col = 1;
				if (col >= 0)
					gda_data_model_set_value_at (model, col, row, value, NULL);
				gda_value_free (value);
			}
			gda_config_free_list (keys);
		}
		g_free (root);
        }
        gda_config_free_list (sections);

	return model;
}

/*
 * make a list of all the providers in the index page
 */
static GdaDataModel *
list_all_providers (MainData *data)
{
        GList *providers;
	GdaDataModel *model;

	model = gda_data_model_array_new_with_g_types (4,
						       G_TYPE_STRING,
						       G_TYPE_STRING,
						       G_TYPE_STRING,
						       G_TYPE_STRING);
	gda_data_model_set_column_title (model, 0, _("Provider"));
	gda_data_model_set_column_title (model, 1, _("Description"));
	gda_data_model_set_column_title (model, 2, _("DSN parameters"));
	gda_data_model_set_column_title (model, 3, _("File"));

        for (providers = gda_config_get_provider_list (); providers; providers = providers->next) {
                GdaProviderInfo *info = (GdaProviderInfo *) providers->data;

		GValue *value;
		gint row;

		row = gda_data_model_append_row (model, NULL);

		value = gda_value_new_from_string( info->id, G_TYPE_STRING);
		gda_data_model_set_value_at (model, 0, row, value, NULL);
		gda_value_free (value);

		value = gda_value_new_from_string (info->description, G_TYPE_STRING);
		gda_data_model_set_value_at (model, 1, row, value, NULL);
		gda_value_free (value);

		if (info->gda_params) {
			GSList *params;
			GString *string = g_string_new ("");
			for (params = info->gda_params->parameters; 
			     params; params = params->next) {
				gchar *tmp;
				
				g_object_get (G_OBJECT (params->data), "string_id", &tmp, NULL);
				if (params != info->gda_params->parameters)
					g_string_append (string, ", ");
				g_string_append (string, tmp);
				g_free (tmp);
			}
			value = gda_value_new_from_string (string->str, G_TYPE_STRING);
			g_string_free (string, TRUE);
			gda_data_model_set_value_at (model, 2, row, value, NULL);
			gda_value_free (value);
		}

		value = gda_value_new_from_string (info->location, G_TYPE_STRING);
		gda_data_model_set_value_at (model, 3, row, value, NULL);
		gda_value_free (value);
        }

	return model;
}

static gchar **args_as_string_func (const gchar *str);
static gchar **args_as_string_set (const gchar *str);

static GdaInternalCommandResult *extra_command_copyright (GdaConnection *cnc, GdaDict *dict, 
							  const gchar **args,
							  GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_quit (GdaConnection *cnc, GdaDict *dict, 
						     const gchar **args,
						     GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_cd (GdaConnection *cnc, GdaDict *dict, 
						   const gchar **args,
						   GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_set_output (GdaConnection *cnc, GdaDict *dict, 
							   const gchar **args,
							   GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_set_output_format (GdaConnection *cnc, GdaDict *dict, 
								  const gchar **args,
								  GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_set_input (GdaConnection *cnc, GdaDict *dict, 
							  const gchar **args,
							  GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_echo (GdaConnection *cnc, GdaDict *dict, 
						     const gchar **args,
						     GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_qecho (GdaConnection *cnc, GdaDict *dict, 
						      const gchar **args,
						      GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_list_dsn (GdaConnection *cnc, GdaDict *dict, 
							 const gchar **args,
							 GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_list_providers (GdaConnection *cnc, GdaDict *dict, 
							       const gchar **args,
							       GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_change_cnc_dsn (GdaConnection *cnc, GdaDict *dict, 
							       const gchar **args,
							       GError **error, MainData *data);

static GdaInternalCommandResult *extra_command_edit_buffer (GdaConnection *cnc, GdaDict *dict, 
							    const gchar **args,
							    GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_reset_buffer (GdaConnection *cnc, GdaDict *dict, 
							     const gchar **args,
							     GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_show_buffer (GdaConnection *cnc, GdaDict *dict, 
							    const gchar **args,
							    GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_exec_buffer (GdaConnection *cnc, GdaDict *dict, 
							    const gchar **args,
							    GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_write_buffer (GdaConnection *cnc, GdaDict *dict, 
							     const gchar **args,
							     GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_query_buffer_to_dict (GdaConnection *cnc, GdaDict *dict, 
								     const gchar **args,
								     GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_query_buffer_from_dict (GdaConnection *cnc, GdaDict *dict, 
								       const gchar **args,
								       GError **error, MainData *data);

static GdaInternalCommandResult *extra_command_set (GdaConnection *cnc, GdaDict *dict, 
						    const gchar **args,
						    GError **error, MainData *data);
static GdaInternalCommandResult *extra_command_unset (GdaConnection *cnc, GdaDict *dict, 
						      const gchar **args,
						      GError **error, MainData *data);

static GdaInternalCommandsList *
build_internal_commands_list (MainData *data)
{
	GdaInternalCommandsList *commands = g_new0 (GdaInternalCommandsList, 1);
	GdaInternalCommand *c;

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("General");
	c->name = "s [FILE]";
	c->description = _("Show commands history, or save it to file");
	c->args = NULL;
	c->command_func = gda_internal_command_history;
	c->user_data = NULL;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Dictionary");
	c->name = "dict_sync";
	c->description = _("Synchronize dictionary with database structure");
	c->args = NULL;
	c->command_func = gda_internal_command_dict_sync;
	c->user_data = NULL;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Dictionary");
	c->name = "dict_save";
	c->description = _("Save dictionary");
	c->args = NULL;
	c->command_func = gda_internal_command_dict_save;
	c->user_data = NULL;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Information");
	c->name = "dt [TABLE]";
	c->description = _("List all tables and views (or named table or view)");
	c->args = NULL;
	c->command_func = gda_internal_command_list_tables_views;
	c->user_data = NULL;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Information");
	c->name = "dq [QUERY] [+]";
	c->description = _("List all queries (or named query) in dictionary");
	c->args = NULL;
	c->command_func = gda_internal_command_list_queries;
	c->user_data = NULL;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	/* specific commands */
	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("General");
	c->name = _("c DSN [USER [PASSWORD]]");
	c->description = _("Connect to another defined data source (DSN, see \\dsn_list)");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc)extra_command_change_cnc_dsn;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("General");
	c->name = "dsn_list";
	c->description = _("List configured data sources (DSN)");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_list_dsn;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("General");
	c->name = "providers_list";
	c->description = _("List installed database providers");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_list_providers;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Input/Output");
	c->name = _("i FILE");
	c->description = _("Execute commands from file");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_set_input;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Input/Output");
	c->name = _("o [FILE]");
	c->description = _("Send output to a file or |pipe");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_set_output;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Input/Output");
	c->name = _("echo [TEXT]");
	c->description = _("Send output to stdout");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_echo;
	c->user_data = data;
	c->arguments_delimiter_func = args_as_string_func;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Input/Output");
	c->name = _("qecho [TEXT]");
	c->description = _("Send output to output stream");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_qecho;
	c->user_data = data;
	c->arguments_delimiter_func = args_as_string_func;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("General");
	c->name = _("q");
	c->description = _("Quit");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_quit;
	c->user_data = NULL;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("General");
	c->name = _("cd [DIR]");
	c->description = _("Change the current working directory");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_cd;
	c->user_data = NULL;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("General");
	c->name = _("copyright");
	c->description = _("Show usage and distribution terms");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_copyright;
	c->user_data = NULL;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = "e [FILE]";
	c->description = _("Edit the query buffer (or file) with external editor");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_edit_buffer;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = "r [FILE]";
	c->description = _("Reset the query buffer (clear buffer of with contents of file)");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_reset_buffer;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = "p";
	c->description = _("Show the contents of the query buffer");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_show_buffer;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = "g";
	c->description = _("Execute contents of query buffer");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_exec_buffer;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = _("w FILE");
	c->description = _("Write query buffer to file");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_write_buffer;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = _("w_dict [QUERY_NAME]");
	c->description = _("Save query buffer to dictionary");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_query_buffer_to_dict;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = _("r_dict QUERY_NAME");
	c->description = _("Set named query from dictionary into query buffer");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_query_buffer_from_dict;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = _("set [NAME [VALUE|_null_]]");
	c->description = _("Set or show internal parameter, or list all if no parameters");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_set;
	c->user_data = data;
	c->arguments_delimiter_func = args_as_string_set;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Query buffer");
	c->name = _("unset NAME");
	c->description = _("Unset (delete) internal parameter");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_unset;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("Formatting");
	c->name = _("H [HTML|XML]");
	c->description = _("Set output format");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) extra_command_set_output_format;
	c->user_data = data;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	/* comes last */
	c = g_new0 (GdaInternalCommand, 1);
	c->group = _("General");
	c->name = "?";
	c->description = _("List all available commands");
	c->args = NULL;
	c->command_func = (GdaInternalCommandFunc) gda_internal_command_help;
	c->user_data = commands;
	c->arguments_delimiter_func = NULL;
	commands->commands = g_slist_prepend (commands->commands, c);

	return commands;
}

static GdaInternalCommandResult *
extra_command_set_output (GdaConnection *cnc, GdaDict *dict, 
			  const gchar **args,
			  GError **error, MainData *data)
{
	if (set_output_file (data, args[0], error)) {
		GdaInternalCommandResult *res;
		
		res = g_new0 (GdaInternalCommandResult, 1);
		res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
		return res;
	}
	else
		return NULL;
}

static GdaInternalCommandResult *
extra_command_set_output_format (GdaConnection *cnc, GdaDict *dict, 
				 const gchar **args,
				 GError **error, MainData *data)
{
	GdaInternalCommandResult *res;
	const gchar *format = NULL;

	if (args[0] && *args[0])
		format = args[0];
	
	data->output_format = OUTPUT_FORMAT_DEFAULT;
	if (format) {
		if ((*format == 'X') || (*format == 'x'))
			data->output_format = OUTPUT_FORMAT_XML;
		else if ((*format == 'H') || (*format == 'h'))
			data->output_format = OUTPUT_FORMAT_HTML;
	}

	if (!data->output_stream) {
		res = g_new0 (GdaInternalCommandResult, 1);
		res->type = GDA_INTERNAL_COMMAND_RESULT_TXT_STDOUT;
		res->u.txt = g_string_new ("");
		switch (data->output_format) {
		case OUTPUT_FORMAT_DEFAULT:
			g_string_assign (res->u.txt, ("Output format is default\n"));
			break;
		case OUTPUT_FORMAT_HTML:
			g_string_assign (res->u.txt, ("Output format is HTML\n"));
			break;
		case OUTPUT_FORMAT_XML:
			g_string_assign (res->u.txt, ("Output format is XML\n"));
			break;
		default:
			TO_IMPLEMENT;
		}
	}
	else {
		res = g_new0 (GdaInternalCommandResult, 1);
		res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
	}
	return res;
}

static GdaInternalCommandResult *
extra_command_set_input (GdaConnection *cnc, GdaDict *dict, 
			 const gchar **args,
			 GError **error, MainData *data)
{
	if (set_input_file (data, args[0], error)) {
		GdaInternalCommandResult *res;
		
		res = g_new0 (GdaInternalCommandResult, 1);
		res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
		return res;
	}
	else
		return NULL;
}

static GdaInternalCommandResult *
extra_command_echo (GdaConnection *cnc, GdaDict *dict, 
		    const gchar **args,
		    GError **error, MainData *data)
{
	GdaInternalCommandResult *res;
	
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_TXT_STDOUT;
	res->u.txt = g_string_new (args[0]);
	return res;
}

static GdaInternalCommandResult *
extra_command_qecho (GdaConnection *cnc, GdaDict *dict, 
		     const gchar **args,
		     GError **error, MainData *data)
{
	GdaInternalCommandResult *res;
	
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_TXT;
	res->u.txt = g_string_new (args[0]);
	return res;
}

static 
GdaInternalCommandResult *extra_command_list_dsn (GdaConnection *cnc, GdaDict *dict, 
						  const gchar **args,
						  GError **error, MainData *data)
{
	GdaInternalCommandResult *res;
	
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_DATA_MODEL;
	res->u.model = list_all_sections (data);
	return res;
}

static 
GdaInternalCommandResult *extra_command_list_providers (GdaConnection *cnc, GdaDict *dict, 
							const gchar **args,
							GError **error, MainData *data)
{
	GdaInternalCommandResult *res;
	
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_DATA_MODEL;
	res->u.model = list_all_providers (data);
	return res;
}

static 
GdaInternalCommandResult *extra_command_change_cnc_dsn (GdaConnection *cnc, GdaDict *dict, 
							const gchar **args,
							GError **error, MainData *data)
{
	const gchar *user = NULL, *pass = NULL;

	if (args[1]) {
		user = args[1];
		if (args[2])
			pass = args[2];
	}
	if (open_connection (data, args[0], NULL, NULL, user, pass, error)) {
		GdaInternalCommandResult *res;
		
		res = g_new0 (GdaInternalCommandResult, 1);
		res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
		return res;
	}
	else
		return NULL;
}

static GdaInternalCommandResult *
extra_command_copyright (GdaConnection *cnc, GdaDict *dict, 
			 const gchar **args,
			 GError **error, MainData *data)
{
	GdaInternalCommandResult *res;
	
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_TXT;
	res->u.txt = g_string_new ("This program is free software; you can redistribute it and/or modify\n"
				   "it under the terms of the GNU General Public License as published by\n"
				   "the Free Software Foundation; either version 2 of the License, or\n"
				   "(at your option) any later version.\n\n"
				   "This program is distributed in the hope that it will be useful,\n"
				   "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
				   "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
				   "GNU General Public License for more details.\n");
	return res;
}

static GdaInternalCommandResult *
extra_command_quit (GdaConnection *cnc, GdaDict *dict, 
		    const gchar **args,
		    GError **error, MainData *data)
{
	GdaInternalCommandResult *res;
	
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_EXIT;
	return res;
}

static 
GdaInternalCommandResult *
extra_command_cd (GdaConnection *cnc, GdaDict *dict, 
		  const gchar **args,
		  GError **error, MainData *data)
{
	const gchar *dir = NULL;
#define DIR_LENGTH 256
	static char start_dir[DIR_LENGTH];
	static gboolean init_done = FALSE;

	if (!init_done) {
		init_done = TRUE;
		memset (start_dir, 0, DIR_LENGTH);
		if (getcwd (start_dir, DIR_LENGTH) <= 0) {
			/* default to $HOME */
#ifdef G_OS_WIN32
			TO_IMPLEMENT;
			strncpy (start_dir, "/", 2);
#else
			struct passwd *pw;
			
			pw = getpwuid (geteuid ());
			if (!pw) {
				g_set_error (error, 0, 0, _("Could not get home directory: %s"), strerror (errno));
				return NULL;
			}
			else {
				gint l = strlen (pw->pw_dir);
				if (l > DIR_LENGTH - 1)
					strncpy (start_dir, "/", 2);
				else
					strncpy (start_dir, pw->pw_dir, l + 1);
			}
#endif
		}
	}

	if (args[0]) 
		dir = args[0];
	else 
		dir = start_dir;

	if (dir) {
		if (chdir (dir) == 0) {
			GdaInternalCommandResult *res;

			res = g_new0 (GdaInternalCommandResult, 1);
			res->type = GDA_INTERNAL_COMMAND_RESULT_TXT_STDOUT;
			res->u.txt = g_string_new ("");
			g_string_append_printf (res->u.txt, _("Working directory is now: %s"), dir);
			return res;
		}
		else
			g_set_error (error, 0, 0, _("Could not change working directory to '%s': %s"),
				     dir, strerror (errno));
	}

	return NULL;
}

static 
GdaInternalCommandResult *
extra_command_edit_buffer (GdaConnection *cnc, GdaDict *dict, 
			   const gchar **args,
			   GError **error, MainData *data)
{
	gchar *filename = NULL;
	static gchar *editor_name = NULL;
	gchar *command = NULL;
	gint systemres;
	GdaInternalCommandResult *res = NULL;

	if (!data->query_buffer) 
		data->query_buffer = g_string_new ("");

	if (args[0] && *args[0])
		filename = (gchar *) args[0];
	else {
		/* use a temp file */
		gint fd;
		fd = g_file_open_tmp (NULL, &filename, error);
		if (fd < 0)
			goto end_of_command;
		if (write (fd, data->query_buffer->str, data->query_buffer->len) != data->query_buffer->len) {
			g_set_error (error, 0, 0,
				     _("Could not write to temporary file '%s': %s"),
				     filename, strerror (errno));
			close (fd);
			goto end_of_command;
		}
		close (fd);
	}

	if (!editor_name) {
		editor_name = getenv("GDA_SQL_EDITOR");
		if (!editor_name)
			editor_name = getenv("EDITOR");
		if (!editor_name)
			editor_name = getenv("VISUAL");
		if (!editor_name) {
#ifdef G_OS_WIN32
			editor_name = "notepad.exe";
#else
			editor_name = "vi";
#endif
		}
	}
#ifdef G_OS_WIN32
#ifndef __CYGWIN__
#define SYSTEMQUOTE "\""
#else
#define SYSTEMQUOTE ""
#endif
	command = g_strdup_printf ("%s\"%s\" \"%s\"%s", SYSTEMQUOTE, editor_name, filename, SYSTEMQUOTE);
#else
	command = g_strdup_printf ("exec %s '%s'", editor_name, filename);
#endif

	systemres = system (command);
	if (systemres == -1) {
                g_set_error (error, 0, 0,
			     _("could not start editor '%s'"), editor_name);
		goto end_of_command;
	}
        else if (systemres == 127) {
		g_set_error (error, 0, 0,
			     _("Could not start /bin/sh"));
		goto end_of_command;
	}
	else {
		if (! args[0]) {
			gchar *str;
			
			if (!g_file_get_contents (filename, &str, NULL, error))
				goto end_of_command;
			g_string_assign (data->query_buffer, str);
			g_free (str);
		}
	}
	
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;

 end_of_command:

	g_free (command);
	if (! args[0]) {
		g_unlink (filename);
		g_free (filename);
	}

	return res;
}

static 
GdaInternalCommandResult *
extra_command_reset_buffer (GdaConnection *cnc, GdaDict *dict, 
			    const gchar **args,
			    GError **error, MainData *data)
{
	GdaInternalCommandResult *res = NULL;

	if (!data->query_buffer) 
		data->query_buffer = g_string_new ("");
	else 
		g_string_assign (data->query_buffer, "");

	if (args[0]) {
		const gchar *filename = NULL;
		gchar *str;

		filename = args[0];
		if (!g_file_get_contents (filename, &str, NULL, error))
			return NULL;

		g_string_assign (data->query_buffer, str);
		g_free (str);
	}
	
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;

	return res;
}

static 
GdaInternalCommandResult *
extra_command_show_buffer (GdaConnection *cnc, GdaDict *dict, 
			   const gchar **args,
			   GError **error, MainData *data)
{
	GdaInternalCommandResult *res = NULL;

	if (!data->query_buffer) 
		data->query_buffer = g_string_new ("");
	res = g_new0 (GdaInternalCommandResult, 1);
	res->type = GDA_INTERNAL_COMMAND_RESULT_TXT;
	res->u.txt = g_string_new (data->query_buffer->str);

	return res;
}

static 
GdaInternalCommandResult *
extra_command_exec_buffer (GdaConnection *cnc, GdaDict *dict, 
			   const gchar **args,
			   GError **error, MainData *data)
{
	GdaInternalCommandResult *res = NULL;

	if (!data->query_buffer) 
		data->query_buffer = g_string_new ("");
	if (*data->query_buffer->str != 0)
		res = command_execute (data, data->query_buffer->str, error);
	else {
		res = g_new0 (GdaInternalCommandResult, 1);
		res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
	}

	return res;
}

static 
GdaInternalCommandResult *
extra_command_write_buffer (GdaConnection *cnc, GdaDict *dict, 
			    const gchar **args,
			    GError **error, MainData *data)
{
	GdaInternalCommandResult *res = NULL;
	if (!data->query_buffer) 
		data->query_buffer = g_string_new ("");
	if (!args[0]) 
		g_set_error (error, 0, 0, 
			     _("Missing FILE to write to"));
	else {
		if (g_file_set_contents (args[0], data->query_buffer->str, -1, error)) {
			res = g_new0 (GdaInternalCommandResult, 1);
			res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
		}
	}

	return res;
}

static GdaQuery *
find_query_in_dict (GdaDict *dict, const gchar *query_name)
{
	GdaQuery *query = NULL;
	GSList *queries, *list;
	queries = gda_dict_get_queries (dict);
	for (list = queries; list; list = list->next) {
		const gchar *cstr;			
		cstr = gda_object_get_name (GDA_OBJECT (list->data));
		if (cstr && !strcmp (cstr, query_name)) {
			query = GDA_QUERY (list->data);
			break;
		}
	}
	g_slist_free (queries);
	return query;
}

static GdaInternalCommandResult *
extra_command_query_buffer_to_dict (GdaConnection *cnc, GdaDict *dict, 
				    const gchar **args,
				    GError **error, MainData *data)
{
	GdaInternalCommandResult *res = NULL;

	if (!data->query_buffer) 
		data->query_buffer = g_string_new ("");
	if (*data->query_buffer->str != 0) {
		GdaQuery *query = NULL;
		gboolean query_created = FALSE;
		GError *lerror = NULL;
		gchar *qname;

		if (args[0] && *args[0]) {
			qname = (gchar *) args[0];
			query = find_query_in_dict (dict, args[0]);
		}
		else {
			/* find a suitable name for query */
			gint i;
			for (i = 0; ; i++) {
				qname = g_strdup_printf (_("saved_query_%d"), i);
				query = find_query_in_dict (dict, qname);
				if (!query)
					break;
			}
		}
		if (!query) {
			query = gda_query_new_from_sql (data->dict, data->query_buffer->str, &lerror);
			query_created = TRUE;
		}
		else
			gda_query_set_sql_text (query, data->query_buffer->str, &lerror);
		if (lerror) {
			g_object_unref (query);
			g_propagate_error (error, lerror);
			g_error_free (lerror);
		}
		else {
			gda_object_set_name (GDA_OBJECT (query), qname);
			if (!args[0] || !(*args[0]))
				g_free (qname);
			if (query_created) {
				gda_dict_assume_object (data->dict, GDA_OBJECT (query));
				g_object_unref (query);
			}
			res = g_new0 (GdaInternalCommandResult, 1);
			res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
		}
	}
	else
		g_set_error (error, 0, 0,
			     _("Query buffer is empty"));

	return res;
}

static GdaInternalCommandResult *
extra_command_query_buffer_from_dict (GdaConnection *cnc, GdaDict *dict, 
				      const gchar **args,
				      GError **error, MainData *data)
{
	GdaInternalCommandResult *res = NULL;

	if (!data->query_buffer) 
		data->query_buffer = g_string_new ("");

	if (args[0] && *args[0]) {
		GdaQuery *query = find_query_in_dict (dict, args[0]);
		if (query) {
			gchar *str;
			str = gda_renderer_render_as_sql (GDA_RENDERER (query), NULL, NULL,
							  GDA_RENDERER_EXTRA_PRETTY_SQL | GDA_RENDERER_PARAMS_AS_DETAILED, 
							  NULL);
			g_string_assign (data->query_buffer, str);
			g_free (str);
			res = g_new0 (GdaInternalCommandResult, 1);
			res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
		}
		else
			g_set_error (error, 0, 0,
				     _("Could not find query named '%s'"), args[0]);
	}
	else
		g_set_error (error, 0, 0,
			     _("Missing query name"));
		
	return res;
}

static void foreach_param_set (const gchar *pname, GdaParameter *param, GdaDataModel *model);
static GdaInternalCommandResult *
extra_command_set (GdaConnection *cnc, GdaDict *dict, 
		   const gchar **args,
		   GError **error, MainData *data)
{
	GdaInternalCommandResult *res = NULL;
	const gchar *pname = NULL;
	const gchar *value = NULL;

	if (args[0] && *args[0]) {
		pname = args[0];
		if (args[1] && *args[1])
			value = args[1];
	}

	if (pname) {
		GdaParameter *param = g_hash_table_lookup (data->parameters, pname);
		if (param) {
			if (value) {
				/* set param's value */
				if (!strcmp (value, "_null_"))
					gda_parameter_set_value (param, NULL);
				else 
					gda_parameter_set_value_str (param, value);

				res = g_new0 (GdaInternalCommandResult, 1);
				res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
			}
			else {
				res = g_new0 (GdaInternalCommandResult, 1);
				res->type = GDA_INTERNAL_COMMAND_RESULT_PLIST;
				res->u.plist = gda_parameter_list_new (NULL);
				gda_parameter_list_add_param (res->u.plist, gda_parameter_new_copy (param));
			}
		}
		else {
			if (value) {
				/* create parameter */
				if (!strcmp (value, "_null_"))
					value = NULL;
				param = gda_parameter_new_string (pname, value);
				g_hash_table_insert (data->parameters, g_strdup (pname), param);
				res = g_new0 (GdaInternalCommandResult, 1);
				res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
			}
			else 
				g_set_error (error, 0, 0,
					     _("No parameter named '%s' defined"), pname);
		}
	}
	else {
		/* list parameter's values */
		GdaDataModel *model;
		model = gda_data_model_array_new_with_g_types (2,
							       G_TYPE_STRING,
							       G_TYPE_STRING);
		gda_data_model_set_column_title (model, 0, _("Name"));
		gda_data_model_set_column_title (model, 1, _("Value"));
		g_hash_table_foreach (data->parameters, (GHFunc) foreach_param_set, model);
		res = g_new0 (GdaInternalCommandResult, 1);
		res->type = GDA_INTERNAL_COMMAND_RESULT_DATA_MODEL;
		res->u.model = model;
	}
		
	return res;
}

static void 
foreach_param_set (const gchar *pname, GdaParameter *param, GdaDataModel *model)
{
	gint row;
	gchar *str;
	GValue *value;
	row = gda_data_model_append_row (model, NULL);

	value = gda_value_new_from_string (pname, G_TYPE_STRING);
	gda_data_model_set_value_at (model, 0, row, value, NULL);
	gda_value_free (value);
			
	str = gda_parameter_get_value_str (param);
	value = gda_value_new_from_string (str ? str : "(NULL)", G_TYPE_STRING);
	gda_data_model_set_value_at (model, 1, row, value, NULL);
	gda_value_free (value);
}

static GdaInternalCommandResult *
extra_command_unset (GdaConnection *cnc, GdaDict *dict, 
		     const gchar **args,
		     GError **error, MainData *data)
{
	GdaInternalCommandResult *res = NULL;
	const gchar *pname = NULL;

	if (args[0] && *args[0]) 
		pname = args[0];

	if (pname) {
		GdaParameter *param = g_hash_table_lookup (data->parameters, pname);
		if (param) {
			g_hash_table_remove (data->parameters, pname);
			res = g_new0 (GdaInternalCommandResult, 1);
			res->type = GDA_INTERNAL_COMMAND_RESULT_EMPTY;
		}
		else 
			g_set_error (error, 0, 0,
				     _("No parameter named '%s' defined"), pname);
	}
	else 
		g_set_error (error, 0, 0,
			     _("Missing parameter's name"));
		
	return res;
}

static gchar **
args_as_string_func (const gchar *str)
{
	return g_strsplit (str, " ", 2);
}

static gchar **
args_as_string_set (const gchar *str)
{
	return g_strsplit (str, " ", 3);
}
