/* This file is part of libccc
 *
 * AUTHORS
 *     Sven Herzberg  <herzi@gnome-de.org>
 *
 * Copyright (C) 2007  Sven Herzberg
 *
 * This work is provided "as is"; redistribution and modification
 * in whole or in part, in any medium, physical or electronic is
 * permitted without restriction.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * In no event shall the authors or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 */

#include "s-item.h"

#include <gtk/gtk.h>
#include <ccc.h>
#include "cc-test-view.h"

static GtkWidget* window = NULL;
static CcView* view = NULL;
static CcItem* root = NULL;
static CcItem* line = NULL;

static gboolean dirty_sent = FALSE;

static gboolean implemented = FALSE;

static void
debug_dirty (CcItem * item,
	     CcView * view,
	     CcDRect* dirty)
{
	g_print ("<< dirty sent (%s, %s) >>\n",
		 G_OBJECT_TYPE_NAME (item),
		 G_OBJECT_TYPE_NAME (view));
}

static void
pass_dirty (CcItem * item,
	     CcView * view,
	     CcDRect* dirty)
{
	debug_dirty (item, view, dirty);
	dirty_sent = TRUE;
}

static void
setup (void)
{
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	view = g_object_ref_sink (cc_view_widget_new ());
	gtk_container_add (GTK_CONTAINER (window), GTK_WIDGET (view));
	gtk_widget_show_all (window);
	while (gtk_events_pending ()) {
		gtk_main_iteration ();
	}
	root = g_object_ref_sink (cc_item_new ());
	line = g_object_ref_sink (cc_line_new ());
	cc_line_move (CC_LINE (line), 0.0, 0.0);
	cc_shape_set_brush_border (CC_SHAPE (line),
				   cc_brush_color_new (cc_color_new_rgb (0.0, 0.0, 1.0)));
	dirty_sent = FALSE;
}

static void
teardown (void)
{
	gtk_widget_destroy (window);
	g_object_unref (view);
	view = NULL;
	g_object_unref (root);
	root = NULL;
	g_object_unref (line);
	line = NULL;
}

static void
set_dirty_sent (CcItem  * item,
		CcView  * view,
		CcDRect * dirty,
		gboolean* dirty_sent)
{
	*dirty_sent = TRUE;
}

START_TEST(test_dirty_simple)
{
	/* prepare */
	gboolean dirty_sent = FALSE;
	g_signal_connect (line, "dirty",
			  G_CALLBACK (set_dirty_sent), &dirty_sent);

	/* check */
	cc_view_set_root (view, line);
	fail_unless (dirty_sent);

	/* cleanup */
	g_signal_handlers_disconnect_by_func (line, set_dirty_sent, &dirty_sent);
}
END_TEST

START_TEST(test_dirty_lazy)
{
	/* prepare */
	gboolean dirty_sent = FALSE;
	g_signal_connect (line, "dirty",
			  G_CALLBACK (set_dirty_sent), &dirty_sent);
	cc_view_set_root (view, line);

	/* check */
	cc_line_line (CC_LINE (line), 1.0, 1.0);
	fail_unless (dirty_sent);

	/* cleanup */
	/* FIXME: implement:
	 *
	 * view.root = line
	 * change line.bounds
	 *
	 * And make sure the dirty signal gets emitted while changing the bounds
	 */
}
END_TEST

START_TEST(test_dirty_recurse)
{
	/* prepare */
	g_signal_connect (line, "dirty",
			  G_CALLBACK (debug_dirty), NULL);
	g_signal_connect (root, "dirty",
			  G_CALLBACK (pass_dirty), NULL);
	cc_line_line (CC_LINE (line), 1.0, 1.0);
	cc_item_append (root, line);

	/* check */
	cc_view_set_root (view, root);
	fail_unless (dirty_sent);

	/* cleanup */
	g_signal_handlers_disconnect_by_func (line, debug_dirty, NULL);
	g_signal_handlers_disconnect_by_func (root, debug_dirty, NULL);
}
END_TEST

START_TEST(test_dirty_recurse_implied)
{
	/* prepare */
	g_signal_connect (root, "dirty",
			  G_CALLBACK (pass_dirty), NULL);
	cc_view_set_root (view, root);
	cc_line_line (CC_LINE (line), 1.0, 1.0);

	/* check */
	cc_item_append (root, line);
	while (gtk_events_pending ()) {
		gtk_main_iteration ();
	}
	fail_unless (dirty_sent);

	/* cleanup */
	g_signal_handlers_disconnect_by_func (root, debug_dirty, NULL);
}
END_TEST

static void
test_dirty_recurse_lazy_cb (CcItem  * item,
			    CcView  * view,
			    CcDRect * dirty,
			    gboolean* dirty_sent)
{
	*dirty_sent = TRUE;
}

START_TEST(test_dirty_recurse_lazy)
{
	// FIXME: also check that line doesn't emit anything to view
	/* prepare */
	gboolean dirty_sent = FALSE;
	cc_item_append (root, line);
	cc_view_set_root (view, root);
	g_signal_connect (root, "dirty",
			  G_CALLBACK (test_dirty_recurse_lazy_cb), &dirty_sent);

	/* check */
	cc_line_line (CC_LINE (line), 1.0, 1.0);
	fail_unless (dirty_sent);

	/* cleanup */
}
END_TEST

START_TEST(test_no_direct_emission)
{
	/* FIXME: implement this:
	 *
	 * item.append(line)
	 * view.root = line
	 *
	 * And make sure that "dirty" doesn't get emitted from line with the
	 * view parameter.
	 */
	// FIXME: implement this with all the other recurse tests
	fail_unless (implemented);
}
END_TEST

TCase*
tcase_item_dirty (void)
{
	TCase* self = tcase_create ("Dirty Emission");
	tcase_add_checked_fixture (self, setup, teardown);
	tcase_add_test (self, test_dirty_simple);
	tcase_add_test (self, test_dirty_lazy);
	tcase_add_test (self, test_dirty_recurse);
	tcase_add_test (self, test_dirty_recurse_implied);
	tcase_add_test (self, test_dirty_recurse_lazy);
	tcase_add_test (self, test_no_direct_emission);
	// FIXME: add tests for dirty emission on unregister()
	return self;
}

