/*
 * Copyright (C) 2009, 2010, 2011 Openismus GmbH
 *
 * This file is part of Java-libglom.
 *
 * Java-libglom is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * Java-libglom is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Java-libglom.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.glom.libglom.tests;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

import org.glom.libglom.Document;
import org.glom.libglom.Field;
import org.glom.libglom.FieldVector;
import org.glom.libglom.Glom;
import org.glom.libglom.LayoutFieldVector;
import org.glom.libglom.LayoutGroupVector;
import org.glom.libglom.LayoutItem;
import org.glom.libglom.LayoutItemVector;
import org.glom.libglom.LayoutItem_Field;
import org.glom.libglom.NumericFormat;
import org.glom.libglom.SortClause;
import org.glom.libglom.SortFieldPair;
import org.glom.libglom.StringVector;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Simple test to ensure that the generated bindings are working.
 * 
 * @author Murray Cumming, Ben Konrath
 */
public class DocumentTest {

	private static Document document;

	@BeforeClass
	static public void setUp() {
		Glom.libglom_init();
		document = new Document();
		document.set_file_uri("file://" + Glom.GLOM_EXAMPLE_FILE_DIR + "/example_music_collection.glom");
		int error = 0;
		boolean retval = document.load(error);
		assertTrue(retval);
		assertEquals(error, 0);
	}

	@AfterClass
	static public void tearDown() {
		Glom.libglom_deinit();
	}

	@Test
	public void testDocumentInfo() {
		assertThat(document.get_database_title(), is("Music Collection"));
		assertThat(document.get_default_table(), is("artists"));
	}

	@Test
	public void testReadTableNames() {
		StringVector tableNames = document.get_table_names();
		assertEquals(4, tableNames.size());

		String tables = tableNames.get(0);
		for (int i = 1; i < tableNames.size(); i++) {
			tables += ", " + tableNames.get(i);
		}
		assertThat(tables, is("albums, artists, publishers, songs"));
	}

	@Test
	public void testReadTableFieldSizes() {

		FieldVector fields = document.get_table_fields("albums");
		assertEquals(6, fields.size());

		Field field = fields.get(0);
		String titles = field.get_title_or_name();
		for (int i = 1; i < fields.size(); i++) {
			field = fields.get(i);
			titles += ", " + field.get_title_or_name();
		}
		assertThat(titles, is("Album ID, Comments, Name, Artist ID, Publisher ID, Year"));

		fields = document.get_table_fields("artists");
		assertEquals(4, fields.size());

		field = fields.get(0);
		titles = field.get_title_or_name();
		for (int i = 1; i < fields.size(); i++) {
			field = fields.get(i);
			titles += ", " + field.get_title_or_name();
		}
		assertThat(titles, is("Artist ID, Description, Comments, Name"));

		fields = document.get_table_fields("publishers");
		assertEquals(3, fields.size());

		field = fields.get(0);
		titles = field.get_title_or_name();
		for (int i = 1; i < fields.size(); i++) {
			field = fields.get(i);
			titles += ", " + field.get_title_or_name();
		}
		assertThat(titles, is("Publisher ID, Comments, Name"));

		fields = document.get_table_fields("songs");
		assertEquals(4, fields.size());

		field = fields.get(0);
		titles = field.get_title_or_name();
		for (int i = 1; i < fields.size(); i++) {
			field = fields.get(i);
			titles += ", " + field.get_title_or_name();
		}
		assertThat(titles, is("Song ID, Comments, Album ID, Name"));
	}

	@Test
	public void testReadLayoutListInfo() {
		String[] tables = { "albums", "artists", "publishers", "songs" };
		int[] sortClauseSizes = { 0, 1, 1, 1 };
		int[] layoutFieldSizes = { 7, 4, 3, 4 };

		for (int i = 0; i < tables.length; i++) {
			LayoutGroupVector layoutList = document.get_data_layout_groups("list", tables[i]);
			LayoutItemVector layoutItems = layoutList.get(0).get_items();
			LayoutFieldVector layoutFields = new LayoutFieldVector();
			SortClause sortClause = new SortClause();
			int numItems = safeLongToInt(layoutItems.size());
			for (int j = 0; j < numItems; j++) {
				LayoutItem item = layoutItems.get(j);
				LayoutItem_Field field = LayoutItem_Field.cast_dynamic(item);
				if (field != null) {
					layoutFields.add(field);
					Field details = field.get_full_field_details();
					if (details != null && details.get_primary_key()) {
						sortClause.addLast(new SortFieldPair(field, true)); // ascending
					}
				}
			}
			assertEquals(sortClauseSizes[i], sortClause.size());
			assertEquals(layoutFieldSizes[i], safeLongToInt(layoutFields.size()));
		}
	}

	// This tests if getting values from a NumericFormat object is working. This test was failing with a JVM crash when
	// using the glom_sharedptr macro with Glom::UsesRelationship and Glom::FieldFormatting.
	@Test
	public void testGetNumericFormat() {
		StringVector tableNames = document.get_table_names();

		for (int i = 0; i < tableNames.size(); i++) {
			String table = tableNames.get(i);
			LayoutGroupVector layoutList = document.get_data_layout_groups("list", table);
			LayoutItemVector layoutItems = layoutList.get(0).get_items();
			int numItems = safeLongToInt(layoutItems.size());
			for (int j = 0; j < numItems; j++) {
				LayoutItem item = layoutItems.get(j);
				LayoutItem_Field item_field = LayoutItem_Field.cast_dynamic(item);
				if (item_field != null) {
					// don't keep a reference to the FeildFormatting object
					NumericFormat numFormat = item_field.get_formatting_used().getM_numeric_format();

					// get the values
					boolean altForegroundColorForNegatives = numFormat.getM_alt_foreground_color_for_negatives();
					String currencySymbol = numFormat.getM_currency_symbol();
					long decimalPlaces = numFormat.getM_decimal_places();
					boolean decimalPlacesRestricted = numFormat.getM_decimal_places_restricted();
					boolean useThousandsSepator = numFormat.getM_use_thousands_separator();
					String alternativeColorForNegatives = NumericFormat.get_alternative_color_for_negatives();
					long defaultPrecision = NumericFormat.get_default_precision();

					// Simulate a garbage collection
					System.gc();
					System.runFinalization();

					// re-get the values and test
					assertEquals(altForegroundColorForNegatives, numFormat.getM_alt_foreground_color_for_negatives());
					assertEquals(currencySymbol, numFormat.getM_currency_symbol());
					assertEquals(decimalPlaces, numFormat.getM_decimal_places());
					assertEquals(decimalPlacesRestricted, numFormat.getM_decimal_places_restricted());
					assertEquals(useThousandsSepator, numFormat.getM_use_thousands_separator());
					assertEquals(alternativeColorForNegatives, NumericFormat.get_alternative_color_for_negatives());
					assertEquals(defaultPrecision, NumericFormat.get_default_precision());

				}
			}
		}
	}

	// A smoke test for the methods added to LayoutItem_Field for accessing methods in Glom::UsesRelationship.
	@Test
	public void testUsesRelationshipMethods() {
		String table = "albums";
		LayoutGroupVector layoutList = document.get_data_layout_groups("list", table);
		LayoutItemVector layoutItems = layoutList.get(0).get_items();

		String names = null, hasRelationshipNames = null, tablesUsed = null;
		LayoutItem firstItem = layoutItems.get(0);
		LayoutItem_Field firstItemField = LayoutItem_Field.cast_dynamic(firstItem);
		if (firstItemField != null) {
			names = firstItemField.get_name();
			hasRelationshipNames = "" + firstItemField.get_has_relationship_name();
			tablesUsed = firstItemField.get_table_used(table);
		}
		int numItems = safeLongToInt(layoutItems.size());
		for (int j = 1; j < numItems; j++) {
			LayoutItem item = layoutItems.get(j);
			LayoutItem_Field itemField = LayoutItem_Field.cast_dynamic(item);
			if (itemField != null) {
				names += ", " + itemField.get_name();
				hasRelationshipNames += ", " + itemField.get_has_relationship_name();
				tablesUsed += ", " + itemField.get_table_used(table);
			}
		}
		assertEquals(names, "name, year, artist_id, name, publisher_id, name, comments");
		assertEquals(hasRelationshipNames, "false, false, false, true, false, true, false");
		assertEquals(tablesUsed, "albums, albums, albums, artists, albums, publishers, albums");
	}

	/*
	 * This method safely converts longs from libglom into ints. This method was taken from stackoverflow:
	 * 
	 * http://stackoverflow.com/questions/1590831/safely-casting-long-to-int-in-java
	 */
	private static int safeLongToInt(long l) {
		if (l < Integer.MIN_VALUE || l > Integer.MAX_VALUE) {
			throw new IllegalArgumentException(l + " cannot be cast to int without changing its value.");
		}
		return (int) l;
	}

}
