/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <glib/gi18n.h>
#include "query.h"
#include "playlist.h"
#include "playlist-xml.h"
#include "expr.h"

static void apply_limits (Query *query);

static GTimer *timer = NULL;

#define TIMEOUT 100

struct _Query
{
  char *name;

  Expr *expr;

  GList *hits;
  GList *current;

  guint update_timeout_id;

  /* For debugging .*/
  gboolean in_func;
  
  LimitType limit_type;
  MatchType match_type;
  SelectionType selection_type;
  int limit;

  int refcount;
};

typedef struct {
  Query *query;
  QueryUpdateFunc update_func;
  gpointer user_data;
} UpdateData;

static void
query_free (Query *query)
{
  g_return_if_fail (query->refcount == 0);

  g_free (query->name);
  g_list_free (query->hits);

  if (query->update_timeout_id)
    g_warning ("Freeing active query.");
  
  if (query->expr)
    expr_free (query->expr);

  g_free (query);
}

Query *
query_new (Expr *expr)
{
  Query *query;

  if (!timer)
    timer = g_timer_new ();

  query = g_new0 (Query, 1);
  query->refcount = 1;
  query->expr = expr;
  
  return query;
}

Query *
query_ref (Query *query)
{
  query->refcount++;

  return query;
}

void
query_unref (Query *query)
{
  query->refcount--;

  if (query->refcount == 0)
    query_free (query);
}

void
query_set_name (Query *query, const char *name)
{
  g_return_if_fail (query != NULL);

  g_free (query->name);
  query->name = g_strdup (name);
}

const char *
query_get_name (Query *query)
{
  g_return_val_if_fail (query != NULL, NULL);

  return query->name;
}

void
query_set_expr (Query *query, Expr *expr)
{
  g_return_if_fail (query != NULL);

  query_cancel (query);

  if (query->expr)
    expr_free (query->expr);
  
  query->expr = expr;
}

Expr *
query_get_expr (Query *query)
{
  g_return_val_if_fail (query != NULL, NULL);

  return query->expr;
}

void
query_set_simple (Query      *query,
		  const char *title,
		  const char *artist,
		  const char *album)
{
  Expr *v, *c;
  Expr *e, *tmp;
  
  g_return_if_fail (query != NULL);

  query_cancel (query);
  
  if (query->expr)
    expr_free (query->expr);
 
  e = NULL;

  if (title)
    {
      v = expr_variable_new (VARIABLE_TITLE);
      c = expr_constant_new (constant_string_new (title));

      e = expr_binary_new (EXPR_OP_CONTAINS, v, c);
    }
  
  if (artist)
    {
      v = expr_variable_new (VARIABLE_ARTIST);
      c = expr_constant_new (constant_string_new (artist));

      tmp = expr_binary_new (EXPR_OP_CONTAINS, v, c);

      if (e)
	e = expr_binary_new (EXPR_OP_OR, e, tmp);
      else
	e = tmp;
    }
  
  if (album)
    {
      v = expr_variable_new (VARIABLE_ALBUM);
      c = expr_constant_new (constant_string_new (album));

      tmp = expr_binary_new (EXPR_OP_CONTAINS, v, c);
      
      if (e)
	e = expr_binary_new (EXPR_OP_OR, e, tmp);
      else
	e = tmp;
    }

  query->expr = e;
}

static gboolean
update_timeout_cb (UpdateData *data)
{
  Query *query;
  GList *next;

  query = data->query;

  query->in_func = TRUE;
  next = data->update_func (query, FALSE, query->current, data->user_data);
  query->in_func = FALSE;
 
  if (next == NULL)
    {
      query->update_timeout_id = 0;
      query_cancel (query);
      
      return FALSE;
    }

  query->current = next;
  
  return TRUE;
}

static void
update_destroy_notify (UpdateData *data)
{
  data->query->update_timeout_id = 0;

#if 0
  g_print ("Query done after %.2f seconds\n", g_timer_elapsed (timer, NULL));
#endif
  
  query_unref (data->query);
  g_free (data);
}

void
query_run (Query           *query,
	   GList           *songs,
	   QueryUpdateFunc  update_func,
	   gpointer         user_data)
{
  UpdateData *update_data;
  GList *l;

  g_return_if_fail (query != NULL);
  g_return_if_fail (update_func != NULL);

  g_timer_start (timer);
  
  query_cancel (query);

  g_list_free (query->hits);
  query->hits = NULL;
  
  for (l = songs; l; l = l->next)
    {
      Song *song = l->data;

      if (!query->expr || expr_evaluate (query->expr, song))
	{
	  query->hits = g_list_prepend (query->hits, song);
	  song->random = g_random_int ();
	}
    }

  apply_limits (query);

  query->current = query->hits;

  query_ref (query);
	
  /* Run the first burst immediately. */
  query->in_func = TRUE;
  l = update_func (query, TRUE, query->current, user_data);
  
  query->in_func = FALSE;
  if (l)
    {
      update_data = g_new0 (UpdateData, 1);

      update_data->query = query;
      update_data->update_func = update_func;
      update_data->user_data = user_data;

      query->current = l;

      /* Higher than drawing which is done in HIGH_IDLE + 20 */
      query->update_timeout_id = g_idle_add_full (G_PRIORITY_HIGH_IDLE + 15, 
						  (GSourceFunc) update_timeout_cb,
						  update_data,
						  (GDestroyNotify) update_destroy_notify);
    }
  else
    query_unref (query);
}

gboolean
query_cancel (Query *query)
{
  guint id;
  
  if (query->in_func)
    g_warning ("Don't call query_cancel() from within the update function.");

  /* The query may be freed after the source_remove call so be careful... */
  id = query->update_timeout_id;
  query->update_timeout_id = 0;  

  if (id)
    g_source_remove (id);

  return (id != 0);
}

/*
 * Limits
 */

static int
compare_int (int a, int b)
{
  if (a < b)
    return -1;
  else if (a > b)
    return 1;
  else
    return 0;
}

static int
compare_func (Song  *song_a,
	      Song  *song_b,
	      Query *query)
{
  switch (query->selection_type)
    {
    case SELECTION_TYPE_DATE_ADDED:
      return -compare_int (song_a->date_added, song_b->date_added);
      
    case SELECTION_TYPE_MOST_RECENTLY_PLAYED:
      return -compare_int (song_a->last_played, song_b->last_played);
      
    case SELECTION_TYPE_LEAST_RECENTLY_PLAYED:
      return compare_int (song_a->last_played, song_b->last_played);
      
    case SELECTION_TYPE_RATING:
      return compare_int (song_a->year, song_b->year);

    case SELECTION_TYPE_RANDOM:
      return compare_int (song_a->random, song_b->random);

    case SELECTION_TYPE_LAST_PLAYED:
      return compare_int (song_a->last_played, song_b->last_played);

    case SELECTION_TYPE_PLAY_COUNT:
      return -compare_int (song_a->play_count, song_b->play_count);

    case SELECTION_TYPE_QUALITY:
      /*g_warning ("Quality type is not implemented yet.");*/
      return 0;

    case SELECTION_TYPE_YEAR:
      return compare_int (song_a->year, song_b->year);

    case SELECTION_TYPE_NONE:
      /*g_warning ("Must select on something.");*/
      return 0;
    }

  return 0;
}

static void
apply_limits (Query *query)
{
  GList *sorted;
  GList *list, *l;
  Song *song;
  int limit;

  if (query->limit_type == LIMIT_TYPE_NONE)
    return;

  sorted = g_list_sort_with_data (query->hits, (GCompareDataFunc) compare_func, query);
    
  list = NULL;
  limit = 0;
  for (l = sorted; l; l = l->next)
    {
      song = l->data;

      switch (query->limit_type)
	{
	case LIMIT_TYPE_SONGS:
	  limit++;
	  break;
	case LIMIT_TYPE_MINUTES:
	case LIMIT_TYPE_HOURS:
	  limit += song->length;
	  break;
	case LIMIT_TYPE_MB:
	case LIMIT_TYPE_GB:
	  limit += song->filesize;
	  break;
	case LIMIT_TYPE_NONE:
	  g_assert_not_reached ();
	}

      if (limit <= query->limit)
	list = g_list_prepend (list, song);
      else
	break;
    }
  
  g_list_free (sorted);
  query->hits = g_list_reverse (list);
}

void
query_set_selection_type (Query         *query,
			  SelectionType  selection)
{
  g_return_if_fail (query != NULL);

  query->selection_type = selection;
}

SelectionType
query_get_selection_type (Query *query)
{
  g_return_val_if_fail (query != NULL, SELECTION_TYPE_NONE);

  return query->selection_type;
}

void
query_set_limit_type (Query *query, LimitType type)
{
  g_return_if_fail (query != NULL);

  query->limit_type = type;
}

LimitType
query_get_limit_type (Query *query)
{
  g_return_val_if_fail (query != NULL, LIMIT_TYPE_NONE);

  return query->limit_type;
}

void
query_set_match_type (Query *query, MatchType type)
{
  g_return_if_fail (query != NULL);

  query->match_type = type;
}

MatchType
query_get_match_type (Query *query)
{
  g_return_val_if_fail (query != NULL, MATCH_TYPE_NONE);

  return query->match_type;
}

void
query_set_limit (Query *query, int limit)
{
  g_return_if_fail (query != NULL);

  query->limit = limit;
}

int
query_get_limit (Query *query)
{
  g_return_val_if_fail (query != NULL, 0);

  return query->limit;
}

const char *
limit_type_to_string (LimitType type)
{
 switch (type)
    {
    case LIMIT_TYPE_NONE:
      return "none";
    case LIMIT_TYPE_SONGS:
      return "songs";
    case LIMIT_TYPE_MB:
      return "mb";
    case LIMIT_TYPE_GB:
      return "gb";
    case LIMIT_TYPE_MINUTES:
      return "minutes";
    case LIMIT_TYPE_HOURS:
      return "hours";

    default:
      return "invalid-limit-type";
    }
}

LimitType
limit_type_from_string (const char *str)
{
  LimitType type;
  
  if (strcmp (str, "songs") == 0)
    type = LIMIT_TYPE_SONGS;
  else if (strcmp (str, "mb") == 0)
    type = LIMIT_TYPE_MB;
  else if (strcmp (str, "gb") == 0)
    type = LIMIT_TYPE_GB;
  else if (strcmp (str, "minutes") == 0)
    type = LIMIT_TYPE_MINUTES;
  else if (strcmp (str, "hours") == 0)
    type = LIMIT_TYPE_HOURS;
  else
    type = LIMIT_TYPE_NONE;
  
  return type;
}

const char *
match_type_to_string (MatchType type)
{
 switch (type)
    {
    case MATCH_TYPE_NONE:
      return "none";
    case MATCH_TYPE_ALL:
      return "all";
    case MATCH_TYPE_ANY:
      return "any";
    default:
      return "invalid-match-type";
    }
}

MatchType
match_type_from_string (const char *str)
{
  MatchType type;
  
  if (strcmp (str, "any") == 0)
    type = MATCH_TYPE_ANY;
  else if (strcmp (str, "all") == 0)
    type = MATCH_TYPE_ALL;
  else
    type = MATCH_TYPE_NONE;
  
  return type;
}

const char *
selection_type_to_string (SelectionType type)
{
  switch (type)
    {
    case SELECTION_TYPE_DATE_ADDED:
      return "date-added";
    case SELECTION_TYPE_MOST_RECENTLY_PLAYED:
      return "most-recently-played";
    case SELECTION_TYPE_LEAST_RECENTLY_PLAYED:
      return "least-recently-played";
    case SELECTION_TYPE_RATING:
      return "rating";
    case SELECTION_TYPE_QUALITY:
      return "quality";
    case SELECTION_TYPE_RANDOM:
      return "random";
    case SELECTION_TYPE_YEAR:
      return "year";
    case SELECTION_TYPE_PLAY_COUNT:
      return "play-count";
    
    default:
      return "invalid-selection-type";
    }
}

SelectionType
selection_type_from_string (const char *str)
{
  SelectionType type;
  
  if (strcmp (str, "date-added") == 0)
    type = SELECTION_TYPE_DATE_ADDED;
  else if (strcmp (str, "most-recently-played") == 0)
    type = SELECTION_TYPE_MOST_RECENTLY_PLAYED;
  else if (strcmp (str, "least-recently-played") == 0)
    type = SELECTION_TYPE_LEAST_RECENTLY_PLAYED;
  else if (strcmp (str, "rating") == 0)
    type = SELECTION_TYPE_RATING;
  else if (strcmp (str, "quality") == 0)
    type = SELECTION_TYPE_QUALITY;
  else if (strcmp (str, "year") == 0)
    type = SELECTION_TYPE_YEAR;
  else if (strcmp (str, "random") == 0)
    type = SELECTION_TYPE_RANDOM;
  else if (strcmp (str, "play-count") == 0)
    type = SELECTION_TYPE_PLAY_COUNT;
  else
    type = SELECTION_TYPE_NONE;

  return type;
}

void
query_create_default_queries (void)
{
  Expr *c, *v, *expr;
  Query *query;
  Playlist *playlist;
  GList *playlists;
  
  playlists = playlist_xml_load_playlists ();

  /* Recently Played */
  c = expr_constant_new (constant_int_new (0));
  v = expr_variable_new (VARIABLE_PLAY_COUNT);
  expr = expr_binary_new (EXPR_OP_GT, v, c);

  query = query_new (expr);
  query_set_selection_type (query, SELECTION_TYPE_LAST_PLAYED);
  query_set_limit_type (query, LIMIT_TYPE_SONGS);
  query_set_limit (query, 50);
  
  playlist = playlist_new (PLAYLIST_TYPE_SMART, _("Recently Played"));
  playlist_set_query (playlist, query);
  playlists = g_list_append (playlists, playlist);
  
  /* Most Played */
  c = expr_constant_new (constant_int_new (0));
  v = expr_variable_new (VARIABLE_PLAY_COUNT);
  expr = expr_binary_new (EXPR_OP_GT, v, c);

  query = query_new (expr);
  query_set_selection_type (query, SELECTION_TYPE_PLAY_COUNT);
  query_set_limit_type (query, LIMIT_TYPE_SONGS);
  query_set_limit (query, 50);
  
  playlist = playlist_new (PLAYLIST_TYPE_SMART, _("Most Played"));
  playlist_set_query (playlist, query);
  playlists = g_list_append (playlists, playlist);
  
  playlist_xml_save_playlists (playlists);
}
