;;      $Id$
;;;; 	Copyright (C) 1996, 1997,1998 Christopher Lee
;;;; 
;;;; This program is free software; you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation; either version 2, or (at your option)
;;;; any later version.
;;;; 
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License
;;;; along with this software; see the file COPYING.  If not, write to
;;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;; 

;; $Id$	
;; This is a utility for generating glue code for interfacing a C-library
;;  to Guile.
;;

(define-module (gnome-gwrap g-wrap)
  :use-module (gnome-gwrap output-file)
  :use-module (gnome-gwrap sorting))

(define c-file '())
(define h-file '())
(define doc-file '())
(define mod-prefix #f)
(define gen-header-file #f)

(define c++-file? #f)

(define *types* (list))
(define-public (add-type scheme-sym c-type-str fn-to-scm fn-from-scm fn-isa)
  (let ((type (make-c-type c-type-str fn-to-scm fn-from-scm fn-isa)))
    (set! *types* 
	  (cons (cons scheme-sym type)
		*types*))
    type))
(define-public (add-new-type scheme-sym the-type)
  (set! *types* 
	(cons (cons scheme-sym the-type)
	      *types*)))

(define-public (get-type scheme-sym)
  (let ((type-cc (assq scheme-sym *types*)))
    (if (not type-cc)
	(begin (display *types*)
	       (error "get-type: type not found:" scheme-sym)))
    (cdr type-cc)))

(define (fn-option options name else-thunk)
  (let ((option (assq name options)))
    (if option
	(cadr option)
	(else-thunk))))

(define (make-index-generator kind)
  (let ((elements '())
	(category #f)
	(title #f))
    (let ((add-element (lambda (el)
			  (set! elements (cons el elements))))
	  (elements->html
	   (lambda ()
	     (if (null? elements)
		 (list "<p>(no " category " defined)</p>\n")
		 (list
		  "<h3>" title "</h3>"
		  (separate-by
		   (map 
		    (lambda (item)
		      (list 
		       "<a href=\"#" category "-" item "\">" item "</a>"))
		    (sort elements string<?))
		   " |\n")
		  "\n")))))
      (let ((info
	     (assq kind '((constants "constants" "Constants")
			  (functions "functions" "Functions")
			  (types     "types"     "Types")))))
	(if (not info)
	    (error 
	     "make-index-generator: use constants, functions or types"))
	(set! category (cadr info))
	(set! title (caddr info)))

      (lambda (dispatch-command)
	(case dispatch-command
	  ((add) add-element)
	  ((output) elements->html)
	  (else
	   (error "index-generator: don't know command " dispatch-command))))
      )))
	  

(define false-fn (lambda () #f))
(define true-fn (lambda () #t))

(define type-index-generator #f)
(define function-index-generator #f)
(define constant-index-generator #f)

(define *files* '())

(define (gw:make-outfile-text sym name text sym-assq)
  (let ((outfile (make-outfile-txt sym name text sym-assq)))
    (set! *files* (cons (list sym outfile) *files*))
    outfile))

(define (get-file file-sym)
  (let ((file-pair (assq file-sym *files*)))
    (if (not file-pair) (error "get-file failed for " file))
    (cadr file-pair)))

(define (gw:write file-sym section-sym . text-lst)
  (outfile:add-to-section (get-file file-sym) section-sym text-lst))

(define (gw:trans-write file-sym section-sym trans-assq text)
  (gw:write file-sym section-sym (translate-str text trans-assq)))

(define (prefix-name name)
  (if mod-prefix
      (string->symbol (string-append mod-prefix ":" name))
      name))

(define c-file-tmpl "\
/* Generated by G-Wrap: an experimental Guile function-wrapper engine */

#ifdef __cplusplus
extern \"C\" {
#endif
#include <libguile.h>
#include <libguile/__scm.h>
#ifdef __cplusplus
}
#endif
%includes%
#include \"%fname%.h\"

/* Internal function declarations */
%fn-declarations%

/* Type headers */
%type-headers%

/* Wrapper function implementations */
%fn-wrappers%

/* Initialization code */
%init-fn-decl%
%fn-inits%

%type-inits%

%file-end%
}
")
(define h-file-tmpl "\
/* Generated by G-Wrap: an experimental Guile function-wrapper engine */

#ifdef __cplusplus
extern \"C\" {
#endif

void init_%fname% (void);
%declarations%
#ifdef __cplusplus
}
#endif
")
(define doc-file-tmpl "\
<html> <head>\n<title>Documentation for %fname%</title>
</head>
<body text=\"#000000\" bgcolor=\"#ffffff\">
<h1> Documentation for %fname%</h1>
<!-- Generated by G-Wrap -->

<h2>Index</h2>
%type-index%
%fn-index%
%const-index%

<h2>Types</h2>
<dl>
%types%
</dl>

<h2>Constants</h2>
<dl>
%constants%
</dl>

<h2>Functions</h2>
<dl>
%functions%
</dl>
")
(define gen-header-tmpl "\
#ifndef __%header-def-sym%__
#define __%header-def-sym%__
/* This header was automatically generated by G-WRAP version XXX.
 *  You should not edit this file manually.
 */

%includes%
#ifdef __cplusplus
extern \"C\" {
#endif

%declarations%
#ifdef __cplusplus
}
#endif
#endif
")


(define-public (gwrap-open-module fname . options)
  (let ((c++?         (fn-option options 'c++ false-fn))
	(guile-module (fn-option options 'guile-module false-fn))
	(guile-prefix (fn-option options 'guile-prefix false-fn))
	(call-on-init (fn-option options 'call-on-init false-fn))
	(trans-assq   `((fname ,fname))))
    (if c++? (set! c++-file? #t))
    (if guile-prefix (set! mod-prefix guile-prefix))
    (set! c-file
	  (gw:make-outfile-text 
	   'c-file (string-append fname (if c++? ".cc" ".c"))
	   c-file-tmpl trans-assq))
    (set! h-file
	  (gw:make-outfile-text
	   'h-file (string-append fname ".h")
	   h-file-tmpl trans-assq))
    (set! type-index-generator (make-index-generator 'types))
    (set! function-index-generator (make-index-generator 'functions))
    (set! constant-index-generator (make-index-generator 'constants))
    (set! doc-file
	  (gw:make-outfile-text
	   'doc-file (string-append fname ".html")
	   doc-file-tmpl
	   (append `((type-index  ,type-index-generator)
		     (fn-index    ,function-index-generator)
		     (const-index ,constant-index-generator))
		   trans-assq)))
    ;; Write a module initialization function, if needed
    (cond (guile-module
	   (gw:trans-write 
	    'c-file 'init-fn-decl
	    `((fname ,fname)
	      (uc-module ,(separate-by guile-module "_"))
	      (sp-module ,(separate-by guile-module " ")))
	    "\
void scm_init_%uc-module%_module (void) {
}

")))
    (gw:trans-write 
     'c-file 'init-fn-decl
     `((fname ,fname)
       (init  ,(if guile-module 
		   "/* init happens in module init fn above */"
		   ""))
       (init-call ,(if call-on-init
		       (list call-on-init " ();")
		       "/* nothing extra called on init */")))
     "\
void init_%fname% (void) {
  %init%
  %init-call%
")))

(define-public (gwrap-include-local-header x)
  (gwrap-c-output-c 'includes
		    "#include \"" x "\"\n"))

(define-public (gwrap-include-global-header x)
  (gwrap-c-output-c 'includes
		    "#include <" x ">\n"))

(define-public (gwrap-wrapped-header-include-local-header x)
  (gw:write 'gen-header-file 'includes "#include \"" x "\"\n"))

(define-public (gwrap-wrapped-header-include-global-header x)
  (gw:write 'gen-header-file 'includes "#include <" x ">\n"))

(define-public (gwrap-close-module)
  (outfile:close c-file)
  (outfile:close h-file)
  (outfile:close doc-file)
  (if gen-header-file (outfile:close gen-header-file)))

(define (gwrap-c-output-c where . lst)
  (outfile:add-to-section c-file where lst))

(define (gwrap-c-output-h where . lst)
  (outfile:add-to-section h-file where lst))

(define (gwrap-c-output-doc where . lst)
  (outfile:add-to-section doc-file where lst))

(define (gwrap-c-doc-type type description)
  (outfile:add-to-section 
   doc-file 'types 
   (list
    "  <dt> <a name=\"types-" type "\">"
    "<strong>type <em>" type "</em></strong></a>\n"
    "  <dd> " description "\n")))

(define (gwrap-c-doc-function scheme-name declaration description)
  (outfile:add-to-section 
   doc-file 'functions
   (list
    "  <dt> <a name=\"functions-" scheme-name "\">"
    "<strong>" declaration "</strong></a>\n"
    "  <dd> " description "\n")))

(define (gwrap-c-doc-constant constant type description)
  (outfile:add-to-section 
   doc-file 'constants
   (list
    "  <dt> <a name=\"constants-" constant "\"><strong>constant <tt>" constant 
    "</tt> (type <em>" type "</em></strong>)</a>\n"
    "  <dd> " description "\n")))

(define-public (new-constant sym-name type varb . options)
  (set! sym-name (prefix-name sym-name))
  (let ((description (fn-option options 'doc (lambda () '()))))
    (gwrap-c-doc-constant sym-name type description))
  (if constant-index-generator
      ((constant-index-generator 'add) sym-name))
  (gwrap-c-output-c
   'type-inits
   "  scm_sysintern (\"" sym-name "\", "
   (make-conversion-to-scm (get-type type) varb)
   ");\n"))

(define-public (gwrap-generate-wrapped-header headername)
  (let ((subs
	 `((header-def-sym 
	    ,(str-translate (string-upcase! (string-copy headername))
			    "- ." (vector "_" "_" "_"))))))
    (set! gen-header-file
	  (gw:make-outfile-text 
	   'gen-header-file headername gen-header-tmpl subs))))

(define-public (new-type scheme-sym c-name c-print-name c-die-name c-eq-name 
			 . options)
  (let ((c-gc-name   (fn-option options 'c-gc-name   false-fn))
	(pass-c-ptr? (fn-option options 'pass-c-ptr? false-fn))
	(doc         (fn-option options 'doc (lambda () ""))))
    (if type-index-generator
	((type-index-generator 'add) scheme-sym))
    (gwrap-c-doc-type scheme-sym doc)
    (let ((subs `((c-name       ,c-name)
		  (c-die-name   ,c-die-name)
		  (c-print-name ,c-print-name)
		  (c-eq-name    ,c-eq-name)
		  (t16-name     ,(string-append "t16_" c-name))
		  (smob-name    ,(string-append c-name "_smob"))
		  (type-name    ,(if pass-c-ptr?
				     (string-append c-name "*")
				     c-name))
		  (mark-name    ,(if (not c-gc-name) 
				     "scm_mark0"
				     (string-append c-name "_mark0"))))))
      (gw:trans-write
       'c-file 'fn-declarations subs "\

static scm_sizet gwrapu_%c-die-name% (SCM x);
static int gwrapu_%c-print-name% (SCM x, SCM port, scm_print_state* pstate);
static SCM gwrapu_%c-eq-name% (SCM x, SCM y);\n")
      (gw:trans-write
       'c-file 'type-headers subs "\
long %t16-name%;\n
#ifdef GWRAP_OLD_GUILE_SMOB
static scm_smobfuns %smob-name% = {
  %mark-name%, gwrapu_%c-die-name%, gwrapu_%c-print-name%,
  gwrapu_%c-eq-name%};\n
#endif\n\n")
      (gw:trans-write
       'h-file 'declarations subs "extern long %t16-name%;\n")
      (gw:trans-write
       'c-file 'type-inits subs "\
#ifdef GWRAP_OLD_GUILE_SMOB
  %t16-name% = scm_newsmob(&%smob-name%);\n
#else 
  %t16-name% = scm_make_smob_type(\"%smob-name%\", 0);
  scm_set_smob_mark(%t16-name%, %mark-name%);
  scm_set_smob_free(%t16-name%, gwrapu_%c-die-name%);
  scm_set_smob_print(%t16-name%, gwrapu_%c-print-name%);
  scm_set_smob_equalp(%t16-name%, gwrapu_%c-eq-name%);
#endif\n\n")
      (gw:trans-write
       'h-file 'declarations subs "SCM %c-name%_to_SCM(%type-name% x);\n")
      (gw:trans-write
       'c-file 'fn-wrappers subs "\
SCM %c-name%_to_SCM(%type-name% x) {
  SCM scm;
  SCM_NEWCELL(scm);
  SCM_CAR(scm) = %t16-name%;
  SCM_CDR(scm) = (SCM)x;
  return scm;
}\n")
      (let ((the-type (apply make-wrapper-c-type 
			     (append
			      (list scheme-sym c-name)
			      options))))
	(add-new-type scheme-sym the-type)
	(gw:trans-write
	 'h-file 'declarations subs "SCM is_a_%c-name% (SCM x);\n")
	(gw:trans-write
	 'c-file 'fn-wrappers
	 (cons `(isa-check:x ,(make-isa-check the-type "x"))
	       subs)
	 "\
SCM is_a_%c-name% (SCM x) {
  return ( %isa-check:x% ? SCM_BOOL_T : SCM_BOOL_F);
}\n\n")
	(gw:trans-write
	 'c-file 'fn-inits
	 (cons `(fn-type-cast 
		 ,(if c++-file? "(SCM (*) (...))" "(SCM (*) ())"))
	       subs)
	 "\
  scm_make_gsubr( \"%c-name%?\", 1, 0, 0, %fn-type-cast% is_a_%c-name%);\n")
	(gw:trans-write
	 'c-file 'fn-wrappers
	 (append
	  `((scm->c:x     ,(make-conversion-from-scm the-type "x"))
	    (scm->c:y     ,(make-conversion-from-scm the-type "y")))
	  subs)
	 "\
int gwrapu_%c-print-name%(SCM x, SCM port, scm_print_state* pstate) {
  %c-print-name% (%scm->c:x%, port, SCM_WRITINGP(pstate));
  return 1;
}

scm_sizet gwrapu_%c-die-name%(SCM x) {
  %c-die-name% (%scm->c:x%);
  return sizeof(%c-name%);
}

SCM gwrapu_%c-eq-name% (SCM x, SCM y) {
  return (%c-eq-name% (%scm->c:x%, %scm->c:y%) ? SCM_BOOL_T : SCM_BOOL_F);
}\n\n")	 
	(if c-gc-name
	    (gw:trans-write
	     'c-file 'fn-wrappers 
	     (append
	       `((c-gc-name    ,c-gc-name)
		 (type->c      ,(make-conversion-from-scm the-type "obj")))
	       subs)
	     "\
SCM %c-name%_mark0(SCM obj) {
  %c-gc-name% (%type->c%);
  return SCM_BOOL_F;
}\n\n"))
	))))

;;; Utility function

(define-public (gwrap-assume-types-wrapped lst . options)
  (for-each (lambda (x) (apply make-wrapper-c-type (append x options)))
	    lst))

(define (make-wrapper-c-type scheme-sym c-name . options)
  (let ((pass-c-ptr? (fn-option options 'pass-c-ptr? false-fn)))
    (let ((smob-name (string-append c-name "_smob"))
	  (type-name (if pass-c-ptr?
			 (string-append c-name "*")
			 c-name))
	  (t16-name  (string-append "t16_" c-name)))
      (add-type
       scheme-sym
       type-name
       ;fn-convert-to-scm 
       (lambda (x) (list c-name "_to_SCM(" x ")"))
       ;fn-convert-from-scm 
       (lambda (x) (list "((" type-name ")SCM_CDR(" x "))"))
       ;fn-scm-is-a
       (lambda (x) 
	 (list "(SCM_NIMP(" x ") && (SCM_TYP16(" x ") == " t16-name
	       "))"))))))

(define-public (new-function scheme-sym
			     ret-type c-name param-list
			     description)
  (new-function-old c-name scheme-sym ret-type 
		    (map arg-type-of param-list)
		    (list
		     (param-list->description-head 
		      scheme-sym (ret-type-type-of ret-type) param-list)
		     description)
		    `(doc  ,description)
		    `(args ,param-list)))

(define (param-list->description-head scheme-sym ret-type param-list)
  (list
   (list 
    "(" scheme-sym (map (lambda (x) (list " " (cadr x))) param-list) ")\n")
   (if (null? param-list)
       ""
       (list (separate-by
	      (map (lambda (x) (list (cadr x) " is a " (car x))) param-list)
	      ", ")
	     ".\n"))
   (if (eq? 'void ret-type)
       " No return value.\n"
       (list " Returns " ret-type ".\n"))))

(define (split-at-char char str)
  (let ((len (string-length str)))
    (let loop ((i 0)
	       (start 0)
	       (strings '()))
      (cond
       ((= i len)
	(reverse (cons (substring str start i) strings))) ;; return line
       ((eq? (string-ref str i) char)
	(loop (+ i 1) (+ i 1) (cons (substring str start i) strings)))
       (else
	(loop (+ i 1) start strings))))))

(define (gen-c-comment input-text)
  (let ((text (split-at-char #\newline (flatten-string input-text))))
    (cond
     ((null? text) '())
     (else
      (let loop ((txt (cdr text))
		 (out (list (list "/* " (car text) "\n"))))
	(cond 
	 ((null? txt) (reverse (cons " */\n" out))) ;; return line
	 (else
	  (loop (cdr txt)
		(cons (list " * " (car txt) "\n") out)))))))))


(define (new-function-old c-name scheme-name ret-type types description
			  . extras)
  (let ((ret-type-type (ret-type-type-of ret-type))
        (ret-type-options (ret-type-options-of ret-type)))
    
    (set! scheme-name (prefix-name scheme-name))
    (if function-index-generator
        ((function-index-generator 'add) scheme-name))
    (gwrap-c-doc-function scheme-name
                          (caar description) 
                          (list "<em>" (cdar description) "</em><br>\n"
                                (cdr description)))
    (set! types (map get-type types))
    (set! ret-type-type (get-type ret-type-type))
    
    (let ((orig-doc  (fn-option extras 'doc (lambda () "")))
          (orig-args (fn-option extras 'args (lambda () '()))))
      (if gen-header-file
          (let ((subs `((doc    ,(gen-c-comment orig-doc))
                        (ret    ,(c-name-of ret-type-type))
                        (fnname ,c-name)
                        (args   ,(separate-by 
                                  (map 
                                   (lambda (arg)
                                     (list (c-name-of (get-type (car arg)))
                                           " " (cdr arg)))
                                   orig-args)
                                  ", ")))))
            (gw:trans-write 'gen-header-file 'declarations subs
                            "%doc%%ret% %fnname% (%args%);\n\n")))
      
      (let ((nargs (length types))
            (params (make-params types))
            (fn-c-wrapper (string-append "gwrap_" c-name))
            (fn-c-string  (string-append "gwrap_" c-name "_s"))
            (use-extra-params? (> (length types) 10)))
        (let ((subs
               `((fn-c-string  ,fn-c-string)
                 (scheme-name  ,scheme-name)
                 (fn-c-wrapper ,fn-c-wrapper)
                 (param-decl
                  ,(make-param-declarations params use-extra-params?))
                 (ret-var-decl
                  ,(if (eq? ret-type-type (get-type 'void))
                       "/* no return variable */"
                       (list (c-name-of ret-type-type)  " ret;")))
                 (c-param-protos ,(make-c-param-protos params))
                 (extra-param-assertions
                  ,(if use-extra-params?
                       (make-extra-param-assertions 
                        (- (length params) 9) scheme-name)
                       ""))
                 (param-assertions
                  ,(make-param-assertions params fn-c-string))
                 (param-assignments
                  ,(make-param-assignments params))
                 (ret-val-assignment
                  ,(if (eq? ret-type-type (get-type 'void))
                       (list "   " c-name "( " (make-param-list params) " );\n")
                       (list "   ret = " c-name 
                             "( " (make-param-list params) " );\n")))
                 (ret-val->scm ,(c-to-scm ret-type-type "ret"))
                 (param-cleanup ,(make-param-cleanup params orig-args))
                 (ret-val-cleanup
                  ,(cond
                    ((memq 'no-cleanup ret-type-options)
                     (list "  /* no-cleanup ordered for return value */\n"))
                    ((memq 'cleanup ret-type-options)
                     (list "  " (make-c-cleanup ret-type-type "ret") ";" ))
                    ((c-cleanup-ret-default?-of ret-type-type)
                     (list "  " (make-c-cleanup ret-type-type "ret" ";" )))
                    (else
                     (list "  /* no-cleanup default for return value */\n")))))))
          (gw:trans-write
           'c-file 'fn-wrappers subs "\
static char * %fn-c-string% = \"%scheme-name%\";
static SCM %fn-c-wrapper%  (%param-decl%) {
  SCM gw_scm_result;
  %ret-var-decl%
%c-param-protos%
     
  /* Type checks */
%extra-param-assertions%
%param-assertions%
  /* Type conversions */
%param-assignments%
  /* Call function */
  SCM_DEFER_INTS;
%ret-val-assignment%
  SCM_ALLOW_INTS;

  /* Cleanup C params (do it now b/c it may reclaim memory needed below) */
%param-cleanup%

  gw_scm_result = %ret-val->scm%;

  /* Cleanup C return val */
%ret-val-cleanup%

  return(gw_scm_result);
}\n\n"))
        
        (gwrap-c-output-c
         'fn-inits
         "  scm_c_define_gsubr( "
         fn-c-string ", "
         (if use-extra-params? 9 nargs) ", 0, " 
         (if use-extra-params? "1, " "0, ")
         (if c++-file?
             "(SCM (*) (...))"
             "(SCM (*) ())") fn-c-wrapper ");\n")
	(if (not (eqv? (string-ref scheme-name 0) #\%))
	    (gwrap-c-output-c
	     'fn-inits
	     "  scm_c_export( "
	     fn-c-string ", " 
	     "NULL);\n")
	    )))))

;;; Utility functions

(define (make-params types)
  (let ((extras? (> (length types) 10)))
    (let loop ((t types) (n 0))
      (cond ((null? t) '())
	    (else
	     (cons
	      (make-param (string-append "param" (number->string n))
			  (car t) 
			  n
			  extras?)
	      (loop (cdr t) (+ n 1))))))))

(define (make-param-declarations params extras?)
  (let loop ((params params)
	     (index  0))
    (cond ((null? params) 
	   '())
	  ((and (= index 9) extras?)
	   "SCM scm_restargs ")
	  (else
	   (cons
	    (list
	     "SCM " (s-name-of (car params)) 
	     (if (null? (cdr params))
		 " "
		 ", "))
	    (loop (cdr params) (+ index 1)))))))

(define (c-type-string param)
  (c-name-of (type-of param)))

(define (make-c-param-protos params)  
  (cond ((null? params) '())
	(else
	 (cons
	  (list "  " (c-name-of (type-of (car params))) 
		" " (name-of (car params)) ";\n")
	  (make-c-param-protos (cdr params))))))

(define (make-extra-param-assertions n-extra-params procname)
  (list
   "  if ( " n-extra-params 
   " != SCM_INUM(scm_length(scm_restargs)) ) {\n"
   "    scm_wrong_num_args(scm_makfrom0str(\"" procname "\"));\n"
   "  }\n"
   ))

(define (make-param-assertions params fn-c-string)
  (cond ((null? params) '())
	(else
	 (cons
	  (let ((param (car params)))
	    (list
	     "  SCM_ASSERT("
	     (make-isa-check (type-of param) (s-name-of param))  ","
	     (s-name-of param) ","
	     "SCM_ARG" 
	     (if (< (number-of param) 5)
		 (+ 1 (number-of param))
		 "n")
	     ","
	     fn-c-string ");\n"))
	  (make-param-assertions (cdr params) fn-c-string)))))

(define (make-param-list params)  
  (cond ((null? params) '())
	(else
	 (cons
	  (list 
	   (name-of (car params))
	   (if (null? (cdr params))
	       " "
	       ", "))
	  (make-param-list (cdr params))))))

(define-public (c-to-scm ret-type var)
  (make-conversion-to-scm ret-type var))

(define (make-param-assignments params)
  (cond ((null? params) '())
        (else
         (cons
          (list
           "  " (name-of (car params)) " = " 
           (make-conversion-from-scm (type-of (car params))
                                     (s-name-of (car params)))
           ";\n")
          (make-param-assignments (cdr params))))))

(define (make-param-cleanup params orig-args)
  (if (null? params)
      '()
      (cons
       (cond
        ((memq 'no-cleanup (arg-options-of (car orig-args)))
         (list "  /* no-cleanup ordered for " (name-of (car params)) " */\n"))
        ((memq 'cleanup (arg-options-of (car orig-args)))
         (list "  " (make-c-cleanup (type-of (car params))
                                    (name-of (car params))) ";\n"))
        ((c-cleanup-arg-default?-of (type-of (car params)))
         (list "  " (make-c-cleanup (type-of (car params))
                                    (name-of (car params))) ";\n"))         
        (else
         (list "  /* no-cleanup default for " (name-of (car params)) " */\n")))
       
       (make-param-cleanup (cdr params) (cdr orig-args)))))

;;; Function parameters
(define (make-rest-arg-s-name count)
  (list
   "SCM_CAR("
   (let loop ((i 0))
     (cond ((>= i count) "scm_restargs")
	   (else
	    (list
	     "SCM_CDR(" (loop (+ i 1)) ")"))))
   ")"))

(define (make-param name type number extras?)
  (vector name
	  (if (and extras? (>= number 9))
	      (make-rest-arg-s-name (- number 9))
	      (string-append "scm_" name))
	  type 
	  number))

(define (name-of x)   (vector-ref x 0))
(define (s-name-of x) (vector-ref x 1))
(define (type-of x)   (vector-ref x 2))
(define (number-of x) (vector-ref x 3))

;;; C types

;; A return-type can be a symbol (naming the type) or a list of the form
;; '(type [ options ... ]).

(define-public (ret-type-type-of r) (if (pair? r) (car r) r))
(define-public (ret-type-options-of r) (if (pair? r) (cdr r) '()))

;; An arg is the (int foo [ options ... ]) bit.
(define-public (arg-type-of arg) (car arg))
(define-public (arg-name-of arg) (cadr arg))
(define-public (arg-options-of arg) (cddr arg))

(define-public (make-c-type c-name 
			    fn-convert-to-scm fn-convert-from-scm fn-scm-is-a)
  (vector 'simple c-name fn-convert-to-scm fn-convert-from-scm fn-scm-is-a))

(define-public (make-complex-c-type c-name 
                                    fn-convert-to-scm
                                    fn-convert-from-scm
                                    fn-scm-is-a
                                    c-cleanup-arg-default?
                                    c-cleanup-ret-default?
                                    fn-c-cleanup)
  (vector 'complex
          c-name 
          fn-convert-to-scm fn-convert-from-scm fn-scm-is-a
          c-cleanup-arg-default? c-cleanup-ret-default? fn-c-cleanup))

(define-public (c-type-type-of x) (vector-ref x 0))
(define-public (c-name-of x) (vector-ref x 1))
(define-public (fn-to-scm-of x) (vector-ref x 2))
(define-public (fn-from-scm-of x) (vector-ref x 3))
(define-public (fn-isa-check-of x) (vector-ref x 4))
(define-public (c-cleanup-arg-default?-of x)
  (if (eq? (c-type-type-of x) 'simple)
      #f
      (vector-ref x 5)))
(define-public (c-cleanup-ret-default?-of x)
  (if (eq? (c-type-type-of x) 'simple)
      #f
      (vector-ref x 6)))
(define-public (fn-c-cleanup-of x)
  (if (eq? (c-type-type-of x) 'simple)
      (lambda (x) (list "/* simple type: no cleanup */"))
      (vector-ref x 7)))

(define-public (make-conversion-to-scm type var)
  ((fn-to-scm-of type) var))
(define-public (make-conversion-from-scm type var)
  ((fn-from-scm-of type) var))
(define-public (make-isa-check type var)
  ((fn-isa-check-of type) var))
(define-public (make-c-cleanup type var)
  ((fn-c-cleanup-of type) var))
