/*
 *  Copyright (C) 1999, 2000 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "dialogs.h"
#include "download-dirs.h"

/* Structure that keeps the widgets with the information of the new download
 * file */
typedef struct 
{
    GtkWidget *dlg;              /* GNOME dialog */
    GtkWidget *url;              /* Entry */
    GtkWidget *download_dir;     /* Entry */
    GtkWidget *disable_proxy;    /* Toggle button */
    GtkWidget *disable_auto_dl;  /* Check button */
} NewDlData;


/* Call back function used when the OK and CANCEL buttons get pressed */
static void
ok_new_file (GtkWidget *widget, NewDlData *new_dl_data)
{
    FileData *file_data;
    gchar *url, *dir, *filename;
    gint length;
    gboolean disable_proxy, disable_auto_dl;
    GtkWidget *error;

    /* Get the URL data and make sure it's right */
    url = g_strdup (gtk_entry_get_text (GTK_ENTRY (new_dl_data->url)));
    if (!file_data_url_valid(url)) {
	error = gnome_message_box_new (
	    _ ("Invalid URL"), GNOME_MESSAGE_BOX_ERROR, 
	    GNOME_STOCK_BUTTON_OK, NULL);
	gnome_dialog_set_parent (GNOME_DIALOG (error), 
                                 GTK_WINDOW (new_dl_data->dlg));
	gnome_dialog_run_and_close (GNOME_DIALOG (error));
        g_free (url);
	return;
    }

    /* Get download filename data */
    filename = &url[strlen (url)];
    while (*filename != '/' && filename != url)
	filename--;
    filename++;

    /* TODO: If filename == NULL instead of reminding the user that it is not
     * possible yet to do multiple downloads, when the group stuff is 
     * implemented this should enter a new dialog like dialog-multiple and 
     * ask there for download depth and get the available files to download. 
     * Then the user can remove what he doesn't want. */
    if (strlen (filename) == 0) {
	error = gnome_message_box_new (
	    _ ("Multiple download not implemented yet"), 
            GNOME_MESSAGE_BOX_ERROR, 
	    GNOME_STOCK_BUTTON_OK, NULL);
	gnome_dialog_set_parent (GNOME_DIALOG (error), 
                                 GTK_WINDOW (new_dl_data->dlg));
	gnome_dialog_run_and_close (GNOME_DIALOG (error));
        g_free (url);
	return;
    }

    /* Get download directory data */
    dir = gtk_entry_get_text (GTK_ENTRY (new_dl_data->download_dir));

    /* Get the disable proxy option data */
    disable_proxy =
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
            new_dl_data->disable_proxy));

    /* Get the disable auto download option data */
    disable_auto_dl = 
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
	    new_dl_data->disable_auto_dl));

    /* Create the file data with the information */
    file_data = file_data_create (url, dir, disable_proxy,
                                  disable_auto_dl);
    
    /* Add the new download file to the file list (if not a duplicate) */
    if (file_list_add (GTK_CLIST (dl_file_list), file_data) == FALSE) {
        /* URL on the file list */
        gchar *error_msg;
        
        error_msg = g_strconcat (_ ("URL:\n"), url, "\n",
                                 _ ("already in the list"), NULL);
        error = gnome_message_box_new (error_msg, GNOME_MESSAGE_BOX_ERROR,
                                       GNOME_STOCK_BUTTON_OK, NULL);
        gnome_dialog_run_and_close (GNOME_DIALOG (error));
        file_data_free (file_data);
        g_free (error_msg);
    }
    
    /* If dialog new or add changes default download dir update the def dl 
     * dir */
    if (gtm_pref.addnew_changes_def_dl_dir) {
        g_free (gtm_pref.def_download_dir);
        length = strlen (dir);
        if (length != 0 && dir[length - 1] != '/')
            gtm_pref.def_download_dir = g_strconcat (dir, "/", NULL);
        else
            gtm_pref.def_download_dir = g_strdup (dir);
    }

    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (new_dl_data->dlg);
    g_free (url);
    g_free (new_dl_data);
}

static void
cancel_new_file (GtkWidget *widget, NewDlData *new_dl_data)
{
    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (new_dl_data->dlg);
    g_free (new_dl_data);
}

/* If possible, calculate the directory by applying pattern matching
   to the URL */
static void
match_cb (GtkWidget *widget, NewDlData *new_dl_data)
{
    gchar *url, *dir;
    url = g_strdup (gtk_entry_get_text (GTK_ENTRY (new_dl_data->url)));
    if (download_dirs_suggest_dir (url, &dir)) {
        gtk_entry_set_text (GTK_ENTRY (new_dl_data->download_dir), dir);
    }
    g_free(url);
}

/* Dialog used to add a new file to download */
void
dialog_new (void)
{
    GtkWidget *label, *file_entry, *hbox0, *match_button, *hbox1;
    NewDlData *new_dl_data;

    new_dl_data = g_new (NewDlData, 1);

    /* Create dialog box */
    new_dl_data->dlg = 
	gnome_dialog_new (_ ("GTM - New file to download"),
			  GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL,
			  NULL);
    gnome_dialog_set_parent (GNOME_DIALOG (new_dl_data->dlg), 
			     GTK_WINDOW (gtm_app));

    /* Set its contents */
    label = gtk_label_new (_ ("Please enter URL of the file to download:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
			label, FALSE, FALSE, 0);
    new_dl_data->url = gtk_entry_new ();
    gtk_widget_set_usize (new_dl_data->url, 300, -2);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
			new_dl_data->url, FALSE, FALSE, 0);
    gtk_widget_grab_focus (new_dl_data->url);

    label = gtk_label_new (_ ("Download dir:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
			label, FALSE, FALSE, 0);
    file_entry = gnome_file_entry_new ("NewDownloadDir", 
                                       _("GTM - New Download Dir"));
    new_dl_data->download_dir = 
	gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (file_entry));
    gnome_file_entry_set_directory (GNOME_FILE_ENTRY (file_entry), 1);
    gtk_entry_set_text (GTK_ENTRY (new_dl_data->download_dir), 
			gtm_pref.def_download_dir);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox),
			file_entry, FALSE, FALSE, 0);

    /* match button, only show if user has defined some rules */
    if (download_dirs_get_total_dirs() > 0) {
        match_button = gtk_button_new_with_label 
            (_ ("Suggest directory by pattern matching"));
        hbox0 = gtk_hbox_new(FALSE, FALSE);
        gtk_box_pack_start (GTK_BOX (hbox0), match_button,
                            TRUE, FALSE, 0);
        gtk_box_pack_start
            (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox),
             hbox0, FALSE, FALSE, 0);
        gtk_signal_connect (GTK_OBJECT (match_button), 
                            "clicked", 
                            GTK_SIGNAL_FUNC (match_cb),
                            (gpointer) new_dl_data);
    }

    hbox1 = gtk_hbox_new(FALSE, FALSE);
    new_dl_data->disable_proxy = 
	gtk_check_button_new_with_label (_ ("Disable proxy"));
    gtk_box_pack_start (GTK_BOX (hbox1), new_dl_data->disable_proxy,
                        FALSE, FALSE, 0);
    new_dl_data->disable_auto_dl =
        gtk_check_button_new_with_label (_ ("Disable auto download"));
    gtk_box_pack_start (GTK_BOX (hbox1), new_dl_data->disable_auto_dl,
                        FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox),
                        hbox1, FALSE, FALSE, 0);

    /* Configure the dialog box */
    gnome_dialog_set_close (GNOME_DIALOG (new_dl_data->dlg), FALSE);
    gnome_dialog_set_default (GNOME_DIALOG (new_dl_data->dlg), GNOME_OK);

    /* Set the signal handlers */
    gnome_dialog_button_connect (GNOME_DIALOG (new_dl_data->dlg), GNOME_OK,
				 GTK_SIGNAL_FUNC (ok_new_file),
				 (gpointer) new_dl_data);
    gnome_dialog_button_connect (GNOME_DIALOG (new_dl_data->dlg), GNOME_CANCEL,
				 GTK_SIGNAL_FUNC (cancel_new_file),
				 (gpointer) new_dl_data);
    gtk_signal_connect (GTK_OBJECT (new_dl_data->url), "activate",
			GTK_SIGNAL_FUNC (ok_new_file), 
			(gpointer) new_dl_data);

    /* Show the dialog box and every thing it contains */
    gtk_widget_show_all (new_dl_data->dlg);
}

