/*
 *  Copyright (C) 1999, 2000 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "gtm_applet.h"
#include "dialogs.h"

/* Structure that keeps the widgets with the information of the new download
 * file */
typedef struct 
{
    GtkWidget *dlg;              /* GNOME dialog */
    GtkWidget *url_list;         /* GtkCList */
    GtkWidget *download_dir;     /* Entry */
    GtkWidget *disable_proxy;    /* Toggle button */
    GtkWidget *disable_auto_dl;  /* Check button */
} NewDlData;

#define ASC2HEXD(x) (((x) >= '0' && (x) <= '9') ? \
                     ((x) - '0') : (toupper (x) - 'A' + 10))


/* Call back function used when the OK and CANCEL buttons get pressed */
static void
ok_add_files (GtkWidget *widget, NewDlData *new_dl_data)
{
    gchar *url, *dir, *error_msg, empty[] = "";
    gint length, pos;
    gboolean disable_proxy, disable_auto_dl;

    /* Get download directory data */
    if (!gtm_applet_prop.disable_dl_dir)
        dir = gtk_entry_get_text (GTK_ENTRY (new_dl_data->download_dir));
    else
        dir = empty;

    /* Get the disable proxy option data */
    disable_proxy =
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
            new_dl_data->disable_proxy));

    /* Get the disable auto download option data */
    disable_auto_dl = 
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
	    new_dl_data->disable_auto_dl));

    /* Add the URL from the url list to the file list */
    pos = 0;
    error_msg = g_strconcat (_ ("URL:\n"), NULL);
    while ((url = (gchar *)
            gtk_clist_get_row_data (GTK_CLIST (new_dl_data->url_list),
                                    pos++)) != NULL)
        /* Add the new download file to the file list (if not a duplicate) */
        if (gtm_add_url (url, dir, disable_proxy, disable_auto_dl) == FALSE)
            /* URL on the file list */
            error_msg = g_strconcat (error_msg, url, "\n", NULL);
    
    /* If there where errors give them to the user */
    if (strcmp (error_msg, _("URL:\n")) != 0) {
        GtkWidget *error;
            
        error_msg = g_strconcat (error_msg, _ ("already in the list"),
                                 NULL);
        error = gnome_message_box_new (error_msg, GNOME_MESSAGE_BOX_ERROR,
                                       GNOME_STOCK_BUTTON_OK, NULL);
        gnome_dialog_run_and_close (GNOME_DIALOG (error));
    }

    /* If dialog new or add changes default download dir update the def dl 
     * dir */
    if (gtm_applet_prop.addnew_changes_def_dl_dir) {
        g_free (gtm_applet_prop.def_download_dir);
        length = strlen (dir);
        if (length != 0 && dir[length - 1] != '/')
            gtm_applet_prop.def_download_dir = g_strconcat (dir, "/", NULL);
        else
            gtm_applet_prop.def_download_dir = g_strdup (dir);
    }

    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (new_dl_data->dlg);
    g_free (error_msg);
    g_free (new_dl_data);
}

static void
cancel_add_files (GtkWidget *widget, NewDlData *new_dl_data)
{
    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (new_dl_data->dlg);
    g_free (new_dl_data);
}


/* Function to replace occurrences of %?? in the corresponding character for
 * the given URL. */
static void
decode_url (gchar *url)
{
    gchar *orig, *dest;
    
    orig = dest = url;
    while (*orig != '\0') {
        if (*orig == '%' && *(orig+1) != '\0' && *(orig+2) != '\0'
            && isxdigit (*(orig+1)) && isxdigit (*(orig+2))) {
            *dest = (ASC2HEXD (*(orig+1)) << 4) + ASC2HEXD (*(orig+2));
            orig += 2;
        } else
            *dest = *orig;
        orig++;
        dest++;
    }
    *dest = '\0';
}


/* Function to create a list of valid URL */
static GList *
url_list_build (gchar *uri_list)
{
    GList *urls, *p_url;
    gchar *url, *at, *slash;

    urls = gnome_uri_list_extract_uris (uri_list);
    p_url = urls;
    while (p_url) {
        url = p_url->data;
        decode_url (url);
        
        /* First remove "file:" and if there is a "/#ftp:" change it
         * to "ftp://" */
        if (strncmp (url, "file:", 5) == 0)
            url += 5;
        if (strncmp (url, "/#ftp:", 6) == 0)
            memcpy (url, "ftp://", 6);
        if (strncmp (url, "ftp://", 6) == 0) {
            /* It is a FTP url so we need to remove the login/password info
             * because with wget we cannot support it without compromising
             * user information. */
            at = strchr (url, '@');
            slash = strchr (url + 6, '/');
            if (at != NULL && (slash == NULL || at < slash)) {
                /* Remove the login */
                url = g_strconcat ("ftp://", at + 1, NULL);
                g_free (p_url->data);
                p_url->data = url;
            }
        }

        /* Remove URL from the list if it is not a valid URL */
        if (strncasecmp (url, "http://", 7) != 0
            && strncasecmp (url, "ftp://", 6) != 0) {
            g_free (p_url->data);
            p_url->data = NULL;
            p_url = g_list_next (p_url);
            urls = g_list_remove (urls, NULL);
        } else
            p_url = g_list_next (p_url);
    }

    return urls;
}


/* Function to create a url list */
static GtkWidget *
url_list_create (gchar *uri_list)
{
    GtkWidget *url_list;
    GList *urls, *p_url;
    gchar *data[1];
    gint row;

    url_list = gtk_clist_new (1);
    gtk_clist_set_column_auto_resize (GTK_CLIST (url_list), 0, TRUE);
    gtk_widget_set_usize (url_list, 300, 100);
    urls = url_list_build (uri_list);
    for (p_url = urls; p_url; p_url = g_list_next (p_url)) {
        /* Add the URL to the url list */
        data[0] = p_url->data;
        row = gtk_clist_append (GTK_CLIST (url_list), data);
        gtk_clist_set_row_data_full (GTK_CLIST (url_list), row,
                                     g_strdup (p_url->data),
                                     (GtkDestroyNotify) g_free);
    }
    gnome_uri_list_free_strings (urls);

    return url_list;
}


/* Dialog used to add files to download */
void
dialog_add (gchar *uri_list)
{
    GtkWidget *label, *file_entry, *hbox1, *sw;
    NewDlData *new_dl_data;

    if (gtm_applet_prop.confirm_dnd_url) {
        new_dl_data = g_new (NewDlData, 1);

        /* Create dialog box */
        new_dl_data->dlg = 
            gnome_dialog_new (_ ("GTM Applet - Add files to download"),
                              GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL,
                              NULL);

        /* Set it's contents */
        label = gtk_label_new (_ ("URLs to be added"));
        gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
                            label, FALSE, FALSE, 0);
        new_dl_data->url_list = url_list_create (uri_list);
        sw = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_add (GTK_CONTAINER (sw),
                           GTK_WIDGET (new_dl_data->url_list));
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC);
        gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
                            sw, TRUE, TRUE, 0);

        if (!gtm_applet_prop.disable_dl_dir) {
            label = gtk_label_new (_ ("Download dir:"));
            gtk_box_pack_start (GTK_BOX (
                GNOME_DIALOG (new_dl_data->dlg)->vbox),
                                label, FALSE, FALSE, 0);
            file_entry = gnome_file_entry_new ("AddDownloadDir", 
                                               _("GTM - Add Download Dir"));
            new_dl_data->download_dir = 
                gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (file_entry));
            gnome_file_entry_set_directory (GNOME_FILE_ENTRY (file_entry), 1);
            gtk_entry_set_text (GTK_ENTRY (new_dl_data->download_dir), 
                                gtm_applet_prop.def_download_dir);
            gtk_box_pack_start (GTK_BOX (
                GNOME_DIALOG (new_dl_data->dlg)->vbox),
                                file_entry, FALSE, FALSE, 0);
        }

        hbox1 = gtk_hbox_new (FALSE, FALSE);
        new_dl_data->disable_proxy = 
            gtk_check_button_new_with_label (_ ("Disable proxy"));
        gtk_box_pack_start (GTK_BOX (hbox1), new_dl_data->disable_proxy, 
                            FALSE, FALSE, 0);
        new_dl_data->disable_auto_dl =
            gtk_check_button_new_with_label (_ ("Disable auto download"));
        gtk_box_pack_start (GTK_BOX (hbox1), new_dl_data->disable_auto_dl,
                            FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox),
                            hbox1, FALSE, FALSE, 0);

        /* Configure the dialog box */
        gnome_dialog_set_close (GNOME_DIALOG (new_dl_data->dlg), FALSE);
        gnome_dialog_set_default (GNOME_DIALOG (new_dl_data->dlg), GNOME_OK);

        /* Set the signal handlers */
        gnome_dialog_button_connect (GNOME_DIALOG (new_dl_data->dlg), GNOME_OK,
                                     GTK_SIGNAL_FUNC (ok_add_files),
                                     (gpointer) new_dl_data);
        gnome_dialog_button_connect (GNOME_DIALOG (new_dl_data->dlg),
                                     GNOME_CANCEL,
                                     GTK_SIGNAL_FUNC (cancel_add_files),
                                     (gpointer) new_dl_data);

        /* Show the dialog box and every thing it contains */
        gtk_widget_show_all (new_dl_data->dlg);
    } else {
        GList *urls, *p_url;
        gchar *error_msg, empty[] = "";

        /* Build a valid list of URL */
        urls = url_list_build (uri_list);
        error_msg = g_strconcat (_ ("URL:\n"), NULL);
        for (p_url = urls; p_url; p_url = g_list_next (p_url))
            /* Add the new download file to the file list
             * (if not a duplicate) */
            if (!gtm_applet_prop.disable_dl_dir) {
                if (gtm_add_url (p_url->data, gtm_applet_prop.def_download_dir,
                                 FALSE, FALSE) == FALSE)
                    /* URL on the file list */
                    error_msg = g_strconcat (error_msg,
                                             p_url->data, "\n", NULL);
            } else {
                if (gtm_add_url (p_url->data, empty, FALSE, FALSE) == FALSE)
                    /* URL on the file list */
                    error_msg = g_strconcat (error_msg,
                                             p_url->data, "\n", NULL);
            }

        /* If there where errors give them to the user */
        if (strcmp (error_msg, _("URL:\n")) != 0) {
            GtkWidget *error;

            error_msg = g_strconcat (error_msg, _ ("already in the list"),
                                     NULL);
            error = gnome_message_box_new (error_msg, GNOME_MESSAGE_BOX_ERROR,
                                           GNOME_STOCK_BUTTON_OK, NULL);
            gnome_dialog_run_and_close (GNOME_DIALOG (error));
        }

        /* Free unused memory */
        gnome_uri_list_free_strings (urls);
        g_free (error_msg);
    }
}
