/*
 *  Copyright (C) 2005 Robert Staudinger
 *
 *  This software is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "nsString.h"
#include "nsCOMPtr.h"
#include "nsICommandManager.h"
#include "nsIInterfaceRequestorUtils.h"

#include <gtkmozedit/gme-command-manager-private.h>
#include <gtkmozedit/gme-command-params-private.h>
#include <gtkmozedit/gme-observer-private.h>
#include <gtkmozedit/dom/gme-dom-window-private.h>

enum {
	PROP_0,
	PROP_WRAPPED_PTR,
	_NUM_PROPS
};

static GmeSupportsClass *gme_command_manager_parent_class = NULL;

static void
instance_init (GmeCommandManager *self)
{
	self->wrapped_ptr = NULL;
	self->is_disposed = FALSE;
}

static void
instance_dispose (GObject *instance)
{
	GmeCommandManager *self = GME_COMMAND_MANAGER (instance);

	if (self->is_disposed)
		return;

	if (self->wrapped_ptr) NS_RELEASE (self->wrapped_ptr);
	self->wrapped_ptr = NULL;
	self->is_disposed = TRUE;

	gme_command_manager_parent_class->dispose (G_OBJECT (self));
}

static void
set_property (GObject      *object,
	      guint         prop_id,
	      GValue const *value,
	      GParamSpec   *pspec)
{
	GmeCommandManager *self = GME_COMMAND_MANAGER (object);
	nsISupports *ptr = NULL;
	nsCOMPtr<nsICommandManager> cmd_mgr;
	nsresult rv;

	switch (prop_id) {
	case PROP_WRAPPED_PTR:
		ptr = reinterpret_cast<nsISupports*>(g_value_get_pointer (value));
		/* strange, CallQueryInterface seems to fail while the below solution 
		rv = CallQueryInterface (ptr, &self->wrapped_ptr);
		*/
		cmd_mgr = do_GetInterface (ptr, &rv);
		if (NS_SUCCEEDED (rv)) {
			self->wrapped_ptr = cmd_mgr;
			NS_ADDREF (self->wrapped_ptr);
			/* constuction param, init parent */
			gme_supports_private_set_wrapped_ptr (GME_SUPPORTS (self), self->wrapped_ptr);
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
get_property (GObject    *object,
	      guint       prop_id,
	      GValue     *value,
	      GParamSpec *pspec)
{
	GmeCommandManager *self = GME_COMMAND_MANAGER (object);

	switch (prop_id) {
	case PROP_WRAPPED_PTR:
		g_value_set_pointer (value, (gpointer) self->wrapped_ptr);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
class_init (GmeCommandManagerClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	/* vfuncs */
	klass->dispose = instance_dispose;

	/* hook gobject vfuncs */
	gobject_class->dispose = instance_dispose;

	gme_command_manager_parent_class = (GmeSupportsClass*) g_type_class_peek_parent (klass);

	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	g_object_class_install_property (gobject_class,
		PROP_WRAPPED_PTR,
		g_param_spec_pointer ("wrapped-ptr", _("Wrapped Pointer"),
			_("Pointer to the wrapped c++ object"),
			(GParamFlags)(G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY)));
}

GType
gme_command_manager_get_gtype (void)
{
        static GType type = 0;
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (GmeCommandManagerClass),
                        NULL,           /* base_init */
                        NULL,           /* base_finalize */
                        (GClassInitFunc) class_init,
                        NULL,           /* class_finalize */
                        NULL,           /* class_data */
                        sizeof (GmeCommandManager),
                        0,              /* n_preallocs */
                        (GInstanceInitFunc) instance_init,
                };
                type = g_type_register_static (GME_TYPE_SUPPORTS, "GmeCommandManager", &info, (GTypeFlags)0);
        }
        return type;
}

void 
gme_command_manager_private_set_wrapped_ptr (GmeCommandManager *self, 
				      nsICommandManager *wrapped_ptr)
{
	g_assert (self && wrapped_ptr);
	self->wrapped_ptr = wrapped_ptr;
	NS_ADDREF (self->wrapped_ptr);
	/* constuction param, init parent */
	gme_supports_private_set_wrapped_ptr (GME_SUPPORTS (self), self->wrapped_ptr);
}

GmeCommandManager* 
gme_command_manager_new (nsICommandManager *wrapped_ptr)
{
	return GME_COMMAND_MANAGER (g_object_new (GME_TYPE_COMMAND_MANAGER, "wrapped-ptr", wrapped_ptr, NULL));
}

gboolean 
gme_command_manager_add_command_observer (GmeCommandManager *self, 
					  GmeObserver *observer, 
					  const gchar *cmd)
{
	nsresult rv;
	g_assert (self && observer);

	rv = self->wrapped_ptr->AddCommandObserver (observer->wrapped_ptr, cmd);
	return NS_SUCCEEDED (rv);
}

gboolean 
gme_command_manager_remove_command_observer (GmeCommandManager *self, 
					     GmeObserver *observer, 
					     const gchar *cmd)
{
	nsresult rv;
	g_assert (self && observer);

	rv = self->wrapped_ptr->RemoveCommandObserver (observer->wrapped_ptr, cmd);
	return NS_SUCCEEDED (rv);
}

gboolean 
gme_command_manager_do_command  (GmeCommandManager *self, 
				 const gchar *cmd, 
				 GmeCommandParams *params, 
				 GmeDOMWindow *win)
{
	nsresult rv;
	g_assert (self && params && win);

	rv = self->wrapped_ptr->DoCommand (cmd, params->wrapped_ptr, win->wrapped_ptr);
	return NS_SUCCEEDED (rv);
}

gboolean   
gme_command_manager_get_command_state (GmeCommandManager *self, 
				       const gchar *cmd, 
				       GmeDOMWindow *win, 
				       GmeCommandParams *params)
{
	nsresult rv;
	g_assert (self && params && win);

	rv = self->wrapped_ptr->GetCommandState (cmd, win->wrapped_ptr, params->wrapped_ptr);
	return NS_SUCCEEDED (rv);	
}

gboolean 
gme_command_manager_is_command_enabled (GmeCommandManager *self, 
					const gchar *cmd, 
					GmeDOMWindow *win)
{
	gboolean is_enabled;
	nsresult rv;
	g_assert (self && win);

	rv = self->wrapped_ptr->IsCommandEnabled (cmd, win->wrapped_ptr, &is_enabled);
	if (NS_SUCCEEDED (rv)) {
		return is_enabled;
	}
	return FALSE;
}

gboolean 
gme_command_manager_is_command_supported (GmeCommandManager *self, 
					  const gchar *cmd, 
					  GmeDOMWindow *win)
{
	gboolean is_supported;
	nsresult rv;
	g_assert (self && win);

	rv = self->wrapped_ptr->IsCommandSupported (cmd, win->wrapped_ptr, &is_supported);
	if (NS_SUCCEEDED (rv)) {
		return is_supported;
	}
	return FALSE;
}
