
#include <gtkmm/treeselection.h>
#include <gtkmm/private/treeselection_p.h>

// -*- c++ -*-
/* $Id: treeselection.ccg,v 1.17 2002/05/09 18:34:40 daniel Exp $ */

/* Copyright 1998-1999 The Gtk-- Development Team
 * Copyright 2001      Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtkmm/treeview.h>
#include <gtk/gtktreeselection.h>


namespace
{

extern "C"
{

void proxy_foreach_iter_callback(GtkTreeModel* model, GtkTreePath*, GtkTreeIter* iter, void* data)
{
  typedef Gtk::TreeSelection::SlotForeachIter SlotType;
  SlotType& slot = *static_cast<SlotType*>(data);

  try
  {
    slot(Gtk::TreeModel::iterator(model, iter));
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }
}

void proxy_foreach_path_callback(GtkTreeModel*, GtkTreePath* path, GtkTreeIter*, void* data)
{
  typedef Gtk::TreeSelection::SlotForeachPath SlotType;
  SlotType& slot = *static_cast<SlotType*>(data);

  try
  {
    slot(Gtk::TreeModel::Path(path, true));
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }
}

} // extern "C"


// This Signal Proxy allows the C++ coder to specify
// a SigC::Slot instead of a static function.
class SignalProxy_Select
{
public:
  typedef Gtk::TreeSelection::SlotSelect SlotType;

  SignalProxy_Select(const SlotType& slot) : slot_(slot) {}
  ~SignalProxy_Select();

  static gboolean gtk_callback(GtkTreeSelection*, GtkTreeModel* model, GtkTreePath* path,
                               gboolean path_currently_selected, void* data);

  static void gtk_callback_destroy(void* data);

private:
  SlotType slot_;
};

SignalProxy_Select::~SignalProxy_Select()
{}

gboolean SignalProxy_Select::gtk_callback(GtkTreeSelection*, GtkTreeModel* model, GtkTreePath* path,
                                          gboolean path_currently_selected, void* data)
{
  SignalProxy_Select *const self = static_cast<SignalProxy_Select*>(data);

  try
  {
    return (self->slot_)(Glib::wrap(model, true), // take copy
                         Glib::wrap(path,  true), // take copy
                         path_currently_selected);
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }

  return 0;
}

void SignalProxy_Select::gtk_callback_destroy(void* data)
{
  delete static_cast<SignalProxy_Select*>(data);
}

} // anonymous namespace


namespace Gtk
{

void TreeSelection::set_select_function(const SlotSelect& slot)
{
  // Create a signal proxy.  A pointer to this will be passed
  // through the callback's data parameter.  It will be deleted
  // when SignalProxy_Select::gtk_callback_destroy() is called.
  SignalProxy_Select *const pSignalProxy = new SignalProxy_Select(slot);

  gtk_tree_selection_set_select_function(gobj(),
      &SignalProxy_Select::gtk_callback, pSignalProxy,
      &SignalProxy_Select::gtk_callback_destroy);
}

Glib::RefPtr<TreeModel> TreeSelection::get_model()
{
  GtkTreeView *const tree_view = gtk_tree_selection_get_tree_view(gobj());
  return Glib::wrap(gtk_tree_view_get_model(tree_view), true);
}

TreeModel::iterator TreeSelection::get_selected()
{
  TreeModel::iterator iter;
  GtkTreeModel* model_gobject = 0;

  gtk_tree_selection_get_selected(gobj(), &model_gobject, iter.gobj());

  iter.set_model_gobject(model_gobject);
  return iter;
}

TreeModel::iterator TreeSelection::get_selected(Glib::RefPtr<TreeModel>& model)
{
  TreeModel::iterator iter;
  GtkTreeModel* model_gobject = 0;

  gtk_tree_selection_get_selected(gobj(), &model_gobject, iter.gobj());

  model = Glib::wrap(model_gobject, true);

  iter.set_model_refptr(model);
  return iter;
}

void TreeSelection::selected_foreach(const TreeSelection::SlotForeachIter& slot)
{
  SlotForeachIter slot_copy (slot);
  gtk_tree_selection_selected_foreach(gobj(), &proxy_foreach_iter_callback, &slot_copy);
}

void TreeSelection::selected_foreach(const TreeSelection::SlotForeachPath& slot)
{
  SlotForeachPath slot_copy (slot);
  gtk_tree_selection_selected_foreach(gobj(), &proxy_foreach_path_callback, &slot_copy);
}

} // namespace Gtk


namespace
{

const Glib::SignalProxyInfo TreeSelection_signal_changed_info =
{
  "changed",
  (GCallback) &Glib::SignalProxyNormal::slot0_void_callback,
  (GCallback) &Glib::SignalProxyNormal::slot0_void_callback
};

} // anonymous namespace


namespace Glib
{

Glib::RefPtr<Gtk::TreeSelection> wrap(GtkTreeSelection* object, bool take_copy /* = false */)
{
  return Glib::RefPtr<Gtk::TreeSelection>( dynamic_cast<Gtk::TreeSelection*> (Glib::wrap_auto ((GObject*)(object), take_copy)) );
  //We use dynamic_cast<> in case of multiple inheritance.
}

} /* namespace Glib */


namespace Gtk
{


/* The *_Class implementation: */

GType TreeSelection_Class::get_type()
{
  if(!gtype_) // create the GType if necessary
  {
    // TODO: This is currently just optimized away, apparently with no harm.
    // Is it actually needed?
    // Make sure that the parent type has been created.
    CppClassParent::CppObjectType::get_type();

    // Create the wrapper type, with the same class/instance size as the base type.
    register_derived_type(gtk_tree_selection_get_type(), (GClassInitFunc) &class_init_function);

    // Add derived versions of interfaces, if the C type implements any interfaces:
  }

  return gtype_;
}

void TreeSelection_Class::class_init_function(BaseClassType* klass)
{
  CppClassParent::class_init_function((CppClassParent::BaseClassType*) klass);
  klass->changed = &changed_callback;
}


void TreeSelection_Class::changed_callback(GtkTreeSelection* self)
{
  CppObjectType *const obj = dynamic_cast<CppObjectType*>(
      Glib::ObjectBase::_get_current_wrapper((GObject*)self));

  if(obj)
  {
    try
    {
      obj->on_changed();
    }
    catch(...)
    {
      Glib::exception_handlers_invoke();
    }
  }
  else
  {
    BaseClassType *const base = static_cast<BaseClassType*>(
        g_type_class_peek_parent(G_OBJECT_GET_CLASS(self)) // Get the parent class of the object class.
);
    g_assert(base != 0);

    if(base->changed)
      (*base->changed)(self);
  }
}


Glib::ObjectBase* TreeSelection_Class::wrap_new(GObject* o)
{
  return new TreeSelection((GtkTreeSelection*)(o));
}


/* The implementation: */

GtkTreeSelection* TreeSelection::gobj_copy()
{
  reference();
  return gobj();
}

Glib::RefPtr<TreeSelection> TreeSelection::wrap_specific_type(GtkTreeSelection* gobject, bool take_copy /* = false */) //static
{
  Glib::RefPtr<TreeSelection> refPtr;

  if(gobject)
  {
    //Check for an existing wrapper:
    TreeSelection* pCppObject = dynamic_cast<TreeSelection*>(Glib::ObjectBase::_get_current_wrapper(G_OBJECT(gobject)));
    if(pCppObject)
    {
      //Return the existing wrapper:
      refPtr = Glib::RefPtr<TreeSelection>(pCppObject);
    }
    else
    {
      //Create a new wrapper:
      refPtr = Glib::RefPtr<TreeSelection>( new TreeSelection(gobject) );
    }

    if(take_copy && refPtr)
      refPtr->reference();
  }

  return refPtr;
}

TreeSelection::TreeSelection(GtkTreeSelection* castitem)
: Glib::Object((GObject*)(castitem))
{}

TreeSelection::~TreeSelection()
{
}

TreeSelection::CppClassType TreeSelection::treeselection_class_; //Initialize static member.

GType TreeSelection::get_type()
{
  return treeselection_class_.get_type();
}

GType TreeSelection::get_base_type()
{
  return gtk_tree_selection_get_type();
}

void TreeSelection::set_mode(SelectionMode type)
{
  gtk_tree_selection_set_mode(gobj(), ((GtkSelectionMode)(type)));
}

SelectionMode TreeSelection::get_mode() const
{
  return ((SelectionMode)(gtk_tree_selection_get_mode(const_cast<GtkTreeSelection*>(gobj()))));
}

TreeView* TreeSelection::get_tree_view()
{
  return Glib::wrap(gtk_tree_selection_get_tree_view(gobj()));
}

void TreeSelection::select(const TreeModel::Path& path)
{
  gtk_tree_selection_select_path(gobj(), const_cast<GtkTreePath*>((path).gobj()));
}

void TreeSelection::select(const TreeModel::iterator& iter)
{
  gtk_tree_selection_select_iter(gobj(), const_cast<GtkTreeIter*>((iter).gobj()));
}

void TreeSelection::select(const TreeModel::Path& start_path, const TreeModel::Path& end_path)
{
  gtk_tree_selection_select_range(gobj(), const_cast<GtkTreePath*>((start_path).gobj()), const_cast<GtkTreePath*>((end_path).gobj()));
}

void TreeSelection::unselect(const TreeModel::Path& path)
{
  gtk_tree_selection_unselect_path(gobj(), const_cast<GtkTreePath*>((path).gobj()));
}

void TreeSelection::unselect(const TreeModel::iterator& iter)
{
  gtk_tree_selection_unselect_iter(gobj(), const_cast<GtkTreeIter*>((iter).gobj()));
}

bool TreeSelection::is_selected(const TreeModel::Path& path) const
{
  return gtk_tree_selection_path_is_selected(const_cast<GtkTreeSelection*>(gobj()), const_cast<GtkTreePath*>((path).gobj()));
}

bool TreeSelection::is_selected(const TreeModel::iterator& iter) const
{
  return gtk_tree_selection_iter_is_selected(const_cast<GtkTreeSelection*>(gobj()), const_cast<GtkTreeIter*>((iter).gobj()));
}

void TreeSelection::select_all()
{
  gtk_tree_selection_select_all(gobj());
}

void TreeSelection::unselect_all()
{
  gtk_tree_selection_unselect_all(gobj());
}


Glib::SignalProxy0<void> TreeSelection::signal_changed()
{
  return Glib::SignalProxy0<void>(this, &TreeSelection_signal_changed_info);
}


void Gtk::TreeSelection::on_changed()
{
  BaseClassType *const base = static_cast<BaseClassType*>(
      g_type_class_peek_parent(G_OBJECT_GET_CLASS(gobject_)) // Get the parent class of the object class.
);
  g_assert(base != 0);

  if(base->changed)
    (*base->changed)(gobj());
}


} // namespace Gtk

