/* Images
 *
 * GtkImage is used to display an image; the image can be in a number of formats.
 * Typically, you load an image into a GdkPixbuf, then display the pixbuf.
 *
 * This demo code shows some of the more obscure cases, in the simple
 * case a call to gtk_image_new_from_file() is all you need.
 *
 * If you want to put image data in your program as a C variable,
 * use the make-inline-pixbuf program that comes with GTK+.
 * This way you won't need to depend on loading external files, your
 * application binary can be self-contained.
 */

#include <gtkmm.h>
#include <stdio.h>
#include <errno.h>

class Example_Images : public Gtk::Window
{
public:
  Example_Images();
  virtual ~Example_Images();

protected:
  virtual void start_progressive_loading(Gtk::Image& image);

  //signal handler:
  virtual bool on_timeout();
  virtual void on_loader_area_prepared();
  virtual void on_loader_area_updated(int x, int y, int width, int height);

  //Member widgets:
  Gtk::VBox m_VBox;
  Gtk::Label m_Label_Image, m_Label_Animation, m_Label_Progressive;
  Gtk::Frame m_Frame_Image, m_Frame_Animation, m_Frame_Progressive;
  Gtk::Alignment m_Alignment_Image, m_Alignment_Animation, m_Alignment_Progressive;
  Gtk::Image m_Image_Progressive;
  Glib::RefPtr<Gdk::PixbufLoader> m_refPixbufLoader;

  SigC::Connection m_TimeoutConnection;
  FILE* m_image_stream;
};

//Called by DemoWindow;
Gtk::Window* do_images()
{
  return new Example_Images();
}

Example_Images::Example_Images()
: m_VBox(false, 8),
  m_Alignment_Image(0.5, 0.5, 0, 0),
  m_Alignment_Animation(0.5, 0.5, 0, 0)
{
  m_image_stream = 0;

  set_title("Images");
  set_border_width(8);

  m_VBox.set_border_width(8);
  add(m_VBox);


  /* Image */

  m_Label_Image.set_markup("<u>Image loaded from a file</u>");
  m_VBox.pack_start(m_Label_Image, false, false);

  m_Frame_Image.set_shadow_type(Gtk::SHADOW_IN);

  /* The alignment keeps the frame from growing when users resize
   * the window
   */
  m_Alignment_Image.add(m_Frame_Image);
  m_VBox.pack_start(m_Alignment_Image, false, false);

  Gtk::Image* pImage = Gtk::manage( new Gtk::Image(Glib::ustring("./gtk-logo-rgb.gif")) );		
  m_Frame_Image.add(*pImage);


  /* Animation */

  m_Label_Animation.set_markup("<u>Animation loaded from a file</u>");
  m_VBox.pack_start(m_Label_Animation, false, false);

  m_Frame_Animation.set_shadow_type(Gtk::SHADOW_IN);

  /* The alignment keeps the frame from growing when users resize
  * the window
  */
  m_Alignment_Animation.add(m_Frame_Animation);
  m_VBox.pack_start(m_Alignment_Animation, false, false);

  pImage = Gtk::manage( new Gtk::Image(Glib::ustring("./floppybuddy.gif")) );		
  m_Frame_Animation.add(*pImage);


  /* Progressive */

  m_Label_Progressive.set_markup("<u>Progressive image loading</u>");
  m_VBox.pack_start(m_Label_Progressive, false, false);

  m_Frame_Progressive.set_shadow_type(Gtk::SHADOW_IN);

  /* The alignment keeps the frame from growing when users resize
  * the window
  */
  m_Alignment_Progressive.add(m_Frame_Progressive);
  m_VBox.pack_start(m_Alignment_Progressive, false, false);

  /* Create an empty image for now; the progressive loader
   * will create the pixbuf and fill it in.
   */
  m_Frame_Progressive.add(m_Image_Progressive);

  start_progressive_loading(m_Image_Progressive);

  show_all();
}

Example_Images::~Example_Images()
{
}

void Example_Images::start_progressive_loading(Gtk::Image& image)
{
  m_TimeoutConnection = Gtk::Main::signal_timeout().connect(
      SigC::slot(*this, &Example_Images::on_timeout), 150);
}

bool Example_Images::on_timeout()
{
  /* This shows off fully-paranoid error handling, so looks scary.
   * You could factor out the error handling code into a nice separate
   * function to make things nicer.
   */

  if (m_image_stream)
  {
    guchar buf[256];
    GError* error = NULL;

    size_t bytes_read = fread (buf, 1, 256, m_image_stream);

    if (ferror (m_image_stream))
  	{
      Glib::ustring strMsg = "Failure reading image file 'alphatest.png': ";
      strMsg += g_strerror (errno);

      Gtk::MessageDialog dialog(strMsg, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE);
      dialog.run();

  	  fclose (m_image_stream);
  	  m_image_stream = NULL;
    	
  	  m_TimeoutConnection.disconnect();

  	  return false; /* uninstall the timeout */
  	}

    if (!m_refPixbufLoader->write(buf, bytes_read, error))
  	{
      Glib::ustring strMsg = "Failed to load image: ";
      strMsg += error->message;

      Gtk::MessageDialog dialog(strMsg, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE);
      dialog.run();

  	  fclose (m_image_stream);
  	  m_image_stream = NULL;
    	
  	  m_TimeoutConnection.disconnect();

  	  return false; /* uninstall the timeout */
  	}

    if (feof (m_image_stream))
   	{
   	  fclose (m_image_stream);
   	  m_image_stream = NULL;

   	  /* Errors can happen on close, e.g. if the image
   	   * file was truncated we'll know on close that
   	   * it was incomplete.
   	   */
   	  error = NULL;
   	  if (!m_refPixbufLoader->close(error))
   	  {
        Glib::ustring strMsg = "Failed to load image: ";
        strMsg += error->message;

        Gtk::MessageDialog dialog(strMsg, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE);
        dialog.run();

        m_refPixbufLoader.clear();
   	    m_TimeoutConnection.disconnect();
     	
   	    return false; /* uninstall the timeout */
   	  }
	
      m_refPixbufLoader.clear();
    }
  }
  else
  {
    m_image_stream = fopen ("./alphatest.png", "r");
    if (m_image_stream == NULL)
  	{
  	  Glib::ustring strMsg = "Unable to open image file 'alphatest.png': ";
      strMsg += g_strerror (errno);

      Gtk::MessageDialog dialog(strMsg, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE);
      dialog.run();

      m_TimeoutConnection.disconnect();

  	  return false; /* uninstall the timeout */
  	}

    if (m_refPixbufLoader)
  	{
  	  m_refPixbufLoader->close();
  	  m_refPixbufLoader.clear();
  	}

    m_refPixbufLoader = Gdk::PixbufLoader::create();
    m_refPixbufLoader->signal_area_prepared().connect(SigC::slot(*this, &Example_Images::on_loader_area_prepared));
    m_refPixbufLoader->signal_area_updated().connect(SigC::slot(*this, &Example_Images::on_loader_area_updated));
  }

  /* leave timeout installed */
  return true;
}

void Example_Images::on_loader_area_prepared()
{
  Glib::RefPtr<Gdk::Pixbuf> refPixbuf = m_refPixbufLoader->get_pixbuf();

  /* Avoid displaying random memory contents, since the pixbuf
   * isn't filled in yet.
   */
  refPixbuf->fill(0xaaaaaaff);
  m_Image_Progressive.set(refPixbuf);
}

void Example_Images::on_loader_area_updated(int x, int y, int width, int height)
{
  /* We know the pixbuf inside the GtkImage has changed, but the image
   * itself doesn't know this; so queue a redraw.  If we wanted to be
   * really efficient, we could use a drawing area or something
   * instead of a GtkImage, so we could control the exact position of
   * the pixbuf on the display, then we could queue a draw for only
   * the updated area of the image.
   */

  m_Image_Progressive.queue_draw();
}



