/*
 * Copyright (C) 2005 Ronald S. Bultje <rbultje@ronald.bitfreak.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mixerpad.h"

enum
{
  ARG_0,
  ARG_Z_ORDER,
#define DEFAULT_Z_ORDER 0
  ARG_ALPHA,
#define DEFAULT_ALPHA 1.0
  ARG_V_ALIGN,
#define DEFAULT_V_ALIGN GST_IMAGE_MIXER_PAD_V_ALIGN_BOTTOM
  ARG_H_ALIGN,
#define DEFAULT_H_ALIGN GST_IMAGE_MIXER_PAD_H_ALIGN_CENTER
  ARG_V_OFFSET,
#define DEFAULT_V_OFFSET 6
  ARG_H_OFFSET
#define DEFAULT_H_OFFSET 6
};

GST_DEBUG_CATEGORY_EXTERN (imagemixer_debug);
#define GST_CAT_DEFAULT imagemixer_debug

G_DEFINE_TYPE (GstImageMixerPad, gst_image_mixer_pad, GST_TYPE_REAL_PAD);

#define GST_IMAGE_MIXER_PAD_TYPE_V_ALIGN \
  (gst_image_mixer_pad_v_align_get_type ())
static GType
gst_image_mixer_pad_v_align_get_type (void)
{
  static GType v_align_type = 0;

  if (G_UNLIKELY (v_align_type == 0)) {
    static GEnumValue v_align_types[] = {
      {GST_IMAGE_MIXER_PAD_V_ALIGN_TOP, "top", "At the top"},
      {GST_IMAGE_MIXER_PAD_V_ALIGN_MIDDLE, "middle", "In the middle"},
      {GST_IMAGE_MIXER_PAD_V_ALIGN_BOTTOM, "bottom", "At the bottom"},
      {0, NULL, NULL},
    };

    v_align_type =
        g_enum_register_static ("GstImageMixerPadVAlign", v_align_types);
  }

  return v_align_type;
}

#define GST_IMAGE_MIXER_PAD_TYPE_H_ALIGN \
  (gst_image_mixer_pad_h_align_get_type ())
static GType
gst_image_mixer_pad_h_align_get_type (void)
{
  static GType h_align_type = 0;

  if (G_UNLIKELY (h_align_type == 0)) {
    static GEnumValue h_align_types[] = {
      {GST_IMAGE_MIXER_PAD_H_ALIGN_LEFT, "left", "At the left"},
      {GST_IMAGE_MIXER_PAD_H_ALIGN_CENTER, "center", "In the center"},
      {GST_IMAGE_MIXER_PAD_H_ALIGN_RIGHT, "right", "At the right"},
      {0, NULL, NULL},
    };

    h_align_type =
        g_enum_register_static ("GstImageMixerPadHAlign", h_align_types);
  }

  return h_align_type;
}

static void
gst_image_mixer_pad_set_property (GObject * obj,
    guint prop_id, const GValue * value, GParamSpec * pspec)
{
  GstImageMixerPad *ipad = GST_IMAGE_MIXER_PAD (obj);

  switch (prop_id) {
    case ARG_Z_ORDER:
      ipad->z_order = g_value_get_int (value);
      break;
    case ARG_ALPHA:
      ipad->alpha = g_value_get_float (value);
      break;
    case ARG_H_ALIGN:
      ipad->h_align = g_value_get_enum (value);
      break;
    case ARG_V_ALIGN:
      ipad->v_align = g_value_get_enum (value);
      break;
    case ARG_H_OFFSET:
      ipad->h_offset = g_value_get_int (value);
      break;
    case ARG_V_OFFSET:
      ipad->v_offset = g_value_get_int (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (obj, prop_id, pspec);
      break;
  }
}

static void
gst_image_mixer_pad_get_property (GObject * obj,
    guint prop_id, GValue * value, GParamSpec * pspec)
{
  GstImageMixerPad *ipad = GST_IMAGE_MIXER_PAD (obj);

  switch (prop_id) {
    case ARG_Z_ORDER:
      g_value_set_int (value, ipad->z_order);
      break;
    case ARG_ALPHA:
      g_value_set_float (value, ipad->alpha);
      break;
    case ARG_H_ALIGN:
      g_value_set_enum (value, ipad->h_align);
      break;
    case ARG_V_ALIGN:
      g_value_set_enum (value, ipad->v_align);
      break;
    case ARG_H_OFFSET:
      g_value_set_int (value, ipad->h_offset);
      break;
    case ARG_V_OFFSET:
      g_value_set_int (value, ipad->v_offset);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (obj, prop_id, pspec);
      break;
  }
}

static void
gst_image_mixer_pad_class_init (GstImageMixerPadClass * klass)
{
  GObjectClass *oklass = G_OBJECT_CLASS (klass);

  oklass->set_property = gst_image_mixer_pad_set_property;
  oklass->get_property = gst_image_mixer_pad_get_property;

  g_object_class_install_property (oklass, ARG_Z_ORDER,
      g_param_spec_int ("z-order", "Z-Order",
          "Z Order of the sub-picture (0 = top, 10000 = bottom)",
          -10000, 10000, DEFAULT_Z_ORDER, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_ALPHA,
      g_param_spec_float ("alpha", "Alpha",
          "Alpha of the sub-picture (0.0 = transparent, 1.0 = fully visible)",
          0.0, 1.0, DEFAULT_ALPHA, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_V_ALIGN,
      g_param_spec_enum ("v-align", "Vert. alignment",
          "Vertical alignment of the sub-picture",
          GST_IMAGE_MIXER_PAD_TYPE_V_ALIGN,
          DEFAULT_V_ALIGN, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_H_ALIGN,
      g_param_spec_enum ("h-align", "Hor. alignment",
          "Horizontal alignment of the sub-picture",
          GST_IMAGE_MIXER_PAD_TYPE_H_ALIGN,
          DEFAULT_H_ALIGN, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_V_OFFSET,
      g_param_spec_int ("v-offset", "Vert. offset",
          "Vertical offset of the sub-picture (respective to alignment)",
          G_MININT, G_MAXINT, DEFAULT_V_OFFSET, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_H_OFFSET,
      g_param_spec_int ("h-offset", "Hor. offset",
          "Horizontal offset of the sub-picture (respective to alignment)",
          G_MININT, G_MAXINT, DEFAULT_H_OFFSET, G_PARAM_READWRITE));
}

static void
gst_image_mixer_pad_init (GstImageMixerPad * ipad)
{
  ipad->z_order = DEFAULT_Z_ORDER;
  ipad->h_offset = DEFAULT_H_OFFSET;
  ipad->v_offset = DEFAULT_V_OFFSET;
  ipad->h_align = DEFAULT_H_ALIGN;
  ipad->v_align = DEFAULT_V_ALIGN;
  ipad->alpha = DEFAULT_ALPHA;

  ipad->fmt.width = 0;
  ipad->fmt.height = 0;
  ipad->fmt.fourcc = 0;

  ipad->current_data = NULL;
}
