/* GStreamer x264 plugin
 * Copyright (C) 2005 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/video/video.h>
#include "x264enc.h"

enum
{
  ARG_0,
  /* general */
  ARG_MAX_KEY_INTERVAL,
  ARG_MIN_KEY_INTERVAL,
  ARG_BFRAMES,
  ARG_ENCODING_TYPE,
  /* rate control */
  ARG_CBR,
  ARG_BITRATE,
  ARG_QMIN,
  ARG_QMAX,
  ARG_QSTEP,
  /* analysis */
  ARG_ME_METHOD
};

static GstStaticPadTemplate sink_templ = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK, GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("I420")));
static GstStaticPadTemplate src_templ = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC, GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-h264, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], " "framerate = (double) [ 1, max ]"));

static GstPadLinkReturn gst_x264enc_setcaps (GstPad * pad,
    const GstCaps * caps);
static GstCaps *gst_x264enc_getcaps (GstPad * pad);
static void gst_x264enc_chain (GstPad * pad, GstData * data);
static GstElementStateReturn gst_x264enc_change_state (GstElement * element);

static void gst_x264enc_get_property (GObject * obj, guint prop_id,
    GValue * val, GParamSpec * pspec);
static void gst_x264enc_set_property (GObject * obj, guint prop_id,
    const GValue * val, GParamSpec * pspec);

GST_BOILERPLATE (GstX264Enc, gst_x264enc, GstElement, GST_TYPE_ELEMENT);

#define GST_TYPE_X264ENC_ENCODING (gst_x264enc_encoding_get_type ())
static GType
gst_x264enc_encoding_get_type (void)
{
  static GType encoding_type = 0;

  if (encoding_type == 0) {
    static GEnumValue encoding_types[] = {
      {0, "cavlc", "Context-adaptive variable length coding"},
      {1, "cabac", "Context-adaptive binary arithmetic coding"},
      {0, NULL, NULL}
    };

    encoding_type =
        g_enum_register_static ("GstX264EncEncoding", encoding_types);
  }

  return encoding_type;
}

#define GST_TYPE_X264ENC_MEMETHOD (gst_x264enc_memethod_get_type ())
static GType
gst_x264enc_memethod_get_type (void)
{
  static GType memethod_type = 0;

  if (memethod_type == 0) {
    static GEnumValue memethod_types[] = {
      {X264_ME_DIA, "dia", "Diamond Search, Radius 1"},
      {X264_ME_HEX, "hex", "Hexagon Search, Radius 2"},
      {X264_ME_UMH, "umh", "Uneven Cross Multi-Hexagon Search"},
      {X264_ME_ESA, "esa", "ESA (?)"},
      {0, NULL, NULL}
    };

    memethod_type =
        g_enum_register_static ("GstX264EncMEMethod", memethod_types);
  }

  return memethod_type;
}

#ifndef GST_DISABLE_GST_DEBUG
static void
gst_x264enc_log (void *data, int level, const char *str, va_list args)
{
  gint g_level;

  switch (level) {
    case X264_LOG_ERROR:
      g_level = GST_LEVEL_ERROR;
      break;
    case X264_LOG_WARNING:
      g_level = GST_LEVEL_WARNING;
      break;
    case X264_LOG_INFO:
      g_level = GST_LEVEL_INFO;
      break;
    case X264_LOG_DEBUG:
      g_level = GST_LEVEL_DEBUG;
      break;
    default:
      return;
  }

  gst_debug_log_valist (x264_debug, g_level, "", "", 0, NULL, str, args);
}
#endif

static void
gst_x264enc_base_init (gpointer klass)
{
  GstElementClass *eklass = GST_ELEMENT_CLASS (klass);
  static GstElementDetails details = GST_ELEMENT_DETAILS ("x264 encoder",
      "Codec/Encoder/Video",
      "Encoder for h.264/AVC streams based on x264",
      "Ronald S. Bultje <rbultje@ronald.bitfreak.net>");

  gst_element_class_set_details (eklass, &details);
  gst_element_class_add_pad_template (eklass,
      gst_static_pad_template_get (&sink_templ));
  gst_element_class_add_pad_template (eklass,
      gst_static_pad_template_get (&src_templ));
}

static void
gst_x264enc_class_init (GstX264EncClass * klass)
{
  GObjectClass *oklass = G_OBJECT_CLASS (klass);

  GST_ELEMENT_CLASS (klass)->change_state = gst_x264enc_change_state;
  oklass->set_property = gst_x264enc_set_property;
  oklass->get_property = gst_x264enc_get_property;

  /* encoder properties */
  g_object_class_install_property (oklass, ARG_MAX_KEY_INTERVAL,
      g_param_spec_int ("max-key-interval", "Max. keyframe interval",
          "Maximum number of frames after which a keyframe (IDR) is forced",
          1, 1000, 250, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_MIN_KEY_INTERVAL,
      g_param_spec_int ("min-key-interval", "Min. keyframe interval",
          "Minimum number of frames after which a keyframe (IDR) is allowed",
          1, 1000, 25, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_BFRAMES,
      g_param_spec_int ("bframes", "No. B-frames",
          "Max. number of B-frames between two reference frames",
          0, 10, 0, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_ENCODING_TYPE,
      g_param_spec_enum ("encoding-type", "Encoding type",
          "General compression type for the output bitstream",
          GST_TYPE_X264ENC_ENCODING, 1, G_PARAM_READWRITE));

  g_object_class_install_property (oklass, ARG_CBR,
      g_param_spec_boolean ("cbr", "Constant bitrate",
          "Follow bitrate rather than quality for rate control",
          FALSE, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_BITRATE,
      g_param_spec_int ("bitrate", "Bitrate",
          "Bitrate in kbit/sec", 1, 16 * 1024, 1000, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_QMIN,
      g_param_spec_int ("qmin", "Min. QP", "Minimum QP value",
          0, 63, 10, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_QMAX,
      g_param_spec_int ("qmax", "Max. QP", "Maximum QP value",
          0, 63, 51, G_PARAM_READWRITE));
  g_object_class_install_property (oklass, ARG_QSTEP,
      g_param_spec_int ("qstep", "Max. QP step",
          "Maximum difference between QP values of two frames",
          0, 63, 4, G_PARAM_READWRITE));

  g_object_class_install_property (oklass, ARG_ME_METHOD,
      g_param_spec_enum ("me-method", "ME method",
          "Motion estimation method",
          GST_TYPE_X264ENC_MEMETHOD, 1, G_PARAM_READWRITE));
}

static void
gst_x264enc_init (GstX264Enc * x264)
{
  /* pads */
  x264->sinkpad =
      gst_pad_new_from_template (gst_static_pad_template_get (&sink_templ),
      "sink");
  gst_pad_set_chain_function (x264->sinkpad, gst_x264enc_chain);
  gst_pad_set_link_function (x264->sinkpad, gst_x264enc_setcaps);
  gst_pad_set_getcaps_function (x264->sinkpad, gst_x264enc_getcaps);
  gst_element_add_pad (GST_ELEMENT (x264), x264->sinkpad);

  x264->srcpad =
      gst_pad_new_from_template (gst_static_pad_template_get (&src_templ),
      "src");
  gst_pad_use_explicit_caps (x264->srcpad);
  gst_element_add_pad (GST_ELEMENT (x264), x264->srcpad);

  /* x264 stuff */
  x264->ctx = NULL;
  x264_param_default (&x264->params);
  x264->resync = TRUE;
#ifndef GST_DISABLE_GST_DEBUG
  x264->params.pf_log = gst_x264enc_log;
  x264->params.p_log_private = x264;
  x264->params.i_log_level = X264_LOG_DEBUG;
  x264->params.b_visualize = 0;
#else
  x264->params.i_log_level = X264_LOG_NONE;
#endif

  /* FIXME: handle events (EOS) for flush of cached B-frames */
}

static GstCaps *
gst_x264enc_getcaps (GstPad * pad)
{
  GstX264Enc *x264 = GST_X264ENC (GST_OBJECT_PARENT (pad));
  GstCaps *othercaps;
  GstStructure *s;

  /* build */
  othercaps = gst_pad_get_allowed_caps (x264->srcpad);
  s = gst_caps_get_structure (othercaps, 0);
  gst_caps_set_simple (othercaps,
      "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('I', '4', '2', '0'), NULL);
  gst_structure_set_name (s, "video/x-raw-yuv");

  return othercaps;
}

static GstPadLinkReturn
gst_x264enc_setcaps (GstPad * pad, const GstCaps * caps)
{
  GstX264Enc *x264 = GST_X264ENC (GST_OBJECT_PARENT (pad));
  gdouble fps;
  gint width, height, par_n = 1, par_d = 1;
  const GValue *par;
  GstStructure *s;

  /* extract properties */
  s = gst_caps_get_structure (caps, 0);
  gst_structure_get_int (s, "width", &width);
  gst_structure_get_int (s, "height", &height);
  gst_structure_get_double (s, "framerate", &fps);
  if ((par = gst_structure_get_value (s, "pixel-aspect-ratio"))) {
    par_n = gst_value_get_fraction_numerator (par);
    par_d = gst_value_get_fraction_denominator (par);
  }

  /* now setup encoder */
  x264->resync = TRUE;
  x264->params.i_width = width;
  x264->params.i_height = height;
  x264->params.i_csp = X264_CSP_I420;
  x264->params.i_fps_num = fps * 1001;
  x264->params.i_fps_den = 1001;
  if (par) {
    x264->params.vui.i_sar_width = par_n;
    x264->params.vui.i_sar_height = par_d;
  }
  x264->fps = fps;

  return GST_PAD_LINK_OK;
}

static GstBuffer *
gst_x264enc_nal (GstX264Enc * x264, x264_nal_t * nal, int num)
{
  GstBuffer *buf;
  guint8 *data;
  gint n, res, size = 0;

  /* predict size, roughly */
  for (n = 0; n < num; n++)
    size += nal[n].i_payload * 2 + 10;
  buf = gst_buffer_new_and_alloc (size);
  data = GST_BUFFER_DATA (buf);

  GST_DEBUG_OBJECT (x264, "Prepare output of %d nal units, total size=%d",
      num, size);

  /* output */
  for (n = 0; n < num; n++) {
    if ((res = x264_nal_encode (data, &size, 1, &nal[n])) < 0) {
      gst_buffer_unref (buf);
      return NULL;
    }

    GST_DEBUG_OBJECT (x264, "Output nal %d/%d, size=%d, "
        "data=0x%02x%02x%02x-0x%02x%02x%02x, priority=%d, type=%d",
        n, num, res, res > 0 ? data[0] : 0x00, res > 1 ? data[1] : 0x00,
        res > 2 ? data[2] : 0x00, res > 5 ? data[res - 3] : 0x00,
        res > 4 ? data[res - 2] : 0x00, res > 3 ? data[res - 1] : 0x00,
        nal[n].i_ref_idc, nal[n].i_type);

    size -= res;
    data += res;
  }
  GST_BUFFER_SIZE (buf) = (data - GST_BUFFER_DATA (buf));

  GST_DEBUG_OBJECT (x264, "Finished output of %d nal units, size=%d",
      num, GST_BUFFER_SIZE (buf));

  return buf;
}

static void
gst_x264enc_chain (GstPad * pad, GstData * data)
{
  GstX264Enc *x264 = GST_X264ENC (GST_OBJECT_PARENT (pad));
  GstBuffer *buf = GST_BUFFER (data), *outbuf;
  x264_picture_t in, out;
  x264_nal_t *nal;
  gint nnals, res;

  /* (re)init encoder */
  if (x264->resync) {
    if (!x264->ctx) {
      GstCaps *caps;
      GstBuffer *ext = NULL;

      GST_DEBUG_OBJECT (x264, "Initializing encoder");

      if (!(x264->ctx = x264_encoder_open (&x264->params)) ||
          (res = x264_encoder_headers (x264->ctx, &nal, &nnals)) < 0 ||
          (nnals > 0 && !(ext = gst_x264enc_nal (x264, nal, nnals)))) {
        gst_buffer_unref (buf);
        GST_ELEMENT_ERROR (x264, LIBRARY, INIT, (NULL), (NULL));
        return;
      }

      /* build */
      caps = gst_caps_new_simple ("video/x-h264",
          "width", G_TYPE_INT, x264->params.i_width,
          "height", G_TYPE_INT, x264->params.i_height,
          "framerate", G_TYPE_DOUBLE, x264->fps, NULL);
      if (ext != NULL)
        gst_caps_set_simple (caps, "codec_data", GST_TYPE_BUFFER, ext, NULL);
      res = gst_pad_set_explicit_caps (x264->srcpad, caps);
      gst_caps_free (caps);
      if (!res) {
        gst_buffer_unref (buf);
        GST_ELEMENT_ERROR (x264, CORE, NEGOTIATION, (NULL), (NULL));
        return;
      }
    } else {
      GST_DEBUG_OBJECT (x264, "reconfiguring encoder");

      if (x264_encoder_reconfig (x264->ctx, &x264->params)) {
        gst_buffer_unref (buf);
        GST_ELEMENT_ERROR (x264, LIBRARY, INIT, (NULL), (NULL));
        return;
      }
    }
    x264->resync = FALSE;
  }

  /* put picture in x264-form */
  in.i_type = X264_TYPE_AUTO;
  in.i_pts = GST_BUFFER_TIMESTAMP (buf);
  in.i_qpplus1 = 0;
  in.img.i_csp = X264_CSP_I420;
  in.img.i_plane = 3;
  in.img.plane[0] = GST_BUFFER_DATA (buf);
  in.img.plane[1] = GST_BUFFER_DATA (buf) +
      x264->params.i_width * x264->params.i_height;
  in.img.plane[2] = GST_BUFFER_DATA (buf) +
      x264->params.i_width * x264->params.i_height * 5 / 4;
  in.img.i_stride[0] = x264->params.i_width;
  in.img.i_stride[1] = in.img.i_stride[2] = x264->params.i_width / 2;

  GST_DEBUG_OBJECT (x264, "Encoding picture at %dx%d",
      x264->params.i_width, x264->params.i_height);

  /* actually encode */
  if (x264_encoder_encode (x264->ctx, &nal, &nnals, &in, &out) < 0 ||
      !(outbuf = gst_x264enc_nal (x264, nal, nnals))) {
    gst_buffer_unref (buf);
    GST_ERROR_OBJECT (x264, "Failed to encode picture");
    return;
  }
  gst_buffer_stamp (outbuf, buf);
  gst_buffer_unref (buf);
  GST_BUFFER_TIMESTAMP (outbuf) = out.i_pts;
  if (out.i_type == X264_TYPE_IDR)
    GST_BUFFER_FLAG_SET (outbuf, GST_BUFFER_KEY_UNIT);

  GST_DEBUG_OBJECT (x264, "Output buffer time=%" GST_TIME_FORMAT
      ", size=%d bytes, key=%s, nnals=%d",
      GST_TIME_ARGS (out.i_pts), GST_BUFFER_SIZE (outbuf),
      GST_BUFFER_FLAG_IS_SET (outbuf, GST_BUFFER_KEY_UNIT) ? "yes" : "no",
      nnals);

  /* output */
  gst_pad_push (x264->srcpad, GST_DATA (outbuf));
}

static GstElementStateReturn
gst_x264enc_change_state (GstElement * element)
{
  GstX264Enc *x264 = GST_X264ENC (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_PAUSED_TO_READY:
      if (x264->ctx) {
        x264_encoder_close (x264->ctx);
        x264->ctx = NULL;
      }
      x264->resync = TRUE;
      break;
    default:
      break;
  }

  return GST_CALL_PARENT_WITH_DEFAULT (GST_ELEMENT_CLASS, change_state,
      (element), GST_STATE_SUCCESS);
}

static void
gst_x264enc_get_property (GObject * obj, guint prop_id,
    GValue * val, GParamSpec * pspec)
{
  GstX264Enc *x264 = GST_X264ENC (obj);

  switch (prop_id) {
    case ARG_MAX_KEY_INTERVAL:
      g_value_set_int (val, x264->params.i_keyint_max);
      break;
    case ARG_MIN_KEY_INTERVAL:
      g_value_set_int (val, x264->params.i_keyint_min);
      break;
    case ARG_BFRAMES:
      g_value_set_int (val, x264->params.i_bframe);
      break;
    case ARG_ENCODING_TYPE:
      g_value_set_enum (val, x264->params.b_cabac);
      break;
    case ARG_CBR:
      g_value_set_boolean (val, x264->params.rc.b_cbr);
      break;
    case ARG_BITRATE:
      g_value_set_int (val, x264->params.rc.i_bitrate);
      break;
    case ARG_QMIN:
      g_value_set_int (val, x264->params.rc.i_qp_min);
      break;
    case ARG_QMAX:
      g_value_set_int (val, x264->params.rc.i_qp_max);
      break;
    case ARG_QSTEP:
      g_value_set_int (val, x264->params.rc.i_qp_step);
      break;
    case ARG_ME_METHOD:
      g_value_set_enum (val, x264->params.analyse.i_me_method);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (obj, prop_id, pspec);
      break;
  }
}

static void
gst_x264enc_set_property (GObject * obj, guint prop_id,
    const GValue * val, GParamSpec * pspec)
{
  GstX264Enc *x264 = GST_X264ENC (obj);

  switch (prop_id) {
    case ARG_MAX_KEY_INTERVAL:
      x264->params.i_keyint_max = g_value_get_int (val);
      break;
    case ARG_MIN_KEY_INTERVAL:
      x264->params.i_keyint_min = g_value_get_int (val);
      break;
    case ARG_BFRAMES:
      x264->params.i_bframe = g_value_get_int (val);
      break;
    case ARG_ENCODING_TYPE:
      x264->params.b_cabac = g_value_get_enum (val);
      break;
    case ARG_CBR:
      x264->params.rc.b_cbr = g_value_get_boolean (val);
      break;
    case ARG_BITRATE:
      x264->params.rc.i_bitrate = g_value_get_int (val);
      break;
    case ARG_QMIN:
      x264->params.rc.i_qp_min = g_value_get_int (val);
      break;
    case ARG_QMAX:
      x264->params.rc.i_qp_max = g_value_get_int (val);
      break;
    case ARG_QSTEP:
      x264->params.rc.i_qp_step = g_value_get_int (val);
      break;
    case ARG_ME_METHOD:
      x264->params.analyse.i_me_method = g_value_get_enum (val);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (obj, prop_id, pspec);
      return;                   /* no resync needed here */
  }

  x264->resync = TRUE;
}
